/*
 ** mio_sink_multi.c
 ** Multiple I/O compound sink, for output fanout case
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999 
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _AIRFRAME_SOURCE_
#include <airframe/mio_sink_multi.h>

static gboolean mio_sink_next_multi(
    MIOSource       *source,
    MIOSink         *sink,
    uint32_t        *flags,
    GError          **err)
{
    MIOSink         *ms = NULL, *cs = NULL;
    GError          *err2 = NULL;
    uint32_t        i, j;
    
    for (i = 0; i < mio_smc(sink); i++) {
        ms = &mio_smn(sink, i);
        if (!ms->next_sink(source, ms, flags, err)) {
            /* on error, close all sinks that already went next. */
            for (j = 0; j < i; j++) {
                cs = &mio_smn(sink, j);
                if (!cs->close_sink(source, cs, flags, &err2)) {
                    /* error closing an opened sink... bail for now */
                    g_error("panic on multiple sink next: "
                            "couldn't close sink %s: %s on error "
                            "while opening sink %s: %s", 
                            cs->spec, err2->message, 
                            ms->spec, (*err)->message);
                }
            }
            
            /* all sinks opened by this operation closed. */
            return FALSE;
        }
    }

    /* done. */
    return TRUE;
}

static gboolean mio_sink_close_multi(
    MIOSource       *source,
    MIOSink         *sink,
    uint32_t        *flags,
    GError          **err)
{
    GString         *errstr = NULL;
    GError          *err2 = NULL;
    uint32_t        errcount = 0;
    MIOSink         *ms = NULL;
    uint32_t        i;
    
    /* close subordinate sinks */
    for (i = 0; i < mio_smc(sink); i++) {
        ms = &mio_smn(sink, i);
        if (!ms->close_sink(source, ms, flags, &err2)) {
            if (!errstr) errstr = g_string_new("");
            g_string_append_printf(errstr, "%s\n", err2->message);
            errcount++;
            g_clear_error(&err2);
        }
    }
    
    /* report errors */
    if (errcount) {
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_MULTIPLE, "%u error(s) closing sink: %s", errcount, errstr->str);
        g_string_free(errstr, TRUE);
        return FALSE;
    }
    
    /* done */
    return TRUE;
}

static void mio_sink_free_multi(
    MIOSink                 *sink)
{
    MIOSink         *ms = NULL;
    uint32_t        i;

    for (i = 0; i < mio_smc(sink); i++) {
        ms = &mio_smn(sink, i);
        ms->free_sink(ms);
    }

    if (sink->spec) g_free(sink->spec);
    if (sink->vsp) g_free(sink->vsp);
}

gboolean mio_sink_init_multi(
    MIOSink         *sink,
    const char      *spec,
    MIOType         vsp_type,
    void            *cfg,
    GError          **err)
{
    uint32_t        vsp_count = GPOINTER_TO_UINT(cfg);

    /* choose default type */
    if (vsp_type == MIO_T_ANY) vsp_type = MIO_T_SINKARRAY;

    /* Ensure type is valid */
    if (vsp_type != MIO_T_SINKARRAY) {
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot create multiple sink: type mismatch");
        return FALSE;
    }
    
    /* Ensure array length is valid */
    if (!vsp_count) {
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot create multiple sink: zero sinks");
        return FALSE;
    }
    
    /* initialize multi sink */
    if (spec) {
        sink->spec = g_strdup(spec);
    } else {
        sink->spec = NULL;
    }
    sink->name = NULL;
    sink->vsp_type = vsp_type;
    sink->vsp = g_new0(MIOSink, vsp_count);
    sink->ctx = NULL;
    sink->cfg = cfg;
    sink->next_sink = mio_sink_next_multi;
    sink->close_sink = mio_sink_close_multi;
    sink->free_sink = mio_sink_free_multi;
    sink->opened = FALSE;
    sink->active = FALSE;
    sink->iterative = TRUE;
    
    return TRUE;
}
