/*
 ** mio_sink_tcp.c
 ** Multiple I/O active TCP stream sink
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999 
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _AIRFRAME_SOURCE_
#include <airframe/mio_sink_tcp.h>
#include "mio_common_net.h"

gboolean mio_sink_init_tcp(
    MIOSink         *sink,
    const char      *spec,
    MIOType         vsp_type,
    void            *cfg,
    GError          **err)
{
    char            *splitspec = NULL, *hostaddr = NULL, *svcaddr = NULL;
    gboolean        ok = TRUE;
    
    /* choose default type */
    if (vsp_type == MIO_T_ANY) vsp_type = MIO_T_SOCK_STREAM;
    
    /* initialize UDP sink */
    sink->spec = g_strdup(spec);
    sink->name = NULL;
    sink->vsp_type = vsp_type;
    sink->vsp = NULL;
    sink->ctx = NULL;
    sink->cfg = cfg;
    sink->next_sink = mio_sink_next_common_net;
    sink->close_sink = mio_sink_close_common_net;
    sink->free_sink = mio_sink_free_common_net;
    sink->opened = FALSE;
    sink->active = FALSE;
    sink->iterative = FALSE;
    
    /* Ensure type is valid */
    if (vsp_type != MIO_T_SOCK_STREAM) {
        ok = FALSE;
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot create TCP sink: type mismatch");
        goto end;
    }
    
    /* Parse specifier */
    splitspec = spec ? g_strdup(spec) : NULL;
    mio_init_ip_splitspec(splitspec, FALSE, (char *)cfg,
                          &hostaddr, &svcaddr, &(sink->name));

    /* Check for no host */
    if (!hostaddr) {
        ok = FALSE;
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot create TCP sink: no output specifier");
        goto end;
    }
    
    /* Do lookup and create context */
    if (!(sink->ctx = mio_init_ip_lookup(hostaddr, svcaddr, SOCK_STREAM,
                                         IPPROTO_TCP, FALSE, err))) {
        ok = FALSE;
        goto end;
    }

end:
    if (splitspec) g_free(splitspec);    
    return ok;
}
