/*
 ** mio_stdio.c
 ** Multiple I/O standard in source / standard out sink
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999 
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _AIRFRAME_SOURCE_
#include <airframe/mio_stdio.h>

gboolean mio_source_check_stdin(
    MIOSource       *source,
    uint32_t        *flags,
    GError          **err)
{
    /* terminate the application if standard input has been closed. */
    if (!source->name) {
        *flags |= MIO_F_CTL_TERMINATE;
        return FALSE;
    }
    
    return TRUE;
}

gboolean mio_source_close_stdin(
    MIOSource       *source,
    uint32_t        *flags,
    GError          **err)
{
    source->name = NULL;
    return TRUE;
}

gboolean mio_source_init_stdin(
    MIOSource       *source,
    const char      *spec,
    MIOType         vsp_type,
    void            *cfg,
    GError          **err)
{
    /* match spec */
    if (strcmp(spec, "-")) {
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot open stdin source: spec mismatch");
        return FALSE;
    }

    /* choose default type */
    if (vsp_type == MIO_T_ANY) vsp_type = MIO_T_FP;

    /* initialize source */
    source->spec = "-";
    source->name = "-";
    source->vsp_type = vsp_type;
    source->cfg = NULL;
    source->ctx = NULL;
    source->next_source = mio_source_check_stdin;
    source->close_source = mio_source_close_stdin;
    source->free_source = NULL;
    source->opened = FALSE;
    source->active = FALSE;

    /* set up source pointer as appropriate */
    switch (vsp_type) {
    case MIO_T_NULL:
        source->vsp = NULL;
        break;
    case MIO_T_FD:
        source->vsp = GINT_TO_POINTER(0);
        break;
    case MIO_T_FP:
        source->vsp = stdin;
        break;
    default:
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot open stdin source: type mismatch");
        return FALSE;
    }
    
    return TRUE;
}

gboolean mio_sink_init_stdout(
    MIOSink         *sink,
    const char      *spec,
    MIOType         vsp_type,
    void            *cfg,
    GError          **err)
{
    /* match spec */
    if (strcmp(spec, "-")) {
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot open stdout sink: spec mismatch");
        return FALSE;
    }
    
    /* choose default type */
    if (vsp_type == MIO_T_ANY) vsp_type = MIO_T_FP;

    /* initialize sink */
    sink->spec = "-";
    sink->name = "-";
    sink->vsp_type = vsp_type;
    sink->cfg = NULL;
    sink->ctx = NULL;
    sink->next_sink = NULL;
    sink->close_sink = NULL;
    sink->free_sink = NULL;
    sink->opened = FALSE;
    sink->active = FALSE;
    sink->iterative = FALSE;
    
    /* set up sink pointer as appropriate */
    switch (vsp_type) {
    case MIO_T_NULL:
        sink->vsp = NULL;
        break;
    case MIO_T_FD:
        sink->vsp = GINT_TO_POINTER(1);
        break;
    case MIO_T_FP:
        sink->vsp = stdout;
        break;
    default:
        g_set_error(err, MIO_ERROR_DOMAIN, MIO_ERROR_ARGUMENT,
                    "Cannot open stdout sink: type mismatch");
        return FALSE;
    }
    
    return TRUE;
}
