/*
 ** privconfig.c
 ** Generic privilege configuration support.
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999 
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _AIRFRAME_SOURCE_
#include <airframe/privconfig.h>

static char *RCSID __attribute__ ((unused)) = 
    "$Id: daeconfig.c 3509 2006-04-21 14:09:13Z bht $";

static char *opt_user = NULL;
static char *opt_group = NULL;

static uid_t new_user = 0;
static gid_t new_group = 0;

static gboolean did_become = FALSE;


AirOptionEntry privc_optentries[] = {
    AF_OPTION( "become-user", 'U', 0, AF_OPT_TYPE_STRING, &opt_user,
               "Become user after setup if started as root", NULL ),
    AF_OPTION( "become-group", (char)0, 0, AF_OPT_TYPE_STRING, &opt_group,
               "Become group after setup if started as root", NULL ),
    AF_OPTION_END
};

gboolean privc_add_option_group(
    AirOptionCtx *aoctx)
{
    g_assert(aoctx != NULL);

    air_option_context_add_group(aoctx, "privilege", "Privilege Options:",
                                 "Show help for privilege options",
                                 privc_optentries);

    return TRUE;
}

gboolean privc_setup(
    GError          **err) 
{
    struct passwd   *pwe = NULL;
    struct group    *gre = NULL;

    if (geteuid() == 0) {
        /* We're root. Parse user and group names. */
        if (opt_user) {
            if (!(pwe = getpwnam(opt_user))) {
                g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_SETUP,
                            "Cannot become user %s: %s.",
                            opt_user, strerror(errno));
                return FALSE;
            }
            
            /* By default, become new user's user and group. */
            new_user = pwe->pw_uid;
            new_group = pwe->pw_gid;
            if (opt_group) {
                if (!(gre = getgrnam(opt_group))) {
                    g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_SETUP,
                                "Cannot become group %s: %s.",
                                opt_group, strerror(errno));
                    return FALSE;
                }
            
                /* Override new group if set */
                new_group = gre->gr_gid;
            }
        }
    } else {
        /* We're not root. If we have options, the user is confused, and
           we should straighten him out by killing the process. */
        if (opt_user) {
            g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_SETUP,
                        "Cannot become user %s: not root.",
                        opt_user);
            return FALSE;
        }
        if (opt_group) {
            g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_SETUP,
                        "Cannot become group %s: not root.",
                        opt_user);
            return FALSE;
        }
    }
    
    /* All done. */
    return TRUE;
}

gboolean privc_configured()
{
    return (new_user) ? TRUE : FALSE;
}

gboolean privc_become(
    GError          **err)
{
    /* Die if we've already become */
    if (did_become) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_ALREADY,
                    "not dropping privileges, already did so");
        return FALSE;
    }

    /* Short circuit if we're not root */
    if (geteuid() != 0) return TRUE;

    /* Allow app to warn if not dropping */
    if (new_user == 0) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_NODROP,
                    "not dropping privileges (use --become-user to do so)");
        return FALSE;
    }
    
    /* Okay. Do the drop. */
    
    /* Drop ancillary group privileges while we're still root */
    if (setgroups(1, &new_group) < 0) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_FAILED,
                    "couldn't drop ancillary groups: %s", strerror(errno));
        return FALSE;
    }
    
#if LINUX_PRIVHACK
    /* Change to group */
    if (setregid(new_group, new_group) < 0) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_FAILED,
                    "couldn't become group %u: %s", new_group, strerror(errno));
        return FALSE;
    }
        
    /* Lose root privileges */
    if (setreuid(new_user, new_user) < 0) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_FAILED,
                    "couldn't become user %u: %s", new_user, strerror(errno));
        return FALSE;
    }
#else
    /* Change to group */
    if (setgid(new_group) < 0) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_FAILED,
                    "couldn't become group %u: %s", new_group, strerror(errno));
        return FALSE;
    }
        
    /* Lose root privileges */
    if (setuid(new_user) < 0) {
        g_set_error(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_FAILED,
                    "couldn't become user %u: %s", new_user, strerror(errno));
        return FALSE;
    }
#endif

    /* All done. */
    did_become = TRUE;
    return TRUE;
}
