/*@internal
 *
 ** @file decode.h
 ** YAF Layer 2 and Layer 3 decode routines
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 *
 * Packet decoding interface for YAF. This file's single function
 * decodes IPv4 and IPv6 packets within loopback, raw, Ethernet, Linux SLL
 * ("cooked"), and C-HDLC frames, encapsulated within MPLS, 802.1q VLAN,
 * and/or GRE. It provides high-performance partial reassembly of IPv4
 * and IPv6 fragments to properly generate flows from fragmented data, and to
 * support the export of the first N bytes of a given flow.
 *
 * 
 * The structures filled in by yfDecodePkt() are used within the flow generator,
 * and are suitable for other similar purposes.
 */
 
#ifndef _YAF_DECODE_H_
#define _YAF_DECODE_H_

#include <yaf/autoinc.h>
#include <yaf/yafcore.h>

/** Fragmentation information structure */
typedef struct yfIPFragInfo_st {
    /** Fragment ID. This is a 32-bit integer to support both IPv4 and IPv6. */
    uint32_t        ipid;
    /** Fragment offset within the reassembled datagram. */
    uint16_t        offset;
    /** IP header length. Used to calculate total fragment length. */
    uint16_t        iphlen;
    /** 
     * Decoded header length. Number of bytes at the start of the packet _not_
     * represented in the associated packet data.
     */
    uint16_t        l4hlen;
    /** 
     * Fragmented packet flag. Set if the packet is a fragment, 
     * clear if it is complete.
     */
    uint8_t         frag;
    /** 
     * More fragments flag. Set if this fragment is not the last in the packet.
     */
    uint8_t         more;
} yfIPFragInfo_t;

/** Maximum MPLS label count */
#define YF_MPLS_LABEL_COUNT_MAX     10

/** Datalink layer information structure */
typedef struct yfL2Info_st {
    /** Source MAC address */
    uint8_t         smac[6];
    /** Destination MAC address */
    uint8_t         dmac[6];
    /** VLAN tag */
    uint16_t        vlan_tag;
    /** MPLS label count */
    uint32_t        mpls_count;
    /** MPLS label stack */
    uint32_t        mpls_label[YF_MPLS_LABEL_COUNT_MAX];
} yfL2Info_t;

/** TCP information structure */
typedef struct yfTCPInfo_st {
    /** TCP sequence number */
    uint32_t        seq;
    /** TCP flags */
    uint8_t         flags;
} yfTCPInfo_t;

/** Full packet information structure. Used in the packet ring buffer. */
typedef struct yfPBuf_st {
    /** Packet timestamp in epoch milliseconds */
    uint64_t        ptime;
    /** Flow key containing decoded IP and transport headers. */
    yfFlowKey_t     key;
    /** Packet IP length. */
    uint16_t        iplen;
    /** Interface number packet was decoded from. Currently unused. */
    uint16_t        ifnum;
    /** TCP information structure. */
    yfTCPInfo_t     tcpinfo;
    /** Decoded layer 2 information. */
    yfL2Info_t      l2info;
    /** Length of payload available in captured payload buffer. */
    size_t          paylen;
    /** 
     * Captured payload buffer. Note that this in a convenience field;
     * the actual field is larger than one byte. */
    uint8_t         payload[1];
} yfPBuf_t;

/** Size of a packet buffer with payload capture disabled */
#define YF_PBUFLEN_NOL2INFO offsetof(yfPBuf_t, l2info)

/** Size of a packet buffer with payload capture disabled */
#define YF_PBUFLEN_NOPAYLOAD offsetof(yfPBuf_t, paylen)

/** Size of a packet buffer minus payload buffer */
#define YF_PBUFLEN_BASE offsetof(yfPBuf_t, payload)

struct yfDecodeCtx_st;
/** An opaque decode context */
typedef struct yfDecodeCtx_st yfDecodeCtx_t;

/** Ethertype for IP version 4 packets. */
#define YF_TYPE_IPv4    0x0800
/** Ethertype for IP version 6 packets. */
#define YF_TYPE_IPv6    0x86DD
/** 
 * Pseudo-ethertype for any IP version packets. 
 * Used as the reqtype argument to yfDecodeIP().
 */
#define YF_TYPE_IPANY   0x0000

/** IPv6 Next Header for Hop-by-Hop Options */
#define YF_PROTO_IP6_HOP    0
/** IPv4 Protocol Identifier and IPv6 Next Header for ICMP */
#define YF_PROTO_ICMP       1
/** IPv4 Protocol Identifier and IPv6 Next Header for TCP */
#define YF_PROTO_TCP        6
/** IPv4 Protocol Identifier and IPv6 Next Header for UDP */
#define YF_PROTO_UDP        17
/** IPv6 Next Header for Routing Options */
#define YF_PROTO_IP6_ROUTE  43
/** IPv6 Next Header for Fragment Options */
#define YF_PROTO_IP6_FRAG   44
/** IPv4 Protocol Identifier and IPv6 Next Header for GRE */
#define YF_PROTO_GRE        47
/** IPv4 Protocol Identifier and IPv6 Next Header for ICMP6 */
#define YF_PROTO_ICMP6      58
/** IPv6 Next Header for Destination Options */
#define YF_PROTO_IP6_DOPT   60

/** TCP FIN flag. End of connection. */
#define YF_TF_FIN	0x01
/** TCP SYN flag. Start of connection. */
#define YF_TF_SYN	0x02
/** TCP FIN flag. Abnormal end of connection. */
#define YF_TF_RST	0x04
/** TCP PSH flag. */
#define YF_TF_PSH	0x08
/** TCP ACK flag. Acknowledgment number is valid. */
#define YF_TF_ACK	0x10
/** TCP URG flag. Urgent pointer is valid. */
#define YF_TF_URG	0x20
/** TCP ECE flag. Used for explicit congestion notification. */
#define YF_TF_ECE	0x40
/** TCP CWR flag. Used for explicit congestion notification. */
#define YF_TF_CWR	0x80

/**
 * Allocate a decode context. Decode contexts are used to store decoder
 * internal state, configuration, and statistics.
 *
 * @param datalink libpcap DLT_ constant describing the layer 2 headers on the
 *                 packet in pkt. Supported datalink types are DLT_EN10MB
 *                 (Ethernet), DLT_CHDLC (Cisco HDLC), DLT_LINUX_SLL (Linux
 *                 "cooked" capture interface), DLT_RAW (raw IP packet, no
 *                 layer 2), DLT_NULL (loopback), and DLT_LOOP (OpenBSD
 *                 loopback).
 * @param reqtype  Required IP packet ethertype. Pass YF_TYPE_IPv4 to decode
 *                 only IPv4 packets, YF_TYPE_IPv6 to decode only IPv6 packets,
 *                 or YP_TYPE_IPANY to decode both IPv4 and IPv6 packets.
 * @param gremode  TRUE to enable GREv1 decoding; otherwise, GRE packets 
 *                 will be left encapsulated.
 * @return a new decode context
 */

yfDecodeCtx_t *yfDecodeCtxAlloc(
    int             datalink,
    uint16_t        reqtype,
    gboolean        gremode);

/** Free a decode context.
 *
 * @param ctx A decode context allocated with yfDecodeCtxAlloc()
 */

void yfDecodeCtxFree(
    yfDecodeCtx_t           *ctx);

/**
 * Decode a packet into a durable packet buffer. It is assumed the packet 
 * is encapsulated within a link layer frame described by the datalink 
 * parameter. It fills in the pbuf structure, copying payload if necessary. 
 *
 * @param ctx      Decode context obtained from yfDecodeCtxAlloc()
 *                 containing decoder configuration and internal state.
 * @param ptime    Packet observation time in epoch milliseconds. Use 
 *                 yfDecodeTimeval() or yfDecodeTimeNTP() to get epoch 
 *                 milliseconds from a struct timeval or a 64-bit NTP 
 *                 timestamp, respectively.
 * @param caplen   Length of the packet to decode pkt. 
 * @param pkt      Pointer to packet to decode. Is assumed to start with the 
 *                 layer 2 header described by the datalink parameter.
 * @param fraginfo Pointer to IP Fragment information structure which will be
 *                 filled in with fragment id and offset information from the
 *                 decoded IP headers. MAY be NULL if the caller does not
 *                 require fragment information; in this case, all fragmented
 *                 packets will be dropped.
 * @param pbuflen  Total length of the packet buffer pbuf. Use the YF_PBUFLEN_
 *                 macros to set this value. YF_PUBFLEN_NOFRAG disables 
 *                 fragment decode, layer 2 decode, and payload capture. 
 *                 YF_PBUFLEN_NOL2INFO disables layer 2 decode and payload 
 *                 capture. YF_PBUFLEN_NOPAYLOAD disables payload capture only. 
 *                 To enable full decode including payload, use YF_PBUFLEN_BASE 
 *                 plus the payload length. The buffer at pbuf MUST be able 
 *                 to contain pbuflen bytes.
 * @param pbuf     Packet buffer to decode packet into. Will contain copies of
 *                 all packet data and payload; this buffer is durable.
 * @return TRUE on success (a packet of the required type was decoded and
 *         all the decode structures are valid), FALSE otherwise. Failures
 *         are counted in the decode statistics which can be logged with the
 *         yfDecodeDumpStats() call;
 */

gboolean yfDecodeToPBuf(
    yfDecodeCtx_t           *ctx,
    uint64_t                ptime,
    size_t                  caplen,
    const uint8_t           *pkt,
    yfIPFragInfo_t          *fraginfo,
    size_t                  pbuflen,
    yfPBuf_t                *pbuf);

/**
 * Utility call to convert a struct timeval (as returned from pcap) into a 
 * 64-bit epoch millisecond timestamp suitable for use with yfDecodeToPBuf.
 *
 * @param tv        Pointer to struct timeval to convert
 * @return the corresponding timestamp in epoch milliseconds
 */

uint64_t yfDecodeTimeval(
    const struct timeval    *tv);

/**
 * Utility call to convert an NTP timestamp (as returned from DAG) into a 
 * 64-bit epoch millisecond timestamp suitable for use with
 *
 * @param tv        Pointer to struct timeval to convert
 * @return the corresponding timestamp in epoch milliseconds
 */

uint64_t yfDecodeTimeNTP(
    uint64_t                ntp);

/**
 * Print decoder statistics to the log.
 *
 * @param ctx decode context to print statistics from
 */
 
void yfDecodeDumpStats(
    yfDecodeCtx_t       *ctx); 

/* end idem */
#endif 
