/*
 ** picq.c
 ** General pickable queue implementation
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 * @file
 *
 * Generic Pickable Queue. A pickable queue's elements may be removed from any 
 * point in the queue, and added to the queue's head or tail. Together with a 
 * hash table to locate mid-queue elements, this can be used to implement idle 
 * timeout of its elements. 
 *
 * Where present, a <tt>vq</tt> argument is a void pointer to a queue. A 
 * queue's first two members must be a pointer to the tail (last node) of the 
 * queue and a pointer to the head (first node) of the queue.
 *
 * Where present, a <tt>vn</tt> argument is a void pointer to a queue node. 
 * A queue node's first two elements must be a pointer to the previous node
 * in the queue and a pointer to the next node in the queue.
 */

/* idem hack */
#ifndef _YAF_PICQ_H_
#define _YAF_PICQ_H_
#include <yaf/autoinc.h>


/**
 * Pick a node from a given pickable queue. The node is removed from the queue,
 * and its previous and next pointers are set to NULL. It is assumed that the 
 * node is actually an element of the given queue; undefined behavior may 
 * result if this is not the case.
 *
 * @param vq queue to remove from
 * @param vn node to remove 
 */
 
void piqPick (
    void        *vq,
    void        *vn);

/**
 * Enqueue a node at the head of a given pickable queue. The node must not be
 * an element in another queue; that is, its own previous and next pointers 
 * must be NULL. To move a node from one queue to another, use piqPick() 
 * first. 
 *
 * @param vq queue to enqueue to
 * @param vn node to enqueue
 */

void piqEnQ(
    void        *vq,
    void        *vn);

/**
 * Enqueue a node at the tail of a given pickable queue. The node must not be
 * an element in another queue; that is, its own previous and next pointers 
 * must be NULL. To move a node from one queue to another, use piqPick() 
 * first. 
 *
 * @param vq queue to enqueue to
 * @param vn node to enqueue
 */

void piqUnshift(
    void        *vq,
    void        *vn);

/**
 * Dequeue a node from the head of a given pickable queue. Analogous to finding
 * the head, picking it, then returning it. The node is removed from the queue,
 * and its previous and next pointers are set to NULL. Returns NULL if the
 * queue is empty.
 *
 * @param vq queue to remove from
 * @return the dequeued head of the queue, or NULL if empty.
 */

void *piqShift(
    void        *vq);
    
/**
 * Dequeue a node from the tail of a given pickable queue. Analogous to finding
 * the tail, picking it, then returning it. The node is removed from the queue,
 * and its previous and next pointers are set to NULL. Returns NULL if the
 * queue is empty.
 *
 * @param vq queue to remove from
 * @return the dequeued tail of the queue, or NULL if empty.
 */

void *piqDeQ(
    void        *vq);

/* end idem */
#endif
