/*
 ** yafrag.h
 ** YAF Active Fragment Table
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 * @file
 *
 * Fragment reassembly interface for YAF. [TODO - new frontmatter]
 * 
 * This facility is used by the YAF flow generator.
 */

#ifndef _YAF_FRAG_H_
#define _YAF_FRAG_H_

#include <yaf/autoinc.h>
#include <yaf/decode.h>
#include <yaf/yafcore.h>

struct yfFragTab_st;
/** 
 * A fragment table. Opaque. Create with yfFragTabAlloc() and free with
 * yfFragTabFree().
 */
typedef struct yfFragTab_st yfFragTab_t;

/**
 * Allocate a fragment table. 
 *
 * @param idle_ms   idle timeout in milliseconds. A fragmented packet for which
 *                  no fragments are received over an idle timeout is dropped.
 *                  Most host IPv4 implementations use 30 seconds (30000); it is
 *                  recommended to use the same here.
 * @param max_frags maximum number of unreassembled fragmented packets. 
 *                  Fragmented packets exceeding this limit will be dropped in 
 *                  least-recent order. Used to limit resource usage of a 
 *                  fragment table. A value of 0 disables fragment count limits.
 * @param max_payload   maximum octets of payload to capture per fragmented 
 *                      packet. A value of 0 disables payload reassembly.
 *
 * @return a new fragment table.
 */

yfFragTab_t *yfFragTabAlloc(
    uint32_t        idle_ms,
    uint32_t        max_frags,
    uint32_t        max_payload);

/**
 * Free a fragment table. Discards any outstanding fragmented packets within.
 *
 * @param fragtab a fragment table.
 */

void yfFragTabFree(
    yfFragTab_t         *fragtab);

/**
 * Defragment a fragment returned by yfDecodeToPBuf(). This adds the fragment to
 * the given fragment table. If the fragment completes a fragmented packet,
 * copies the assembled packet into the given pbuf, overwriting it, and
 * returns TRUE. If the packet is not fragmented (that is, if fraginfo->frag
 * is 0), has no effect and returns TRUE.
 *
 * @param fragtab   fragment table to add fragment to
 * @param fraginfo  fragment information structure filled in by yfDecodeToPBuf()
 * @param pbuflen   size of the packet buffer pbuf
 * @param pbuf      packet buffer. On call, contains decoded fragmented packet 
 *                  to add to the fragment table. If this call returns TRUE,
 *                  on return, contains assembled packet.
 * @return  TRUE if pbuf is valid and contains an assembled packet, 
 *          FALSE otherwise.
 */
 
gboolean yfDefragPBuf(
    yfFragTab_t         *fragtab,
    yfIPFragInfo_t      *fraginfo,
    size_t              pbuflen,
    yfPBuf_t            *pbuf);

/**
 * Print fragment reassembler statistics to the log.
 *
 * @param fragtab fragment table to dump stats for
 */

void yfFragDumpStats(
    yfFragTab_t         *fragtab);

#endif
