/*
 ** yaftab.h
 ** YAF Active Flow Table
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/*
 * This is the documentation for the _old_ yaftab.h; it is no longer current,
 * and should not be read by anyone.
 *
 * Flow generation interface for YAF. This facility works by maintaining a 
 * current flow table. Packets may be added to the active flows within this 
 * table using the yfFlowPkt() call. Completed flows may be written to an
 * IPFIX message buffer using yfFlowFlush(). 
 *
 * The flow table is configured by a number of global variables. 
 * 
 * <tt>yaf_idle</tt> sets
 * the idle timeout in seconds. A flow that receives no packets for the idle
 * timeout is assumed to be complete. The idle timeout is set to 300 seconds
 * (five minutes) by default. 
 *
 * <tt>yaf_active</tt> sets the active timeout in seconds. 
 * The maximum duration of a flow is the active timeout; additional packets 
 * for the same flow will be counted as part of a new flow. The active timeout
 * is set to 1800 seconds (half an hour) by default.
 *
 * <tt>yaf_flowlim</tt> sets the maximum size of the flow table; flows exceeding
 * this limit will be expired in least-recent order, as if they were idle. The
 * flow limit defaults to zero, for no limit. Use this global to limit resource
 * usage by the flow table.
 *
 * <tt>yaf_paylen</tt> sets the number of bytes of payload to capture from the
 * start of each flow. The payload length defaults to zero, which disables 
 * payload capture.
 *
 * <tt>yaf_uniflow</tt>, if TRUE, exports flows in uniflow mode, using the
 * record adjacency export method described in section 3 of 
 * draft-ietf-ipfix-biflow. Defaults to FALSE.
 *
 * <tt>yaf_macmode</tt>, if TRUE, exports layer 2 information with each flow;
 * presently this is limited to VLAN tags but may be expanded to include the
 * MPLS stack and MAC addresses in the future. Defaults to FALSE.
 *
 * <tt>yaf_silkmode</tt>, if TRUE, enables SiLK compatibility mode. In this 
 * mode, totalOctetCount and reverseTotalOctetCount are clamped to 32 bits. 
 * Any packet that would cause either of these counters to overflow 32 bits 
 * will force an active timeout. The high-order bit of the flowEndReason IE 
 * is set on any flow created on a counter overflow, as above, or on an active
 * timeout. Defaults to FALSE.
 *
 * <tt>yaf_reqtype</tt> limits the flow table to collecting IPv4 or IPv6 flows
 * only. Set to YF_TYPE_IPv4 for IPv4 flows only, YF_TYPE_IPv6 for IPv6 flows
 * only, or YF_TYPE_IPANY (the default) to collect both IPv4 and IPv6 flows.
 *
 * This facility is used by YAF to assemble packets into flows.
 */

/**
 * @file
 *
 * Flow generation interface for YAF. [TODO - frontmatter]
 *
 * This facility is used by YAF to assemble packets into flows.
 */

#ifndef _YAF_TAB_H_
#define _YAF_TAB_H_

#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>


struct yfFlowTab_st;
/** 
 * A flow table. Opaque. Create with yfFlowTabAlloc() and free with
 * yfFlowTabFree().
 */
typedef struct yfFlowTab_st yfFlowTab_t;

/**
 *yfFlowTabAlloc
 *
 * Allocate a flow table. 
 *
 * @param idle_ms   idle timeout in milliseconds. A flow that receives no 
 *                  packets for the idle timeout is assumed to be complete.
 * @param active_ms active timeout in milliseconds. The maximum duration of a
 *                  flow is the active timeout; additional packets 
 *                  for the same flow will be counted as part of a new flow.
 * @param max_flows maximum number of active flows. Flows exceeding this limit 
 *                  will be expired in least-recent order, as if they were idle. 
 *                  Used to limit resource usage of a flow table. A value of 0
 *                  disables flow count limits.
 * @param max_payload   maximum octets of payload to capture per flow direction.
 *                      Requires at least max_payload octets of payload to be
 *                      available in each packet buffer passed to yfFlowPBuf().
 *                      A value of 0 disables payload capture and export.
 * @param uniflow   If TRUE, export biflows using record adjacency (two uniflows
 *                  exported back-to-back. Use this for interoperability with
 *                  IPFIX collectors that do not implement RFC 5103.
 * @param silkmode  If TRUE, clamp totalOctetCount and maxTotalOctetCount to 32
 *                  bits and force active timeout on overflow. Set high order
 *                  bit in flowEndReason for each flow created on an overflow
 *                  or active timeout. Breaks IPFIX interoperability; use for
 *                  direct export to SiLK rwflowpack or flowcap.
 *
 * @param applabelmode If TRUE, then the payload, (as limited by max_payload,)
 *                     is sent through various plugins and code in order to
 *                     determine which protocol is running on the flow by doing
 *                     only payload inspection and exporting payload relevent 
 *                     information.
 *
 * @param entropymode  If TRUE, then a Shannon Entropy measurement is made over
 *                     the captured payload (as limited by max_payload).  The
 *                     entropy value is exported as two values one for forward
 *                     payload and one for reverse payload.
 *
 * @return a new flow table.
 */
yfFlowTab_t *yfFlowTabAlloc(
    uint64_t        idle_ms,
    uint64_t        active_ms,
    uint32_t        max_flows,
    uint32_t        max_payload,
    gboolean        uniflow,
    gboolean        silkmode,
	gboolean        applabelmode,
    gboolean        entropymode);

/**
 * Free a previously allocated flow table. Discards any outstanding active flows
 * without closing or flushing them; use yfFlowTabFlushAll() before yfFlowFree()
 * to do this. 
 *
 * @param flowtab a flow table allocated by yfFlowTabAlloc()
 */

void yfFlowTabFree(
    yfFlowTab_t     *flowtab);

/**
 * Add a decoded packet buffer to a given flow table. Adds the packet to 
 * the flow to which it belongs, creating a new flow if necessary. Causes
 * the flow to which it belongs to time out if it is longer than the active
 * timeout.  Closes the flow if the flow closure conditions (TCP RST, TCP FIN
 * four-way teardown) are met. 
 *
 * @param flowtab   flow table to add the packet to
 * @param pbuflen   size of the packet buffer pbuf
 * @param pbuf      packet buffer containing decoded packet to add.
 */
 
void yfFlowPBuf(
    yfFlowTab_t                 *flowtab,
    size_t                      pbuflen,
    yfPBuf_t                    *pbuf);

/**
 * Flush closed flows in the given flow table to the given IPFIX Message 
 * Buffer. Causes any idle flows to time out, removing them from the active 
 * flow table; also enforces the flow table's resource limit. If close is TRUE, 
 * additionally closes all active flows and flushes as well.
 *
 * @param flowtab   flow table to flush closed flows from
 * @param fbuf      message buffer to write flushed flows to.
 * @param close     close all active flows before flushing
 * @param err       An error description pointer; must not be NULL.
 * @return TRUE on success, FALSE otherwise.
 */

gboolean yfFlowTabFlush(
    yfFlowTab_t     *flowtab,
    fBuf_t          *fbuf,
    gboolean        close,
    GError          **err);

/**
 * Get the current packet clock from a flow table.
 * 
 * @param flowtab a flow table
 * @return current packet clock
 */

uint64_t yfFlowTabCurrentTime(
    yfFlowTab_t     *flowtab);

/**
 * Print flow table statistics to the log.
 *
 * @param flowtab flow table to dump stats for
 * @param timer a GTimer containing the runtime 
 *              (for packet and flow rate logging). May be NULL to suppress
 *              rate logging.
 */

void yfFlowDumpStats(
    yfFlowTab_t     *flowtab,
    GTimer          *timer);

#endif
