/**
 * @internal
 *
 * @file dnsplugin.c
 *
 * provides a plugin to the ipfix payload classifier to attempt to determine if a packet payload
 * is a DNS packet (see RFC 1035)
 *
 * @note defining PAYLOAD_INSPECTION at compile time will attempt to better inspection of the packet
 *       payload at a cost of deeper inspection;  even with PAYLOAD_INSPECTION enabled, it is possible
 *       that this may not be 100% correct in ID'ing the packets
 *
 *
 * @author $Author: inacio_svn $
 * @date $Date: 2008-01-08 15:31:44 -0500 (Tue, 08 Jan 2008) $
 * @Version $Revision: 10048 $
 *
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2007-2008 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Chris Inacio <inacio@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 *
 */
#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>



typedef struct ycDnsScanMessageHeader_st {
    uint16_t            id;

    uint16_t            qr:1;
    uint16_t            opcode:4;
    uint16_t            aa:1;
    uint16_t            tc:1;
    uint16_t            rd:1;
    uint16_t            ra:1;
    uint16_t            z:3;
    uint16_t            rcode:4;

    uint16_t            qdcount;
    uint16_t            ancount;
    uint16_t            nscount;
    uint16_t            arcount;
} ycDnsScanMessageHeader_t;


#define DNS_PORT_NUMBER 53
#define DNS_NAME_COMPRESSION 0xc0

/** this field defines the number of octects we fuzz the size of the 
    DNS to the IP+TCP+payload size with; we don't record any TCP
    options, so it is possible to have a few extra bytes in the
    calculation, and we won't say that's bad until that is larger
    than the following constant */
#define DNS_TCP_FLAG_SLACK 8

#define PAYLOAD_INSPECTION 1



/**
 * local prototypes
 *
 */
static void         ycDnsScanRebuildHeader (
    uint8_t * payload,
    ycDnsScanMessageHeader_t * header);

#ifdef PAYLOAD_INSPECTION
static uint16_t     ycDnsScanCheckResourceRecord (
    uint8_t * payload,
    uint16_t * offset,
    unsigned int payloadSize);
#endif

/**
 * dnsScanner_LTX_ycDnsScanScan
 *
 * scans a payload to determine if the payload is a dns request/reply.  It checks the structure for
 * self referential integrity, but it can't guarantee that the payload is actually DNS, it could be
 * some degenerate random data
 *
 * name abomination has been achieved by combining multiple naming standards until the prefix to
 * the function name is dnsplugin_LTX_ycDnsScan --- it's a feature
 *
 * @param argc NOT USED
 * @param argv NOT USED
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match DNS_PORT_NUMBER (53) for a match 
 *
 */
uint16_t
dnsplugin_LTX_ycDnsScanScan (
    int argc,
    char *argv[],
    uint8_t * payload,
    unsigned int payloadSize,
    yfFlow_t * flow,
    yfFlowVal_t * val)
{
    unsigned int        loop;
    ycDnsScanMessageHeader_t header;

    /* what is the minimum DNS message size? */
    if (payloadSize < sizeof (ycDnsScanMessageHeader_t)) {
        /* fprintf(stderr, " <dns exit 1> "); */
        return 0;
    }

    ycDnsScanRebuildHeader (payload, &header);

    /* all opcodes above 2 ar reserved for future use (rfc1035) */
    if (header.opcode > 2) {
        /* fprintf(stderr, " <dns exit 2> "); */
        return 0;
    }
    /* all return codes above 5 are reserved for future use (rfc1035) (we're
     * only sanity checking it if it is a response) */
    if ((header.rcode > 5) && (1 == header.qr)) {
        /* fprintf(stderr, " <dns exit 3> "); */
        return 0;
    }
    /* rfc states that Z is reserved for future use and must be zero */
    if (0 != header.z) {
        /* fprintf(stderr, " <dns exit 4> "); */
        return 0;
    }
    /* should we check the body reponse here?? If we have limited snarf
     * lengths, we could reject packets as not being "real" DNS packets, when
     * in fact they are */
#   ifdef PAYLOAD_INSPECTION
    {
        /* parse through the rest of the DNS message, only the header is fixed
         * in size */
        uint16_t            payloadOffset = sizeof (ycDnsScanMessageHeader_t);

        /* the the query enteries */
        for (loop = 0; loop < header.qdcount; loop++) {
            uint8_t             sizeOct = *(payload + payloadOffset);
            uint16_t            qtype;
            uint16_t            qclass;

            while (0 != sizeOct && payloadOffset < payloadSize) {
                if (DNS_NAME_COMPRESSION == (sizeOct & DNS_NAME_COMPRESSION)) {
                    payloadOffset += sizeof (uint16_t);
                } else {
                    payloadOffset += sizeOct + 1;
                }
                if (payloadOffset > payloadSize) {
                    return 0;
                }
                sizeOct = *(payload + payloadOffset);
            }
            if (payloadOffset > payloadSize) {
                /* this is either a DNS fragment, or a malformed DNS */
                /* fprintf(stderr, " <dns exit 5> "); */
                return 0;
            }
            /* get past the terminating 0 length in the name */
            payloadOffset++;

            /* check the query type */
#           if HAVE_ALIGNED_ACCESS_REQUIRED
            qtype = ((*(payload + payloadOffset)) << 8) |
                    ((*(payload + payloadOffset + 1)) );
            qtype = ntohs(qtype);
#           else
            qtype = ntohs(*((uint16_t *)(payload + payloadOffset)));
#           endif

            if (qtype > 16 && ((qtype < 252) && (qtype > 255))) {
                /* fprintf(stderr, " <dns exit 6> "); */
                return 0;
            }
            payloadOffset += sizeof (uint16_t);

            if (payloadOffset > payloadSize) {
                return 0;
            }

            /* check the class code */
#           if HAVE_ALIGNED_ACCESS_REQUIRED
            qclass = ((*(payload + payloadOffset)) << 8) | 
                     ((*(payload + payloadOffset + 1)) );
            qclass = ntohs (qclass);
#           else
            qclass = ntohs(*((uint16_t *)(payload+payloadOffset)));
#           endif
            if (qclass > 4 && qclass != 255) {
                /* fprintf(stderr, " <dns exit 7, qclass = %d> ", qclass); */
                return 0;
            }
            payloadOffset += sizeof (uint16_t);
        }

        /* check each record for the answer record count */
        for (loop = 0; loop < header.ancount; loop++) {
            uint16_t            rc;

            rc =
              ycDnsScanCheckResourceRecord (payload, &payloadOffset,
                                            payloadSize);
            if (0 == rc) {
                /* fprintf(stderr, " <dns exit 8> "); */
                return rc;
            }
        }


        /* check each record for the name server resource record count */
        for (loop = 0; loop < header.nscount; loop++) {
            uint16_t            rc;

            rc =
              ycDnsScanCheckResourceRecord (payload, &payloadOffset,
                                            payloadSize);
            if (0 == rc) {
                /* fprintf(stderr, " <dns exit 9> "); */
                return rc;
            }
        }

        /* check each record for the additional record count */
        for (loop = 0; loop < header.arcount; loop++) {
            uint16_t            rc;

            rc =
              ycDnsScanCheckResourceRecord (payload, &payloadOffset,
                                            payloadSize);
            if (0 == rc) {
                /* fprintf(stderr, " <dns exit 10> "); */
                return rc;
            }
        }

    }
#   endif


    /* this is the DNS port code */
    /* fprintf(stderr, " <dns exit 11 match> "); */
    return DNS_PORT_NUMBER;
}


/**
 * ycDnsScanRebuildHeader
 *
 * This function handles the endianess of the received message and
 * deals with machine alignment issues by not mapping a network
 * octect stream directly into the DNS structure
 *
 * @param payload a network stream capture
 * @param header a pointer to a client allocated dns message
 *        header structure
 *
 *
 */
static
  void
ycDnsScanRebuildHeader (
    uint8_t * payload,
    ycDnsScanMessageHeader_t * header)
{
    uint16_t           *tempArray = (uint16_t *) header;
    uint16_t            bitmasks = ntohs (*((uint16_t *) (payload + 2)));
    unsigned int        loop;

    memcpy (tempArray, payload, sizeof (ycDnsScanMessageHeader_t));
    for (loop = 0; loop < sizeof (ycDnsScanMessageHeader_t) / sizeof (uint16_t);
         loop++) {
        *(tempArray + loop) = ntohs (*(tempArray + loop));
    }

    header->qr = bitmasks & 0x8000 ? 1 : 0;
    header->opcode = (bitmasks & 0x7800) >> 11;
    header->aa = bitmasks & 0x0400 ? 1 : 0;
    header->tc = bitmasks & 0x0200 ? 1 : 0;
    header->rd = bitmasks & 0x0100 ? 1 : 0;
    header->ra = bitmasks & 0x0080 ? 1 : 0;
    header->z = (bitmasks & 0x0070) >> 4;
    header->rcode = bitmasks & 0x000f;

}




#ifdef PAYLOAD_INSPECTION
static
                    uint16_t
ycDnsScanCheckResourceRecord (
    uint8_t * payload,
    uint16_t * offset,
    unsigned int payloadSize)
{
    uint8_t             nameSize = *(payload + (*offset));
    uint16_t            rrType;
    uint16_t            rrClass;
    uint16_t            rdLength;

    while ((0 != nameSize) && (*offset < payloadSize)) {
        if (DNS_NAME_COMPRESSION == (nameSize & DNS_NAME_COMPRESSION)) {
            *offset += sizeof (uint16_t);
        } else {
            *offset += nameSize + 1;
        }
        if (*offset > payloadSize) {
            return 0;
        }
        nameSize = *(payload + (*offset));
    }
    if (*offset > payloadSize) {
        return 0;
    }
    /* check the type */
#   if HAVE_ALIGNED_ACCESS_REQUIRED
    rrType = ((*(payload + (*offset))) << 8) |
             ((*(payload + (*offset) + 1)) );
    rrType = ntohs (rrType);
#   else
    rrType = ntohs(*(uint16_t*)(payload + (*offset)));
#   endif
    *offset += sizeof (uint16_t);
    if (rrType > 16 || *offset > payloadSize) {
        return 0;
    }
    /* check the class */
#   if HAVE_ALIGNED_ACCESS_REQUIRED
    rrClass = ((*(payload + (*offset))) << 8) |
              ((*(payload + (*offset) + 1)) );
    rrClass = ntohs (rrClass);
#   else
    rrClass = ntohs(*(uint16_t*)(payload + (*offset)));
#   endif
    *offset += sizeof (uint16_t);
    if (rrClass > 4) {
        return 0;
    }
    /* skip past the time to live */
    *offset += sizeof (uint32_t);

    if (*offset > payloadSize) {
        return 0;
    }

    /* get the record data length, (so we can skip ahead the right amount) */
#   if HAVE_ALIGNED_ACCESS_REQUIRED
    rdLength = ((*(payload + (*offset))) << 8) |
               ((*(payload + (*offset) + 1)) );
    rdLength = ntohs (rdLength);
#   else
    rdLength = ntohs(*(uint16_t*)(payload + (*offset)));
#   endif
    *offset += sizeof (uint16_t);

    /* not going to try to parse the data record, what's in there depends on
     * the class and type fields, but the rdlength field always tells us how
     * many bytes are in it */
    *offset += rdLength;

    if (*offset > payloadSize) {
        return 0;
    }
    /* the record seems intact enough */
    return DNS_PORT_NUMBER;
}
#endif
