/**
 * @internal
 *
 * @file ircplugin.c
 *
 * this provides IRC payload packet recognition for use within YAF
 * It is based on RFC 2812 and some random limited packet capture.
 *
 *
 * @author $Author$
 * @date $Date$
 * @Version $Revision$
 *
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2007-2008 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Chris Inacio <inacio@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 *
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>


#include <pcre.h>


#define IRC_PORT 194

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre *ircMsgRegex = NULL;
static pcre *ircJoinRegex = NULL;
static pcre *ircRegex = NULL;
static unsigned int pcreInitialized = 0;


/**
 * static local functions
 *
 */
static uint16_t ycIrcScanInit (void);
static int ycDebugBinPrintf(uint8_t *data, uint16_t size);


/**
 * ircplugin_LTX_ycIrcScanScan
 *
 * scans a given payload to see if it conforms to our idea of what IRC traffic
 * looks like.
 *
 *
 * name abomination has been achieved by combining multiple naming standards until the prefix to
 * the function name is dnsplugin_LTX_ycDnsScan --- it's a feature
 *
 * @param argc NOT USED
 * @param argv NOT USED
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match DNS_PORT_NUMBER (53) for a match 
 *
 */
uint16_t
ircplugin_LTX_ycIrcScanScan (
    int argc,
    char *argv[],
    uint8_t * payload,
    unsigned int payloadSize,
    yfFlow_t * flow,
    yfFlowVal_t * val)
{



    int rc;
#   define NUM_CAPT_VECTS 60
    int vects[NUM_CAPT_VECTS];

 
    if (0 == pcreInitialized) {
        if (0 == ycIrcScanInit()) {
            return 0;
        }
    }

    rc = pcre_exec(ircMsgRegex, NULL, (char *)payload, payloadSize, 0, 0, vects, 
                   NUM_CAPT_VECTS);
    if (rc <= 0) {
        rc = pcre_exec(ircJoinRegex, NULL, (char *)payload, payloadSize, 0, 0, vects, 
                       NUM_CAPT_VECTS);
    }
    if (rc <= 0) {
        rc = pcre_exec(ircRegex, NULL, (char *)payload, payloadSize, 0, 0, vects, 
                       NUM_CAPT_VECTS);
    }


    /** at some point in the future, this is the place to extract protocol 
        information like message targets and join targets, etc.*/

    if (flow->key.addr.v4.sip == 0xd1b19222 || flow->key.addr.v4.sip == 0xd1b19222) {
        printf("payloadSize %d", payloadSize);
        if (rc > 0)
        {
            unsigned int loop;
            ycDebugBinPrintf(payload, payloadSize);
            printf("=> ");
            for (loop=1; loop < rc; loop++)
            {
                const char *targetString;
                pcre_get_substring((char *)payload, vects, rc, loop, &targetString);
                printf ("[%d](",loop);
                ycDebugBinPrintf((uint8_t*)targetString, payloadSize);
                printf(") ");
                pcre_free_substring(targetString);
            }
            /* printf(" matched a pattern\n\n\n"); /* */
            printf("\n");
        }
        else
        {
            printf("<=(%d)", rc);
            ycDebugBinPrintf(payload, payloadSize);
            printf("\n");
        }
    }

        

    if (rc >0) {
        return IRC_PORT;
    }
    
    
    return 0;
}



/**
 * ycIrcScanInit
 *
 * this initializes the PCRE expressions needed to search the payload for
 * IRC
 *
 *
 * @sideeffect sets the initialized flag on success
 *
 * @return 1 if initialization is complete correctly, 0 otherwise
 */
static
uint16_t
ycIrcScanInit ()
{
    const char *errorString;
    int errorPos;

    const char ircMsgRegexString[] = "^(?:(:[^: \\n\\r]+)(?:\\ ))?"
                                     "(PRIVMSG|NOTICE) \\ "
                                     "([^: \\n\\r]+|:.*) (?:\\ )"
                                     "([^: \\n\\r]+\\ |:.*)";
    const char ircJoinRegexString[] = "^(?:(:[^\\: \\n\\r]+)(?:\\ ))?"
                                      "(JOIN) \\ "
                                      "([^: \\n\\r]+\\ |:.*)\\s";
    const char ircRegexString[] = "^(?:(:[^: \\n\\r]+)(?:\\ ))?"
                                  "(\\d{3}|PASS|OPER|QUIT|SQUIT|NICK"
                                  "|MODE|USER|SERVICE|JOIN|NAMES|INVITE"
                                  "|PART|TOPIC|LIST|KICK|PRIVMSG|NOTICE"
                                  "|MOTD|STATS|CONNECT|INFO|LUSERS|LINKS"
                                  "|TRACE|VERSION|TIME|ADMIN|SERVLIST"
                                  "|SQUERY|WHO|WHOWAS|WHOIS|KILL|PING"
                                  "|PONG|ERROR|AWAY|DIE|SUMMON|REHASH"
                                  "|RESTART|USERS)(?:[ \r\n])";



    ircRegex = pcre_compile(ircRegexString, PCRE_EXTENDED|PCRE_ANCHORED, &errorString, &errorPos, NULL);
    ircMsgRegex = pcre_compile(ircMsgRegexString, PCRE_EXTENDED|PCRE_ANCHORED, &errorString, &errorPos, NULL);
    ircJoinRegex = pcre_compile(ircJoinRegexString, PCRE_EXTENDED|PCRE_ANCHORED, &errorString, &errorPos, NULL);

    if (NULL != ircRegex && NULL != ircMsgRegex && NULL != ircJoinRegex) {
        pcreInitialized = 1;
    }

    
    return pcreInitialized;
}

static
int
ycDebugBinPrintf(uint8_t *data, uint16_t size)
{
    uint16_t loop;
    int numPrinted = 0;
    
    if (0 == size) {
        return 0;
    }

    for (loop=0; loop < size; loop++) {
        if (isprint(*(data+loop)) && !iscntrl(*(data+loop))){
            printf("%c", *(data+loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data+loop) || '\r' == *(data+loop) || '\0' == *(data+loop)) {
            break;
        }
        numPrinted++;
    }
    
    return numPrinted;
}

