/**
 * @file tlsplugin.c
 *
 *
 * This recognizes SSL & TLS packets
 *
 *
 * @author $Author: inacio_svn $
 * @date $Date: 2008-01-08 15:31:44 -0500 (Tue, 08 Jan 2008) $
 * @Version $Revision: 10048 $
 *
 *
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2007-2008 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Chris Inacio <inacio@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 *
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>


/** defining the header structure for SSLv2 is pointless, because the 
    first field of the record is variable length, either 2 or 3 bytes
    meaning that the first step has to be to figure out how far offset
    all of the other fields are.  Further, the client can send a v2
    client_hello stating that it is v3/TLS 1.0 capable, and the server
    can respond with v3/TLS 1.0 record formats
    */


/** this defines the record header for SSL V3 negotiations, 
    it also works for TLS 1.0 */
typedef struct sslv3RecordHeader_st {
    uint8_t             contentType;
    uint8_t             protocolMajor;
    uint8_t             protocolMinor;
    uint16_t            length;
} sslv3RecordHeader_t;


#define TLS_PORT_NUMBER  0xcdcd
#define SSL2_PORT_NUMBER 0xcdce
#define SSL3_PORT_NUMBER 0xcdcf





/**
 * tlsplugin_LTX_ycTlsScanScan
 *
 * the scanner for recognizing SSL/TLS packets
 *
 * @param argc number of string arguments in argv
 * @param argv string arguments for this plugin (first two are library 
 *             name and function name)
 * @param payload the packet payload
 * @param payloadSize size of the packet payload
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 *
 * @return TLS_PORT_NUMBER for TLS packets, 
 *         SSL2_PORT_NUMBER for SSLv2 packets,
 *         SSL3_PORT_NUMBER for SSLv3 packets,
 *         otherwise 0
 */
uint16_t
tlsplugin_LTX_ycTlsScanScan (
    int argc,
    char *argv[],
    uint8_t * payload,
    unsigned int payloadSize,
    yfFlow_t * flow,
    yfFlowVal_t * val)
{

    /* every SSL/TLS header has to be at least 2 bytes long... */
    if (payloadSize < 2) {
        return 0;
    }

    /* understanding how to determine between SSLv2 and SSLv3/TLS is "borrowed" 
     * from OpenSSL payload byte 0 for v2 is the start of the length field, but 
     * its MSb is always reserved to tell us how long the length field is, and
     * in some cases, the second MSb is reserved as well */

    /* when length is 2 bytes in size (MSb == 1), and the message type code is
     * 0x01 (client_hello) we know we're doing SSL v2 */
    if ((payload[0] & 0x80) && (0x01 == payload[2])) {
        /** @bug need to do more validity checks */

        /* SSLv2 (client_hello) */
        return SSL2_PORT_NUMBER;
    } else if (payloadSize >= 3) {
        if ((0x00 == (payload[0] & 0x80)) && (0x00 == (payload[0] & 0x40))
            && (0x01 == payload[3])) {
            /** @bug need to do more validity checks */


            /* SSLv2 (client_hello) */
            return SSL2_PORT_NUMBER;
        } else if (payloadSize >= 9) {
            if ((payload[0] == 0x16) && /* handshake request */
                (payload[1] == 0x03) && /* ssl major version is 3 */
                (payload[5] == 0x01) && /* handshake command is client_hello */
                (((payload[3] == 0) && (payload[4] < 5)) ||     /* payload
                                                                 * length */
                 (payload[9] == payload[1]))    /* don't know what the hell
                                                 * this is, payload length
                                                 * equals major version #???? */
              ) {
                /** @bug need to do more validity checks */


                /* SSLv3 / TLS */
                return SSL3_PORT_NUMBER;
            }

        }
    }
    return 0;
}
