/*
 ** yaf.c
 ** Yet Another Flow generator
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <airframe/logconfig.h>
#include <airframe/privconfig.h>
#include <airframe/airutil.h>
#include <airframe/airopt.h>
#include <yaf/yafcore.h>
#include <yaf/yaftab.h>
#include <yaf/yafrag.h>

#include "yafcap.h"
#include "yafstat.h"
#include "yafctx.h"
#if YAF_ENABLE_DAG
#include "yafdag.h"
#endif 
#if YAF_ENABLE_APPLABEL
#include "yafapplabel.h"
#endif

/* I/O configuration */
static yfConfig_t	yaf_config = YF_CONFIG_INIT;
static int			yaf_opt_rotate = 0;
static uint64_t		yaf_rotate_ms = 0;
static gboolean     yaf_opt_caplist_mode = FALSE;
static char			*yaf_opt_ipfix_transport = NULL;
static gboolean		yaf_opt_ipfix_tls = FALSE;

/* GOption managed flow table options */
static int          yaf_opt_idle = 300;
static int          yaf_opt_active = 1800;
static int          yaf_opt_max_flows = 0;
static int          yaf_opt_max_payload = 0;
static gboolean     yaf_opt_payload_export = FALSE;
static gboolean	    yaf_opt_applabel_mode = FALSE;
#if YAF_ENABLE_APPLABEL
static char	   *yaf_opt_applabel_rules = NULL;
#endif
static gboolean	    yaf_opt_entropy_mode = FALSE;
static gboolean     yaf_opt_uniflow_mode = FALSE;
static gboolean     yaf_opt_silk_mode = FALSE;

/* GOption managed fragment table options */
static int          yaf_opt_max_frags = 0;
static gboolean     yaf_opt_nofrag = FALSE;

/* GOption managed decoder options and derived decoder config */
static gboolean     yaf_opt_ip4_mode = FALSE;
static gboolean     yaf_opt_ip6_mode = FALSE;
static uint16_t     yaf_reqtype;
static gboolean     yaf_opt_gre_mode = FALSE;
static gboolean		yaf_opt_mac_mode = FALSE;

/* GOption managed core export options */
static gboolean		yaf_opt_ip6map_mode = FALSE;

/* global quit flag */
int	yaf_quit = 0;

/* Runtime functions */

typedef void *(*yfLiveOpen_fn)(const char *, int, int *, GError **);
static yfLiveOpen_fn yaf_liveopen_fn = NULL;

typedef gboolean (*yfLoop_fn)(yfContext_t *);
static yfLoop_fn yaf_loop_fn = NULL;

typedef void (*yfClose_fn)(void *);
static yfClose_fn yaf_close_fn = NULL;


#define THE_LAME_80COL_FORMATTER_STRING "\n\t\t\t\t\t"

/* Local derived configutation */

AirOptionEntry yaf_optent_core[] = {
    AF_OPTION( "in", 'i', 0, AF_OPT_TYPE_STRING, &yaf_config.inspec, 
               "Input (file, - for stdin; interface)", "inspec"),
    AF_OPTION( "out", 'o', 0, AF_OPT_TYPE_STRING, &yaf_config.outspec, 
               "Output (file, - for stdout; file prefix,"
               THE_LAME_80COL_FORMATTER_STRING"address)", "outspec"),
    AF_OPTION( "live", 'P', 0, AF_OPT_TYPE_STRING, &yaf_config.livetype,
               "Capture from interface in -i; type is "
               THE_LAME_80COL_FORMATTER_STRING"[pcap] or dag", "type"),
    AF_OPTION( "caplist", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_caplist_mode,
               "Read ordered list of input files from "
               THE_LAME_80COL_FORMATTER_STRING"file in -i", NULL),
    AF_OPTION( "rotate", 'R', 0, AF_OPT_TYPE_INT, &yaf_opt_rotate,
               "Rotate output files every n seconds ", "sec" ),
    AF_OPTION( "lock", 'k', 0, AF_OPT_TYPE_NONE, &yaf_config.lockmode, 
               "Use exclusive .lock files on output for"
               THE_LAME_80COL_FORMATTER_STRING"concurrency", NULL ),
    AF_OPTION( "ipfix", (char)0, 0, AF_OPT_TYPE_STRING, &yaf_opt_ipfix_transport,
               "Export via IPFIX (tcp, udp, sctp) to CP "
               THE_LAME_80COL_FORMATTER_STRING"at -o", "protocol" ),
    AF_OPTION_END
};
    
AirOptionEntry yaf_optent_dec[] = {
    AF_OPTION( "no-frag", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_nofrag,
               "Disable IP fragment reassembly", NULL ),
    AF_OPTION( "max-frags", (char)0, 0, AF_OPT_TYPE_INT, &yaf_opt_max_frags,
               "Maximum size of fragment table [0]", "fragments" ),
    AF_OPTION( "ip4-only", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ip4_mode,
               "Only process IPv4 packets", NULL ),
    AF_OPTION( "ip6-only", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ip6_mode,
               "Only process IPv6 packets", NULL ),
    AF_OPTION( "gre-decode", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_gre_mode,
               "Decode GRE encapsulated packets", NULL ),
    AF_OPTION( "mac", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_mac_mode,
               "Export MAC-layer information", NULL ),
    AF_OPTION_END
};

AirOptionEntry yaf_optent_flow[] = {
    AF_OPTION( "idle-timeout", 'I', 0, AF_OPT_TYPE_INT, &yaf_opt_idle,
               "Idle flow timeout [300, 5m]", "sec" ),
    AF_OPTION( "active-timeout", 'A', 0, AF_OPT_TYPE_INT, &yaf_opt_active,
               "Active flow timeout [1800, 30m]", "sec" ),
    AF_OPTION( "max-flows", (char)0, 0, AF_OPT_TYPE_INT, &yaf_opt_max_flows,
			   "Maximum size of flow table [0]", "flows" ),
    AF_OPTION_END
};

AirOptionEntry yaf_optent_exp[] = {
    AF_OPTION( "silk", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_silk_mode,
               "Clamp octets to 32 bits, note continued in"
               THE_LAME_80COL_FORMATTER_STRING"flowEndReason",
               NULL ),
    AF_OPTION( "uniflow", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_uniflow_mode,
               "Write uniflows for compatibility", NULL ),
    AF_OPTION( "force-ip6-export", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ip6map_mode,
               "Export all IPv4 addresses as IPv6 in "
               THE_LAME_80COL_FORMATTER_STRING"::FFFF/96 [N/A]", NULL ),
    AF_OPTION( "observation-domain", (char)0, 0, AF_OPT_TYPE_INT, &yaf_config.odid,
               "Set observationDomainID on exported"
               THE_LAME_80COL_FORMATTER_STRING"messages [1]", "odId" ),
    AF_OPTION_END
};

AirOptionEntry yaf_optent_ipfix[] = {
    AF_OPTION( "ipfix-port", (char)0, 0, AF_OPT_TYPE_STRING, &(yaf_config.connspec.svc),
               "Select IPFIX export port [4739, 4740]", "port" ),
    AF_OPTION( "tls", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ipfix_tls,
               "Use TLS/DTLS to secure IPFIX export", NULL ),
    AF_OPTION( "tls-ca", (char)0, 0, AF_OPT_TYPE_STRING,
               &(yaf_config.connspec.ssl_ca_file),
               "Specify TLS Certificate Authority file", "cafile" ),
    AF_OPTION( "tls-cert", (char)0, 0, AF_OPT_TYPE_STRING,
               &(yaf_config.connspec.ssl_cert_file),
               "Specify TLS Certificate file", "certfile" ),
    AF_OPTION( "tls-key", (char)0, 0, AF_OPT_TYPE_STRING,
               &(yaf_config.connspec.ssl_key_file),
               "Specify TLS Private Key file", "keyfile" ),
    AF_OPTION_END
};

#if YAF_ENABLE_PAYLOAD
AirOptionEntry yaf_optent_payload[] = {
    AF_OPTION( "max-payload", 's', 0, AF_OPT_TYPE_INT, &yaf_opt_max_payload,
               "Maximum payload to capture per flow [0]", "octets" ),
    AF_OPTION( "export-payload", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_payload_export,
			    "Export full captured payload per flow", NULL),
#if YAF_ENABLE_ENTROPY
    AF_OPTION( "entropy", (char)0, 0, AF_OPT_TYPE_NONE, &yaf_opt_entropy_mode, 
	           "Export Shannon entropy of captured payload", NULL),
#endif
#if YAF_ENABLE_APPLABEL
    AF_OPTION( "applabel-rules", 0, 0,
                AF_OPT_TYPE_STRING, &yaf_opt_applabel_rules, 
                "specify the name of the application labeler"
                THE_LAME_80COL_FORMATTER_STRING"rules file", 
                "file"),
    AF_OPTION("applabel", 0, 0,
                AF_OPT_TYPE_NONE, &yaf_opt_applabel_mode,
                "enable the packet inspection protocol"
                THE_LAME_80COL_FORMATTER_STRING"application labeler engine",
                NULL ),
#endif
    AF_OPTION_END
};
#endif


static void yfParseOptions(
    int             *argc,
    char            **argv[]) {

    AirOptionCtx    *aoctx = NULL;
    GError          *err = NULL;

    aoctx = air_option_context_new("", argc, argv, yaf_optent_core);
    
    air_option_context_add_group(aoctx, "decode", "Decoder Options:",
                                 "Show help for packet decoder options", 
                                 yaf_optent_dec);
    air_option_context_add_group(aoctx, "flow", "Flow table Options:",
                                 "Show help for flow table options", 
                                 yaf_optent_flow);
    air_option_context_add_group(aoctx, "export", "Export Options:",
                                 "Show help for export format options", 
                                 yaf_optent_exp);
    air_option_context_add_group(aoctx, "ipfix", "IPFIX Options:",
                                 "Show help for IPFIX export options", 
                                 yaf_optent_ipfix);
#if YAF_ENABLE_PAYLOAD                                 
    air_option_context_add_group(aoctx, "payload", "Payload Options:", 
                                 "Show the help for payload options",
                                 yaf_optent_payload);
#endif
    
    privc_add_option_group(aoctx);
    logc_add_option_group(aoctx, "yaf", VERSION);

    air_option_context_set_help_enabled(aoctx);
    air_option_context_parse(aoctx);


    /* set up logging and privilege drop */
    if (!logc_setup(&err)) {
        air_opterr("%s", err->message);
    }

    if (!privc_setup(&err)) {
        air_opterr("%s", err->message);
    }

#if YAF_ENABLE_APPLABEL
    if (TRUE == yaf_opt_applabel_mode) {
        if (yaf_opt_max_payload == 0) {
            g_warning("--applabel requires --max-payload.");
            yaf_opt_applabel_mode = FALSE;
        } else {
            if (!yfAppLabelInit(yaf_opt_applabel_rules, &err)) {
                if (NULL != err) {
                    g_warning("application labeler config error: %s",
                              err->message);
                    g_warning("application labeler engine will not operate");
                    g_clear_error(&err);
                }
            }
        }
    }
#endif

#if YAF_ENABLE_ENTROPY
	if (TRUE == yaf_opt_entropy_mode) {
		if (yaf_opt_max_payload == 0) {
			g_warning("--entropy requires --max-payload.");
            yaf_opt_entropy_mode = FALSE;
		}
	}
#endif

    /* process ip4mode and ip6mode */
    if (yaf_opt_ip4_mode && yaf_opt_ip6_mode) {
        g_warning("cannot run in both ip4-only and ip6-only modes; "
                  "ignoring these flags");
        yaf_opt_ip4_mode = FALSE;
        yaf_opt_ip6_mode = FALSE;
    }
    
    if (yaf_opt_ip4_mode) {
        yaf_reqtype = YF_TYPE_IPv4;
    } else if (yaf_opt_ip6_mode) {
        yaf_reqtype = YF_TYPE_IPv6;
    } else {
        yaf_reqtype = YF_TYPE_IPANY;
    }
	
	/* process core library options */
	if (yaf_opt_payload_export) {
		yfWriterExportPayload(TRUE);
	}
	
	if (yaf_opt_ip6map_mode) {
		yfWriterExportMappedV6(TRUE);
	}
    
    /* Pre-process input options */
    if (yaf_config.livetype) {
        /* can't use caplist with live */
        if (yaf_opt_caplist_mode) {
            air_opterr("Please choose only one of --live or --caplist");
        }
        
        /* select live capture type */
        if ((*yaf_config.livetype == (char)0) ||
            (strncmp(yaf_config.livetype, "pcap", 4) == 0))
        {
            /* live capture via pcap (--live=pcap or --live) */
            yaf_liveopen_fn = (yfLiveOpen_fn)yfCapOpenLive; 
            yaf_loop_fn = (yfLoop_fn)yfCapMain;
            yaf_close_fn = (yfClose_fn)yfCapClose;           
#if YAF_ENABLE_DAG
        } else if (strncmp(yaf_config.livetype, "dag", 3) == 0) {
            /* live capture via dag (--live=dag) */
            yaf_liveopen_fn = (yfLiveOpen_fn)yfDagOpenLive;
            yaf_loop_fn = (yfLoop_fn)yfDagMain;
            yaf_close_fn = (yfClose_fn)yfDagClose;
#endif
        } else {
            /* unsupported live capture type */
            air_opterr("Unsupported live capture type %s", yaf_config.livetype);
        }
        
        /* Require an interface name for live input */
        if (!yaf_config.inspec) {
            air_opterr("--live requires interface name in --in");
        }
        
    } else {
        /* Use pcap loop and close functions */
        yaf_loop_fn = (yfLoop_fn)yfCapMain;
        yaf_close_fn =(yfClose_fn)yfCapClose;           
    
        /* Default to stdin for no input */
        if (!yaf_config.inspec || !strlen(yaf_config.inspec)) {
            yaf_config.inspec = "-";
        }
    }

    /* calculate live rotation delay in milliseconds */
    yaf_rotate_ms = yaf_opt_rotate * 1000;
    
    /* Pre-process output options */
    if (yaf_opt_ipfix_transport) {
        /* set default port */
        if (!yaf_config.connspec.svc) {
            yaf_config.connspec.svc = yaf_opt_ipfix_tls ? "4740" : "4739";
        }
        
        /* Require a hostname for IPFIX output */
        if (!yaf_config.outspec) {
            air_opterr("--ipfix requires hostname in --out");
        }
        
        /* set hostname */
        yaf_config.connspec.host = yaf_config.outspec;
        
        if ((*yaf_opt_ipfix_transport == (char)0) ||
            (strcmp(yaf_opt_ipfix_transport, "sctp") == 0)) 
        {
            if (yaf_opt_ipfix_tls) {
                yaf_config.connspec.transport = FB_DTLS_SCTP;
            } else {
                yaf_config.connspec.transport = FB_SCTP;
            }
        } else if (strcmp(yaf_opt_ipfix_transport, "tcp") == 0) {
            if (yaf_opt_ipfix_tls) {
                yaf_config.connspec.transport = FB_TLS_TCP;
            } else {
                yaf_config.connspec.transport = FB_TCP;
            }
        } else if (strcmp(yaf_opt_ipfix_transport, "udp") == 0) {
            if (yaf_opt_ipfix_tls) {
                yaf_config.connspec.transport = FB_DTLS_UDP;
            } else {
                yaf_config.connspec.transport = FB_UDP;
            }
        } else {
            air_opterr("Unsupported IPFIX transport protocol %s", yaf_opt_ipfix_transport);
        }

        /* grab TLS password from environment */
        if (yaf_opt_ipfix_tls) {
            yaf_config.connspec.ssl_key_pass = getenv("YAF_TLS_PASS");
        }

        /* mark that a network connection is requested for this spec */
        yaf_config.ipfixNetTrans = TRUE;
		
    } else {
        if (!yaf_config.outspec || !strlen(yaf_config.outspec)) {
            if (yaf_rotate_ms) {
                /* Require a path prefix for IPFIX output */
                air_opterr("--rotate requires prefix in --out");
            } else {
                /* Default to stdout for no output without rotation */
                yaf_config.outspec = "-";
            }
        }
    }
    
    /* Check for stdin/stdout is terminal */
    if ((strlen(yaf_config.inspec) == 1) && yaf_config.inspec[0] == '-') {
        /* Don't open stdout if it's a terminal */
        if (isatty(fileno(stdin))) {
            air_opterr("Refusing to read from terminal on stdin");
        }
    }

    if ((strlen(yaf_config.outspec) == 1) && yaf_config.outspec[0] == '-') {
        /* Don't open stdout if it's a terminal */
        if (isatty(fileno(stdout))) {
            air_opterr("Refusing to write to terminal on stdout");
        }
    }
}

static void yfQuit() {
    yaf_quit++;
}

static void yfQuitInit() {
    struct sigaction sa, osa;

    /* install quit flag handlers */
    sa.sa_handler = yfQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGINT,&sa,&osa)) {
        g_error("sigaction(SIGINT) failed: %s", strerror(errno));
    }

    sa.sa_handler = yfQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGTERM,&sa,&osa)) {
        g_error("sigaction(SIGTERM) failed: %s", strerror(errno));
    }
}

int main (
    int             argc,
    char            *argv[])
{
    GError          *err = NULL;
	yfContext_t		ctx = YF_CTX_INIT;
    int             datalink;
    gboolean        loop_ok = TRUE;
    
    /* check structure alignment */
    yfAlignmentCheck();

    /* parse options */
    yfParseOptions(&argc, &argv);
	ctx.cfg = &yaf_config;

    /* Set up quit handler */
    yfQuitInit();
    
    /* open interface if we're doing live capture */
    if (yaf_liveopen_fn) {
        /* open interface */
        if (!(ctx.pktsrc = yaf_liveopen_fn(yaf_config.inspec, 
										   yaf_opt_max_payload + 96, 
										   &datalink, &err))) {
            g_warning("Cannot open interface %s: %s", yaf_config.inspec, err->message);
            exit(1);
        }
        
        /* drop privilege */
        if (!privc_become(&err)) {
            if (g_error_matches(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_NODROP)) {
                g_warning("running as root in --live mode, "
                          "but not dropping privilege");
                g_clear_error(&err);
            } else {
                yaf_close_fn(ctx.pktsrc);
                g_warning("Cannot drop privilege: %s", err->message);
                exit(1);
            }
        }
    } else {
        if (yaf_opt_caplist_mode) {
            /* open input file list */
            if (!(ctx.pktsrc = yfCapOpenFileList(yaf_config.inspec, &datalink, &err))) {
                g_warning("Cannot open packet file list file %s: %s", 
                         yaf_config.inspec, err->message);
                exit(1);
            }
        } else {
            /* open input file */
            if (!(ctx.pktsrc = yfCapOpenFile(yaf_config.inspec, &datalink, &err))) {
                g_warning("Cannot open packet file %s: %s", 
                          yaf_config.inspec, err->message);
                exit(1);
            }
        }
    }
	
	/* Calculate packet buffer size */
    if (yaf_opt_max_payload) {
        ctx.pbuflen = YF_PBUFLEN_BASE + yaf_opt_max_payload;
    } else if (yaf_opt_mac_mode) {
        ctx.pbuflen = YF_PBUFLEN_NOPAYLOAD;
    } else {
        ctx.pbuflen = YF_PBUFLEN_NOL2INFO;
    }
    
    /* Allocate a packet ring. */
    ctx.pbufring = rgaAlloc(ctx.pbuflen, 128);
 
	/* Set up decode context */
    ctx.dectx = yfDecodeCtxAlloc(datalink, yaf_reqtype, yaf_opt_gre_mode);
	
	/* Set up flow table */
	ctx.flowtab = yfFlowTabAlloc(yaf_opt_idle * 1000,
                                 yaf_opt_active * 1000,
                                 yaf_opt_max_flows,
                                 yaf_opt_max_payload,
								 yaf_opt_uniflow_mode,
                                 yaf_opt_silk_mode,
								 yaf_opt_applabel_mode,
								 yaf_opt_entropy_mode);

	/* Set up fragment table */
	ctx.fragtab = yfFragTabAlloc(30000,
                                 yaf_opt_max_frags,
                                 yaf_opt_max_payload);

     
    /* We have a packet source, an output stream, 
       and all the tables we need. Run with it. */
    yfStatInit(&ctx);
    loop_ok = yaf_loop_fn(&ctx);
    yfStatComplete();
    yfCapDumpStats();
#if YAF_ENABLE_DAG
    yfDagDumpStats();
#endif

    /* Close packet source */
    yaf_close_fn(ctx.pktsrc);
    
    /* Print exit message */
    if (loop_ok) {
        g_debug("yaf terminating");
    } else {
        g_warning("yaf terminating on error: %s", ctx.err->message);
    }
    
    return loop_ok ? 0 : 1;
}
