/*
 ** yafcap.c
 ** YAF libpcap input support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/yaftab.h>
#include "yafout.h"

#include <airframe/airlock.h>
#include <pcap.h>

#include "yafcap.h"
#include "yaflush.h"
#include "yafstat.h"

struct yfCapSource_st {
    pcap_t          *pcap;
    FILE            *lfp;
    gboolean        is_live;
    int             datalink;
};

/* Quit flag support */
extern int yaf_quit;

/* Statistics */
uint32_t            yaf_pcap_drop = 0;

/* One second timeout for capture loop */
#define YAF_CAP_TIMEOUT 1000

/* Process at most 64 packets at once */
#define YAF_CAP_COUNT   64

static gboolean yfCapCheckDatalink(
    pcap_t                  *pcap,
    int                     *datalink,
    GError                  **err)
{
    /* verify datalink */
    *datalink = pcap_datalink(pcap);
    switch (*datalink) {
#ifdef DLT_EN10MB
      case DLT_EN10MB:
#endif
#ifdef DLT_C_HDLC
      case DLT_C_HDLC:
#endif
#ifdef DLT_LINUX_SLL
      case DLT_LINUX_SLL:
#endif
#ifdef DLT_PPP
      case DLT_PPP:
#endif
#ifdef DLT_PPP_ETHER
      case DLT_PPP_ETHER:
#endif
#ifdef DLT_RAW
      case DLT_RAW:
#endif
#ifdef DLT_NULL
      case DLT_NULL:
#endif
#ifdef DLT_LOOP
      case DLT_LOOP:
#endif
        break;
      case -1:
          g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
              "Unable to access pcap datalink, (superuser access?)");
          return FALSE;
      default:
        g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                    "Unsupported pcap datalink type %u", *datalink);
        return FALSE;
    }
    
    return TRUE;
}

pcap_t *yfCapOpenFileInner(
    const char              *path,
    int                     *datalink,
    GError                  **err)
{
    pcap_t                  *pcap;
    static char             pcap_errbuf[PCAP_ERRBUF_SIZE];
    
    if ((strlen(path) == 1) && path[0] == '-') {
        /* Don't open stdin if it's a terminal */
        if (isatty(fileno(stdin))) {
            g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                "Refusing to read from terminal on stdin");
            return NULL;
        }
    }
    
    pcap = pcap_open_offline(path, pcap_errbuf);
    if (!pcap) {
        g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                    "%s", pcap_errbuf);
        return NULL;
    }
    
    if (!yfCapCheckDatalink(pcap, datalink, err)) {
        pcap_close(pcap);
        return NULL;
    }
    
    g_debug("Reading packets from %s", path);
    
    return pcap;
}

yfCapSource_t *yfCapOpenFile(
    const char              *path,
    int                     *datalink,
    GError                  **err)
{
    yfCapSource_t           *cs;

    cs = g_new0(yfCapSource_t, 1);
    cs->pcap = yfCapOpenFileInner(path, datalink, err);
    cs->is_live = FALSE;
    cs->lfp = NULL;
    cs->datalink = *datalink;
    
    if (!cs->pcap) {
        g_free(cs);
        cs = NULL;
    }
    
    return cs;
}

static gboolean yfCapFileListNext(
    yfCapSource_t           *cs,
    GError                  **err)
{
    static char             cappath[FILENAME_MAX+1];
    size_t                  cappath_len;
    int                     this_datalink;

    /* close the present pcap if necessary */
    if (cs->pcap) {
        pcap_close(cs->pcap);
        cs->pcap = NULL;
    }

    /* keep going until we get an actual opened pcap file */
    while (1) {

        /* get the next line from the name list file */
        if (!fgets(cappath, FILENAME_MAX, cs->lfp)) {
            if (feof(cs->lfp)) {
                g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_EOF, 
                    "End of pcap file list");
            } else {
                g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                    "Couldn't read pcap file list: %s", strerror(errno));
            }
            return FALSE;
        }

        /* ensure filename is null terminated */
        cappath[FILENAME_MAX] = (char)0;

        /* skip comments and blank lines */
        if (cappath[0] == '\n' || cappath[0] == '#') {
            continue;
        }
        
        /* remove trailing newline */
        cappath_len = strlen(cappath);
        if (cappath[cappath_len-1] == '\n') {
            cappath[cappath_len-1] = (char)0;
        }

        /* we have what we think is a filename. try opening it. */
        cs->pcap = yfCapOpenFileInner(cappath, &this_datalink, err);
        if (!cs->pcap) {
            g_warning("skipping pcap file %s due to error: %s.", 
                cappath, (*err)->message);
            g_clear_error(err);
            continue;
        }

        /* make sure the datalink matches all the others */
        if (cs->datalink == -1) {
            cs->datalink = this_datalink;
        } else if (cs->datalink != this_datalink) {
            g_warning("skipping pcap file %s due to mismatched "
                "datalink type %u (expecting %u).", 
                cappath, this_datalink, cs->datalink);
            pcap_close(cs->pcap);
            cs->pcap = NULL;
            continue;
        }

        /* We have a file. All is well. */
        return TRUE;
    }
}

yfCapSource_t *yfCapOpenFileList(
    const char              *path,
    int                     *datalink,
    GError                  **err)
{
    yfCapSource_t           *cs;
    
    /* allocate a new capsource */
    cs = g_new0(yfCapSource_t, 1);

    /* handle file list from stdin */
    if ((strlen(path) == 1) && path[0] == '-') {
        /* Don't open stdin if it's a terminal */
        if (isatty(fileno(stdin))) {
            g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                "Refusing to read from terminal on stdin");
            g_free(cs);
            return NULL;
        }
        cs->lfp = stdin;
    } else {
        /* open file list file */
        cs->lfp = fopen(path, "r");
        if (!cs->lfp) {
            g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                "Couldn't open pcap file list: %s", strerror(errno));
            g_free(cs);
            return NULL;
        }
    }

    /* note we're not live */
    cs->is_live = FALSE;
    
    /* note we have no datalink yet */
    cs->datalink = -1;

    /* open the first pcap file in the file list */
    if (!yfCapFileListNext(cs, err)) {
        fclose(cs->lfp);
        g_free(cs);
        return NULL;
    }
    
    /* copy datalink back out of capsource */
    *datalink = cs->datalink;

    /* all done */
    return cs;
}

yfCapSource_t *yfCapOpenLive(
    const char              *ifname,
	int						snaplen,
    int                     *datalink,
    GError                  **err)
{
    yfCapSource_t           *cs;
    pcap_t                  *pcap;
    static char             pcap_errbuf[PCAP_ERRBUF_SIZE];
    
    pcap = pcap_open_live(ifname, snaplen, 1, 
                          YAF_CAP_TIMEOUT, pcap_errbuf);
    if (!pcap) {
        g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                    "%s", pcap_errbuf);
        return NULL;
    }
    
    if (!yfCapCheckDatalink(pcap, datalink, err)) {
        pcap_close(pcap);
        return NULL;
    }
    
    cs = g_new0(yfCapSource_t, 1);
    cs->pcap = pcap;
    cs->is_live = TRUE;
    cs->lfp = NULL;
    cs->datalink = *datalink;

    return cs;
}

void yfCapClose(
    yfCapSource_t                 *cs)
{
    if (cs->pcap) {
        pcap_close(cs->pcap);
    }
    if (cs->lfp) {
        fclose(cs->lfp);
    }
    g_free(cs);
}

static void yfCapUpdateStats(pcap_t *pcap) {
    struct pcap_stat ps;
    
    if (pcap_stats(pcap, &ps) != 0) {
        g_warning("couldn't get statistics: %s", pcap_geterr(pcap));
        return;
    }
    
    yaf_pcap_drop = ps.ps_drop;
}

void yfCapDumpStats() {
    if (yaf_pcap_drop) {
        g_warning("Live capture device dropped %u packets.", yaf_pcap_drop);
    }
}

static void yfCapHandle(
    yfContext_t                 *ctx,
    const struct pcap_pkthdr    *hdr,
    const uint8_t               *pkt)
{
    yfPBuf_t                    *pbuf;
    yfIPFragInfo_t              fraginfo_buf, 
                                *fraginfo = ctx->fragtab ? 
                                            &fraginfo_buf : NULL;

    /* get next spot in ring buffer */
    pbuf = (yfPBuf_t *)rgaNextHead(ctx->pbufring);
    g_assert(pbuf);

    /* Decode packet into packet buffer */
    if (!yfDecodeToPBuf(ctx->dectx, 
                        yfDecodeTimeval(&(hdr->ts)),
                        hdr->caplen, pkt, 
                        fraginfo, ctx->pbuflen, pbuf))      
    {
        /* Couldn't decode packet; counted in dectx. Skip. */
        return;
    }

    /* Handle fragmentation if necessary */
    if (fraginfo && fraginfo->frag) {
        if (!yfDefragPBuf(ctx->fragtab, fraginfo, 
                          ctx->pbuflen, pbuf))
        {
            /* No complete defragmented packet available. Skip. */
            return;
        }
    }

}

void *yfCapMain(
    yfContext_t             *ctx)
{
    AirLock                 lockbuf = AIR_LOCK_INIT, *lock = NULL;
    gboolean                ok = TRUE;
    yfCapSource_t           *cs = (yfCapSource_t *)ctx->pktsrc;
    int                     pcrv = 0;

    /* set up output locking in lock mode */
    if (ctx->cfg->lockmode) {
        lock = &lockbuf;
    }

    /* FIXME need to rework simulation mode for new world.
       removed for now */
    
    /* process input until we're done */
    while (!yaf_quit) {

        /* Process some packets */
        pcrv = pcap_dispatch(cs->pcap, YAF_CAP_COUNT, (pcap_handler)yfCapHandle, (void *)ctx);
        
        /* Handle the aftermath */
        if (pcrv == 0) {
            /* No packet available */
            if (cs->lfp) {
                /* Advance to next capfile */
                if (!yfCapFileListNext(cs, &(ctx->err))) {
                    if (!g_error_matches(ctx->err, YAF_ERROR_DOMAIN, 
											          YAF_ERROR_EOF)) {
                        ok = FALSE;
                    }
                    break;
                }
            } else if (!cs->is_live) {
                /* EOF in single capfile mode; break */
                break;
            } else {
                /* Live, no packet processed (timeout). Skip. */
               continue;
            }
        } else if (pcrv < 0) {
            /* An error occurred reading packets. */
            g_set_error(&(ctx->err), YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                        "Couldn't read next pcap record from %s: %s",
                        ctx->cfg->inspec, pcap_geterr(cs->pcap));
            ok = FALSE;
            break;
        } 
        /* Process the packet buffer */
        if (ok && !yfProcessPBufRing(ctx, &(ctx->err))) {
            ok = FALSE;
            break;
        }
    }

    /* Update packet drop statistics for live capture */
    if (cs->is_live) {
        yfCapUpdateStats(cs->pcap);
    }
    
    /* Handle final flush */
    return (void *)yfFinalFlush(ctx, ok, &(ctx->err));
}
