/*
 *@internal
 ** yafcore.c
 ** YAF core I/O routines
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <airframe/airutil.h>

/** These are the template ID's for the templates that YAF uses to
    select the output. Template ID's are maintained for a set of
    basic flow types data 
    * BASE which gets various additions added as the flow requires, 
    * FULL base plus the internal fields are added
    * EXT (extended) which has the additional records in the 
      yaf_extime_spec (extended time specification) 
      
    WARNING: these need to be adjusted according to changes in the
    general & special dimensions */
#define YAF_FLOW_BASE_TID   0xB800
#define YAF_FLOW_FULL_TID   0xBC00
#define YAF_FLOW_EXT_TID    0xBBFF

/** The dimensions are flags which determine which sets of fields will
    be exported out to an IPFIX record.  They are entries in a bitmap
    used to control the template. e.g. TCP flow information (seq num,
    tcp flags, etc.) only get added to the output record when the 
    YTF_TCP flag is set; it only gets set when the transport protocol
    is set to 0x06. */
    
/** General dimensions */
#define YTF_BIF         0x0001
#define YTF_PAY         0x0002
#define YTF_TCP         0x0004
#define YTF_MAC         0x0008
#define YTF_APPLBL      0x0010
#define YTF_ENTROPY     0x0020

/* Special dimensions */
#define YTF_FLE         0x0040
#define YTF_RLE         0x0080
#define YTF_IP4         0x0100
#define YTF_IP6         0x0200
#define YTF_INTERNAL    0x0400
#define YTF_ALL         0x077F /* this has to be everything _except_ RLE enabled */

/** If any of the FLE/RLE values are larger than this constant
    then we have to use FLE, otherwise, we choose RLE to 
    conserve space/bandwidth etc.*/
#define YAF_RLEMAX      (1L << 31)

#define YF_PRINT_DELIM  "|"

/* IPFIX information elements in 6871 space for YAF */
static fbInfoElement_t yaf_info_elements[] = {
    FB_IE_INIT("initialTCPFlags", 6871, 14, 1, FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("unionTCPFlags", 6871, 15, 1, FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("payload", 6871, 18, FB_IE_VARLEN, FB_IE_F_REVERSIBLE),
    FB_IE_INIT("reverseFlowDeltaMilliseconds", 6871, 21, 4, FB_IE_F_ENDIAN),
    FB_IE_INIT("silkAppLabel", 6871, 33, 2, FB_IE_F_ENDIAN),
    FB_IE_INIT("payloadEntropy", 6871, 35, 1, FB_IE_F_REVERSIBLE),
    FB_IE_NULL
};

/* IPFIX definition of the full YAF flow record */
static fbInfoElementSpec_t yaf_flow_spec[] = {
    /* Millisecond start and end (epoch) (native time) */
    { "flowStartMilliseconds",              0, 0 },
    { "flowEndMilliseconds",                0, 0 },
    /* Counters */
    { "octetTotalCount",                    8, YTF_FLE },
    { "reverseOctetTotalCount",             8, YTF_FLE | YTF_BIF },
    { "packetTotalCount",                   8, YTF_FLE },
    { "reversePacketTotalCount",            8, YTF_FLE | YTF_BIF },
    /* Reduced-length counters */
    { "octetTotalCount",                    4, YTF_RLE },
    { "reverseOctetTotalCount",             4, YTF_RLE | YTF_BIF },
    { "packetTotalCount",                   4, YTF_RLE },
    { "reversePacketTotalCount",            4, YTF_RLE | YTF_BIF },
    /* 5-tuple and flow status */
    { "sourceIPv6Address",                  0, YTF_IP6 },
    { "destinationIPv6Address",             0, YTF_IP6 },
    { "sourceIPv4Address",                  0, YTF_IP4 },
    { "destinationIPv4Address",             0, YTF_IP4 },
    { "sourceTransportPort",                0, 0 },
    { "destinationTransportPort",           0, 0 },
    { "protocolIdentifier",                 0, 0 },
    { "flowEndReason",                      0, 0 },
#   if defined(YAF_ENABLE_APPLABEL) && !defined(YAF_ENABLE_ENTROPY)
    /* protocol app labeler information */
    {"silkAppLabel",                        0, YTF_APPLBL },
    { "paddingOctets",                      4, YTF_INTERNAL },
#   elif defined(YAF_ENABLE_ENTROPY) && !defined(YAF_ENABLE_APPLABEL)
    /* entropy fields */
    { "payloadEntropy",                     0, YTF_ENTROPY },
    { "reversePayloadEntropy",              0, YTF_ENTROPY | YTF_BIF },
    { "paddingOctets",                      4, YTF_INTERNAL },
#   elif defined(YAF_ENABLE_ENTROPY) && defined(YAF_ENABLE_APPLABEL)
    /* applabeler and entropy fields */
    {"silkAppLabel",                        0, YTF_APPLBL },
    { "payloadEntropy",                     0, YTF_ENTROPY },
    { "reversePayloadEntropy",              0, YTF_ENTROPY | YTF_BIF },
    { "paddingOctets",                      2, YTF_INTERNAL },
#   else
    /* Internal-only padding */
    { "paddingOctets",                      6, YTF_INTERNAL },
#   endif
    /* Round-trip time */
    { "reverseFlowDeltaMilliseconds",       0, YTF_BIF },
    /* TCP-specific information */
    { "tcpSequenceNumber",                  0, YTF_TCP },
    { "reverseTcpSequenceNumber",           0, YTF_TCP | YTF_BIF },
    { "initialTCPFlags",                    0, YTF_TCP },
    { "unionTCPFlags",                      0, YTF_TCP },
    { "reverseInitialTCPFlags",             0, YTF_TCP | YTF_BIF },
    { "reverseUnionTCPFlags",               0, YTF_TCP | YTF_BIF },
    /* MAC-specific information */
    { "vlanId",                             0, YTF_MAC },
    { "reverseVlanId",                      0, YTF_MAC | YTF_BIF },
#   if YAF_ENABLE_PAYLOAD
    /* Variable-length payload fields */
    { "payload",                            0, YTF_PAY },
    { "reversePayload",                     0, YTF_PAY },
#   endif
    FB_IESPEC_NULL
};

/* IPFIX definition of the YAF flow record time extension */
static fbInfoElementSpec_t yaf_extime_spec[] = {
    /* Microsecond start and end (RFC1305-style) (extended time) */
    { "flowStartMicroseconds",              0, 0 },
    { "flowEndMicroseconds",                0, 0 },    
    /* Second start, end, and duration (extended time) */
    { "flowStartSeconds",                   0, 0 },
    { "flowEndSeconds",                     0, 0 },    
    /* Flow durations (extended time) */
    { "flowDurationMicroseconds",           0, 0 },
    { "flowDurationMilliseconds",           0, 0 },
    /* Microsecond delta start and end (extended time) */
    { "flowStartDeltaMicroseconds",         0, 0 },
    { "flowEndDeltaMicroseconds",           0, 0 },
    /* Delta counter support (uniflow only) */
    { "octetDeltaCount",                    0, 0 },
    { "packetDeltaCount",                   0, 0 },
    FB_IESPEC_NULL
};

/* IPv6-mapped IPv4 address prefix */
static uint8_t yaf_ip6map_pfx[12] =
    { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xFF, 0xFF };

/* Full YAF flow record. */
typedef struct yfIpfixFlow_st {
    uint64_t    flowStartMilliseconds;
    uint64_t    flowEndMilliseconds;
    uint64_t    octetTotalCount;
    uint64_t    reverseOctetTotalCount;
    uint64_t    packetTotalCount;
    uint64_t    reversePacketTotalCount;
    uint8_t     sourceIPv6Address[16];
    uint8_t     destinationIPv6Address[16];
    uint32_t    sourceIPv4Address;
    uint32_t    destinationIPv4Address;
    uint16_t    sourceTransportPort;
    uint16_t    destinationTransportPort;
    uint8_t     protocolIdentifier;
    uint8_t     flowEndReason;
#   if defined(YAF_ENABLE_APPLABEL) && !defined(YAF_ENABLE_ENTROPY)
    uint16_t    silkAppLabel;
    uint8_t     paddingOctets[4];
#   elif defined(YAF_ENABLE_ENTROPY) && !defined(YAF_ENABLE_APPLABEL)
    uint8_t     entropy;
    uint8_t     reverseEntropy;
    uint8_t     paddingOctets[4];
#   elif defined(YAF_ENABLE_ENTROPY) && defined(YAF_ENABLE_APPLABEL)
    uint16_t    silkAppLabel;
    uint8_t     entropy;
    uint8_t     reverseEntropy;
    uint16_t    paddingOctets;
#   else
    uint8_t     paddingOctets[6];
#   endif
    int32_t     reverseFlowDeltaMilliseconds;
    uint32_t    tcpSequenceNumber;
    uint32_t    reverseTcpSequenceNumber;
    uint8_t     initialTCPFlags;
    uint8_t     unionTCPFlags;
    uint8_t     reverseInitialTCPFlags;
    uint8_t     reverseUnionTCPFlags;
    uint16_t    vlanId;
    uint16_t    reverseVlanId;
#   if YAF_ENABLE_PAYLOAD
    fbVarfield_t payload;
    fbVarfield_t reversePayload;
#   endif
} yfIpfixFlow_t;

typedef struct yfIpfixExtFlow_st {
    yfIpfixFlow_t   f;
    uint64_t    flowStartMicroseconds;
    uint64_t    flowEndMicroseconds;
    uint32_t    flowStartSeconds;
    uint32_t    flowEndSeconds;
    uint32_t    flowDurationMicroseconds;
    uint32_t    flowDurationMilliseconds;
    uint32_t    flowStartDeltaMicroseconds;
    uint32_t    flowEndDeltaMicroseconds;
    uint64_t    octetDeltaCount;
    uint64_t    packetDeltaCount;
} yfIpfixExtFlow_t;

/* Core library configuration variables */
static gboolean yaf_core_export_payload = FALSE;
static gboolean yaf_core_map_ipv6 = FALSE;

/**
 * yfAlignmentCheck
 *
 * this checks the alignment of the template and corresponding record
 * ideally, all this magic would happen at compile time, but it 
 * doesn't currently, (can't really do it in C,) so we do it at
 * run time
 *
 *
 * @param err a Glib error structure pointer initialized with an
 *        empty error on input, if an alignment error is detected
 *        then a new error will be put into the pointer.
 *
 */
void yfAlignmentCheck()
{
    size_t prevOffset = 0;
    size_t prevSize = 0;

#define DO_SIZE(S_,F_) (SIZE_T_CAST)sizeof(((S_ *)(0))->F_)
#define EA_STRING(S_,F_) "alignment error in struct " #S_ " for element "   \
                         #F_ " offset %#"SIZE_T_FORMATX" size %"            \
                         SIZE_T_FORMAT" (pad %"SIZE_T_FORMAT")",            \
                         (SIZE_T_CAST)offsetof(S_,F_), DO_SIZE(S_,F_),      \
                         (SIZE_T_CAST)(offsetof(S_,F_) % DO_SIZE(S_,F_)) 
#define EG_STRING(S_,F_) "gap error in struct " #S_ " for element " #F_     \
                         " offset %#"SIZE_T_FORMATX" size %"SIZE_T_FORMAT,  \
                         (SIZE_T_CAST)offsetof(S_,F_),                      \
                         DO_SIZE(S_,F_) 
#define RUN_CHECKS(S_,F_,A_) {                                          \
        if (((offsetof(S_,F_) % DO_SIZE(S_,F_)) != 0) && A_) {          \
            g_error(EA_STRING(S_,F_));                                  \
        }                                                               \
        if (offsetof(S_,F_) != (prevOffset+prevSize)) {                 \
            g_error(EG_STRING(S_,F_));                                      \
            return;                                                     \
        }                                                               \
        prevOffset = offsetof(S_,F_);                                   \
        prevSize = DO_SIZE(S_,F_);                                      \
        /* fprintf(stderr, "%17s %30s %#5x %3d %#5x\n", #S_, #F_,       \
                offsetof(S_,F_), DO_SIZE(S_,F_),                        \
                offsetof(S_,F_)+DO_SIZE(S_,F_)); */                     \
    }
    
    RUN_CHECKS(yfIpfixFlow_t,flowStartMilliseconds,1);
    RUN_CHECKS(yfIpfixFlow_t,flowEndMilliseconds,1);
    RUN_CHECKS(yfIpfixFlow_t,octetTotalCount,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseOctetTotalCount,1);
    RUN_CHECKS(yfIpfixFlow_t,packetTotalCount,1);
    RUN_CHECKS(yfIpfixFlow_t,reversePacketTotalCount,1);
    RUN_CHECKS(yfIpfixFlow_t,sourceIPv6Address,1);
    RUN_CHECKS(yfIpfixFlow_t,destinationIPv6Address,1);
    RUN_CHECKS(yfIpfixFlow_t,sourceIPv4Address,1);
    RUN_CHECKS(yfIpfixFlow_t,destinationIPv4Address,1);
    RUN_CHECKS(yfIpfixFlow_t,sourceTransportPort,1);
    RUN_CHECKS(yfIpfixFlow_t,destinationTransportPort,1);
    RUN_CHECKS(yfIpfixFlow_t,protocolIdentifier,1);
    RUN_CHECKS(yfIpfixFlow_t,flowEndReason,1);
#   if defined(YAF_ENABLE_APPLABEL) && !defined(YAF_ENABLE_ENTROPY)
    RUN_CHECKS(yfIpfixFlow_t,silkAppLabel,1);
    RUN_CHECKS(yfIpfixFlow_t,paddingOctets,0);
#   elif defined(YAF_ENABLE_ENTROPY) && !defined(YAF_ENABLE_APPLABEL)
    RUN_CHECKS(yfIpfixFlow_t,entropy,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseEntropy,1);
    RUN_CHECKS(yfIpfixFlow_t,paddingOctets,0);
#   elif defined(YAF_ENABLE_ENTROPY) && defined(YAF_ENABLE_APPLABEL)
    RUN_CHECKS(yfIpfixFlow_t,silkAppLabel,1);
    RUN_CHECKS(yfIpfixFlow_t,entropy,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseEntropy,1);
    RUN_CHECKS(yfIpfixFlow_t,paddingOctets,0);
#   else
    RUN_CHECKS(yfIpfixFlow_t,paddingOctets,0);
#   endif
    RUN_CHECKS(yfIpfixFlow_t,reverseFlowDeltaMilliseconds,1);
    RUN_CHECKS(yfIpfixFlow_t,tcpSequenceNumber,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseTcpSequenceNumber,1);
    RUN_CHECKS(yfIpfixFlow_t,initialTCPFlags,1);
    RUN_CHECKS(yfIpfixFlow_t,unionTCPFlags,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseInitialTCPFlags,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseUnionTCPFlags,1);
    RUN_CHECKS(yfIpfixFlow_t,vlanId,1);
    RUN_CHECKS(yfIpfixFlow_t,reverseVlanId,1);
#   if YAF_ENABLE_PAYLOAD
    RUN_CHECKS(yfIpfixFlow_t,payload,0);
    RUN_CHECKS(yfIpfixFlow_t,reversePayload,0);
#   endif

    prevOffset = 0;
    prevSize = 0;

    RUN_CHECKS(yfIpfixExtFlow_t,f,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowStartMicroseconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowEndMicroseconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowStartSeconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowEndSeconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowDurationMicroseconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowDurationMilliseconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowStartDeltaMicroseconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,flowEndDeltaMicroseconds,1);
    RUN_CHECKS(yfIpfixExtFlow_t,octetDeltaCount,1);
    RUN_CHECKS(yfIpfixExtFlow_t,packetDeltaCount,1);


#undef DO_SIZE
#undef EA_STRING
#undef EG_STRING
#undef RUN_CHECKS

}

void yfWriterExportPayload(
	gboolean			payload_mode)
{
	yaf_core_export_payload = payload_mode;
}

void yfWriterExportMappedV6(
	gboolean			map_mode)
{
	yaf_core_map_ipv6 = map_mode;
}


/**
 * yfFlowPrepare
 *
 * initialize the state of a flow to be "clean" so that they
 * can be reused
 *
 */
void yfFlowPrepare(
    yfFlow_t          *flow)
{
#   if YAF_ENABLE_PAYLOAD
    flow->val.paylen = 0;
    flow->val.payload = NULL;
    flow->rval.paylen = 0;
    flow->rval.payload = NULL;
#   endif
}


/**
 *yfFlowCleanup
 *
 * cleans up after a flow is no longer needed by deallocating
 * the dynamic memory allocated to the flow (think payload)
 *
 */
void yfFlowCleanup(
    yfFlow_t          *flow)
{
#if YAF_ENABLE_PAYLOAD
    if (flow->val.payload) {
        g_free(flow->val.payload);
        flow->val.payload = NULL;
    }
    
    if (flow->rval.payload) {
        g_free(flow->rval.payload);
        flow->rval.payload = NULL;
    }
#endif
}

/**
 *yfPayloadCopyIn
 *
 *
 *
 *
 */
static void yfPayloadCopyIn(
    fbVarfield_t     *payvar,
    yfFlowVal_t       *val)
{
#   if YAF_ENABLE_PAYLOAD
    if (payvar->len) {
        if (!val->payload) {
            val->payload = g_malloc0(payvar->len);
        } else {
            val->payload = g_realloc(val->payload, payvar->len);
        }
        val->paylen = payvar->len;
        memcpy(val->payload, payvar->buf, payvar->len);
    } else {
        if (val->payload) g_free(val->payload);
        val->payload = NULL;
        val->paylen = 0;
    }
#   endif
}

/**
 *yfInfoModel
 *
 */
static fbInfoModel_t *yfInfoModel()
{
    static fbInfoModel_t *yaf_model = NULL;

    if (!yaf_model) {
        yaf_model = fbInfoModelAlloc();
        fbInfoModelAddElementArray(yaf_model, yaf_info_elements);
    }
    
    return yaf_model;
}

/**
 *yfInitExporterSession
 *
 *
 */
static fbSession_t *yfInitExporterSession(
    uint32_t        domain,
    GError          **err)
{
    fbInfoModel_t   *model = yfInfoModel();
    fbTemplate_t    *tmpl = NULL;
    fbSession_t     *session = NULL;
    
    /* Allocate the session */
    session = fbSessionAlloc(model);
    
    /* set observation domain */
    fbSessionSetDomain(session, domain);
    
    /* Create the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, yaf_flow_spec, YTF_ALL, err)) 
        return NULL;

    /* Add the full record template to the session */
    if (!fbSessionAddTemplate(session, TRUE, YAF_FLOW_FULL_TID, tmpl, err)) {
        return NULL;
    }
    
    /* Done. Return the session. */
    return session;
}

/**
 *yfWriterForFile
 *
 *
 */
fBuf_t *yfWriterForFile(
    const char              *path,
    uint32_t                domain,
    GError                  **err)
{
    fBuf_t                  *fbuf = NULL;
    fbExporter_t            *exporter;
    fbSession_t             *session;
    
    /* Allocate an exporter for the file */
    exporter = fbExporterAllocFile(path);
    
    /* Create a new buffer */
    if (!(session = yfInitExporterSession(domain, err))) goto err;
    fbuf = fBufAllocForExport(session, exporter);
    
    /* write YAF flow templates */
    if (!fbSessionExportTemplates(session, err)) goto err;
    
    /* set internal template */
    if (!fBufSetInternalTemplate(fbuf, YAF_FLOW_FULL_TID, err)) goto err;
    
    /* all done */
    return fbuf;
    
  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return NULL;
}

/**
 *yfWriterForFP
 *
 *
 *
 */
fBuf_t *yfWriterForFP(
    FILE                    *fp,
    uint32_t                domain,
    GError                  **err)
{
    fBuf_t                  *fbuf = NULL;
    fbExporter_t            *exporter;
    fbSession_t             *session;
    
    /* Allocate an exporter for the file */
    exporter = fbExporterAllocFP(fp);
    
    /* Create a new buffer */
    if (!(session = yfInitExporterSession(domain, err))) goto err;
    fbuf = fBufAllocForExport(session, exporter);
    
    /* write YAF flow templates */
    if (!fbSessionExportTemplates(session, err)) goto err;
    
    /* set internal template */
    if (!fBufSetInternalTemplate(fbuf, YAF_FLOW_FULL_TID, err)) goto err;
    
    /* all done */
    return fbuf;
    
  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return NULL;
}

/**
 *yfWriterForSpec
 *
 *
 *
 */
fBuf_t *yfWriterForSpec(
    fbConnSpec_t            *spec,
    uint32_t                domain,
    GError                  **err)
{
    fBuf_t                  *fbuf = NULL;
    fbSession_t             *session;
    fbExporter_t            *exporter;

    /* initialize session and exporter */
    if (!(session = yfInitExporterSession(domain, err))) goto err;
    exporter = fbExporterAllocNet(spec);
    fbuf = fBufAllocForExport(session, exporter);

    /* set observation domain */
    fbSessionSetDomain(session, domain);

    /* write YAF flow templates */
    if (!fbSessionExportTemplates(session, err)) goto err;

    /* set internal template */
    if (!fBufSetInternalTemplate(fbuf, YAF_FLOW_FULL_TID, err)) goto err;
        
    /* all done */
    return fbuf;
    
  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return NULL;
}

/**
 *yfSetExportTemplate
 *
 *
 *
 */
static gboolean yfSetExportTemplate(
    fBuf_t              *fbuf,
    uint16_t            tid,
    GError              **err)
{
    fbSession_t         *session = NULL;
    fbTemplate_t        *tmpl = NULL;

    /* Try to set export template */
    if (fBufSetExportTemplate(fbuf, tid, err)) {
        return TRUE;
    }
    
    /* Check for error other than missing template */
    if (!g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_TMPL)) {
        return FALSE;
    }
    
    /* Okay. We have a missing template. Clear the error and try to load it. */
    g_clear_error(err);
    session = fBufGetSession(fbuf);
    tmpl = fbTemplateAlloc(yfInfoModel());
    
    if (!fbTemplateAppendSpecArray(tmpl, yaf_flow_spec,
                                   (tid & (~YAF_FLOW_BASE_TID)), err)) {
        return FALSE;
    }
    
    if (!fbSessionAddTemplate(session, FALSE, tid, tmpl, err)) {
        return FALSE;
    }
    
    /* Template should be loaded. Try setting the template again. */
    return fBufSetExportTemplate(fbuf, tid, err);
}

/**
 *yfWriteFlow
 *
 *
 *
 */
gboolean yfWriteFlow(
    fBuf_t              *fbuf,
    yfFlow_t            *flow,
    GError              **err)
{
    yfIpfixFlow_t       rec;
    uint16_t            wtid;
    
    /* copy time */
    rec.flowStartMilliseconds = flow->stime;
    rec.flowEndMilliseconds = flow->etime;
    rec.reverseFlowDeltaMilliseconds = flow->rdtime;

    /* copy addresses */
    if (yaf_core_map_ipv6 && (flow->key.version == 4)) {
        memcpy(rec.sourceIPv6Address, yaf_ip6map_pfx, 
               sizeof(yaf_ip6map_pfx));
        *(uint32_t *)(&(rec.sourceIPv6Address[sizeof(yaf_ip6map_pfx)])) =
            g_htonl(flow->key.addr.v4.sip);
        memcpy(rec.destinationIPv6Address, yaf_ip6map_pfx, 
               sizeof(yaf_ip6map_pfx));
        *(uint32_t *)(&(rec.destinationIPv6Address[sizeof(yaf_ip6map_pfx)])) =
            g_htonl(flow->key.addr.v4.dip);
    } else if (flow->key.version == 4) {
        rec.sourceIPv4Address = flow->key.addr.v4.sip;
        rec.destinationIPv4Address = flow->key.addr.v4.dip;
    } else if (flow->key.version == 6) {
        memcpy(rec.sourceIPv6Address, flow->key.addr.v6.sip, 
               sizeof(rec.sourceIPv6Address));
        memcpy(rec.destinationIPv6Address, flow->key.addr.v6.dip, 
               sizeof(rec.destinationIPv6Address));
    } else {
        g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_ARGUMENT,
                    "Illegal IP version %u", flow->key.version);
    }

    /* copy key and counters */
    rec.sourceTransportPort = flow->key.sp;
    rec.destinationTransportPort = flow->key.dp;
    rec.protocolIdentifier = flow->key.proto;
    rec.flowEndReason = flow->reason;
    rec.tcpSequenceNumber = flow->val.isn;
    rec.reverseTcpSequenceNumber = flow->rval.isn;
    rec.initialTCPFlags = flow->val.iflags;
    rec.unionTCPFlags = flow->val.uflags;
    rec.reverseInitialTCPFlags = flow->rval.iflags;
    rec.reverseUnionTCPFlags = flow->rval.uflags;
    rec.vlanId = flow->val.tag;
    rec.reverseVlanId = flow->rval.tag;
    rec.octetTotalCount = flow->val.oct;
    rec.reverseOctetTotalCount = flow->rval.oct;
    rec.packetTotalCount = flow->val.pkt;
    rec.reversePacketTotalCount = flow->rval.pkt;
    
#if YAF_ENABLE_PAYLOAD
    /* point to payload */
    rec.payload.buf = flow->val.payload;
    rec.payload.len = flow->val.paylen;
    rec.reversePayload.buf = flow->rval.payload;
    rec.reversePayload.len = flow->rval.paylen;
	
	/* copy payload-derived information */
#if YAF_ENABLE_APPLABEL
	rec.silkAppLabel = flow->appLabel;	
#endif
#if YAF_ENABLE_ENTROPY
	rec.entropy = flow->val.entropy;
	rec.reverseEntropy = flow->rval.entropy;
#endif
#endif

    /* select template */
    wtid = YAF_FLOW_BASE_TID;
    if (rec.protocolIdentifier == YF_PROTO_TCP) {
        wtid |= YTF_TCP;
    }

#if YAF_ENABLE_PAYLOAD
    if ((TRUE == yaf_core_export_payload) &&
	    (rec.payload.len || rec.reversePayload.len)) {
        wtid |= YTF_PAY;
    }

#if YAF_ENABLE_APPLABEL
    if (rec.silkAppLabel) {
        wtid |= YTF_APPLBL;
    }
#endif

#if YAF_ENABLE_ENTROPY
    if (rec.entropy || rec.reverseEntropy) {
        wtid |= YTF_ENTROPY;
    }
#endif
#endif

    if (rec.reversePacketTotalCount) {
        wtid |= YTF_BIF;
    }
    if (rec.vlanId) {
        wtid |= YTF_MAC;
    }
    if (rec.octetTotalCount < YAF_RLEMAX && 
        rec.reverseOctetTotalCount < YAF_RLEMAX &&
        rec.packetTotalCount < YAF_RLEMAX &&
        rec.reversePacketTotalCount < YAF_RLEMAX) {
        wtid |= YTF_RLE;
    } else {
        wtid |= YTF_FLE;
    }
    if (yaf_core_map_ipv6 || (flow->key.version == 6)) {
        wtid |= YTF_IP6;
    } else {
        wtid |= YTF_IP4;
    }

    if (!yfSetExportTemplate(fbuf, wtid, err)) {
        return FALSE;
    }
    
    /* Now append the record to the buffer */
    return fBufAppend(fbuf, (uint8_t *)&rec, sizeof(rec), err);

}

/**
 *yfWriterClose
 *
 *
 *
 */
gboolean yfWriterClose(
    fBuf_t          *fbuf,
    gboolean        flush,
    GError          **err)
{
    gboolean        ok = TRUE;
    
    if (flush) {
        ok = fBufEmit(fbuf, err);
    }
    
    fBufFree(fbuf);
    
    return ok;
}
    
/**
 *yfInitCollectorSession
 *
 *
 *
 */
static fbSession_t *yfInitCollectorSession(
    GError          **err)
{
    fbInfoModel_t   *model = yfInfoModel();
    fbTemplate_t    *tmpl = NULL;
    fbSession_t     *session = NULL;
    
    /* Allocate the session */
    session = fbSessionAlloc(model);
    
    /* Add the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, yaf_flow_spec, YTF_ALL, err)) 
        return NULL;
    if (!fbSessionAddTemplate(session, TRUE, YAF_FLOW_FULL_TID, tmpl, err))
        return NULL;
    
    /* Add the extended record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, yaf_flow_spec, YTF_ALL, err)) 
        return NULL;
    if (!fbTemplateAppendSpecArray(tmpl, yaf_extime_spec, YTF_ALL, err)) 
        return NULL;
    if (!fbSessionAddTemplate(session, TRUE, YAF_FLOW_EXT_TID, tmpl, err))
        return NULL;
    
    /* Done. Return the session. */
    return session;
}

/**
 *yfReaderForFP
 *
 *
 *
 */
fBuf_t *yfReaderForFP(
    fBuf_t          *fbuf,
    FILE            *fp,
    GError          **err)
{
    fbSession_t     *session;
    fbCollector_t   *collector;
        
    /* Allocate a collector for the file */
    collector = fbCollectorAllocFP(NULL, fp);
    
    /* Allocate a buffer, or reset the collector */
    if (fbuf) {
        fBufSetCollector(fbuf, collector);
    } else {        
        if (!(session = yfInitCollectorSession(err))) goto err;
        fbuf = fBufAllocForCollection(session, collector);
    }

    /* FIXME do a preread? */
    
    return fbuf;

  err:
    /* free buffer if necessary */
    if (fbuf) fBufFree(fbuf);
    return NULL;
}

/**
 *yfListenerForSpec
 *
 *
 *
 */
fbListener_t *yfListenerForSpec(
    fbConnSpec_t            *spec,
    fbListenerAppInit_fn    appinit,
    fbListenerAppFree_fn    appfree,
    GError                  **err)
{
    fbSession_t     *session;

    if (!(session = yfInitCollectorSession(err))) return NULL;

    return fbListenerAlloc(spec, session, appinit, appfree, err);
}

/**
 *yfReadFlow
 *
 * read an IPFIX record in, with respect to fields YAF cares about
 *
 */
gboolean yfReadFlow(
    fBuf_t          *fbuf,
    yfFlow_t        *flow,
    GError          **err)
{
    yfIpfixFlow_t       rec;
    size_t              len;

    /* read next YAF record */
    len = sizeof(yfIpfixFlow_t);
    if (!fBufSetInternalTemplate(fbuf, YAF_FLOW_FULL_TID, err))
        return FALSE;
    if (!fBufNext(fbuf, (uint8_t *)&rec, &len, err)) 
        return FALSE;

    /* copy time */
    flow->stime = rec.flowStartMilliseconds;
    flow->etime = rec.flowEndMilliseconds;
    flow->rdtime = rec.reverseFlowDeltaMilliseconds;

    /* copy addresses */
    if (rec.sourceIPv4Address || rec.destinationIPv4Address) {
        flow->key.version = 4;
        flow->key.addr.v4.sip = rec.sourceIPv4Address;
        flow->key.addr.v4.dip = rec.destinationIPv4Address;
    } else if (rec.sourceIPv6Address || rec.destinationIPv6Address) {
        flow->key.version = 6;
        memcpy(flow->key.addr.v6.sip, rec.sourceIPv6Address,  
               sizeof(flow->key.addr.v6.sip));
        memcpy(flow->key.addr.v6.dip, rec.destinationIPv6Address, 
               sizeof(flow->key.addr.v6.dip));
    } else {
        /* Hmm. Default to v4 null addressing for now. */
        flow->key.version = 4;
        flow->key.addr.v4.sip = 0;
        flow->key.addr.v4.dip = 0;
    }
    
    /* copy key and counters */
    flow->key.sp = rec.sourceTransportPort;
    flow->key.dp = rec.destinationTransportPort;
    flow->key.proto = rec.protocolIdentifier;
    flow->val.oct = rec.octetTotalCount;
    flow->val.pkt = rec.packetTotalCount;
    flow->val.isn = rec.tcpSequenceNumber;
    flow->val.iflags = rec.initialTCPFlags;
    flow->val.uflags = rec.unionTCPFlags;
    flow->val.tag = rec.vlanId;
    flow->rval.oct = rec.reverseOctetTotalCount;
    flow->rval.pkt = rec.reversePacketTotalCount;
    flow->rval.isn = rec.reverseTcpSequenceNumber;
    flow->rval.iflags = rec.reverseInitialTCPFlags;
    flow->rval.uflags = rec.reverseUnionTCPFlags;
    flow->rval.tag = rec.reverseVlanId;
    flow->reason = rec.flowEndReason;

#if YAF_ENABLE_PAYLOAD
#if YAF_ENABLE_APPLABEL
    /* set the app label if the app labeler engine is available*/
    flow->appLabel = rec.silkAppLabel;
#endif

#if YAF_ENABLE_ENTROPY
    flow->val.entropy = rec.entropy;
    flow->rval.entropy = rec.reverseEntropy;
#endif

    /* copy payload */
    yfPayloadCopyIn(&rec.payload, &flow->val);
    yfPayloadCopyIn(&rec.reversePayload, &flow->rval);
#   endif
    
    return TRUE;
}

/**
 *yfNTPDecode
 *
 * decodes a 64-bit NTP time variable and returns it in terms of
 * milliseconds
 *
 *
 */
static uint64_t yfNTPDecode(
    uint64_t        ntp)
{
    double          dntp;
    uint64_t        millis;
    
    if (!ntp) return 0;
    
    dntp = (ntp & 0xFFFFFFFF00000000LL) >> 32;
    dntp += ((ntp & 0x00000000FFFFFFFFLL) * 1.0) / (2LL << 32);
    millis = dntp * 1000;
    return millis;
}

/**
 *yfReadFlowExtended
 *
 * read an IPFIX flow record in (with respect to fields YAF cares about) using YAF's
 * extended precision time recording
 *
 */
gboolean yfReadFlowExtended(
    fBuf_t                  *fbuf,
    yfFlow_t                *flow,
    GError                  **err)
{
    yfIpfixExtFlow_t        rec;
    size_t                  len;
    
    /* read next YAF record; retrying on missing template or EOF. */
    len = sizeof(yfIpfixExtFlow_t);
    if (!fBufSetInternalTemplate(fbuf, YAF_FLOW_EXT_TID, err))
        return FALSE;

    while(1) {
        if (fBufNext(fbuf, (uint8_t *)&rec, &len, err)) {
            break;
        } else {
            if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_TMPL)) {
                /* try again on missing template */
                g_debug("skipping IPFIX data set: %s", (*err)->message);
                g_clear_error(err);
                continue;
            } else {
                /* real, actual error */
                return FALSE;
            }
        }
    }
    
    /* Run the Gauntlet of Time. */
    if (rec.f.flowStartMilliseconds) {
        flow->stime = rec.f.flowStartMilliseconds;
        if (rec.f.flowEndMilliseconds >= rec.f.flowStartMilliseconds) {
            flow->etime = rec.f.flowEndMilliseconds;
        } else {
            flow->etime = flow->stime + rec.flowDurationMilliseconds;
        }
    } else if (rec.flowStartMicroseconds) {
        /* Decode NTP-format microseconds */
        flow->stime = yfNTPDecode(rec.flowStartMicroseconds);
        if (rec.flowEndMicroseconds >= rec.flowStartMicroseconds) {
            flow->etime = yfNTPDecode(rec.flowEndMicroseconds);
        } else {
            flow->etime = flow->stime + (rec.flowDurationMicroseconds / 1000);
        }
    } else if (rec.flowStartSeconds) {
        /* Seconds? Well. Okay... */
        flow->stime = rec.flowStartSeconds * 1000;
        flow->etime = rec.flowEndSeconds * 1000;
    } else if (rec.flowStartDeltaMicroseconds) {
        /* Handle delta microseconds. */
        flow->stime = fBufGetExportTime(fbuf) * 1000 - 
                      rec.flowStartDeltaMicroseconds / 1000;
        if (rec.flowEndDeltaMicroseconds && 
            rec.flowEndDeltaMicroseconds <= rec.flowStartDeltaMicroseconds) {
            flow->etime = fBufGetExportTime(fbuf) * 1000 - 
                          rec.flowEndDeltaMicroseconds / 1000;
        } else {
            flow->etime = flow->stime + (rec.flowDurationMicroseconds / 1000);
        }
    } else {
        /* Out of time. Use current timestamp, zero duration */
        struct timeval ct;
        g_assert(!gettimeofday(&ct, NULL));
        flow->stime = ((uint64_t)ct.tv_sec * 1000) +
                      ((uint64_t)ct.tv_usec / 1000);
        flow->etime = flow->stime;
    }

    /* copy private time field - reverse delta */
    flow->rdtime = rec.f.reverseFlowDeltaMilliseconds;

    /* copy addresses */
    if (rec.f.sourceIPv4Address || rec.f.destinationIPv4Address) {
        flow->key.version = 4;
        flow->key.addr.v4.sip = rec.f.sourceIPv4Address;
        flow->key.addr.v4.dip = rec.f.destinationIPv4Address;
    } else if (rec.f.sourceIPv6Address || rec.f.destinationIPv6Address) {
        flow->key.version = 6;
        memcpy(flow->key.addr.v6.sip, rec.f.sourceIPv6Address,  
               sizeof(flow->key.addr.v6.sip));
        memcpy(flow->key.addr.v6.dip, rec.f.destinationIPv6Address, 
               sizeof(flow->key.addr.v6.dip));
    } else {
        /* Hmm. Default to v4 null addressing for now. */
        flow->key.version = 4;
        flow->key.addr.v4.sip = 0;
        flow->key.addr.v4.dip = 0;
    }

    /* copy key and counters */
    flow->key.sp = rec.f.sourceTransportPort;
    flow->key.dp = rec.f.destinationTransportPort;
    flow->key.proto = rec.f.protocolIdentifier;
    flow->val.oct = rec.f.octetTotalCount;
    flow->val.pkt = rec.f.packetTotalCount;
    flow->val.isn = rec.f.tcpSequenceNumber;
    flow->val.iflags = rec.f.initialTCPFlags;
    flow->val.uflags = rec.f.unionTCPFlags;
    flow->val.tag = rec.f.vlanId;
    flow->rval.oct = rec.f.reverseOctetTotalCount;
    flow->rval.pkt = rec.f.reversePacketTotalCount;
    flow->rval.isn = rec.f.reverseTcpSequenceNumber;
    flow->rval.iflags = rec.f.reverseInitialTCPFlags;
    flow->rval.uflags = rec.f.reverseUnionTCPFlags;
    flow->rval.tag = rec.f.reverseVlanId;
    flow->reason = rec.f.flowEndReason;
    
    /* Handle delta counters */
    if (!(flow->val.oct)) flow->val.oct = rec.octetDeltaCount;
    if (!(flow->val.pkt)) flow->val.pkt = rec.packetDeltaCount;


#if YAF_ENABLE_APPLABEL
    /* set the app label if the app labeler engine is available*/
    flow->appLabel = rec.f.silkAppLabel;
#endif


#   if YAF_ENABLE_ENTROPY
    flow->val.entropy = rec.f.entropy;
    flow->rval.entropy = rec.f.reverseEntropy;
#   endif

    
#   if YAF_ENABLE_PAYLOAD
    /* copy payload */
    yfPayloadCopyIn(&rec.f.payload, &flow->val);
    yfPayloadCopyIn(&rec.f.reversePayload, &flow->rval);
#   endif
    
    return TRUE;
}

/**
 *yfPrintFlags
 *
 *
 *
 */
static void yfPrintFlags(
    GString             *str,
    uint8_t             flags)
{
    if (flags & YF_TF_ECE) g_string_append_c(str, 'E');
    if (flags & YF_TF_CWR) g_string_append_c(str, 'C');
    if (flags & YF_TF_URG) g_string_append_c(str, 'U');
    if (flags & YF_TF_ACK) g_string_append_c(str, 'A');
    if (flags & YF_TF_PSH) g_string_append_c(str, 'P');
    if (flags & YF_TF_RST) g_string_append_c(str, 'R');
    if (flags & YF_TF_SYN) g_string_append_c(str, 'S');
    if (flags & YF_TF_FIN) g_string_append_c(str, 'F');
    if (!flags) g_string_append_c(str, '0');
}

/**
 *yfPrintString
 *
 *
 *
 */
void yfPrintString(
    GString             *rstr,
    yfFlow_t              *flow)
{
    char                sabuf[AIR_IP6ADDR_BUF_MINSZ], 
                        dabuf[AIR_IP6ADDR_BUF_MINSZ];
    
    /* print start as date and time */
    air_mstime_g_string_append(rstr, flow->stime, AIR_TIME_ISO8601);
    
    /* print end as time and duration if not zero-duration */
    if (flow->stime != flow->etime) {
        g_string_append_printf(rstr, " - ");
        air_mstime_g_string_append(rstr, flow->etime, AIR_TIME_ISO8601_HMS);
        g_string_append_printf(rstr, " (%.3f sec)", 
            (flow->etime - flow->stime) / 1000.0);
    }
    
    /* print protocol and addresses */
    if (flow->key.version == 4) {
        air_ipaddr_buf_print(sabuf, flow->key.addr.v4.sip);
        air_ipaddr_buf_print(dabuf, flow->key.addr.v4.dip);
    } else if (flow->key.version == 6) {
        air_ip6addr_buf_print(sabuf, flow->key.addr.v6.sip);
        air_ip6addr_buf_print(dabuf, flow->key.addr.v6.dip);
    } else {
        sabuf[0] = (char)0;
        dabuf[0] = (char)0;
    }
    switch (flow->key.proto) {
    case YF_PROTO_TCP:
        if (flow->rval.oct) {
            g_string_append_printf(rstr, " tcp %s:%u => %s:%u %08x:%08x ",
                                   sabuf, flow->key.sp, dabuf, flow->key.dp,
                                   flow->val.isn, flow->rval.isn);
        } else {
            g_string_append_printf(rstr, " tcp %s:%u => %s:%u %08x ",
                                   sabuf, flow->key.sp, dabuf, flow->key.dp,
                                   flow->val.isn);
        }
        
        yfPrintFlags(rstr, flow->val.iflags);
        g_string_append_c(rstr,'/');
        yfPrintFlags(rstr, flow->val.uflags);
        if (flow->rval.oct) {
            g_string_append_c(rstr,':');
            yfPrintFlags(rstr, flow->rval.iflags);
            g_string_append_c(rstr,'/');
            yfPrintFlags(rstr, flow->rval.uflags);
        }
        break;
    case YF_PROTO_UDP:
        g_string_append_printf(rstr, " udp %s:%u => %s:%u",
                               sabuf, flow->key.sp, dabuf, flow->key.dp);
        break;        
    case YF_PROTO_ICMP:
        g_string_append_printf(rstr, " icmp [%u:%u] %s => %s", 
                               (flow->key.dp >> 8), (flow->key.dp & 0xFF),
                               sabuf, dabuf);
        break;
    case YF_PROTO_ICMP6:
        g_string_append_printf(rstr, " icmp6 [%u:%u] %s => %s", 
                               (flow->key.dp >> 8), (flow->key.dp & 0xFF),
                               sabuf, dabuf);
        break;
    default:
        g_string_append_printf(rstr, " ip %u %s => %s", 
                               flow->key.proto, sabuf, dabuf);
        break;
    }
    
    
    /* print vlan tags */
    if (flow->val.tag || flow->rval.tag) {
        if (flow->rval.oct) {
            g_string_append_printf(rstr, " vlan %03hx:%03hx",
                flow->val.tag, flow->rval.tag);
        } else {
            g_string_append_printf(rstr, " vlan %03hx",
                flow->val.tag);
        }
    }
    
    /* print flow counters and round-trip time */
    if (flow->rval.pkt) {
        g_string_append_printf(rstr, " (%llu/%llu <-> %llu/%llu) rtt %u ms",
                               (long long unsigned int)flow->val.pkt,
                               (long long unsigned int)flow->val.oct,
                               (long long unsigned int)flow->rval.pkt, 
                               (long long unsigned int)flow->rval.oct,
                               flow->rdtime);
    } else {
        g_string_append_printf(rstr, " (%llu/%llu ->)",
                               (long long unsigned int)flow->val.pkt, 
                               (long long unsigned int)flow->val.oct);
    }
    
    /* end reason flags */
    if ((flow->reason & YAF_END_MASK) == YAF_END_IDLE) 
        g_string_append(rstr," idle");
    if ((flow->reason & YAF_END_MASK) == YAF_END_ACTIVE)
        g_string_append(rstr," active");
    if ((flow->reason & YAF_END_MASK) == YAF_END_FORCED)
        g_string_append(rstr," eof");
    if ((flow->reason & YAF_END_MASK) == YAF_END_RESOURCE) 
        g_string_append(rstr," rsrc");
    
    /* if app label is enabled, print the label */
#   ifdef YAF_ENABLE_APPLABEL
    if (0 != flow->appLabel) {
        g_string_append_printf(rstr, " applabel: %u", flow->appLabel);
    }
#   endif    

    /* if entropy is enabled, print the entropy values */
#   ifdef YAF_ENABLE_ENTROPY
    if (0 != flow->val.entropy || 0 != flow->rval.entropy) {
        g_string_append_printf(rstr, " entropy: %u rev entropy: %u", 
            flow->val.entropy, flow->rval.entropy);
    }
#   endif   

    /* finish line */
    g_string_append(rstr,"\n");

    /* print payload if necessary */
#   if YAF_ENABLE_PAYLOAD
    if (flow->val.payload) {
        air_hexdump_g_string_append(rstr, "  -> ", 
            flow->val.payload, flow->val.paylen);
    }
    if (flow->rval.payload) {
        air_hexdump_g_string_append(rstr, " <-  ", 
            flow->rval.payload, flow->rval.paylen);
    }
#   endif
}

/**
 *yfPrintDelimitedString
 *
 *
 *
 */
void yfPrintDelimitedString(
    GString                 *rstr,
    yfFlow_t                *flow)
{
    char                sabuf[AIR_IP6ADDR_BUF_MINSZ], 
                        dabuf[AIR_IP6ADDR_BUF_MINSZ];
    GString             *fstr = NULL;
    
    /* print time and duration */
    air_mstime_g_string_append(rstr, flow->stime, AIR_TIME_ISO8601);
    g_string_append_printf(rstr, "%s", YF_PRINT_DELIM);
    air_mstime_g_string_append(rstr, flow->etime, AIR_TIME_ISO8601); 
    g_string_append_printf(rstr, "%s%8.3f%s", 
        YF_PRINT_DELIM, (flow->etime - flow->stime) / 1000.0, YF_PRINT_DELIM);

    /* print initial RTT */
    g_string_append_printf(rstr, "%8.3f%s",
        flow->rdtime / 1000.0, YF_PRINT_DELIM);

    /* print five tuple */
    if (flow->key.version == 4) {
        air_ipaddr_buf_print(sabuf, flow->key.addr.v4.sip);
        air_ipaddr_buf_print(dabuf, flow->key.addr.v4.dip);
    } else if (flow->key.version == 6) {
        air_ip6addr_buf_print(sabuf, flow->key.addr.v6.sip);
        air_ip6addr_buf_print(dabuf, flow->key.addr.v6.dip);
    } else {
        sabuf[0] = (char)0;
        dabuf[0] = (char)0;
        
    }
    g_string_append_printf(rstr, "%3u%s%40s%s%5u%s%40s%s%5u%s", 
        flow->key.proto, YF_PRINT_DELIM,
        sabuf, YF_PRINT_DELIM, flow->key.sp, YF_PRINT_DELIM,
        dabuf, YF_PRINT_DELIM, flow->key.dp, YF_PRINT_DELIM);

    /* print tcp flags */
    fstr = g_string_new("");
    yfPrintFlags(fstr, flow->val.iflags);
    g_string_append_printf(rstr, "%8s%s", fstr->str, YF_PRINT_DELIM);
    g_string_truncate(fstr, 0);
    yfPrintFlags(fstr, flow->val.uflags);
    g_string_append_printf(rstr, "%8s%s", fstr->str, YF_PRINT_DELIM);
    g_string_truncate(fstr, 0);
    yfPrintFlags(fstr, flow->rval.iflags);
    g_string_append_printf(rstr, "%8s%s", fstr->str, YF_PRINT_DELIM);
    g_string_truncate(fstr, 0);
    yfPrintFlags(fstr, flow->rval.uflags);
    g_string_append_printf(rstr, "%8s%s", fstr->str, YF_PRINT_DELIM);
    g_string_free(fstr, TRUE);
    
    /* print tcp sequence numbers */
    g_string_append_printf(rstr, "%08x%s%08x%s", 
        flow->val.isn, YF_PRINT_DELIM, flow->rval.isn, YF_PRINT_DELIM);

    /* print vlan tags */
    g_string_append_printf(rstr, "%03hx%s%03hx%s", 
        flow->val.tag, YF_PRINT_DELIM, flow->rval.tag, YF_PRINT_DELIM);
    
    /* print flow counters */
    g_string_append_printf(rstr, "%8llu%s%8llu%s%8llu%s%8llu%s",
        (long long unsigned int)flow->val.pkt, YF_PRINT_DELIM, 
        (long long unsigned int)flow->val.oct, YF_PRINT_DELIM,
        (long long unsigned int)flow->rval.pkt, YF_PRINT_DELIM, 
        (long long unsigned int)flow->rval.oct, YF_PRINT_DELIM);

    /* if app label is enabled, print the label */
#   ifdef YAF_ENABLE_APPLABEL
    g_string_append_printf(rstr, "%5u%s", flow->appLabel, YF_PRINT_DELIM);
#   endif    

    /* if entropy is enabled, print the entropy values */
#   ifdef YAF_ENABLE_ENTROPY
    g_string_append_printf(rstr, "%3u%s%3u%s", 
        flow->val.entropy, YF_PRINT_DELIM, 
        flow->rval.entropy, YF_PRINT_DELIM);
#   endif   


    /* end reason flags */
    if ((flow->reason & YAF_END_MASK) == YAF_END_IDLE) 
        g_string_append(rstr,"idle ");
    if ((flow->reason & YAF_END_MASK) == YAF_END_ACTIVE)
        g_string_append(rstr,"active ");
    if ((flow->reason & YAF_END_MASK) == YAF_END_FORCED)
        g_string_append(rstr,"eof ");
    if ((flow->reason & YAF_END_MASK) == YAF_END_RESOURCE) 
        g_string_append(rstr,"rsrc ");


    
    /* finish line */
    g_string_append(rstr,"\n");
}

/**
 *yfPrint
 *
 *
 *
 */
gboolean yfPrint(
    FILE                *out,
    yfFlow_t            *flow,
    GError              **err)
{
    GString             *rstr = NULL;
    int                 rc = 0;

    rstr = g_string_new("");

    yfPrintString(rstr, flow);

    rc = fwrite(rstr->str, rstr->len, 1, out);
    
    if (rc != 1) {
        g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                    "error printing flow: %s", strerror(errno));
    }
    
    g_string_free(rstr, TRUE);
    
    return (rc == 1);

}

/**
 *yfPrintDelimited
 *
 *
 *
 */
gboolean yfPrintDelimited(
    FILE                *out,
    yfFlow_t            *flow,
    GError              **err)
{
    GString             *rstr = NULL;
    int                 rc = 0;

    rstr = g_string_new("");

    yfPrintDelimitedString(rstr, flow);

    rc = fwrite(rstr->str, rstr->len, 1, out);
    
    if (rc != 1) {
        g_set_error(err, YAF_ERROR_DOMAIN, YAF_ERROR_IO,
                    "error printing delimited flow: %s", strerror(errno));
    }
    
    g_string_free(rstr, TRUE);
    
    return (rc == 1);

}

