/*
 ** yaflush.c
 ** YAF unified flow/flush logic
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2008 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */
 
#define _YAF_SOURCE_
#include "yaflush.h"
#include "yafout.h"
#include "yafstat.h"

gboolean yfProcessPBufRing(
     yfContext_t        *ctx,
     GError             **err)
{
     AirLock             *lock = NULL;
     yfPBuf_t            *pbuf = NULL;
     gboolean            ok = TRUE;
     uint64_t            ctime;

     /* point to lock buffer if we need it */
     if (ctx->cfg->lockmode) {
         lock = &ctx->lockbuf;
     }

    /* Open output if we need to */
    if (!ctx->fbuf) {
        if (!(ctx->fbuf = yfOutputOpen(ctx->cfg, lock, err))) {
            ok = FALSE;
            goto end;
        }
    }

    /* Dump statistics if requested */
    yfStatDumpLoop(ctx);

    /* process packets from the ring buffer */
    while ((pbuf = (yfPBuf_t *)rgaNextTail(ctx->pbufring))) {

        /* Skip time zero packets (these are marked invalid) */
        if (!pbuf->ptime) {
            continue;
        }
        
        /* Add the packet to the flow table */
        yfFlowPBuf(ctx->flowtab, ctx->pbuflen, pbuf);
        
    }

    /* Flush the flow table */
    if (!yfFlowTabFlush(ctx->flowtab, ctx->fbuf, FALSE, err)) {
        ok = FALSE;
        goto end;
    }

    /* Close output file for rotation if necessary */
    if (ctx->cfg->rotate_ms) {
        ctime = yfFlowTabCurrentTime(ctx->flowtab);
        if (ctx->last_rotate_ms) {
            if (ctime - ctx->last_rotate_ms > ctx->cfg->rotate_ms) {
                yfOutputClose(ctx->fbuf, lock, TRUE);
                ctx->fbuf = NULL;
                ctx->last_rotate_ms = ctime;
            }
        } else {
            ctx->last_rotate_ms = ctime;
        }
    }

end:
    return ok;
}

gboolean yfFinalFlush(
    yfContext_t         *ctx,
    gboolean            ok,
    GError              **err)
{
	AirLock				*lock = NULL;
	gboolean			frv;

    /* point to lock buffer if we need it */
     if (ctx->cfg->lockmode) {
         lock = &ctx->lockbuf;
     }

    /* handle final flush and close */
    if (ctx->fbuf) {
        if (ok) {
            /* Flush flow buffer and close output file on successful exit */
            frv = yfFlowTabFlush(ctx->flowtab, ctx->fbuf, TRUE, err);
            yfOutputClose(ctx->fbuf, lock, TRUE);
            if (!frv) {
                ok = FALSE;
            }
        } else {
            /* Just close output file on error */
            yfOutputClose(ctx->fbuf, lock, FALSE);
        }
    }
    
    return ok;
}
