/*
 ** yafout.c
 ** YAF IPFIX file and session output support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */
 
#define _YAF_SOURCE_
#include "yafout.h"
#include <yaf/yafcore.h>
#include <yaf/yaftab.h>
#include <airframe/airutil.h>

fBuf_t *yfOutputOpen(
    yfConfig_t      *cfg,
    AirLock         *lock,
    GError          **err)
{
    GString         *namebuf = NULL;
    fBuf_t          *fbuf = NULL;
    static uint32_t serial = 0;
    
    /* Short-circuit IPFIX output over the wire.
       Get a writer for the given connection specifier. */
    if (cfg->ipfixNetTrans) {
        return yfWriterForSpec(&(cfg->connspec), cfg->odid, err);
    }
    
    /* create a buffer for the output filename */
    namebuf = g_string_new("");

    if (cfg->rotate_ms) {
        /* Output file rotation. 
           Generate a filename by adding a timestamp and serial number 
           to the end of the output specifier. */
        g_string_append_printf(namebuf, "%s-", cfg->outspec);
        air_time_g_string_append(namebuf, time(NULL), AIR_TIME_SQUISHED);
        g_string_append_printf(namebuf, "-%05u.yaf", serial++);
    } else {
        /* No output file rotation. Write to the file named by the output
           specifier. */
        g_string_append_printf(namebuf, "%s", cfg->outspec);
    }

    /* lock, but not stdout */
    if (lock) {
        if (!(((strlen(cfg->outspec) == 1) && cfg->outspec[0] != '-'))) {
            if (!air_lock_acquire(lock, namebuf->str, err)) {
                goto err;
            }
        }
    }
     
    /* start a writer on the file */
    if (!(fbuf = yfWriterForFile(namebuf->str, cfg->odid, err))) {
        goto err;
    }
        
    /* all done */
    goto end;
    
  err:
    if (lock) {
        air_lock_release(lock);
    }
    
  end:
    g_string_free(namebuf, TRUE);
    return fbuf;
}

void yfOutputClose(
    fBuf_t                  *fbuf,
    AirLock                 *lock,
    gboolean                flush)
{
    gboolean                rv = TRUE;
    GError                  *err = NULL;
    
    /* Close writer (this frees the buffer) */
    rv = yfWriterClose(fbuf, flush, &err);
        
    if (!rv) {
        g_critical("Error closing output file: %s", err->message);
    }
    
    /* Release lock */
    if (lock) {
        air_lock_release(lock);
    }
}
