/*
 ** yafrag.c
 ** YAF Active Fragment Table
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */
 
#define _YAF_SOURCE_
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <yaf/picq.h>
#include <yaf/yafrag.h>

#define YF_FRAG_L4H_MAX 60
#define YF_FRAG_TIMEOUT 30000
#define YF_FRAGPRUNE_DELAY 5000

typedef struct yfFragRec_st {
    uint16_t                off;
    uint16_t                len;
} yfFragRec_t;

typedef struct yfFragKey_st {
    uint32_t                ipid;
    yfFlowKey_t             f;
} yfFragKey_t;

typedef struct yfFragNode_st {
    struct yfFragNode_st        *p;
    struct yfFragNode_st        *n;
    struct yfFragTab_st         *tab;
    uint64_t                    last_ctime;
    gboolean                    have_first;
    gboolean                    have_last;
    yfFragKey_t                 key;
    yfTCPInfo_t                 tcpinfo;
    yfL2Info_t                  l2info;
    uint16_t                    iplen;
    GArray                      *records;
    size_t                      paylen;
    size_t                      payoff;
    uint8_t                     *payload;
} yfFragNode_t;  

typedef struct yfFragQueue_st {
    yfFragNode_t      *tail;
    yfFragNode_t      *head;
} yfFragQueue_t;

struct yfFragTab_st {
    /* State */
    uint64_t                    ctime;
    uint64_t                    prunetime;
    GHashTable                  *table;
    yfFragQueue_t               fraq;
    uint32_t                    count;
    yfFragNode_t                *assembled;
    /* Configuration */
    uint32_t                    idle_ms;
    uint32_t                    max_frags;
    uint32_t                    max_payload;
    /* Stats */
    uint32_t                    stat_frags;
    uint32_t                    stat_seqrej;
    uint32_t                    stat_packets;
    uint32_t                    stat_dropped;
    uint32_t                    stat_peak;
};

static uint32_t yfFragKeyHash(
    yfFragKey_t       *key)
{
    if (key->f.version == 4) {
        return key->ipid ^ (key->f.proto << 12) ^ (key->f.version << 4) ^
               key->f.addr.v4.sip ^ key->f.addr.v4.dip;
    } else {
        return key->ipid ^ (key->f.proto << 12) ^ (key->f.version << 4) ^
               *((uint32_t *)&(key->f.addr.v6.sip[0])) ^
               *((uint32_t *)&(key->f.addr.v6.sip[4])) ^
               *((uint32_t *)&(key->f.addr.v6.sip[8])) ^
               *((uint32_t *)&(key->f.addr.v6.sip[12])) ^
               *((uint32_t *)&(key->f.addr.v6.dip[0])) ^
               *((uint32_t *)&(key->f.addr.v6.dip[4])) ^
               *((uint32_t *)&(key->f.addr.v6.dip[8])) ^
               *((uint32_t *)&(key->f.addr.v6.dip[12]));
    }
}

static gboolean yfFragKeyEqual(
    yfFragKey_t       *a,
    yfFragKey_t       *b)
{
    if ((a->f.version == b->f.version) && 
        (a->ipid == b->ipid) && 
        (a->f.proto == b->f.proto)) {
        if ((a->f.version     == 4) &&
            (a->f.addr.v4.sip == b->f.addr.v4.sip) && 
            (a->f.addr.v4.dip == b->f.addr.v4.dip))
        {
            return TRUE;
        } else if ((a->f.version == 6) &&
                   (memcmp(a->f.addr.v6.sip, b->f.addr.v6.sip, 16) == 0) &&
                   (memcmp(a->f.addr.v6.dip, b->f.addr.v6.dip, 16) == 0))
        {
            return TRUE;
        } else {
            return FALSE;
        }
    } else {
        return FALSE;
    }
}

static int yfFragSortByOffset(
    yfFragRec_t          *a,
    yfFragRec_t          *b)
{
    return (int)a->off - (int)b->off;
}

static void yfFragAdd(
    yfFragTab_t         *fragtab,
    yfFragNode_t        *fn,
    yfIPFragInfo_t      *fraginfo,
    uint16_t            iplen,
    const uint8_t       *pkt,
    size_t              caplen)
{
    yfFragRec_t         fr;
    ssize_t             frag_payoff, frag_paylen, frag_payover;
    
    /* append a fragment record */
    fr.off = fraginfo->offset;
    fr.len = iplen - fraginfo->iphlen;
    g_array_append_vals(fn->records, &fr, 1);
    
    /* set first and last flag on the fragment node */
    if (fraginfo->offset == 0) {
        fn->have_first = TRUE;
    }
    
    if (!fraginfo->more) {
        fn->have_last = TRUE;
    }
    
    /* Short-circuit no payload copy */
    if (!fragtab->max_payload) {
        return;
    }
    
    /* Offset in payload buffer is fragment offset plus the transport header */
    frag_payoff = fraginfo->offset + fraginfo->l4hlen;
    
    /* Length of payload is IP length minus headers, capped to caplen */
    frag_paylen = iplen - fraginfo->iphlen - fraginfo->l4hlen;
    if (frag_paylen > caplen) {
        frag_paylen = caplen;
    }
    
    /* Cap payload length to payload buffer length */
    frag_payover = (frag_payoff + frag_paylen) - 
                   (fragtab->max_payload + YF_FRAG_L4H_MAX);
    if (frag_payover > 0) {
        frag_paylen -= frag_payover;
    }
    
    /* Short circuit no payload to copy */
    if (frag_paylen <= 0) {
        return;
    }
    
    /* Copy payload into buffer */
    memcpy(fn->payload + frag_payoff, pkt, frag_paylen);
    
    /* Track payload buffer length */
    if (frag_payoff + frag_paylen > fn->paylen) {
        fn->paylen = frag_payoff + frag_paylen;
    }
}

static yfFragNode_t *yfFragGetNode(
    yfFragTab_t         *fragtab,
    yfFlowKey_t         *flowkey,
    yfIPFragInfo_t      *fraginfo)
{
    yfFragNode_t        *fn;
    yfFragKey_t         fragkey;
    
    /* construct a key to look up the frag node */
    memcpy(&fragkey.f, flowkey, sizeof(*flowkey));
    fragkey.ipid = fraginfo->ipid;

    /* get it out of the fragment table */
    fn = g_hash_table_lookup(fragtab->table, &fragkey);
    if (fn) {
        /* and place it at the head of the fragment queue */
        piqPick(&(fragtab->fraq), fn);
        piqEnQ(&(fragtab->fraq), fn);
        fn->last_ctime = fragtab->ctime;
        return fn;
    }
    
    /* no fragment node available; create a new one */
    fn = yg_slice_new0(yfFragNode_t);
    
    /* fill in the fragment node */
    memcpy(&fn->key, &fragkey, sizeof(fragkey));
    
    /* allocate fragment record array */
    fn->records = g_array_sized_new(FALSE, TRUE, sizeof(yfFragRec_t), 4);
    
    if (fragtab->max_payload) {
        /* create payload buffer, accounting for maximum TCP header size */
        fn->payload = yg_slice_alloc0(fragtab->max_payload + YF_FRAG_L4H_MAX);
    }
    
    /* place it at the head of the fragment queue */
    piqEnQ(&(fragtab->fraq), fn);
    fn->last_ctime = fragtab->ctime;

    /* stick it in the fragment table */
    g_hash_table_insert(fragtab->table, &fn->key, fn);
    ++(fragtab->count);
    if (fragtab->count > fragtab->stat_peak) {
        fragtab->stat_peak = fragtab->count;
    }

    /* return the fragment node */
    return fn;
}

static void yfFragNodeFree(
    yfFragTab_t         *fragtab,
    yfFragNode_t        *fn)
{
    if (fn->payload) {
        yg_slice_free1(fragtab->max_payload + YF_FRAG_L4H_MAX, fn->payload);
    }
    if (fn->records) { 
        g_array_free(fn->records, TRUE);
    }
    yg_slice_free(yfFragNode_t, fn);
}

static void yfFragRemoveNode(
    yfFragTab_t         *fragtab,
    yfFragNode_t        *fn,
    gboolean            drop)
{
    g_hash_table_remove(fragtab->table, &(fn->key));
    piqPick(&(fragtab->fraq), fn);
    --(fragtab->count);
    
    if (drop) {
        ++(fragtab->stat_dropped);
        yfFragNodeFree(fragtab, fn);
    } else {
        ++(fragtab->stat_packets);
        g_assert(fragtab->assembled == NULL);
        fragtab->assembled = fn;
    }
}

static void yfFragComplete(
    yfFragTab_t         *fragtab,
    yfFragNode_t        *fn,
    yfIPFragInfo_t      *fraginfo)
{
    yfFragRec_t                 *frag = NULL;
    uint32_t                    i, next_off;

    /* Short circuit unless we have both the first and last fragment */
    if (!fn->have_first || !fn->have_last) {
        return;
    }
    
    /* Sort the fragment array by offset */
    g_array_sort(fn->records, (GCompareFunc)yfFragSortByOffset);
    
    /* Traverse the fragment array to see if the fragments fit */
    for (i = 0, next_off = 0; i < fn->records->len; i++) {
        frag = &g_array_index(fn->records, yfFragRec_t, i);
        if (frag->off <= next_off) {
            next_off = frag->off + frag->len;
        } else {
            /* Fragment gap. Stop. */
            return;
        }
    }
    
    /* 
     * If we're here, the fragments fit. Calculate total IP length.
     * This is the total off the offsets plus the IP header.
     */ 
    fn->iplen = next_off + fraginfo->iphlen;
    
    /* Stuff the fragment in the assembled buffer. */
    yfFragRemoveNode(fragtab, fn, FALSE);
}

static void yfFragQueuePrune(
    yfFragTab_t     *fragtab,
    gboolean        prune_all)
{

    /* Limit prune rate */
    if (fragtab->prunetime && 
        (fragtab->ctime < fragtab->prunetime + YF_FRAGPRUNE_DELAY))
    {
        return;
    }
    fragtab->prunetime = fragtab->ctime;

    /* remove limited fragments */
    while (fragtab->max_frags && 
           fragtab->fraq.tail && 
           fragtab->count >= fragtab->max_frags) 
    {
        yfFragRemoveNode(fragtab, fragtab->fraq.tail, TRUE);
    }
    
    /* remove expired fragments */
    while (fragtab->fraq.tail && 
           (fragtab->ctime - fragtab->fraq.tail->last_ctime > YF_FRAG_TIMEOUT)) 
    {
        yfFragRemoveNode(fragtab, fragtab->fraq.tail, TRUE);
    }
    
    /* remove all fragments */
    while (prune_all && fragtab->fraq.tail) {
        yfFragRemoveNode(fragtab, fragtab->fraq.tail, TRUE);
    }
}

yfFragTab_t *yfFragTabAlloc(
    uint32_t        idle_ms,
    uint32_t        max_frags,
    uint32_t        max_payload)
{
    yfFragTab_t     *fragtab = NULL;
    
    /* Allocate a fragment table */
    fragtab = yg_slice_new0(yfFragTab_t);
    
    /* Fill in the configuration */
    fragtab->idle_ms = idle_ms;
    fragtab->max_frags = max_frags;
    fragtab->max_payload = max_payload;

    /* Allocate key index table */
    fragtab->table = g_hash_table_new((GHashFunc)yfFragKeyHash, 
                                      (GEqualFunc)yfFragKeyEqual);
                                      
    /* Done */
    return fragtab;
}

void yfFragTabFree(
    yfFragTab_t         *fragtab)
{
    yfFragNode_t        *fn = NULL, *nfn = NULL;
    
    /* zip through the active fragment queue freeing nodes */
    for (fn = fragtab->fraq.head; fn; fn = nfn) {
        nfn = fn->n;
        yfFragNodeFree(fragtab, fn);
    }
    
    /* free the key index table */
    g_hash_table_destroy(fragtab->table);
    
    /* now free the flow table */
    yg_slice_free(yfFragTab_t, fragtab);
}

gboolean yfDefragPBuf(
    yfFragTab_t         *fragtab,
    yfIPFragInfo_t      *fraginfo,
    size_t              pbuflen,
    yfPBuf_t            *pbuf)
{
    yfFragNode_t        *fn;
    yfTCPInfo_t         *tcpinfo = &(pbuf->tcpinfo);
    yfL2Info_t          *l2info = (pbuflen >= YF_PBUFLEN_NOPAYLOAD) ? 
                                    &(pbuf->l2info) : NULL;
    uint8_t             *payload = (pbuflen >= YF_PBUFLEN_BASE) ?
                                    pbuf->payload : NULL;
    size_t              paylen = (pbuflen >= YF_PBUFLEN_BASE) ?
                                    pbuf->paylen : 0;

    /* short-circuit unfragmented packets */
    if (!fraginfo || !fraginfo->frag) {
        return TRUE;
    }
    
    /* reject out-of-sequence fragments and mark them invalid */
    if (pbuf->ptime < fragtab->ctime) {
        ++(fragtab->stat_seqrej);
        pbuf->ptime = 0;
        return FALSE;
    }
    
    /* set fragment table packet clock */
    fragtab->ctime = pbuf->ptime;

    /* get a fragment node and place it at the head of the queue */
    fn = yfFragGetNode(fragtab, &(pbuf->key), fraginfo);

    /* stash information from first fragment */
    if (fraginfo->offset == 0) {
        /* ports */
        fn->key.f.sp = pbuf->key.sp;
        fn->key.f.dp = pbuf->key.dp;
        
        /* Layer 4 info */
        fn->payoff = fraginfo->l4hlen;
        if (tcpinfo && fn->key.f.proto == YF_PROTO_TCP) {
            memcpy(&(fn->tcpinfo), tcpinfo, sizeof(*tcpinfo));
        }
        
        /* Layer 2 info */
        if (l2info) {
            memcpy(&(fn->l2info), l2info, sizeof(*l2info));
        }
    }
    
    /* add the fragment to the fragment node */
    yfFragAdd(fragtab, fn, fraginfo, pbuf->iplen, payload, paylen);
    ++(fragtab->stat_frags);
    
    /* move completed fragments to the assembled buffer */
    yfFragComplete(fragtab, fn, fraginfo);
    
    /* drop expired and limited fragments off the end of the queue */
    yfFragQueuePrune(fragtab, FALSE);
    
    /* return and mark packet invalid if no assembled packet available */
    if (!fragtab->assembled) {
        pbuf->ptime = 0;
        return FALSE;
    }
    
    /* copy assembled packet into packet buffer */
    fn = fragtab->assembled;
    fragtab->assembled = NULL;
    
    /* Copy out pointers to stored fragment information */
    if (pbuflen >= YF_PBUFLEN_BASE && fn->payload) {
        /* Payload is stored at an offset into the payload buffer */
        paylen = fn->paylen - fn->payoff; 
        payload = fn->payload + fn->payoff;
        /* Cap payload length to space available in pbuf */
        if (paylen > pbuflen - YF_PBUFLEN_BASE) {
            paylen = pbuflen - YF_PBUFLEN_BASE;
        }
        /* Now stuff it in the packet buffer. */
        pbuf->paylen = paylen;
        memcpy(pbuf->payload, payload, paylen);
    } 
    
    /* Copy other values from fragment node to packet buffer */
    memcpy(&(pbuf->key), &(fn->key.f), sizeof(yfFlowKey_t));
    pbuf->iplen = fn->iplen;
    memcpy(tcpinfo, &(fn->tcpinfo), sizeof(yfTCPInfo_t));
    if (l2info) {
        memcpy(l2info, &(fn->l2info), sizeof(yfL2Info_t));
    }
    
    /* Free the fragment node */
    yfFragNodeFree(fragtab, fn);
    
    /* All done. Packet buffer is valid. */
    return TRUE;
}

void yfFragDumpStats(
    yfFragTab_t         *fragtab)
{
    g_debug("Assembled %u fragments into %u packets:", 
        fragtab->stat_frags, fragtab->stat_packets);
    g_debug("  Expired %u incomplete fragmented packets.", 
        fragtab->stat_dropped);
    g_debug("  Maximum fragment table size %u.", 
        fragtab->stat_peak);
    if (fragtab->stat_seqrej) {
        g_warning("Rejected %u out-of-sequence fragments.", 
                    fragtab->stat_seqrej);
    }
}

