/*
 ** yafscii.c
 ** YAF flow printer
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <airframe/mio.h>
#include <airframe/mio_config.h>
#include <airframe/mio_source_file.h>
#include <airframe/mio_sink_file.h>
#include <airframe/logconfig.h>
#include <airframe/daeconfig.h>
#include <airframe/airutil.h>
#include <airframe/privconfig.h>
#include <yaf/yafcore.h>

typedef struct ytContext_st {
    fBuf_t          *fbuf;
    yfFlow_t        flow;
} ytContext_t;

static uint32_t     yaft_flows = 0;

static gboolean     yaft_tabular = FALSE;

static uint32_t     yaft_cliflags = MIO_F_CLI_FILE_IN |
                                    MIO_F_CLI_DIR_IN |
                                    MIO_F_CLI_DEF_STDIN |
                                    MIO_F_CLI_FILE_OUT |
                                    MIO_F_CLI_DIR_OUT |
                                    MIO_F_CLI_DEF_STDOUT;

AirOptionEntry yaft_optentries[] = {
    AF_OPTION( "tabular", (char)0, 0, AF_OPT_TYPE_NONE, &yaft_tabular,
      "Print flows in tabular format", NULL ),
    AF_OPTION_END
};

static void ytParseOptions(
    int             *argc,
    char            **argv[]) {

    AirOptionCtx *aoctx = NULL;
    
    aoctx = air_option_context_new("", argc, argv, yaft_optentries);
    
    mio_add_option_group(aoctx, yaft_cliflags);
    daec_add_option_group(aoctx);
    privc_add_option_group(aoctx);
    logc_add_option_group(aoctx, "yafscii", VERSION);
    
    air_option_context_set_help_enabled(aoctx);
    
    air_option_context_parse(aoctx);
}

static gboolean ytOpenSource(
    MIOSource               *source,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    ytContext_t          *yx = (ytContext_t *)vctx;

    /* start reading a YAF file */
    if (!(yx->fbuf = yfReaderForFP(yx->fbuf, mio_fp(source), err))) {
        *flags |= MIO_F_CTL_SOURCECLOSE;
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
        } else {
            *flags |= MIO_F_CTL_ERROR;
        }
        return FALSE;
    }

    return TRUE;
}

static gboolean ytProcess(
    MIOSource               *source,
    MIOSink                 *sink,
    void                    *vctx,
    uint32_t                *flags,
    GError                  **err)
{
    ytContext_t          *yx = (ytContext_t *)vctx;
    gboolean                ok;
    
    /* read a flow */
    if ((ok = yfReadFlowExtended(yx->fbuf, &(yx->flow), err))) {
        ++yaft_flows;
        if (yaft_tabular) {
            ok = yfPrintDelimited(mio_fp(sink), &(yx->flow), err);
        } else {
            ok = yfPrint(mio_fp(sink), &(yx->flow), err);
        }
    }

    /* handle error */
    if (!ok) {
        /* check for EOF */
        if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
            g_clear_error(err);
            *flags |= MIO_F_CTL_SOURCECLOSE;
            ok = TRUE;
        } else {
            /* die on actual error */
            *flags |= MIO_F_CTL_ERROR;
        }
    }
    
    return ok;
}


int main (
    int                 argc,
    char                *argv[]) {

    GError              *err = NULL;
    ytContext_t         yx;
    MIOSource           source;
    MIOSink             sink;
    MIOAppDriver        adrv;
    uint32_t            miodflags;
    int                 rv = 0;
    
    /* parse options */
    ytParseOptions(&argc, &argv);
    
    /* check to make sure the data structures are sane */
    yfAlignmentCheck();
    
    /* set up logging */
    if (!logc_setup(&err)) {
        air_opterr("%s", err->message);
    }
    
    /* fork if necessary */
    if (!daec_setup(&err)) {
        air_opterr("%s", err->message);
    }

    /* initialize MIO option flags */
    miodflags = MIO_F_OPT_SINKLINK;
    
    if (!mio_config_source(&source, yaft_cliflags, &miodflags, &err)) {
        air_opterr("Cannot set up input: %s", err->message);
    }
    
    /* set up sink */
    if (!mio_config_sink(&source, &sink, "%s.yaf.txt", yaft_cliflags, 
                         &miodflags, &err)) {
        air_opterr("Cannot set up output: %s", err->message);
    }

    /* initialize yafscii context */
    yfFlowPrepare(&(yx.flow));
    yx.fbuf = NULL;
    
    /* set up an app driver */
    adrv.app_open_source = ytOpenSource;
    adrv.app_open_sink = NULL;
    adrv.app_close_source = NULL;
    adrv.app_close_sink = NULL;
    adrv.app_process = ytProcess;
       
    /* run dispatch loop */
    if (!mio_dispatch_loop(&source, &sink, &adrv, &yx, miodflags, mio_ov_poll,
                           1, mio_ov_poll))
    {
        rv = 1;
    }

    g_debug("read %u flows", yaft_flows);
    g_message("yafscii terminating");

    return rv;
}
