/*
 ** yaftab.c
 ** YAF Active Flow Table
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU General Public License (GPL) Rights pursuant to Version 2, June 1991
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <airframe/logconfig.h>
#include <airframe/daeconfig.h>
#include <airframe/airutil.h>
#include <yaf/picq.h>
#include <yaf/yaftab.h>
#include <yaf/yafrag.h>

#if YAF_ENABLE_APPLABEL
#include "yafapplabel.h"
#endif

#if YAF_ENABLE_ENTROPY
#include <math.h>
#endif

#ifndef YFDEBUG_FLOWTABLE
#define YFDEBUG_FLOWTABLE 0
#endif

#define YAF_STATE_ACTIVE        0x00000000
#define YAF_STATE_RST           0x00000001
#define YAF_STATE_FFIN          0x00000010
#define YAF_STATE_RFIN          0x00000020
#define YAF_STATE_FFINACK       0x00000040
#define YAF_STATE_RFINACK       0x00000080
#define YAF_STATE_FIN           0x000000F0
#define YAF_STATE_ATO           0x00000100

#define YF_FLUSH_DELAY 5000

typedef struct yfFlowNode_st {
    struct yfFlowNode_st        *p;
    struct yfFlowNode_st        *n;
    struct yfFlowTab_t          *flowtab;
    uint32_t                    state;
    yfFlow_t                    f;
} yfFlowNode_t;

typedef struct yfFlowQueue_st {
    yfFlowNode_t      *tail;
    yfFlowNode_t      *head;
} yfFlowQueue_t;

#if YAF_ENABLE_COMPACT_IP4
/* 
 * Compact IPv4 flow structures; allows the flow table to only allocate enough 
 * space for IPv4 addresses for IPv4 flows. Requires the flow key to be the
 * last element of the flow, and the flow to be the last element of the
 * flow node. ALL CHANGES made to yfFlowKey_t and yfFlow_t in yafcore.h MUST
 * be reflected here or I'll not be held responsible for the results.
 */
    
typedef struct yfFlowKeyIPv4_st {
    uint16_t            sp;
    uint16_t            dp;
    uint8_t             proto;
    uint8_t             version;
    union {
        struct {
            uint32_t    sip;
            uint32_t    dip;
        }               v4;
    }                   addr;
} yfFlowKeyIPv4_t;

typedef struct yfFlowIPv4_st {
    uint64_t        stime;
    uint64_t        etime;
    uint32_t        rdtime;
    uint8_t         reason;
#if YAF_ENABLE_APPLABEL
    uint16_t        appLabel;
#endif
    yfFlowVal_t     val;
    yfFlowVal_t     rval;
    yfFlowKeyIPv4_t key;
} yfFlowIPv4_t;

typedef struct yfFlowNodeIPv4_st {
    struct yfFlowNodeIPv4_st    *p;
    struct yfFlowNodeIPv4_st    *n;
    struct yfFlowTab_t          *flowtab;
    uint32_t                    state;
    yfFlowIPv4_t                f;
} yfFlowNodeIPv4_t;

#endif

struct yfFlowTab_st {
    /* State */
    uint64_t        ctime;
    uint64_t        flushtime;
    GHashTable      *table;
    yfFlowQueue_t   aq;
    yfFlowQueue_t   cq;
    uint32_t        count;
    /* Configuration */
    uint64_t        idle_ms;
    uint64_t        active_ms;
    uint32_t        max_flows;
    uint32_t        max_payload;
    gboolean        uniflow;
    gboolean        silkmode;
    gboolean        applabelmode;
    gboolean        entropymode;
    /* Statistics */
    uint64_t        stat_octets;
    uint64_t        stat_packets;
    uint64_t        stat_seqrej;
    uint64_t        stat_flows;
    uint32_t        stat_peak;
    uint32_t        stat_flush;
};

static uint32_t yfFlowKeyHash(
    yfFlowKey_t       *key)
{
    if (key->version == 4) {
        return (key->sp << 16) ^ key->dp ^ 
               (key->proto << 12) ^ (key->version << 4) ^
               key->addr.v4.sip ^ key->addr.v4.dip;
    } else {
        return (key->sp << 16) ^ key->dp ^ 
               (key->proto << 12) ^ (key->version << 4) ^
               *((uint32_t *)&(key->addr.v6.sip[0])) ^
               *((uint32_t *)&(key->addr.v6.sip[4])) ^
               *((uint32_t *)&(key->addr.v6.sip[8])) ^
               *((uint32_t *)&(key->addr.v6.sip[12])) ^
               *((uint32_t *)&(key->addr.v6.dip[0])) ^
               *((uint32_t *)&(key->addr.v6.dip[4])) ^
               *((uint32_t *)&(key->addr.v6.dip[8])) ^
               *((uint32_t *)&(key->addr.v6.dip[12]));
    }
}

static gboolean yfFlowKeyEqual(
    yfFlowKey_t       *a,
    yfFlowKey_t       *b)
{
    if ((a->sp      == b->sp)    &&
        (a->dp      == b->dp)    && 
        (a->proto   == b->proto) &&
        (a->version == b->version))
    {
        if ((a->version     == 4) &&
            (a->addr.v4.sip == b->addr.v4.sip) && 
            (a->addr.v4.dip == b->addr.v4.dip))
        {
            return TRUE;
        } else if ((a->version == 6) &&
                   (memcmp(a->addr.v6.sip, b->addr.v6.sip, 16) == 0) &&
                   (memcmp(a->addr.v6.dip, b->addr.v6.dip, 16) == 0))
        {
            return TRUE;
        } else {
            return FALSE;
        }
    } else {
        return FALSE;
    }
}

static void yfFlowKeyReverse(
    yfFlowKey_t       *fwd,
    yfFlowKey_t       *rev)
{    
    if (fwd->proto == YF_PROTO_ICMP || fwd->proto == YF_PROTO_ICMP6) {
        rev->sp = fwd->sp;
        rev->dp = fwd->dp;
    } else {
        rev->sp = fwd->dp;
        rev->dp = fwd->sp;
    }
    rev->proto = fwd->proto;
    rev->version = fwd->version;
    if (fwd->version == 4) {
        rev->addr.v4.sip = fwd->addr.v4.dip;
        rev->addr.v4.dip = fwd->addr.v4.sip;
    } else if (fwd->version == 6) {
        memcpy(rev->addr.v6.sip, fwd->addr.v6.dip, 16);
        memcpy(rev->addr.v6.dip, fwd->addr.v6.sip, 16);
    }
}

static void yfFlowKeyCopy(
    yfFlowKey_t       *src,
    yfFlowKey_t       *dst)
{
#if YAF_ENABLE_COMPACT_IP4
    if (src->version == 4) {
        memcpy(dst, src, sizeof(yfFlowKeyIPv4_t));
    } else {
#endif
        memcpy(dst, src, sizeof(yfFlowKey_t));
#if YAF_ENABLE_COMPACT_IP4
    }
#endif
}

#if YFDEBUG_FLOWTABLE == 1
static void yfFlowDebug(
    const char        *msg,
    yfFlow_t          *flow)
{
    static GString      *str = NULL;
    
    if (!str) {
        str = g_string_new("");
    }
    
    g_string_printf(str,"%s ",msg);
    yfPrintString(str, flow);
    g_debug("%s", str->str);
}

static void yfFlowTabVerifyIdleOrder(
    yfFlowTab_t         *flowtab)
{
    yfFlowNode_t        *fn = NULL, *nfn = NULL;
    uint64_t            end;
    uint32_t            i;
    
    /* rip through the active queue making sure end time strictly decreases */
    for (fn = flowtab->aq.head, end = flowtab->aq.head->f.etime, i = 0;
         fn; end = fn->f.etime, fn = nfn, ++i) 
    {
        nfn = fn->n;
        if (end > fn->f.etime) {
            g_debug("Flow inversion in active table position %u; "
                    "last end %llu, end %llu in flow:");
            yfFlowDebug("iiv", &(fn->f));
        }
    }
}
#endif

static void yfFlowFree(
    yfFlowTab_t         *flowtab,
    yfFlowNode_t        *fn) 
{
#if YAF_ENABLE_PAYLOAD
    /* free payload if present */
    if (fn->f.val.payload) {
        yg_slice_free1(flowtab->max_payload, fn->f.val.payload);
    }
    if (fn->f.rval.payload) {
        yg_slice_free1(flowtab->max_payload, fn->f.rval.payload);
    }
#endif

    /* free flow */
#if YAF_ENABLE_COMPACT_IP4
    if (fn->f.key.version == 4) {
        yg_slice_free(yfFlowNodeIPv4_t, (yfFlowNodeIPv4_t *)fn);
    } else {
#endif
        yg_slice_free(yfFlowNode_t, fn);
#if YAF_ENABLE_COMPACT_IP4
    }
#endif
}

static void yfFlowTick(
    yfFlowTab_t                     *flowtab,
    yfFlowNode_t                    *fn)
{
    /* move flow node to head of queue */
    if (flowtab->aq.head != fn) {
        piqPick(&flowtab->aq, fn);
        piqEnQ(&flowtab->aq, fn);
    }
}

#if YAF_ENABLE_APPLABEL
static void yfFlowLabelApp(
    yfFlowTab_t                     *flowtab,
    yfFlowNode_t                    *fn)
{
    /* If the app labeler is enabled, let it inspect the packet 
       (for UDP & TCP packets anyway) */
    if (flowtab->applabelmode == TRUE && 
        ((fn->f.key.proto == 6) || (fn->f.key.proto == 17))) {
        yfAppLabelFlow(&(fn->f));
    } else {
        fn->f.appLabel = 0;
    }
}
#endif

#if YAF_ENABLE_ENTROPY
static void yfFlowDoEntropy(
    yfFlowTab_t             *flowtab,
    yfFlowNode_t            *fn)
{
    uint8_t                 entropyDist[256];
    double                  entropyScratch;
    uint32_t                loop;
    
    /* if entropy is enabled, then we need to calculate it */
    /* FIXME deglobalize */
    if (flowtab->entropymode == TRUE) {
        /* forward entropy */
		if (fn->f.val.paylen) {
			entropyScratch = 0.0;
			memset(entropyDist, 0, 256);
			for (loop = 0; loop < fn->f.val.paylen; loop++) {
				entropyDist[fn->f.val.payload[loop]]++;
			}
			for (loop = 0; loop < 256; loop++) {
				if (0 == entropyDist[loop]) {
					continue;
				}
				entropyScratch += ((double)entropyDist[loop] / 
									(double)fn->f.val.paylen) *
								   (log((double)entropyDist[loop] / 
									(double)fn->f.val.paylen)/log(2.0));
			}
			entropyScratch *= -1;
			fn->f.val.entropy = (uint8_t)((entropyScratch / 8.0)*256.0);
        }
		
        /* reverse entropy */
		if (fn->f.rval.paylen) {
			entropyScratch = 0.0;
			memset(entropyDist, 0, 256);
			for (loop = 0; loop < fn->f.rval.paylen; loop++) {
				entropyDist[fn->f.rval.payload[loop]]++;
			}
			for (loop = 0; loop < 256; loop++) {
				if (0 == entropyDist[loop]) {
					continue;
				}
				entropyScratch += ((double)entropyDist[loop] / 
									(double)fn->f.rval.paylen) *
								   (log((double)entropyDist[loop] / 
									(double)fn->f.rval.paylen)/log(2.0));
			}
			entropyScratch *= -1;
			fn->f.rval.entropy = (uint8_t)((entropyScratch / 8.0)*256.0);
		}
        
    } else {
        fn->f.val.entropy = 0;
        fn->f.rval.entropy = 0;
    }    
}
#endif

static void yfFlowClose(
    yfFlowTab_t                     *flowtab,
    yfFlowNode_t                    *fn,
    uint8_t                         reason)
{
    /* remove flow from table */
    g_hash_table_remove(flowtab->table, &(fn->f.key));
    
    /* store closure reason */
    fn->f.reason &= ~YAF_END_MASK;
    fn->f.reason |= reason;
    
    /* remove flow from active queue */
    piqPick(&flowtab->aq, fn);

    /* move flow node to close queue */
    piqEnQ(&flowtab->cq, fn);

#if YAF_ENABLE_PAYLOAD
#if YAF_ENABLE_APPLABEL
    /* do application label processing if necessary */
    if (flowtab->applabelmode) {
        yfFlowLabelApp(flowtab, fn);
    }
#endif

#if YAF_ENABLE_ENTROPY
    /* do entropy calculation if necessary */
    if (flowtab->entropymode) {
        yfFlowDoEntropy(flowtab, fn);
    }
#endif
#endif

    /* count the flow as inactive */
    --(flowtab->count);
}


yfFlowTab_t *yfFlowTabAlloc(
    uint64_t        idle_ms,
    uint64_t        active_ms,
    uint32_t        max_flows,
    uint32_t        max_payload,
    gboolean        uniflow,
    gboolean        silkmode,
    gboolean        applabelmode,
    gboolean        entropymode)
{
    yfFlowTab_t     *flowtab = NULL;

    /* Allocate a flow table */
    flowtab = yg_slice_new0(yfFlowTab_t);
    
/* FIXME consider a better mode selection interface */
/* FIXME max payload should not be settable if payload not enabled */

    /* Fill in the configuration */
    flowtab->idle_ms = idle_ms;
    flowtab->active_ms = active_ms;
    flowtab->max_flows = max_flows;
    flowtab->max_payload = max_payload;
    flowtab->uniflow = uniflow;
    flowtab->silkmode = silkmode;
    flowtab->applabelmode = applabelmode;
    flowtab->entropymode = entropymode;
    
    /* Allocate key index table */
    flowtab->table = g_hash_table_new((GHashFunc)yfFlowKeyHash, 
                                      (GEqualFunc)yfFlowKeyEqual);

    /* Done */
    return flowtab;
}

void yfFlowTabFree(
    yfFlowTab_t             *flowtab)
{
    yfFlowNode_t            *fn = NULL, *nfn = NULL;
    
    /* zip through the close queue freeing flows */
    for (fn = flowtab->cq.head; fn; fn = nfn) {
        nfn = fn->n;
        yfFlowFree(flowtab, fn);
    }
    
    /* now do the same with the active queue */
    for (fn = flowtab->aq.head; fn; fn = nfn) {
        nfn = fn->n;
        yfFlowFree(flowtab, fn);
    }
    
    /* free the key index table */
    g_hash_table_destroy(flowtab->table);
    
    /* now free the flow table */
    yg_slice_free(yfFlowTab_t, flowtab);
}
    
static yfFlowNode_t *yfFlowGetNode(
    yfFlowTab_t             *flowtab,
    yfFlowKey_t             *key,
    yfFlowVal_t             **valp)
{
    yfFlowKey_t             rkey;
    yfFlowNode_t            *fn;
    
    /* Look for flow in table */
    if ((fn = g_hash_table_lookup(flowtab->table, key))) {
        /* Forward flow found. */
        *valp = &(fn->f.val);
        return fn;
    }
    
    /* Okay. Check for reverse flow. */
    yfFlowKeyReverse(key, &rkey);
    if ((fn = g_hash_table_lookup(flowtab->table, &rkey))) {
        /* Reverse flow found. */
        *valp = &(fn->f.rval);
        return fn;
    }
    
    /* Neither exists. Create a new flow and put it in the table. */
#if YAF_ENABLE_COMPACT_IP4
    if (key->version == 4) {
        fn = (yfFlowNode_t *)yg_slice_new0(yfFlowNodeIPv4_t);
    } else {
#endif
        fn = yg_slice_new0(yfFlowNode_t);
#if YAF_ENABLE_COMPACT_IP4
    }
#endif
    
    /* Copy key */
    yfFlowKeyCopy(key, &(fn->f.key));
        
    /* set flow start time */
    fn->f.stime = flowtab->ctime;
       
    /* stuff the flow in the table */
    g_hash_table_insert(flowtab->table, &(fn->f.key), fn);
        
    /* This is a forward flow */
    *valp = &(fn->f.val);
        
    /* Count it */
    ++(flowtab->count);
    if (flowtab->count > flowtab->stat_peak) {
        flowtab->stat_peak = flowtab->count;
    }

    /* All done */
    return fn;
}

static void yfFlowPktGenericTpt(
    yfFlowTab_t                 *flowtab,
    yfFlowNode_t                *fn,
    yfFlowVal_t                 *val,
    const uint8_t               *pkt,
    uint32_t                    caplen)
{
#if YAF_ENABLE_PAYLOAD
    /* Short-circuit nth packet or no payload capture */
    if (!flowtab->max_payload || val->pkt || !caplen) {
        return;
    }

    /* truncate capture length to payload limit */
    if (caplen > flowtab->max_payload) {
        caplen = flowtab->max_payload;
    }

    /* allocate and copy */
    val->payload = yg_slice_alloc0(flowtab->max_payload);
    val->paylen = caplen;
    memcpy(val->payload, pkt, caplen);
#endif
}

static void yfFlowPktTCP(
    yfFlowTab_t                 *flowtab,
    yfFlowNode_t                *fn,
    yfFlowVal_t                 *val,
    const uint8_t               *pkt,
    uint32_t                    caplen,
    yfTCPInfo_t                 *tcpinfo)
{
    uint32_t                    appdata_po;

    /* Update flags in flow record */
    if (val->pkt) {
        /* Union flags */
        val->uflags |= tcpinfo->flags;
    } else {
        /* Initial flags */
        val->iflags = tcpinfo->flags;
        /* Initial sequence number */
        val->isn = tcpinfo->seq;
    }
    
    /* Update flow state for FIN flag */    
    if (val == &(fn->f.val)) {
        if (tcpinfo->flags & YF_TF_FIN)
            fn->state |= YAF_STATE_FFIN;
        if ((fn->state & YAF_STATE_RFIN) && (tcpinfo->flags & YF_TF_ACK))
            fn->state |= YAF_STATE_FFINACK;
    } else {
        if (tcpinfo->flags & YF_TF_FIN)
            fn->state |= YAF_STATE_RFIN;
        if ((fn->state & YAF_STATE_FFIN) && (tcpinfo->flags & YF_TF_ACK))
            fn->state |= YAF_STATE_RFINACK;
    }
    
    /* Update flow state for RST flag */
    if (tcpinfo->flags & YF_TF_RST) {
        fn->state |= YAF_STATE_RST;
    } 

#if YAF_ENABLE_PAYLOAD
    /* short circuit no payload capture, continuation, 
       payload full, or no payload in packet */
    if (!flowtab->max_payload || !(val->iflags & YF_TF_SYN) || 
        val->paylen == flowtab->max_payload || caplen == 0) {
        return;
    }

    /* Find app data offset in payload buffer */
    appdata_po = tcpinfo->seq - (val->isn + 1);
    
    /* Short circuit entire packet after capture filter */
    if (appdata_po >= flowtab->max_payload) return;

    /* truncate payload copy length to capture length */
    if ((appdata_po + caplen) > flowtab->max_payload) {
        caplen = flowtab->max_payload - appdata_po;
        if (caplen > flowtab->max_payload) {
            caplen = flowtab->max_payload;
        }
    }
     
    /* allocate and copy */
    if (!val->payload) {
        val->payload = yg_slice_alloc0(flowtab->max_payload);
    }
    if (val->paylen < appdata_po + caplen) {
        val->paylen = appdata_po + caplen;
    }
    memcpy(val->payload + appdata_po, pkt, caplen);
#endif
}    

void yfFlowPBuf(
    yfFlowTab_t                 *flowtab,
    size_t                      pbuflen,
    yfPBuf_t                    *pbuf)
{
    yfFlowKey_t                 *key = &(pbuf->key);
    yfFlowVal_t                 *val = NULL;
    yfFlowNode_t                *fn = NULL;
    yfTCPInfo_t                 *tcpinfo = &(pbuf->tcpinfo);
    yfL2Info_t                  *l2info = (pbuflen >= YF_PBUFLEN_NOPAYLOAD) ? 
                                    &(pbuf->l2info) : NULL;
    uint8_t                     *payload = (pbuflen >= YF_PBUFLEN_BASE) ?
                                    pbuf->payload : NULL;
    size_t                      paylen = (pbuflen >= YF_PBUFLEN_BASE) ?
                                    pbuf->paylen : 0;

    /* skip and count out of sequence packets */
    if (pbuf->ptime < flowtab->ctime) {
        ++(flowtab->stat_seqrej);
        return;
    }
    
    /* update flow table current time */
    flowtab->ctime = pbuf->ptime;

    /* Count the packet and its octets */
    ++(flowtab->stat_packets);
    flowtab->stat_octets += pbuf->iplen;

    /* Get a flow node for this flow */
    fn = yfFlowGetNode(flowtab, key, &val);
    
    /* Check for active timeout or counter overflow */
    if (((pbuf->ptime - fn->f.stime) > flowtab->active_ms) ||
        (flowtab->silkmode && (val->oct + pbuf->iplen > UINT32_MAX)))
    {
        yfFlowClose(flowtab, fn, YAF_END_ACTIVE);
        /* get a new flow node containing this packet */
        fn = yfFlowGetNode(flowtab, key, &val);
        /* set continuation flag in silk mode */
        if (flowtab->silkmode) fn->f.reason = YAF_ENDF_ISCONT;
    }
    
    /* Calculate reverse RTT */
    if (val->pkt == 0 && val == &(fn->f.rval)) {
        fn->f.rdtime = pbuf->ptime - fn->f.stime;
    }
    
    /* Do payload and TCP stuff */
    if (fn->f.key.proto == YF_PROTO_TCP) {
        /* Handle TCP flows specially (flags, ISN, sequenced payload) */
        yfFlowPktTCP(flowtab, fn, val, payload, paylen, tcpinfo);
    } else if (val->pkt == 0) {
        /* Get first packet payload from non-TCP flows */
        yfFlowPktGenericTpt(flowtab, fn, val, payload, paylen);
    }
    
    /* Note VLAN tag */
    if (val->pkt == 0) {
        if (l2info) {
            val->tag = l2info->vlan_tag;
        } else {
            val->tag = 0;
        }
    }
                
    /* Count packets and octets */
    val->oct += pbuf->iplen;
    val->pkt += 1;
    
    /* update flow end time */
    fn->f.etime = pbuf->ptime;

    /* close flow, or move it to head of queue */
    if ((fn->state & YAF_STATE_FIN) == YAF_STATE_FIN ||
         fn->state & YAF_STATE_RST)
    {
        yfFlowClose(flowtab, fn, YAF_END_CLOSED);
    } else {
        yfFlowTick(flowtab, fn);
    }
}

static void yfUniflow(
    yfFlow_t          *bf,
    yfFlow_t          *uf)
{
    memcpy(uf, bf, sizeof(yfFlow_t));
    memset(&(uf->rval), 0, sizeof(yfFlowVal_t));
    uf->rdtime = 0;
}

static gboolean yfUniflowReverse(
    yfFlow_t          *bf,
    yfFlow_t          *uf)
{
    if (!(bf->rval.pkt)) return FALSE;

    /* calculate reverse time */
    uf->stime = bf->stime + bf->rdtime;
    uf->etime = bf->etime;
    uf->rdtime = 0;

    /* reverse key */
    yfFlowKeyReverse(&bf->key, &uf->key);
    
    /* copy and reverse value */
    memcpy(&(uf->val), &(bf->rval), sizeof(yfFlowVal_t));
    memset(&(uf->rval), 0, sizeof(yfFlowVal_t));

    /* copy reason */
    uf->reason = bf->reason;
    
    /* all done */
    return TRUE;
}

gboolean yfFlowTabFlush(
    yfFlowTab_t     *flowtab,
    fBuf_t          *fbuf,
    gboolean        close,
    GError          **err)
{
    gboolean        wok = TRUE;
    yfFlowNode_t    *fn = NULL;
    yfFlow_t        uf;

    if (!close && flowtab->flushtime && 
        (flowtab->ctime < flowtab->flushtime + YF_FLUSH_DELAY))
    {
        return TRUE;
    }

    flowtab->flushtime = flowtab->ctime;
    
    /* Count the flush */
    ++flowtab->stat_flush;
    
    /* Verify flow table order */
    /* yfFlowTabVerifyIdleOrder(flowtab); */

    /* close idle flows */
    while (flowtab->aq.tail && 
           (flowtab->ctime - flowtab->aq.tail->f.etime > flowtab->idle_ms)) 
    {
        yfFlowClose(flowtab, flowtab->aq.tail, YAF_END_IDLE);
    }

    /* close limited flows */
    while (flowtab->max_flows && 
           flowtab->aq.tail && 
           flowtab->count >= flowtab->max_flows) 
    {
        yfFlowClose(flowtab, flowtab->aq.tail, YAF_END_RESOURCE);
    }

    /* close all flows if flushing all */
    while (close && flowtab->aq.tail) {
        yfFlowClose(flowtab, flowtab->aq.tail, YAF_END_FORCED);
    }

    /* flush flows from close queue */
    while ((fn = piqDeQ(&flowtab->cq))) {                
        /* write flow */
        if (flowtab->uniflow) {
            /* Uniflow mode. Split flow in two and write. */
            yfUniflow(&(fn->f), &uf);
            wok = yfWriteFlow(fbuf, &uf, err);
            if (wok) {
                ++(flowtab->stat_flows);
            }
            if (wok && yfUniflowReverse(&(fn->f), &uf)) {
                wok = yfWriteFlow(fbuf, &uf, err);
                if (wok) {
                    ++(flowtab->stat_flows);
                }
            }
        } else {
            /* Biflow mode. Write flow whole. */
            wok = yfWriteFlow(fbuf, &(fn->f), err);
            if (wok) {
                ++(flowtab->stat_flows);
            }
        }
        
        /* free it */
        yfFlowFree(flowtab, fn);
        
        /* return error if necessary */
        if (!wok) return wok;
    }

    return TRUE;
}

uint64_t yfFlowTabCurrentTime(
    yfFlowTab_t     *flowtab)
{
    return flowtab->ctime;
}

void yfFlowDumpStats(
    yfFlowTab_t     *flowtab,
    GTimer          *timer) 
{
    g_debug("Processed %llu packets into %llu flows:", flowtab->stat_packets, flowtab->stat_flows);
    if (timer) {
        g_debug("  Mean flow rate %.2f/s.",
            (double)flowtab->stat_flows / g_timer_elapsed(timer, NULL));
        g_debug("  Mean packet rate %.2f/s.",
            (double)flowtab->stat_packets / g_timer_elapsed(timer, NULL));
        g_debug("  Virtual bandwidth %.4f Mbps.",
            (((double)flowtab->stat_octets * 8.0) / 1000000) / 
               g_timer_elapsed(timer, NULL));
    }
    g_debug("  Maximum flow table size %u.", flowtab->stat_peak);
    g_debug("  %u flush events.", flowtab->stat_flush);
    if (flowtab->stat_seqrej) {
        g_warning("Rejected %"PRIu64" out-of-sequence packets.", 
                    flowtab->stat_seqrej);
    }
}
