/*
 *  Copyright 2007-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  imapplugin.c
 *
 *  this provides IMAP payload packet recognition for use within YAF
 *  It is based on RFC 3501 and RFC 9051.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Steven Ibarra
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.17
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0934
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <payloadScanner.h>
#include "libmidencryption.h"

#if YAF_ENABLE_HOOKS
#include <yaf/yafhooks.h>
#endif

#define IMAPDEBUG 0
#define IMAP_PORT_NUMBER 143

YC_SCANNER_PROTOTYPE(imapplugin_LTX_ycImapScanScan);

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre *imapRegexApplabel = NULL;
#if YAF_ENABLE_HOOKS
static pcre *imapCapabilityRegex  = NULL;
static pcre *imapLoginRegex = NULL;
static pcre *imapStartTLSRegex = NULL;
static pcre *imapAuthenticateRegex = NULL;
static pcre *imapCommandRegex = NULL;
static pcre *imapExistsRegex = NULL;
static pcre *imapRecentRegex = NULL;
static pcre *imapSTLSReply = NULL;
#endif /* if YAF_ENABLE_HOOKS */

/* pcreInitialized is 1 if initialization succeeded, -1 if initialization
 * failed, and 0 if initialization has not been attempted */
static int pcreInitialized = 0;


/**
 * static local functions
 *
 */

static gboolean
ycImapScanInit(
    void);

#if IMAPDEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size);
#endif /* if IMAPDEBUG */

/**
 * imapplugin_LTX_ycImapScanScan
 *
 * scans a given payload to see if it conforms to our idea of what IMAP traffic
 * looks like.
 *
 *
 *
 * @param argc NOT USED
 * @param argv NOT USED
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match IMAP_PORT_NUMBER (110) for a match
 *
 */
uint16_t
imapplugin_LTX_ycImapScanScan(
    int             argc,
    char           *argv[],
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    int rc;
#define NUM_CAPT_VECTS 60
    int vects[NUM_CAPT_VECTS];

    if (1 != pcreInitialized) {
        if (-1 != pcreInitialized) {
            pcreInitialized = ycImapScanInit() ? 1 : -1;
        }
        if (-1 == pcreInitialized) {
            return 0;
        }
    }

    rc = pcre_exec(imapRegexApplabel, NULL, (char *)payload, payloadSize, 0,
                   0, vects, NUM_CAPT_VECTS);
    if (rc <= 0) {
        return 0;
    }

#if YAF_ENABLE_HOOKS
    yfHookScanPayload(flow, payload, payloadSize, imapCapabilityRegex, 0,
                      YF_IMAP_CAPABILITY, IMAP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, imapLoginRegex, 0,
                      YF_IMAP_LOGIN, IMAP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, imapAuthenticateRegex, 0,
                      YF_IMAP_AUTHENTICATE, IMAP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, imapCommandRegex, 0,
                      YF_IMAP_COMMAND, IMAP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, imapExistsRegex, 0,
                      YF_IMAP_EXIST, IMAP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, imapRecentRegex, 0,
                      YF_IMAP_RECENT, IMAP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, imapStartTLSRegex, 0,
                      YF_IMAP_STARTTLS, IMAP_PORT_NUMBER);

    if (rc == 2) {
        int      tmp;
        uint32_t offset;

        tmp = pcre_exec(imapSTLSReply, NULL, (char *)payload, payloadSize, 0,
                        0, vects, NUM_CAPT_VECTS);
        offset = vects[1];

        if (tmp == 1) {
            yfHookScanPayload(flow, payload, payloadSize, imapSTLSReply, 0,
                              YF_IMAP_STLS_REPLY, IMAP_PORT_NUMBER);
            Mid_Encryption_Scan_Payload(payload, payloadSize, flow, val,
                                        offset, IMAP_PORT_NUMBER);
        }
    } else {
        int      tmp;
        uint32_t offset;

        /* client side */

        tmp = pcre_exec(imapStartTLSRegex, NULL, (char *)payload, payloadSize,
                        0, 0, vects, NUM_CAPT_VECTS);
        offset = vects[1];

        if (tmp == 1) {
            Mid_Encryption_Scan_Payload(payload, payloadSize, flow, val,
                                        offset, IMAP_PORT_NUMBER);
        }
    }
#endif /* if YAF_ENABLE_HOOKS */

    return IMAP_PORT_NUMBER;
}


/**
 * ycImapScanInit
 *
 * this initializes the PCRE expressions needed to search the payload for
 * IMAP
 *
 *
 * @return TRUE if initialization completed correctly, FALSE otherwise
 */
static gboolean
ycImapScanInit(
    void)
{
    GError *err = NULL;
#if YAF_ENABLE_HOOKS

    const char imapCapabilityString[] = "(?m)CAPABILITY ([-a-zA-Z0-9+= ]+)\\b";
    const char imapLoginString[] = "(?m)LOGIN ([a-zA-Z0-9\" ]+)\\b";
    const char imapStartTLSString[] = "(?m)(STARTTLS)\\r\\n";

    const char imapAuthenticateString[] =
        "(?m)AUTHENTICATE ([a-zA-Z0-9]+)\\b\\r\\n";

    const char imapCommandString[] =
        "(?m)((SELECT|COPY|EXAMINE|CREATE|DELETE|RENAME|"
        "(UN)?SUBSCRIBE|LIST|LSUB|STATUS|APPEND)"
        " [-a-zA-Z0-9/.:()\" *\\\\]+)";

    const char  imapExistsString[] = "(?m)^\\x2A? ?([0-9]* EXISTS)\\b";
    const char  imapRecentString[] = "(?m)^\\x2A? ?([0-9]* RECENT)\\b";
    const char  imapSTLSReplyString[] = "OK Begin TLS negotiation[ -~]*\\r\\n";
#endif  /* YAF_ENABLE_HOOKS */

    /* used to determine if this connection looks like IMAP; capture the
     * response to distinguish the server from the client */
    const char imapStringApplabel[] =
        "(?m)(?i)^\\s*(.{1,7}\\s"
        "(CAPABILITY|AUTHENTICATE|STARTTLS|LOGIN|FETCH)|\\*\\s.*IMAP)";

    if (!ycPcreCompile(&imapRegexApplabel, imapStringApplabel, 0, &err)) {
        goto ERROR;
    }

#if YAF_ENABLE_HOOKS
    if (!ycPcreCompile(&imapCapabilityRegex, imapCapabilityString, 0, &err) ||
        !ycPcreCompile(&imapLoginRegex, imapLoginString, 0, &err) ||
        !ycPcreCompile(&imapStartTLSRegex, imapStartTLSString, 0, &err) ||
        !ycPcreCompile(&imapAuthenticateRegex, imapAuthenticateString, 0,
                       &err) ||
        !ycPcreCompile(&imapCommandRegex, imapCommandString, 0, &err) ||
        !ycPcreCompile(&imapExistsRegex, imapExistsString, 0, &err) ||
        !ycPcreCompile(&imapRecentRegex, imapRecentString, 0, &err) ||
        !ycPcreCompile(&imapSTLSReply, imapSTLSReplyString, 0, &err))
    {
        goto ERROR;
    }
#endif  /* YAF_ENABLE_HOOKS */

    return TRUE;
  ERROR:
    g_warning("%s", err->message);
    g_clear_error(&err);
    return FALSE;
}


#if IMAPDEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size)
{
    uint16_t loop;
    int      numPrinted = 0;

    for (loop = 0; loop < size; loop++) {
        if (isprint(*(data + loop)) && !iscntrl(*(data + loop))) {
            printf("%c", *(data + loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data + loop) || '\r' == *(data + loop)
            || '\0' == *(data + loop))
        {
            break;
        }
        numPrinted++;
    }

    return numPrinted;
}
#endif /* if IMAPDEBUG */
