/*
 *  Copyright 2007-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  libmidencryption.c
 *
 *
 *  This recognizes SSL & TLS packets.  Used by TLS streams and streams that
 *  support upgrading to a encrypted stream (mid-encryption) with StartTLS or
 *  similar commands.
 *
 *  Remember to update proxyplugin.c with any changes.
 *  ------------------------------------------------------------------------
 *  Authors: Chris Inacio, Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.17
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0934
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <payloadScanner.h>
#include "libmidencryption.h"

#if YAF_ENABLE_HOOKS
#include <yaf/yafhooks.h>
#endif

/*
 *
 *  Best history of SSL/TLS is from Wikipedia:
 *  https://en.wikipedia.org/wiki/Transport_Layer_Security#History_and_development
 *
 *  SSLv2 was released in 1995, was quickly discovered to have security and
 *  usablity flaws, leading to the release of SSLv3 in 1996.
 *
 *  Some clients continued to use a SSLv2-style client hello but specified the
 *  protocol version as SSLv3 or TLSv1 within the Handshake message.
 *
 *  This code allows for a SSLv2 client hello but assumes all other messages
 *  are SSLv3 or TLSv1 (which are largely compatible).
 *
 *  Additional links for TLS and SSL:
 *
 *  SSLv2 (1995,
 *  https://datatracker.ietf.org/doc/html/draft-hickman-netscape-ssl-00) has
 *  been deprecated since 2011 (RFC6176,
 *  https://www.rfc-editor.org/rfc/rfc6176).
 *
 *  SSLv3 (1996, RFC6101, https://www.rfc-editor.org/rfc/rfc6101) is the basis
 *  for TLS-1.0 (1999, RFC2246, https://www.rfc-editor.org/rfc/rfc2246).
 *
 *  TLS 1.1 (2006, RFC4346, https://www.rfc-editor.org/rfc/rfc4346) was
 *  deprecated around 2020.
 *
 *  TLS 1.2 (2008, RFC5246, https://www.rfc-editor.org/rfc/rfc5246)
 *
 *  TLS 1.3 (2018, RFC8446, https://www.rfc-editor.org/rfc/rfc8446)
 *
 *
 *
 *  Great resources for TLS 1.2 and 1.3:
 *  Illustrated TLS 1.2 - Every byte explained:  https://tls12.xargs.org/
 *  Illustrated TLS 1.3 - Every byte explained:  https://tls13.xargs.org/
 *
 *  SSLv3 and TLS blocks begin with five bytes:
 *
 *    uint8_t       ContentType;
 *    uint16_t      ProtocolVersion;
 *    uint16_t      length;
 *
 *  ProtocolVersion contains major version in most significant byte and minor
 *  version in LSB.  For TLS, MSB is 0x03.
 *
 *  Note: ProtocolVersion is above is often 0x0301 for maximum compatibility,
 *  while the ProtocolVersion within the TLS_HANDSHAKE message (below) is the
 *  requested/selected version.
 *
 *  There are four content types:
 *
 *    TLS_CHANGE_CIPHER_SPEC   0x14
 *    TLS_ALERT                0x15
 *    TLS_HANDSHAKE            0x16
 *    TLS_APPLICATION_DATA     0x17
 *
 *  The handshake messages contain the data we want to extract.
 *
 *  When ContentType is TLS_HANDSHAKE on connection initiation, the next
 *  bytes are:
 *
 *    uint8_t       HandshakeType;
 *    uint24_t      length;
 *    uint16_t      ProtocolVersion;
 *    uint8_t[32]   Random;
 *    uint8_t       SessionIdLength;    // may be zero
 *    [...]         SessionIdContent;
 *
 *  If HandshakeType is HS_CLIENT_HELLO, the remaining bytes are
 *
 *    uint16_t      CipherLen;
 *    uint16_t[..]  Ciphers;            // an array of uint16_t
 *    uint8_t       CompressionLen;     // may be zero
 *    [...]         CompressionContent;
 *    [...]         Extensions;
 *
 *  If HandshakeType is HS_SERVER_HELLO, the remaining bytes are
 *
 *    uint16_t      ChosenCipher;
 *    uint8_t       ChosenCompression;
 *    [...]         Extensions;
 *
 *  Where Extensions, if present, is
 *
 *    uint16_t      ExtensionsLength;
 *    [...]         ExtensionList;
 *
 *  where each entry in ExtensionList begins with
 *
 *    uint16_t      ExtensionId;
 *    uint16_t      length;
 *    [...]         ExtensionContent;
 */

/*
 *  To support an SSLv2 client hello, here is information for SSLv2.
 *  https://datatracker.ietf.org/doc/html/draft-hickman-netscape-ssl-00
 *
 *  SSLv2 blocks begin with a two-byte length that may be encoded in either
 *  two or three bytes.
 *
 *  Two-byte length
 *
 *  The two-byte encoding of length has the most significant bit set to high,
 *  and the length is given by:
 *
 *  len = ( ((data[offset] & 0x7f) << 8) | data[offset + 1] )
 *
 *  Three-byte length
 *
 *  The three-byte encoding of length has the most significant bit set to low,
 *  and still uses two bytes to specify the length:
 *
 *  len = ( ((data[offset] & 0x3f) << 8) | data[offset + 1] )
 *
 *  The third byte (data[offset + 2]), specifies the amount of padding that
 *  has been added to the record.  Padding is added to make records a multiple
 *  of the cipher-block size when the message is encrypted using a
 *  block-cipher.  The padding length is included in the overall record
 *  length.  The padding must be ignored.
 *
 *  If the second most significant bit of the length is high, that
 *  indicates a security escape message, though none(?) are defined.
 *
 *  is_escape = ((data[offset] & 0x40) != 0)
 *
 *  Client Hello
 *
 *  Following the length is the Client Hello:
 *
 *    uint8_t       HandshakeType;      // HS_CLIENT_HELLO
 *    uint16_t      ProtocolVersion;    // 0x0300 <= proto <= 0x0302
 *    uint16_t      CipherLen;          // must be > 0 and a multiple of 3
 *    uint16_t      SessionIdLen;       // must be 0 or 16
 *    uint16_t      ChallengeLen;       // must be 16 <= len <= 32
 *    uint24_t[..]  Ciphers;            // an array of uint24_t
 *    [...]         SessionIdContent;
 *    [...]         ChallengeContent;
 */

/* TLS Content Types */
#define TLS_CHANGE_CIPHER_SPEC   0x14
#define TLS_ALERT                0x15
#define TLS_HANDSHAKE            0x16
#define TLS_APPLICATION_DATA     0x17

/* HandshakeType: subtypes within a TLS_HANDSHAKE */
#define HS_HELLO_REQUEST         0x00
#define HS_CLIENT_HELLO          0x01
#define HS_SERVER_HELLO          0x02
#define HS_SESSION_TICKET        0x04
#define HS_CERTIFICATE           0x0B
#define HS_SERVER_KEY_EXCHANGE   0x0C
#define HS_CERTIFICATE_REQEUST   0x0D
#define HS_SERVER_HELLO_DONE     0x0E
#define HS_CERTIFICATE_VERIFY    0x0F
#define HS_CLIENT_KEY_EXCHANGE   0x10
/* There is a also handshake finished type (0x14), but it is not seen because
 * the content of that handshake message is encrypted.  */

/* Extension types */
#define EXT_SERVER_NAME          0x0000
#define EXT_SUPPORTED_GROUPS     0x000a
#define EXT_ELLIPTIC_FORMATS     0x000b

/* Major version number for TLS-1.x (and SSLv3) */
#define TLS_MAJOR_VER   0x03

/*
 *  Unused
 *
 *  #define SSL_VERSION_3   0x0300
 *  #define TLS_VERSION_10  0x0301
 *  #define TLS_VERSION_11  0x0302
 *  #define TLS_VERSION_12  0x0303
 *  #define TLS_VERSION_13  0x0304
 */


/* this might be more - but I have to have a limit somewhere */
#define MAX_CERTS 10

#if HAVE_ALIGNED_ACCESS_REQUIRED

/* Gets a uint16_t (2-byte unsigned) from `payload_` at `offset_` */
#define GET_UINT16(payload_, offset_)           \
    ((((uint16_t)((uint8_t *)payload_)[(offset_)]) << 8)     \
     | ((uint16_t)((uint8_t *)payload_)[(offset_) + 1]))

/* Gets a uint24_t (3-byte unsigned) from `payload_` at `offset_` */
#define GET_UINT24(payload_, offset_)                   \
    ((((uint32_t)((uint8_t *)payload_)[(offset_)]) << 16)            \
     | (((uint32_t)((uint8_t *)payload_)[(offset_) + 1]) << 8)       \
     | ((uint32_t)((uint8_t *)payload_)[(offset_) + 2]))

#else

/* Gets a uint16_t (2-byte unsigned) from `payload_` at `offset_` */
#define GET_UINT16(payload_, offset_)                   \
    (ntohs(*(uint16_t *)((payload_) + (offset_))))

/* Gets a uint24_t (3-byte unsigned) from `payload_` at `offset_` */
#define GET_UINT24(payload_, offset_)                                   \
    ((ntohl(*(uint32_t *)((payload_) + (offset_))) & 0xFFFFFF00) >> 8)

#endif  /* HAVE_ALIGNED_ACCESS_REQUIRED */


/*
 *  This macro makes it easy to store 32-bit numeric values in the DPI by
 *  wrapping yfHookScanPayload() and storing `value_` in the `dpacketCapt`
 *  member which normally holds the offset of the data to extract from the
 *  payload.  `label_` is the identifier for the stored value.
 *
 *  We use 1 as the length of data to extract, though any value would be fine.
 */
#define yfDpiStoreValue(value_, label_)                 \
    yfHookScanPayload(flow, payload, 1, NULL, (value_), \
                      (label_), portNumber)


/*  Debugging macros */
#ifndef DEBUG_LIBMIDENC
#define DEBUG_LIBMIDENC 0
#endif

#if !DEBUG_LIBMIDENC
#define yfTrace(...)
#define TRACE_END_PAYLOAD(off_, need_, plen_)
#else

/* Wrapper over yfTrace2() that includes function name and line number */
#define yfTrace(...)                            \
    yfTrace2(__func__, __LINE__, __VA_ARGS__)

/* Prototype for yfTrace2() in order to provide __attribute__() */
static void
yfTrace2(const char  *funcname,
         int          linenum,
         uint32_t     offset,
         const char  *format,
         ...)
    __attribute__((format (printf, 4, 5)));

/**
 *    Function to print a g_debug() message with the function number, current
 *    line number, offset into the payload, and a caller-provided print-style
 *    format and any arguments to the format.
 */
static void
yfTrace2(const char  *funcname,
         int          linenum,
         uint32_t     offset,
         const char  *format,
         ...)
{
    GString *gstr = g_string_sized_new(2048);
    va_list args;

    va_start(args, format);
    g_string_printf(gstr, "%s():%d Offset %u(%#06x) ",
                    funcname, linenum, offset, offset);
    g_string_append_vprintf(gstr, format, args);
    va_end(args);

    g_debug("%s", gstr->str);
    g_string_free(gstr, TRUE);
}

/**
 *    Use yfTrace() to log that the payload length (plen_) does not hold an
 *    additional number of needed octets (need_).
 */
#define TRACE_END_PAYLOAD(off_, need_, plen_)                   \
    yfTrace(off_, "Octets needed %u exceeds payload length %u", \
            (need_), (plen_))
#endif  /* DEBUG_LIBMIDENC */


/**
 * SSL Declarations are refreced from payloadScanner.h
 */


static gboolean
decodeSSLv2(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint32_t        offset,
    uint32_t        hs_end_offset,
    uint32_t        portNumber);

static gboolean
decodeTLSv1(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint32_t        offset,
    uint32_t        hs_end_offset,
    uint8_t         type,
    uint32_t        portNumber);

static gboolean
decodeCertificates(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint32_t        offset,
    uint32_t        hs_end_offset,
    uint32_t        portNumber);

/**
 *    Scans `payload` to determine if it contains the initialization of a
 *    TLS/SSL session.  Returns `portNumber` if TRUE and 0 if FALSE.
 *
 *    @param payload      the payload (forward or reverse) to scan
 *    @param payloadSize  the number of octets in `payload`
 *    @param flow         the flow record
 *    @param val          the FlowVal of the forward or reverse side
 *    @param offset       where in `payload` to start the examination
 *    @param portNumber   value to return if `payload` is TLS
 */
uint16_t
Mid_Encryption_Scan_Payload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val,
    uint32_t        offset,
    uint32_t        portNumber)
{
#if YAF_ENABLE_HOOKS
    uint16_t tls_version;
#endif
    uint32_t hs_end_offset;
    uint16_t ssl_length;

    yfTrace(offset, "%s payload length %u",
            ((&flow->val == val) ? "Forward" : "Reverse"), payloadSize);

    /* we need 12 octets to determine message type, version, and additional
     * attributes of the TLS handshake */
    if (offset + 12 > payloadSize) {
        TRACE_END_PAYLOAD(offset, 12, payloadSize);
        return 0;
    }

    /*
     *  To check for a TLS HS_CLIENT_HELLO or TLS HS_SERVER_HELLO, check the
     *  following bytes.  (See comment at start of file for details.)
     *
     *  byte 0       ContentType == TLS_HANDSHAKE
     *  byte 1       MSB of ProtocolVersion == TLS_MAJOR_VER
     *  byte 2       LSB of ProtocolVersion -- ignored
     *  bytes 3,4    ssl_length
     *  byte 5       HandshakeType to be HS_CLIENT_HELLO or HS_SERVER_HELLO
     *  bytes 6,7,8  hello_len
     *  byte 9       MSB of ProtocolVersion within Handshake == TLS_MAJOR_VER
     *  byte 10      LSB of ProtocolVersion within Handshake -- ignored
     *
     *  Next get ssl_length and handshake_len and check whether the sizes are
     *  reasonable.
     */
    if (TLS_HANDSHAKE == payload[offset + 0]
        && TLS_MAJOR_VER == payload[offset + 1]
        && TLS_MAJOR_VER == payload[offset + 9]
        && (HS_CLIENT_HELLO == payload[offset + 5]
            || HS_SERVER_HELLO == payload[offset + 5]))
    {
        uint32_t  hello_len;

        yfTrace(offset, "TLS ContentType %#04x with length %u(%#06x)",
                *(payload + offset), GET_UINT16(payload, offset + 3),
                GET_UINT16(payload, offset + 3));
        yfTrace(offset + 5, "TLS HandshakeType %#04x with length %u(%#08x)",
                *(payload + offset + 5), GET_UINT24(payload, offset + 6),
                GET_UINT24(payload, offset + 6));

        /* length of this record and the offset at the end of the record */
        ssl_length = GET_UINT16(payload, offset + 3);
        hs_end_offset = offset + 5 + ssl_length;

        hello_len = GET_UINT24(payload, offset + 6);
        if (ssl_length > 10 && ssl_length >= hello_len + 4) {
            if (decodeTLSv1(payload, payloadSize, flow, offset + 6,
                            hs_end_offset, payload[offset + 5], portNumber))
            {
                /* SSLv3 / TLS */
#if YAF_ENABLE_HOOKS
                /* major version of the style of the ClientHello */
                yfDpiStoreValue(3, YF_SSL_CLIENT_VERSION);
                /* tls_version is stored within decodeTLSv1() */
#endif /* if YAF_ENABLE_HOOKS */
                return portNumber;
            }
        }
    }
    /* else try older methods */

    /*
     *  Check for an SSLv2 Client Hello with a two-byte length.  The first 11
     *  octets of the payload should resemble the following.  (See details in
     *  the comment at the start of this file.)
     *
     *    byte 0,1      length | 0x8000 -- check for MSb high
     *    byte 2        HandshakeType ==  HS_CLIENT_HELLO
     *    byte 3,4      ProtocolVersion -- check for range 0x0300 -- 0x0302
     *    byte 5,6      CipherLen -- ignore for now
     *    byte 7,8      SessionIdLen -- check equal to 0 or 16
     *    byte 9,10     ChallengeLen -- check whether in range [16--32]
     */
    if ((payload[offset + 0] & 0x80)
        && (HS_CLIENT_HELLO == payload[offset + 2])
        && (TLS_MAJOR_VER == payload[offset + 3])
        && (0x02 >= payload[offset + 4])
        && (0 == payload[offset + 7])
        && ((0 == payload[offset + 8] || 16 == payload[offset + 8]))
        && (0 == payload[offset + 9])
        && ((16 <= payload[offset + 10]) && (32 >= payload[offset + 10])))
    {
        /* get length, clearing the most significant bit */
        ssl_length = GET_UINT16(payload, offset + 0) & ~0x8000;
#if YAF_ENABLE_HOOKS
        tls_version = GET_UINT16(payload, offset + 3);
#endif

        hs_end_offset = offset + 2 + ssl_length;
        offset += 5;

        if (!decodeSSLv2(payload, payloadSize, flow, offset,
                         hs_end_offset, portNumber))
        {
            return 0;
        }

        /* SSLv2 (client_hello) */
#if YAF_ENABLE_HOOKS
        /* major version of the style of the ClientHello */
        yfDpiStoreValue(2, YF_SSL_CLIENT_VERSION);
        /* version advertised in the ClientHello */
        yfDpiStoreValue(tls_version, YF_SSL_RECORD_VERSION);
#endif
        return portNumber;
    }

    /*
     *  Check for an SSLv2 Client Hello with a three-byte length (does this
     *  really occur?).  This very similar to the previous, except the MSb of
     *  the record length is low and a byte of padding at offset+2 shifts
     *  everything that follows.
     *
     *    byte 0,1      length -- check for first two MSb low
     *    byte 2        padding length -- ignored
     *    byte 3        HandshakeType ==  HS_CLIENT_HELLO
     *    byte 4,5      ProtocolVersion -- check for range 0x0300 -- 0x0302
     *    byte 6,7      CipherLen -- ignore for now
     *    byte 8,9      SessionIdLen -- check equal to 0 or 16
     *    byte 10,11    ChallengeLen -- check whether 16 <= len <= 32
     */
    if ((0 == (payload[offset + 0] & 0xc0))
        && (HS_CLIENT_HELLO == payload[offset + 3])
        && (TLS_MAJOR_VER == payload[offset + 4])
        && (0x02 >= payload[offset + 5])
        && (0 == payload[offset + 8])
        && ((0 == payload[offset + 9] || 16 == payload[offset + 9]))
        && (0 == payload[offset + 10])
        && (16 <= payload[offset + 11]) && (32 >= payload[offset + 11]))
    {
        /* get length, clearing the two most significant bits */
        ssl_length = GET_UINT16(payload, offset + 0) & ~0xc000;
#if YAF_ENABLE_HOOKS
        tls_version = GET_UINT16(payload, offset + 4);
#endif

        hs_end_offset = offset + 3 + ssl_length;
        offset += 6;

        if (!decodeSSLv2(payload, payloadSize, flow, offset,
                         hs_end_offset, portNumber))
        {
            return 0;
        }

        /* SSLv2 (client_hello) */
#if YAF_ENABLE_HOOKS
        /* major version of the style of the ClientHello */
        yfDpiStoreValue(2, YF_SSL_CLIENT_VERSION);
        /* version advertised in the ClientHello */
        yfDpiStoreValue(tls_version, YF_SSL_RECORD_VERSION);
#endif
        return portNumber;
    }

    return 0;
}


/**
 *    Decode a TLS 1.x (or SSL v3) stream.  Return TRUE on success or FALSE on
 *    failure.
 *
 *    `offset` is located just after the HandshakeType and on the MSB of the
 *    24-bit length.  See comment at start of file for details.
 *
 *    @param payload       the payload (forward or reverse) to scan
 *    @param payloadSize   the number of octets in `payload`
 *    @param flow          the flow record
 *    @param offset        where in `payload` to start the examination
 *    @param hs_end_offset offset at end of the handshake
 *    @param type          handshake type (HS_CLIENT_HELLO, HS_SERVER_HELLO)
 *    @param portNumber    potential appLabel being examined
 */
static gboolean
decodeTLSv1(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint32_t        offset,
    uint32_t        hs_end_offset,
    uint8_t         type,
    uint32_t        portNumber)
{
    uint32_t        hello_end;
    uint32_t        hello_len;
    uint8_t         session_len;

    g_assert(HS_CLIENT_HELLO == type || HS_SERVER_HELLO == type);

    /* Both Client Hello and Server Hello start with 39 octets of data: 3 for
     * Length, 2 for TLS Protocol Version, 32 octets of Random Data, 1 for
     * Session ID Length, followed by Session ID */
    if (offset + 39 > payloadSize) {
        TRACE_END_PAYLOAD(offset, 39, payloadSize);
        return FALSE;
    }

    hello_len = GET_UINT24(payload, offset);
    offset += 3;

    /* note offset at the end of the hello */
    hello_end = offset + hello_len;

    if (hello_len < 35 || hello_end > hs_end_offset) {
        yfTrace(offset - 3, "Bad lengths: hello_len %u(%#08x) >"
                " hs_end_offset %u || hello_len(%u) < 35",
                hello_len, hello_len, hs_end_offset, hello_len);
        return FALSE;
    }

#if YAF_ENABLE_HOOKS
    uint16_t  tls_version = GET_UINT16(payload, offset);
#endif
    /* skip version & random */
    offset += 34;

    /* get session length and skip the session */
    session_len = *(payload + offset);
    offset += 1 + session_len;

    if (type == HS_CLIENT_HELLO) {
        /* Client Hello */
        uint16_t cipher_suite_len;
        uint8_t  compression_len;

        if (offset + 2 > payloadSize) {
            TRACE_END_PAYLOAD(offset, 2, payloadSize);
            return FALSE;
        }

        /* Store the offset and length of the list of ciphers */
        cipher_suite_len = GET_UINT16(payload, offset);
        if (cipher_suite_len == 0 || (cipher_suite_len & 0x01)) {
            /* must be non-zero and even */
            yfTrace(offset, "Bad cipher_suite_len %u", cipher_suite_len);
            return FALSE;
        }
#if YAF_ENABLE_HOOKS
        uint16_t cipher_offset = offset + 2;
#endif
        offset += 2 + cipher_suite_len;

        /* Get and skip the compression */
        if (offset + 1 > payloadSize) {
            TRACE_END_PAYLOAD(offset, 1, payloadSize);
            return FALSE;
        }
        compression_len = *(payload + offset);
        offset += 1 + compression_len;
        if (offset > payloadSize) {
            TRACE_END_PAYLOAD(offset, 0, payloadSize);
            return FALSE;
        }

#if YAF_ENABLE_HOOKS
        /* Record numeric values in the "offset" value */
        /* Version from the ClientHello */
        yfDpiStoreValue(tls_version, YF_SSL_RECORD_VERSION);
        /* Version used in computing JA3 */
        yfDpiStoreValue(tls_version, YF_SSL_VERSION_JA3);
        /* Record cipher position and length */
        yfHookScanPayload(flow, payload, cipher_suite_len, NULL, cipher_offset,
                          YF_SSL_CIPHER_LIST, portNumber);
#endif /* if YAF_ENABLE_HOOKS */
    } else {
        /* Server Hello */

        /* Expect 3 octets: chosen_cipher(2) and chosen compression(1) */
        if (offset + 3 > payloadSize) {
            TRACE_END_PAYLOAD(offset, 3, payloadSize);
            return FALSE;
        }
#if !YAF_ENABLE_HOOKS
        /* skip these values */
        offset += 3;
#else
        uint16_t  chosen_cipher;
        uint8_t   compression;

        /* chosen cipher is here */
        chosen_cipher = GET_UINT16(payload, offset);
        offset += 2;

        /* chosen compression method */
        compression = *(payload + offset);
        offset++;

        /* Record numeric values in the "offset" value */
        /* Version from the ServerHello */
        yfDpiStoreValue(tls_version, YF_SSL_RECORD_VERSION);
        /* Version used in computing JA3S */
        yfDpiStoreValue(tls_version, YF_SSL_VERSION_JA3S);
        yfDpiStoreValue(chosen_cipher, YF_SSL_SERVER_CIPHER);
        yfDpiStoreValue(compression, YF_SSL_COMPRESSION);
#endif /* if YAF_ENABLE_HOOKS */
    }

    /* if there is more data in the HELLO, then assume we have extensions */
    if ((offset + 2 <= hello_end) && (offset + 2 < payloadSize)) {
        const uint16_t tot_ext_len = GET_UINT16(payload, offset);

#if !YAF_ENABLE_HOOKS
        /* Set offset to the end of the extensions */
        offset += 2 + tot_ext_len;
#else
        /* Store location where extensions begin for computing JA3 */
        yfHookScanPayload(flow, payload, 1, NULL, offset,
                          ((HS_CLIENT_HELLO == type)
                           ? YF_SSL_CLIENT_EXTENSION
                           : YF_SSL_SERVER_EXTENSION), portNumber);

        if (HS_SERVER_HELLO == type) {
            /* do nothing with the server; Set offset to the end of the
             * extensions */
            offset += 2 + tot_ext_len;

            /* FIXME: Add support for getting the supported version extension
             * from the server to know what version is being used. */
        } else {
            uint32_t  end_extensions;
            uint32_t  ext_cur;
            uint32_t  ext_len;
            uint16_t  ext_type;
            uint32_t  ext_next;
            uint16_t  eli_curv_len;
            uint8_t   eli_form_len;

            /* current position as we parse the extensions */
            ext_cur = offset + 2;

            /* Set offset to the end of the extensions */
            offset += 2 + tot_ext_len;

            /* stopping position is at the end of the extentions but do not
             * exceed the payload */
            end_extensions = MIN(offset, payloadSize);

            for (; ext_cur + 4 <= end_extensions; ext_cur = ext_next) {
                /* Get type and length */
                ext_type = GET_UINT16(payload, ext_cur);
                ext_cur += 2;
                ext_len = GET_UINT16(payload, ext_cur);
                ext_cur += 2;
                /* note start of next extension (which also marks end of this
                 * extension) */
                ext_next = ext_cur + ext_len;

                if (ext_len < 2 || ext_next > end_extensions) {
                    /* not enough data avail or length is too long */
                    continue;
                }

                switch (ext_type) {
                  case EXT_SERVER_NAME:
                    /* Server Name extension has a 2 byte length, a 1 one byte
                     * type (only 0==DNS-hostname is defined), a 2 byte string
                     * length, and the string */
                    ext_len = GET_UINT16(payload, ext_cur);
                    if (ext_len < 3
                        || 0 != *(payload + ext_cur + 2)
                        || ext_cur + 2 + ext_len > ext_next)
                    {
                        continue;
                    }
                    ext_cur += 3;
                    ext_len = GET_UINT16(payload, ext_cur);
                    ext_cur += 2;
                    if ((ext_cur + ext_len) <= ext_next) {
                        /* store offset and length to extract the name */
                        yfHookScanPayload(
                            flow, payload, ext_len, NULL, ext_cur,
                            YF_SSL_SERVER_NAME, portNumber);
                    }
                    break;

                  case EXT_SUPPORTED_GROUPS:
                    /* elliptic curve list */
                    /* After grabing the length jump past it and grab the
                     * desired list */
                    eli_curv_len = GET_UINT16(payload, ext_cur);
                    ext_cur += 2;
                    if ((ext_cur + eli_curv_len) <= ext_next) {
                        /* store offset and length to extract the list */
                        yfHookScanPayload(
                            flow, payload, eli_curv_len, NULL, ext_cur,
                            YF_SSL_ELIPTIC_CURVE, portNumber);
                    }
                    break;

                  case EXT_ELLIPTIC_FORMATS:
                    /* elliptic curve point format list */
                    /* After grabing the length jump past it and grab the
                     * desired list */
                    eli_form_len = *(payload + ext_cur);
                    ext_cur += 1;
                    if ((ext_cur + eli_form_len) <= ext_next) {
                        /* store offset and length to extract the list */
                        yfHookScanPayload(
                            flow, payload, eli_form_len, NULL, ext_cur,
                            YF_SSL_ELIPTIC_FORMAT, portNumber);
                    }
                    break;
                }
            }
        }
#endif /* #else if #if !YAF_ENABLE_HOOKS */
    }

    /* Check the remainder of the payload for X509 certificates */
    return decodeCertificates(payload, payloadSize, flow,
                              offset, hs_end_offset, portNumber);
}


/**
 *    Look for certificates in the TLS stream.
 *
 *    `offset` is the current offset into the `payload` which has size
 *    `payloadSize`.
 *
 *    The server may include certificates (HS_CERTIFICATE) in the same TLS
 *    message as the HS_SERVER_HELLO or it may begin a new TLS message for the
 *    certificates.  To know whether a new message must be read, the value
 *    `hs_end_offset` is the byte offset of the end of the message containing
 *    the HS_CLIENT_HELLO or HS_SERVER_HELLO.
 */
static gboolean
decodeCertificates(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint32_t        offset,
    uint32_t        hs_end_offset,
    uint32_t        portNumber)
{
    uint32_t  record_len;
    uint32_t  tot_cert_len;
    uint32_t  sub_cert_len;
    uint32_t  certs_end;
    uint32_t  cert_count = 0;

    /*  At this point we are certain this is a TLS stream.  Running out of
     *  payload at this point is okay (still return TRUE).  We also return
     *  TRUE if a size check is incorrect (e.g., a zero length certificate)
     *  since it is possible a TCP segment was not captured and we are reading
     *  a block of zeros. */

    /*
     *  We require at least 5 bytes for a TLS message or at least 4 bytes for
     *  a sub-handshake messsage.
     */
    while (offset + 5 <= payloadSize) {

        if (offset == hs_end_offset) {
            yfTrace(offset, "TLS ContentType %#04x with length %u(%#06x)",
                    *(payload + offset), GET_UINT16(payload, offset + 3),
                    GET_UINT16(payload, offset + 3));

            /*
             *  Read a TLS message header (type(1), proto-version(2),
             *  length(2)) .  Return if the type is anything other than
             *  TLS_HANDSHAKE because we do not do anything with them.
             *
             *  In addition, once TLS_CHANGE_CIPHER_SPEC is seen, the contents
             *  of the remaining messages are encrypted and only the header
             *  octets are visible.  Attempting to treat the first byte of the
             *  encrypted content of a TLS_HANDSHAKE message as a
             *  HandshakeType will lead to errors.
             */
            switch (*(payload + offset)) {
              case TLS_CHANGE_CIPHER_SPEC:
              case TLS_ALERT:
              case TLS_APPLICATION_DATA:
                return TRUE;

              case TLS_HANDSHAKE:
                /* We need 9 octets: 5 byte TLS header and the 4 byte
                 * HandShake header */
                if (offset + 9 > payloadSize) {
                    TRACE_END_PAYLOAD(offset, 9, payloadSize);
                    return TRUE;
                }
                record_len = GET_UINT16(payload, offset + 3);
                offset += 5;
                hs_end_offset = MIN(offset + record_len, payloadSize);
                break;

              default:
                return TRUE;
            }
        } else if (offset > hs_end_offset) {
            yfTrace(offset, "Offset exceeds end of handshake %u(%#06x)",
                    hs_end_offset, hs_end_offset);
            return TRUE;
        }

        yfTrace(offset, "TLS HandshakeType %#04x with length %u(%#08x)",
                *(payload + offset), GET_UINT24(payload, offset + 1),
                GET_UINT24(payload, offset + 1));

        /*
         *  Read the HandShakeType(1 octet) and its length(3 octets), and
         *  ignore anything other than HS_CERTIFICATE.
         */
        switch (*(payload + offset)) {
          case HS_HELLO_REQUEST:
          case HS_CLIENT_HELLO:
          case HS_SERVER_HELLO:
          case HS_SESSION_TICKET:
          case HS_SERVER_KEY_EXCHANGE:
          case HS_CERTIFICATE_REQEUST:
          case HS_SERVER_HELLO_DONE:
          case HS_CERTIFICATE_VERIFY:
          case HS_CLIENT_KEY_EXCHANGE:
            record_len = GET_UINT24(payload, offset + 1);
            offset += 4 + record_len;
            break;

          case HS_CERTIFICATE:
            /* Certificate Header within a Handshake Message */
            /* Require at least 7 bytes (type(1 byte) + two 24-bit lengths) */
            if (offset + 7 > payloadSize) {
                TRACE_END_PAYLOAD(offset, 7, payloadSize);
                return TRUE;
            }
            /* Length of the record */
            record_len = GET_UINT24(payload, offset + 1);
            /* Total length of all certifiates */
            tot_cert_len = GET_UINT24(payload, offset + 4);
            yfTrace(offset, "Length of all certificates %u(%#08x)",
                    tot_cert_len, tot_cert_len);

            offset += 7;
            if (tot_cert_len > record_len) {
                /* Invalid: tot_cert_len larger than its container */
                yfTrace(offset,
                        "Bad lengths: tot_cert_len(%u) > record_len(%u)",
                        tot_cert_len, record_len);
                return TRUE;
            }
            /* Set an offset to simplify checks against end-of-all-certs
             * and the amount of available payload */
            certs_end = MIN(offset + tot_cert_len, payloadSize);

            while (offset + 3 <= certs_end) {
                /* Length of this certificate */
                sub_cert_len = GET_UINT24(payload, offset);
                yfTrace(offset, "Found cert with length %u(%#08x)",
                        sub_cert_len, sub_cert_len);
                if ((sub_cert_len > tot_cert_len) || (sub_cert_len < 2)) {
                    /* Invalid lengths */
                    yfTrace(offset, "Bad lengths: sub_cert_len(%u) >"
                            " tot_cert_len(%u) || sub_cert_len(%u) < 2",
                            sub_cert_len, tot_cert_len, sub_cert_len);
                    return TRUE;
                }
                if (offset + sub_cert_len > certs_end) {
                    /* just not enough room */
                    yfTrace(offset, "Bad lengths: offset + sub_cert_len(%u) >"
                            " certs_end offset(%u)",
                            (offset + sub_cert_len), certs_end);
                    return TRUE;
                }

                if (cert_count >= MAX_CERTS) {
                    /* out of space */
                    yfTrace(offset, "Reached max certificates");
                    return TRUE;
                }
#if YAF_ENABLE_HOOKS
                /* record the offset of certificate */
                if ((offset + sub_cert_len + 3) <= certs_end) {
                    yfHookScanPayload(flow, payload, 1, NULL, offset,
                                      YF_SSL_CERT_START, portNumber);
                }
#endif /* if YAF_ENABLE_HOOKS */
                cert_count++;
                offset += 3 + sub_cert_len;
            }
            break;

          default:
            return TRUE;
        }
    }

    return TRUE;
}


/**
 *    Decode an SSL v2 Client Hello; remainder of payload should be SSLv3 or
 *    TLS.  Return TRUE on success or FALSE on failure.
 *
 *    `offset` is located just after the HandshakeType and on the MSB of the
 *    16-bit cipher_spec_length.  See comment at start of file for details.
 *
 *
 *    @param payload      the payload (forward or reverse) to scan
 *    @param payloadSize  the number of octets in `payload`
 *    @param flow         the flow record
 *    @param offset       where in `payload` to start the examination
 *    @param datalength   outermost TLS-record length
 *    @param portNumber   potential appLabel being examined
 */
static gboolean
decodeSSLv2(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint32_t        offset,
    uint32_t        hs_end_offset,
    uint32_t        portNumber)
{
    uint16_t  cipher_spec_length;
    uint16_t  session_length;
    uint16_t  challenge_length;

    /* expect three 2-byte lengths */
    if (offset + 6 > payloadSize) {
        return FALSE;
    }

    /* cipher_spec_length */
    cipher_spec_length = GET_UINT16(payload, offset);
    offset += 2;
    if (cipher_spec_length % 3 != 0
        || offset + cipher_spec_length >= hs_end_offset)
    {
        return FALSE;
    }

    /* session length */
    session_length = GET_UINT16(payload, offset);
    offset += 2;
    if (session_length != 0 && session_length != 16) {
        return FALSE;
    }

    /* challenge length */
    challenge_length = GET_UINT16(payload, offset);
    offset += 2;
    if (challenge_length < 16 || challenge_length > 32) {
        return FALSE;
    }

    if (cipher_spec_length > payloadSize) {
        return FALSE;
    }
#if YAF_ENABLE_HOOKS
    /* record the offset and length of the cipher list */
    yfHookScanPayload(flow, payload, cipher_spec_length, NULL, offset,
                      YF_SSL_V2_CIPHER_LIST, portNumber);
#endif
    offset += cipher_spec_length + session_length + challenge_length;

    /* Check the remainder of the payload for X509 certificates */
    return decodeCertificates(payload, payloadSize, flow,
                              offset, hs_end_offset, portNumber);
}
