/*
 *  Copyright 2007-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  dnsplugin.c
 *
 *  provides a plugin to the ipfix payload classifier to attempt to determine
 *  if a packet payload is a DNS packet (see RFC 1035)
 *
 *  defining PAYLOAD_INSPECTION at compile time will attempt to better
 *  inspection of the packet payload at a cost of deeper inspection;  even with
 *  PAYLOAD_INSPECTION enabled, it is possible that this may not be 100%
 *  correct in ID'ing the packets
 *
 *  ------------------------------------------------------------------------
 *  Authors: Chris Inacio
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.18
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1281
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */
#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <payloadScanner.h>

#if YAF_ENABLE_HOOKS
#include <yaf/yafhooks.h>
#endif

/*
 * typedef struct ycDnsScanMessageHeader_st {
 *  uint16_t            id;
 *
 *  uint16_t            qr:1;
 *  uint16_t            opcode:4;
 *  uint16_t            aa:1;
 *  uint16_t            tc:1;
 *  uint16_t            rd:1;
 *  uint16_t            ra:1;
 *  uint16_t            z:1;
 *  uint16_t            ad:1;
 *  uint16_t            cd:1;
 *  uint16_t            rcode:4;
 *
 *  uint16_t            qdcount;
 *  uint16_t            ancount;
 *  uint16_t            nscount;
 *  uint16_t            arcount;
 * } ycDnsScanMessageHeader_t;
 */

#define DNS_PORT_NUMBER 53

/*
 *  Since NETBIOS looks A LOT like DNS, there's no need to create a separate
 *  plugin for it - if we think it's NETBIOS we will return
 *  NETBIOS_PORT_NUMBER.
 *
 *  In addition to NETBIOS, LLMNR and MDNS also use the same header structure
 *  as DNS so a separate plugin was not created, and this plugin returns
 *  LLMNR_PORT_NUMBER and MDNS_PORT_NUMBER if we think are LLMNR or MDNS.
 *
 *  LLMNR refrences were found ussing RFC 4795:
 *  https://www.rfc-editor.org/rfc/rfc4795.html
 *
 *  MDNs refrences were found using RFC 6762:
 *  https://datatracker.ietf.org/doc/html/rfc6762
 */
#define NETBIOS_PORT_NUMBER 137
#define LLMNR_PORT_NUMBER   5355
#define MDNS_PORT_NUMBER    5353

/*
 *  LLMNR IPv4 address: 224.0.0.252
 *  LLMNR IPv6 address: FF02::1:3
 *
 *  MDNs IPv4 address: 224.0.0.251
 *  MDNs IPv6 address: FF02::FB
 */
#define LLMNR_IPV4   0xe00000fc
#define MDNS_IPV4    0xe00000fb
#define LLMNR_IPV6                                      \
    {0xff, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,    \
     0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03}
#define MDN_IPV6                                        \
    {0xff, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,    \
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfb}


/* The EDNS option pseudo-type--used to ignore these */
#define DNS_TYPE_OPT 41

/* DNS class NONE--used to check for valid class */
#define DNS_CLASS_NONE 254

#define DNS_NAME_COMPRESSION 0xc0


/*  Debugging macros */
#ifndef DEBUG_DNSPLUGIN
#define DEBUG_DNSPLUGIN 0
#endif

#if !DEBUG_DNSPLUGIN
#define yfTrace(...)
#define TRACE_END_PAYLOAD(off_, need_, plen_)
#else

/* Wrapper over yfTrace2() that includes function name and line number */
#define yfTrace(...)                            \
    yfTrace2(__func__, __LINE__, __VA_ARGS__)

/* Prototype for yfTrace2() in order to provide __attribute__() */
static void
yfTrace2(const char  *funcname,
         int          linenum,
         uint32_t     offset,
         const char  *format,
         ...)
    __attribute__((format (printf, 4, 5)));

/**
 *    Function to print a g_debug() message with the function number, current
 *    line number, offset into the payload, and a caller-provided print-style
 *    format and any arguments to the format.
 */
static void
yfTrace2(const char  *funcname,
         int          linenum,
         uint32_t     offset,
         const char  *format,
         ...)
{
    GString *gstr = g_string_sized_new(2048);
    va_list args;

    va_start(args, format);
    g_string_printf(gstr, "%s():%d Offset %u(%#06x) ",
                    funcname, linenum, offset, offset);
    g_string_append_vprintf(gstr, format, args);
    va_end(args);

    g_debug("%s", gstr->str);
    g_string_free(gstr, TRUE);
}

/**
 *    Use yfTrace() to log that the payload length (plen_) does not hold an
 *    additional number of needed octets (need_).
 */
#define TRACE_END_PAYLOAD(off_, need_, plen_)                   \
    yfTrace(off_, "Octets needed %u exceeds payload length %u", \
            (unsigned int)(need_), (plen_))
#endif  /* DEBUG_DNSPLUGIN */


/*
 *    Extracts a 16-bit integer from `payload_` at `offset_`, byte-swaps it if
 *    needed, stores it in `value_`, and updates `offset_` (moves forward 2).
 *    The caller should check whether sizeof(uint16_t) is available in
 *    `payload_` before using this macro.
 */
#if HAVE_ALIGNED_ACCESS_REQUIRED
#define READ_U16_INC(value_, payload_, offset_)                 \
    {                                                           \
        value_ = ((( *((payload_) + (offset_))) << 8)           \
                  | (( *((payload_) + (offset_) + 1) )));       \
        value_ = ntohs(value_);                                 \
        (offset_) += sizeof(uint16_t);                          \
    }
#else  /* HAVE_ALIGNED_ACCESS_REQUIRED */
#define READ_U16_INC(value_, payload_, offset_)                         \
    {                                                                   \
        value_ = ntohs(*((uint16_t *)((payload_) + (offset_))));        \
        (offset_) += sizeof(uint16_t);                                  \
    }
#endif  /* HAVE_ALIGNED_ACCESS_REQUIRED */


YC_SCANNER_PROTOTYPE(dnsplugin_LTX_ycDnsScanScan);

#define PAYLOAD_INSPECTION 1

/**
 * local prototypes
 *
 */

#ifdef PAYLOAD_INSPECTION
static uint16_t
ycDnsScanCheckResourceRecord(
    const uint8_t  *payload,
    uint32_t       *offset,
    unsigned int    payloadSize);

static inline uint16_t
ycDnsCheckRRType(
    const uint8_t  *payload,
    uint32_t        payloadSize,
    uint32_t       *offset);
#endif /* ifdef PAYLOAD_INSPECTION */


/**
 * dnsScanner_LTX_ycDnsScanScan
 *
 * scans a payload to determine if the payload is a dns request/reply.
 * It checks the structure for self referential integrity, but it can't
 * guarantee that the payload is actually DNS, it could be
 * some degenerate random data
 *
 * name abomination has been achieved by combining multiple naming standards
 * until the prefix to
 * the function name is dnsplugin_LTX_ycDnsScan --- it's a feature
 *
 * @param argc NOT USED
 * @param argv NOT USED
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match DNS_PORT_NUMBER (53) for a match
 *
 */
uint16_t
dnsplugin_LTX_ycDnsScanScan(
    int             argc,
    char           *argv[],
    const uint8_t  *payload,
    uint32_t        payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    ycDnsScanMessageHeader_t header;
    unsigned int  loop;
    gboolean      netbios = FALSE;
    gboolean      llmnr = FALSE;
    gboolean      mdns = FALSE;
    uint32_t      offset;
    uint16_t      appLabel;
    uint16_t      qtype = 0;
    const uint8_t llmnr_ipv6[16] = LLMNR_IPV6;
    const uint8_t mdns_ipv6[16] =  MDN_IPV6;

#if YAF_ENABLE_HOOKS
    unsigned int recordCount = 0;
    uint16_t     direction;
#endif

    /* return if the applabel has been set and it does not match the
     * applabel(s) returned by this function. */
    switch (flow->appLabel) {
      case NETBIOS_PORT_NUMBER:
        /* The appLabel was previously set to NETBIOS which has no DPI, so
         * return the appLabel. */
        return flow->appLabel;
      case 0:
      case DNS_PORT_NUMBER:
      case LLMNR_PORT_NUMBER:
      case MDNS_PORT_NUMBER:
        /* Either appLabel not set or it is an appLabel processed by this
         * plug-in for which there is DPI */
        break;
      default:
        /* The appLabel is set but it is not one returned by this function, so
         * return 0. */
        return 0;
    }

    if (payloadSize < sizeof(ycDnsScanMessageHeader_t)) {
        TRACE_END_PAYLOAD(0, sizeof(ycDnsScanMessageHeader_t), payloadSize);
        return 0;
    }

    if (flow->key.proto == YF_PROTO_TCP) {
        uint32_t  firstpkt = payloadSize;
        uint32_t  msglen;
        for (loop = 0; loop < val->pkt && loop < YAF_MAX_PKT_BOUNDARY; ++loop) {
            if (val->paybounds[loop] != 0) {
                firstpkt = val->paybounds[loop];
                break;
            }
        }
        msglen = ntohs(*((uint16_t *)(payload)));
        if ((msglen + 2) == firstpkt) {
            /* this is the weird message length in TCP */
            payload += sizeof(uint16_t);
            payloadSize -= sizeof(uint16_t);
        }
    }

    /* this function is defined in payloadScanner.c */
    ycDnsScanRebuildHeader(payload, &header);

    if (flow->key.dp == LLMNR_PORT_NUMBER) {
        if (flow->key.version == 6) {
            if (memcmp(flow->key.addr.v6.dip, llmnr_ipv6, sizeof(llmnr_ipv6))
                == 0)
            {
                yfTrace(0, "Looks like LLMNR over IPv6");
                llmnr = TRUE;
            }
        } else if (flow->key.version == 4
                   && flow->key.addr.v4.dip == LLMNR_IPV4)
        {
            yfTrace(0, "Looks like LLMNR over IPv4");
            llmnr = TRUE;
        }
    } else if (flow->key.dp == MDNS_PORT_NUMBER
               && header.id == 0)
    {
        if (flow->key.version == 6) {
            if (memcmp(flow->key.addr.v6.dip, mdns_ipv6, sizeof(mdns_ipv6))
                == 0)
            {
                yfTrace(0, "Looks like MDNS over IPv6");
                mdns = TRUE;
            }
        } else if (flow->key.version == 4
                   && flow->key.addr.v4.dip == MDNS_IPV4)
        {
            yfTrace(0, "Looks like MDNS over IPv4");
            mdns = TRUE;
        }
    }

    /*
     * treat OpCodes 0-5(except 3) as undecided;
     * treat OpCodes 6-9,15 as NetBIOS;
     * reject OpCodes 3,>=10(except 15)
     * https://www.iana.org/assignments/dns-parameters/dns-parameters.xhtml
     * https://osqa-ask.wireshark.org/questions/2856/display-filter-for-nbns-query-type/
     *
     * NOTE: OpCode 6 may indicate DNS Stateful Operations (DSO) RFC-8490, and
     * this file needs to be updated to handle it.  Given that DSO is
     * radically different from traditional DNS, DSO handling should be in a
     * separate function.  (DSO Differences: Only over TCP; all four counters
     * (QR,AN,NS,AR) are zero; TLV-encoded data follows the DNS header; RCODE
     * may be 11.)
     */

    if (header.opcode <= 5) {
        if (header.opcode == 3) {
            yfTrace(0, "Bad header.opcode %u", header.opcode);
            return 0;
        }
        /* else DNS(0-5) or NetBIOS(0,5-9) */
    } else if (header.opcode <= 9 || header.opcode == 15) {
        yfTrace(0, "Looks like NetBIOS opcode %u", header.opcode);
        netbios = TRUE;
    } else {
        yfTrace(0, "Bad header.opcode %u", header.opcode);
        return 0;
    }

    /* rfc 2136 updates rfc 1035 */
    /* 16-22 are DNSSEC rcodes*/
    if ((header.rcode > 10) && (1 == header.qr)) {
        if ((header.rcode < 16) || (header.rcode > 22)) {
            yfTrace(0, "Bad header.rcode %u", header.rcode);
            return 0;
        }
    }

    /* check to make sure resource records are not empty -
     * gets rid of all 0's payloads */
    if (header.qdcount == 0 && header.ancount == 0 && header.nscount == 0
        && header.arcount == 0)
    {
        if (!(header.rcode > 0 && header.qr == 1)) {
            /* DNS responses that are not errors will have something in them */
            yfTrace(0, ("No entries found in DNS response:"
                        " qr %u, rcode %u, qdcount %u, ancount %u,"
                        " nscount %u, arcount %u"),
                    header.qr, header.rcode, header.qdcount, header.ancount,
                    header.nscount, header.arcount);
            return 0;
        }
    }

    /* query validation */
    if (header.qr == 0) {
        if ((header.rcode > 0 || header.aa != 0 || header.ra != 0 ||
             header.ad != 0))
        {
            /* queries should not have an rcode, an authoritative answer,
             * recursion available, or authenticated data */
            yfTrace(0, ("Invalid settings in DNS query:"
                        " qr %u, rcode %u, aa %u, ra %u, ad %u"),
                    header.qr, header.rcode, header.aa, header.ra, header.ad);
            return 0;
        }
        if (!(header.qdcount > 0)) {
            /* queries should have at least one question */
            yfTrace(0, ("No queries in DNS query:"
                        " qr %u, rcode %u, qdcount %u"),
                    header.qr, header.rcode, header.qdcount);
            return 0;
        }
    }

#ifdef PAYLOAD_INSPECTION
    /* parse through the rest of the DNS message, only the header is fixed
     * in size */
    offset = sizeof(ycDnsScanMessageHeader_t);
    if (offset >= payloadSize) {
        yfTrace(offset, "Offset exceeds payloadSize (%u >= %u)",
                offset, payloadSize);
        return 0;
    }

    /* the the query entries */

    yfTrace(offset, ("DNS Record:"
                     " qr %u, qdcount %u, ancount %u, nscount %u, arcount %u"),
                    header.qr, header.qdcount, header.ancount,
                    header.nscount, header.arcount);

    for (loop = 0; loop < header.qdcount; loop++) {
        uint8_t  sizeOct = *(payload + offset);
        uint16_t qclass;
        uint8_t  comp = 0;            /* turn on if something is compressed */

        while (0 != sizeOct && offset < payloadSize) {
            if (DNS_NAME_COMPRESSION == (sizeOct & DNS_NAME_COMPRESSION)) {
                offset += sizeof(uint16_t);
                /* compression happened so we don't need add 1 later */
                comp = 1;
            } else {
                offset += sizeOct + 1;
            }
            if (offset >= payloadSize) {
                yfTrace(offset, "Offset exceeds payloadSize (%u >= %u)",
                        offset, payloadSize);
                return 0;
            }
            sizeOct = *(payload + offset);
        }

        if (offset >= payloadSize) {
            /* this is either a DNS fragment, or a malformed DNS */
            yfTrace(offset, "Offset exceeds payloadSize (%u >= %u)",
                    offset, payloadSize);
            return 0;
        }

        /* get past the terminating 0 length in the name if NO COMPRESSION*/
        if (!comp) {
            offset++;
        }

        /* check that 2 bytes are available and the query type */
        qtype = ycDnsCheckRRType(payload, payloadSize, &offset);
        if (0 == qtype) {
            /* yfTrace() called in ycDnsCheckRRType() */
            return 0;
        }

        if (qtype == 32
            || (qtype == 33
                && (flow->key.sp == NETBIOS_PORT_NUMBER
                    || flow->key.dp == NETBIOS_PORT_NUMBER)))
        {
            yfTrace(offset, "Looks like NetBIOS qtype %u", qtype);
            netbios = TRUE;
        }

        /* check the class code */
        if ((offset + 2) > payloadSize) {
            TRACE_END_PAYLOAD(offset, 2, payloadSize);
            return 0;
        }
        READ_U16_INC(qclass, payload, offset);

        if (qclass > 4 && qclass != 255) {
            yfTrace(offset, "Bad qclass %u", qclass);
            return 0;
        }

        if (netbios && qclass != 1) {
            yfTrace(offset, "Bad qclass %u for NetBIOS", qclass);
            return 0;
        }

        if (offset > payloadSize) {
            yfTrace(offset, "Offset exceeds payloadSize (%u >= %u)",
                    offset, payloadSize);
            return 0;
        }
    }

    /* check each record for the answer record count */
    for (loop = 0; loop < header.ancount; loop++) {
        uint16_t rc;
        rc = ycDnsScanCheckResourceRecord(payload, &offset, payloadSize);
        if (0 == rc) {
            /* yfTrace() called in ycDnsScanCheckResourceRecord() */
            return rc;
        }

        if (netbios) {
            if (rc != 1 && rc != 2 && rc != 10 && rc != 32 && rc != 33) {
                yfTrace(offset, "Bad rrtype %u for NetBIOS in answers", rc);
                return 0;
            }
        } else if ((rc == 32)
                   || (rc == 33 && header.qdcount == 0))
        {
            yfTrace(offset, "Looks like NetBIOS rrtype %u, qdcount %u",
                    rc, header.qdcount);
            netbios = TRUE;
        }

#if YAF_ENABLE_HOOKS
        if (rc != DNS_TYPE_OPT) {
            recordCount++;
        }
#endif
    }

    /* check each record for the name server resource record count */
    for (loop = 0; loop < header.nscount; loop++) {
        uint16_t rc;
        rc = ycDnsScanCheckResourceRecord(payload, &offset, payloadSize);
        if (0 == rc) {
            /* yfTrace() called in ycDnsScanCheckResourceRecord() */
            return 0;
        }

        if (netbios) {
            if (rc != 1 && rc != 2 && rc != 10 && rc != 32 && rc != 33) {
                yfTrace(offset, "Bad rrtype %u for NetBIOS in servers", rc);
                return 0;
            }
        } else if (rc == 2 && header.qdcount == 0) {
            yfTrace(offset, "Looks like NetBIOS rrtype %u, qdcount %u",
                    rc, header.qdcount);
            netbios = TRUE;
        }

#if YAF_ENABLE_HOOKS
        if (rc != DNS_TYPE_OPT) {
            recordCount++;
        }
#endif
    }

    /* check each record for the additional record count */
    for (loop = 0; loop < header.arcount; loop++) {
        uint16_t rc;
        rc = ycDnsScanCheckResourceRecord(payload, &offset, payloadSize);
        if (0 == rc) {
            /* yfTrace() called in ycDnsScanCheckResourceRecord() */
            return 0;
        }

        if (netbios
            && (rc != 1 && rc != 2 && rc != 10 && rc != 32 && rc != 33))
        {
            yfTrace(offset, "Bad rrtype %u for NetBIOS in additions", rc);
            return 0;
        }

#if YAF_ENABLE_HOOKS
        if (rc != DNS_TYPE_OPT) {
            recordCount++;
        }
#endif
    }

    if (0 == flow->appLabel) {
        /* this is the appLabel phase: skip the DPI */
        goto end;
    }
    /* There is no check for NETBIOS here since the an incoming NETBIOS
     * appLabel would have caused the function to return */
    g_assert(DNS_PORT_NUMBER == flow->appLabel
             || LLMNR_PORT_NUMBER == flow->appLabel
             || MDNS_PORT_NUMBER == flow->appLabel);

#if YAF_ENABLE_HOOKS
    if (val == &(flow->val)) {
        direction = 0;
    } else {
        direction = 1;
    }

#if defined(YAF_ENABLE_DNSAUTH) && defined(YAF_ENABLE_DNSNXDOMAIN)
    if ((header.aa == 1) || (header.rcode == 3)) {
        if (recordCount + header.qdcount) {
            yfHookScanPayload(flow, payload, 0, NULL,
                              (recordCount + header.qdcount), direction,
                              DNS_PORT_NUMBER);
        }
    }
#elif defined(YAF_ENABLE_DNSAUTH) && !defined(YAF_ENABLE_DNSNXDOMAIN)
    if (header.aa == 1) {
        if (recordCount + header.qdcount) {
            yfHookScanPayload(flow, payload, 0, NULL,
                              (recordCount + header.qdcount),
                              direction, DNS_PORT_NUMBER);
        }
    }
#elif defined(YAF_ENABLE_DNSNXDOMAIN) && !defined(YAF_ENABLE_DNSAUTH)
    if (header.rcode == 3) {
        if (recordCount + header.qdcount) {
            yfHookScanPayload(flow, payload, 0, NULL,
                              (recordCount + header.qdcount), direction,
                              DNS_PORT_NUMBER);
        }
    }
#else /* if defined(YAF_ENABLE_DNSAUTH) && defined(YAF_ENABLE_DNSNXDOMAIN) */
    if (header.qr && !(header.rcode)) {
        if (recordCount) {
            yfHookScanPayload(flow, payload, 0, NULL, recordCount, direction,
                              DNS_PORT_NUMBER);
        }
    } else {
        if (recordCount + header.qdcount) {
            yfHookScanPayload(flow, payload, 0, NULL,
                              (recordCount + header.qdcount),
                              direction, DNS_PORT_NUMBER);
        }
    }
#endif /* if defined(YAF_ENABLE_DNSAUTH) && defined(YAF_ENABLE_DNSNXDOMAIN) */
#endif /* if YAF_ENABLE_HOOKS */

#endif /* ifdef PAYLOAD_INSPECTION */

  end:
    if (netbios) {
        appLabel = NETBIOS_PORT_NUMBER;
    } else if (llmnr) {
        appLabel = LLMNR_PORT_NUMBER;
    } else if (mdns) {
        appLabel = MDNS_PORT_NUMBER;
    } else {
        appLabel = DNS_PORT_NUMBER;
    }

    yfTrace(0, "Returning %u", appLabel);
    return appLabel;
}


#ifdef PAYLOAD_INSPECTION
/**
 *    Extracts RR TYPE from `payload` at `offset`.  Checks the value and
 *    returns 0 if value is illegal or not one we support; otherwise returns
 *    the type.  Updates the `offset` (adds 2).
 *
 *    If 2 bytes of payload are not available when this function is called
 *    given the `offset` and `payloadSize`, the function leaves `offset`
 *    unchanged and returns 0.
 */
static inline uint16_t
ycDnsCheckRRType(
    const uint8_t  *payload,
    uint32_t        payloadSize,
    uint32_t       *offset)
{
    uint16_t rrtype;

    if ((*offset + sizeof(uint16_t)) > payloadSize) {
        TRACE_END_PAYLOAD(*offset, 2, payloadSize);
        return 0;
    }

    READ_U16_INC(rrtype, payload, *offset);

    /*
     *  Reject anything marked as Reserved or Private Use
     *  https://www.iana.org/assignments/dns-parameters/dns-parameters.xhtml#dns-parameters-4
     */
    if (0 == rrtype || rrtype > 65279) {
        yfTrace(*offset, "Bad rrtype %u", rrtype);
        return 0;
    }

    return rrtype;
}


static uint16_t
ycDnsScanCheckResourceRecord(
    const uint8_t  *payload,
    uint32_t       *offset,
    unsigned int    payloadSize)
{
    uint16_t nameSize;
    uint16_t rrType;
    uint16_t rrClass;
    uint16_t rdLength;
    gboolean compress_flag = FALSE;

    if (*offset >= payloadSize) {
        yfTrace(*offset, "Offset exceeds payloadSize (%u >= %u)",
                *offset, payloadSize);
        return 0;
    }

    nameSize  = *(payload + (*offset));

    while ((0 != nameSize) && (*offset < payloadSize)) {
        if (DNS_NAME_COMPRESSION == (nameSize & DNS_NAME_COMPRESSION)) {
            *offset += sizeof(uint16_t);
            if (!compress_flag) {
                compress_flag = TRUE;
            }
        } else {
            *offset += nameSize + 1;
        }
        if (*offset >= payloadSize) {
            yfTrace(*offset, "Offset exceeds payloadSize (%u >= %u)",
                    *offset, payloadSize);
            return 0;
        }
        nameSize = *(payload + (*offset));
    }

    if (!compress_flag) {
        *offset += 1;
    }

    /* check the type (checks that 2 bytes are available) */
    rrType = ycDnsCheckRRType(payload, payloadSize, offset);
    if (0 == rrType) {
        /* yfTrace() called in ycDnsCheckRRType() */
        return 0;
    }

    /* check the class */
    if ((*offset + 2) > payloadSize) {
        TRACE_END_PAYLOAD(*offset, 2, payloadSize);
        return 0;
    }
    READ_U16_INC(rrClass, payload, *offset);

    /* OPT Records use class field as UDP payload size */
    if (rrClass > 4 && rrType != DNS_TYPE_OPT) {
        /* rfc 2136 */
        if (rrClass != DNS_CLASS_NONE) {
            yfTrace(*offset, "Bad rrClass %u", rrClass);
            return 0;
        }
    }

    /* skip over the time to live */
    *offset += sizeof(uint32_t);

    /* get the record data length, (so we can skip ahead the right amount) */
    if ((*offset + 2) > payloadSize) {
        TRACE_END_PAYLOAD(*offset, 2, payloadSize);
        return 0;
    }
    READ_U16_INC(rdLength, payload, *offset);

    /* not going to try to parse the data record, what's in there depends on
     * the class and type fields, but the rdlength field always tells us how
     * many bytes are in it */
    *offset += rdLength;

    if (*offset > payloadSize) {
        yfTrace(*offset, "Offset exceeds payloadSize (%u >= %u)",
                *offset, payloadSize);
        return 0;
    }

    /* the record seems intact enough */
    return rrType;
}
#endif /* ifdef PAYLOAD_INSPECTION */
