/*
 *  Copyright 2007-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  nntpplugin.c
 *
 *  this provides NNTP payload packet recognition for use within YAF
 *  It is based on RFC 977 and some random limited packet capture.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Ecoff
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.18
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1281
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <payloadScanner.h>

#if YAF_ENABLE_HOOKS
#include <yaf/yafhooks.h>
#endif


#define NNTP_PORT_NUMBER 119

YC_SCANNER_PROTOTYPE(nntpplugin_LTX_ycNNTPScanScan);

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre  *nntpCommandRegex = NULL;
static pcre  *nntpResponseRegex = NULL;
static pcre  *nntpApplabelRegex = NULL;

/* pcreInitialized is 1 if initialization succeeded, -1 if initialization
 * failed, and 0 if initialization has not been attempted */
static int pcreInitialized = 0;


/**
 * static local functions
 *
 */
static gboolean
ycNNTPScanInit(
    void);

/*static int ycDebugBinPrintf(uint8_t *data, uint16_t size);*/


/**
 * nntpplugin_LTX_ycNNTPScanScan
 *
 * scans a given payload to see if it conforms to our idea of what NNTP traffic
 * looks like.
 *
 *
 *
 * @param argc NOT USED
 * @param argv NOT USED
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match NNTP_PORT_NUMBER (119) for a match
 *
 */
uint16_t
nntpplugin_LTX_ycNNTPScanScan(
    int             argc,
    char           *argv[],
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
#define NUM_CAPT_VECTS 60
    int vects[NUM_CAPT_VECTS];

    /* return if the applabel has been set and it does not match the
     * applabel(s) returned by this function */
    if (0 != flow->appLabel && NNTP_PORT_NUMBER != flow->appLabel) {
        return 0;
    }

    if (1 != pcreInitialized) {
        if (-1 != pcreInitialized) {
            pcreInitialized = ycNNTPScanInit() ? 1 : -1;
        }
        if (-1 == pcreInitialized) {
            return 0;
        }
    }

    if (0 == flow->appLabel) {
        if (pcre_exec(nntpApplabelRegex, NULL, (char *)payload, payloadSize,
                      0, 0, vects, NUM_CAPT_VECTS) > 0)
        {
            return NNTP_PORT_NUMBER;
        }
        return 0;
    }
    g_assert(NNTP_PORT_NUMBER == flow->appLabel);

    /** at some point in the future, this is the place to extract protocol
     *  information like message targets and join targets, etc.*/

#if YAF_ENABLE_HOOKS
    yfHookScanPayload(flow, payload, payloadSize, nntpCommandRegex, 0,
                      173, NNTP_PORT_NUMBER);
    yfHookScanPayload(flow, payload, payloadSize, nntpResponseRegex, 0,
                      172, NNTP_PORT_NUMBER);
#endif /* if YAF_ENABLE_HOOKS */

    return NNTP_PORT_NUMBER;
}


/**
 * ycNNTPScanInit
 *
 * this initializes the PCRE expressions needed to search the payload for
 * NNTP
 *
 *
 * @return TRUE if initialization completed correctly, FALSE otherwise
 */
static gboolean
ycNNTPScanInit(
    void)
{
    /* TODO: RFC-3977 allows UTF-8 but the regexes here assume ASCII */

    /* An NNTP command is a case insensitive keyword, perhaps followed by
     * space or tab and arbitrary ascii arguments, followed by CRLF, not to
     * exceed 512 octets total.  The arguments are limited to 497 octets, and
     * they may include a message-id which is why we include all printable
     * ASCII plus space and tab. */

    /* This macro begins the regex and includes the optional arguments; the
     * regex with the keywords is the argument to this macro */
#define YF_NNTP_CMD(keywords)                                   \
    "(?im)^(" keywords "(?:[ \\t][\\t -~]{0,497})?)\\r\\n"

    /* For the appLabel check, use a limited set of commands that are unique
     * to NNTP to help avoid mislabeling as another protocol. */
    const char  nntpApplabelString[] =
        YF_NNTP_CMD("(?:ARTICLE|GROUP|IHAVE|LAST|MODE|"
                    "NEWGROUPS|NEWNEWS|NEXT|POST|SLAVE)");

    /* The commands to capture in the DPI */
    const char  nntpCommandRegexString[] =
        YF_NNTP_CMD("(?:ARTICLE|GROUP|HELP|IHAVE|LAST|LIST|MODE|"
                    "NEWGROUPS|NEWNEWS|NEXT|POST|QUIT|SLAVE|STAT)");

    /* The first line of a response is a three digit response code, a space,
     * the rest of the response, followed by a CRLF, not to exceed 512 octets.
     * We do not capture the additional lines in a multi-line response. */
    const char  nntpResponseRegexString[] =
        "(?m)^([1-5][0-48-9][0-9] [!-~][\\t -~]{1,505})\\r\\n";

    GError     *err = NULL;

    if (!ycPcreCompile(&nntpCommandRegex, nntpCommandRegexString, 0, &err) ||
        !ycPcreCompile(&nntpResponseRegex, nntpResponseRegexString, 0, &err) ||
        !ycPcreCompile(&nntpApplabelRegex, nntpApplabelString, 0, &err))
    {
        goto ERROR;
    }

    return TRUE;
  ERROR:
    g_warning("%s", err->message);
    g_clear_error(&err);
    return FALSE;
}


#if 0
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size)
{
    uint16_t loop;
    int      numPrinted = 0;

    if (0 == size) {
        return 0;
    }

    for (loop = 0; loop < size; loop++) {
        if (isprint(*(data + loop)) && !iscntrl(*(data + loop))) {
            printf("%c", *(data + loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data + loop) || '\r' == *(data + loop)
            || '\0' == (data + loop))
        {
            break;
        }
        numPrinted++;
    }

    return numPrinted;
}
#endif /* 0 */
