/*
 *  Copyright 2007-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  pop3plugin.c
 *
 *  this provides POP3 payload packet recognition for use within YAF
 *  It is based on RFC 1939 and some random limited packet capture.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Dan Ruef, Emily Ecoff
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.18
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1281
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <payloadScanner.h>
#include "libmidencryption.h"

#if YAF_ENABLE_HOOKS
#include <yaf/yafhooks.h>
#endif

#define POP3DEBUG 0
#define POP3_PORT_NUMBER 110

YC_SCANNER_PROTOTYPE(pop3plugin_LTX_ycPop3ScanScan);

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre *pop3RegexApplabel = NULL;
#if YAF_ENABLE_HOOKS
static pcre *pop3RegexRequest   = NULL;
static pcre *pop3RegexResponse  = NULL;
static pcre *pop3STLS           = NULL;
static pcre *pop3STLSReply      = NULL;
#endif /* if YAF_ENABLE_HOOKS */

/* pcreInitialized is 1 if initialization succeeded, -1 if initialization
 * failed, and 0 if initialization has not been attempted */
static int pcreInitialized = 0;


/**
 * static local functions
 *
 */

static gboolean
ycPop3ScanInit(
    void);

#if POP3DEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size);
#endif /* if POP3DEBUG */

/**
 * pop3plugin_LTX_ycPop3ScanScan
 *
 * scans a given payload to see if it conforms to our idea of what POP3 traffic
 * looks like.
 *
 *
 *
 * @param argc NOT USED
 * @param argv NOT USED
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match POP3_PORT_NUMBER (110) for a match
 *
 */
uint16_t
pop3plugin_LTX_ycPop3ScanScan(
    int             argc,
    char           *argv[],
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    int rc;
#define NUM_CAPT_VECTS 60
    int vects[NUM_CAPT_VECTS];

    /* return if the applabel has been set and it does not match the
     * applabel(s) returned by this function */
    if (0 != flow->appLabel && POP3_PORT_NUMBER != flow->appLabel) {
        return 0;
    }

    if (1 != pcreInitialized) {
        if (-1 != pcreInitialized) {
            pcreInitialized = ycPop3ScanInit() ? 1 : -1;
        }
        if (-1 == pcreInitialized) {
            return 0;
        }
    }

    rc = pcre_exec(pop3RegexApplabel, NULL, (char *)payload, payloadSize, 0,
                   0, vects, NUM_CAPT_VECTS);
    if (0 == flow->appLabel) {
        if (rc > 0) {
            return POP3_PORT_NUMBER;
        }
        return 0;
    }
    g_assert(POP3_PORT_NUMBER == flow->appLabel);

#if YAF_ENABLE_HOOKS
    if (rc == 2) {
        int      tmp;
        uint32_t offset;

        /* server side */
        yfHookScanPayload(flow, payload, payloadSize, pop3RegexResponse, 0,
                          111, POP3_PORT_NUMBER);
        tmp = pcre_exec(pop3STLSReply, NULL, (char *)payload, payloadSize, 0,
                        0, vects, NUM_CAPT_VECTS);
        offset = vects[1];

        if (tmp == 1) {
            yfHookScanPayload(flow, payload, payloadSize, pop3STLSReply, 0,
                              113, POP3_PORT_NUMBER);
            Mid_Encryption_Scan_Payload(payload, payloadSize, flow, val,
                                        offset, POP3_PORT_NUMBER);
        }
    } else {
        int      tmp;
        uint32_t offset;

        /* client side */
        yfHookScanPayload(flow, payload, payloadSize, pop3RegexRequest, 0,
                          110, POP3_PORT_NUMBER);
        tmp = pcre_exec(pop3STLS, NULL, (char *)payload, payloadSize, 0,
                        0, vects, NUM_CAPT_VECTS);
        offset = vects[1];

        if (tmp == 1) {
            yfHookScanPayload(flow, payload, payloadSize, pop3STLS, 0,
                              112, POP3_PORT_NUMBER);
            Mid_Encryption_Scan_Payload(payload, payloadSize, flow, val,
                                        offset, POP3_PORT_NUMBER);
        }
    }
#endif /* if YAF_ENABLE_HOOKS */

    return POP3_PORT_NUMBER;
}


/**
 * ycPop3ScanInit
 *
 * this initializes the PCRE expressions needed to search the payload for
 * POP3
 *
 *
 * @return TRUE if initialization completed correctly, FALSE otherwise
 */
static gboolean
ycPop3ScanInit(
    void)
{
    GError  *err = NULL;
#if YAF_ENABLE_HOOKS
    /* capture everything the client says */
    const char  pop3StringRequest[] =  "(?im)^[ \\t]*([!-~][ !-~]+)";

    /* capture the first line of each response */
    const char  pop3StringResponse[] = "(?m)^((?:\\+OK|-ERR)[ -~]*)";

    const char  pop3StringSTLS[] = "STLS\\r\\n";

    const char  pop3StringSTLSReply[] = "\\+OK Begin TLS negotiation now\\r\\n";
#endif /* if YAF_ENABLE_HOOKS */

    /* used to determine if this connection looks like POP3; capture the
     * response to distinguish the server from the client */
    const char pop3StringApplabel[] =
        "(?i)^\\s*(?:(?:CAPA\\b|STLS|AUTH\\s(?:KERBEROS_V|GSSAPI|SKEY)|"
        "UIDL\\b|APOP\\s|USER\\s)|(\\+OK\\b|-ERR\\b))";

    if (!ycPcreCompile(&pop3RegexApplabel, pop3StringApplabel, 0, &err)) {
        goto ERROR;
    }

#if YAF_ENABLE_HOOKS
    if (!ycPcreCompile(&pop3RegexRequest, pop3StringRequest, 0, &err) ||
        !ycPcreCompile(&pop3RegexResponse, pop3StringResponse, 0, &err) ||
        !ycPcreCompile(&pop3STLS, pop3StringSTLS, 0, &err) ||
        !ycPcreCompile(&pop3STLSReply, pop3StringSTLSReply, 0, &err))
    {
        goto ERROR;
    }
#endif  /* YAF_ENABLE_HOOKS */

    return TRUE;
  ERROR:
    g_warning("%s", err->message);
    g_clear_error(&err);
    return FALSE;
}


#if POP3DEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size)
{
    uint16_t loop;
    int      numPrinted = 0;

    for (loop = 0; loop < size; loop++) {
        if (isprint(*(data + loop)) && !iscntrl(*(data + loop))) {
            printf("%c", *(data + loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data + loop) || '\r' == *(data + loop)
            || '\0' == *(data + loop))
        {
            break;
        }
        numPrinted++;
    }

    return numPrinted;
}
#endif /* if POP3DEBUG */
