/*
 *  Copyright 2006-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  yaf.c
 *  Yet Another Flow generator
 *
 *  ------------------------------------------------------------------------
 *  Authors: Brian Trammell
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.18
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1281
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <airframe/logconfig.h>
#include <airframe/privconfig.h>
#include <airframe/airutil.h>
#include <airframe/airopt.h>
#include <yaf/yafcore.h>
#include <yaf/yaftab.h>
#include <yaf/yafrag.h>

#include "yafcap.h"
#include "yafstat.h"
#include "yafctx.h"
#if YAF_ENABLE_DAG
#include "yafdag.h"
#endif
#if YAF_ENABLE_NAPATECH
#include "yafpcapx.h"
#endif
#if YAF_ENABLE_NETRONOME
#include "yafnfe.h"
#endif
#if YAF_ENABLE_PFRING
#include "yafpfring.h"
#endif
#if YAF_ENABLE_APPLABEL
#include "yafapplabel.h"
#endif
#if YAF_ENABLE_HOOKS
#include <yaf/yafhooks.h>
#endif
#if YAF_ENABLE_P0F
#include "applabel/p0f/yfp0f.h"
#endif
#include <lua.h>
#include <lauxlib.h>
#include <lualib.h>

/* wrap this around string literals that are assigned to variables of type
 * "char *" to quiet compiler warnings */
#define C(String) (char *)String

#define DEFAULT_VXLAN_PORT 4789
#define DEFAULT_GENEVE_PORT 6081

/* I/O configuration */
static yfConfig_t yaf_config = YF_CONFIG_INIT;
static char      *yaf_config_file = NULL;
static int        yaf_opt_rotate = 0;
static int        yaf_opt_stats = 300;
static gboolean   yaf_opt_no_tombstone = FALSE;
static uint16_t   yaf_opt_configured_id = 0;
static gboolean   yaf_opt_caplist_mode = FALSE;
static char      *yaf_opt_ipfix_transport = NULL;
static gboolean   yaf_opt_ipfix_tls = FALSE;
static char      *yaf_pcap_meta_file = NULL;
static gboolean   yaf_index_pcap = FALSE;
static gboolean   yaf_daemon = FALSE;
static char      *yaf_pidfile = NULL;
static char      *yaf_tmp_dir = NULL;
static int        yaf_opt_udp_temp_timeout = 600;
static gboolean   yaf_opt_promisc = FALSE;
#ifdef HAVE_SPREAD
/* spread config options */
static char      *yaf_opt_spread_group = 0;
static char      *yaf_opt_spread_groupby = 0;
#endif

/* GOption managed flow table options */
static int       yaf_opt_idle = 300;
static int       yaf_opt_active = 1800;
static int       yaf_opt_max_flows = 0;
static int       yaf_opt_max_payload = 0;
static char     *yaf_opt_max_payload_str = NULL;
static int       yaf_opt_payload_export_max = 0;
static char     *yaf_opt_payload_export_max_str = NULL;
static gboolean  yaf_opt_payload_export_enable = FALSE;
static gboolean  yaf_opt_force_read_all = FALSE;

#if YAF_ENABLE_APPLABEL
static gboolean yaf_opt_applabel_mode = FALSE;
static char    *yaf_opt_applabel_rules = NULL;
static int      yaf_opt_applabel_check_early = 0;
static char    *yaf_opt_applabel_check_early_str = 0;

/* yaf_opt_applabel_max_payload is the amount of payload to store internally
 * per applabel for DPI detection */
static GArray  *yaf_opt_applabel_max_payload = NULL;
static char    *yaf_opt_applabel_max_payload_str = NULL;
/* yaf_opt_payload_applabel_select is the amount of payload to export in the
 * record (as an octetArray) per applabel */
static GArray  *yaf_opt_payload_applabel_select = NULL;
static char    *yaf_opt_payload_applabel_select_str = NULL;
#endif  /* YAF_ENABLE_APPLABEL */

static gboolean yaf_opt_ndpi = FALSE;
static char    *yaf_ndpi_proto_file = NULL;
static gboolean yaf_opt_entropy_mode = FALSE;
static gboolean yaf_opt_uniflow_mode = FALSE;
static uint16_t yaf_opt_udp_uniflow_port = 0;
static gboolean yaf_opt_silk_mode = FALSE;
static gboolean yaf_opt_p0fprint_mode = FALSE;
#if YAF_ENABLE_P0F
static char    *yaf_opt_p0f_fingerprints = NULL;
#endif
static gboolean yaf_opt_fpExport_mode = FALSE;
static gboolean yaf_opt_udp_multipkt_payload = FALSE;
static gboolean yaf_opt_flowstats_mode = FALSE;
static int      yaf_opt_max_pcap = 25;
static int      yaf_opt_pcap_timer = 0;
static int64_t  yaf_hash_search = 0;
static char    *yaf_stime_search = NULL;
static int      yaf_opt_ingress_int = 0;
static int      yaf_opt_egress_int = 0;
static gboolean yaf_novlan_in_key;
static char    *yaf_opt_time_elements = NULL;

/* GOption managed fragment table options */
static int      yaf_opt_max_frags = 0;
static gboolean yaf_opt_nofrag = FALSE;

/* GOption managed decoder options and derived decoder config */
static gboolean yaf_opt_ip4_mode = FALSE;
static gboolean yaf_opt_ip6_mode = FALSE;
static uint16_t yaf_reqtype;
static gboolean yaf_opt_gre_mode = FALSE;
static gboolean yaf_opt_vxlan_mode = FALSE;
static GArray  *yaf_opt_vxlan_ports = NULL;
static char    *yaf_opt_vxlan_ports_str = NULL;
static gboolean yaf_opt_geneve_mode = FALSE;
static GArray  *yaf_opt_geneve_ports = NULL;
static char    *yaf_opt_geneve_ports_str = NULL;
static gboolean yaf_opt_mac_mode = FALSE;

#ifdef YAF_ENABLE_HOOKS
static char    *pluginName = NULL;
static char    *pluginOpts = NULL;
static char    *pluginConf = NULL;
static gboolean hooks_initialized = FALSE;
#endif /* ifdef YAF_ENABLE_HOOKS */
/* array of configuration information that is passed to flow table */
static void    *yfctx[YAF_MAX_HOOKS];

/* global quit flag */
int             yaf_quit = 0;

/* the (first) shutdown signal we receive */
static volatile int yaf_quit_signal = 0;

/* Runtime functions */

typedef void *(*yfLiveOpen_fn)(
    const char *ifname,
    int         snaplen,
    int        *datalink,
    GError    **err);
static yfLiveOpen_fn yaf_liveopen_fn = NULL;

typedef gboolean (*yfLoop_fn)(
    yfContext_t *ctx);
static yfLoop_fn yaf_loop_fn = NULL;

typedef void (*yfClose_fn)(
    void *pktsrc);
static yfClose_fn yaf_close_fn = NULL;

/* Create an array for the supported live capture types */
static const struct yaf_live_type_st {
    /* argument to --live to select this type */
    const char    *name;
    yfLiveOpen_fn  open_fn;
    yfLoop_fn      loop_fn;
    yfClose_fn     close_fn;
} yaf_live_type[] = {
    {
        "pcap",
        (yfLiveOpen_fn)yfCapOpenLive,
        (yfLoop_fn)yfCapMain,
        (yfClose_fn)yfCapClose
    },
#if YAF_ENABLE_DAG
    {
        "dag",
        (yfLiveOpen_fn)yfDagOpenLive,
        (yfLoop_fn)yfDagMain,
        (yfClose_fn)yfDagClose
    },
#endif /* if YAF_ENABLE_DAG */
#if YAF_ENABLE_NAPATECH
    {
        "napatech",
        (yfLiveOpen_fn)yfPcapxOpenLive,
        (yfLoop_fn)yfPcapxMain,
        (yfClose_fn)yfPcapxClose
    },
#endif /* if YAF_ENABLE_NAPATECH */
#if YAF_ENABLE_NETRONOME
    {
        "netronome",
        (yfLiveOpen_fn)yfNFEOpenLive,
        (yfLoop_fn)yfNFEMain,
        (yfClose_fn)yfNFEClose
    },
#endif /* if YAF_ENABLE_NETRONOME */
#if YAF_ENABLE_PFRING
    {
        "pfring",
        (yfLiveOpen_fn)yfPfRingOpenLive,
        (yfLoop_fn)yfPfRingMain,
        (yfClose_fn)yfPfRingClose
    },
#if YAF_ENABLE_PFRINGZC
    {
        "zc",
        (yfLiveOpen_fn)yfPfRingZCOpenLive,
        (yfLoop_fn)yfPfRingZCMain,
        (yfClose_fn)yfPfRingZCClose
    },
#endif /* if YAF_ENABLE_PFRINGZC */
#endif /* if YAF_ENABLE_PFRING */
    {
        /* sentinel */
        NULL, NULL, NULL, NULL
    }
};


#ifdef USE_GOPTION
#define AF_OPTION_WRAP "\n\t\t\t\t"
#else
#define AF_OPTION_WRAP " "
#endif

/* Local functions prototypes */

static uint32_t
yaf_opt_parse_payload(
    const gchar  *option_name,
    gchar        *payload_str,
    long long     max_value);

#if YAF_ENABLE_APPLABEL
static void
yaf_opt_parse_applabel_max_payload(
    const gchar  *applabel_max_payload);
#endif  /* YAF_ENABLE_APPLABEL */

static void
yaf_opt_ports_str_2_array(
    const gchar  *option_name,
    const gchar  *ports_str,
    GArray       *ports_array);

static void
yaf_opt_remove_array_dups(
    GArray *g);

#if YAF_ENABLE_HOOKS
static void
pluginOptParse(
    GError **err);
#endif /* if YAF_ENABLE_HOOKS */


/* Local derived configuration */

static AirOptionEntry yaf_optent_core[] = {
    AF_OPTION("in", 'i', 0, AF_OPT_TYPE_STRING, &yaf_config.inspec,
              AF_OPTION_WRAP "Set input (file, - for stdin; interface) [-]",
              "inspec"),
    AF_OPTION("out", 'o', 0, AF_OPT_TYPE_STRING, &yaf_config.outspec,
              AF_OPTION_WRAP "Set output (file, - for stdout; file prefix;"
              AF_OPTION_WRAP "address) [-]",
              "outspec"),
    AF_OPTION("config", 'c', 0, AF_OPT_TYPE_STRING, &yaf_config_file,
              AF_OPTION_WRAP "Read YAF configuration settings from Lua file",
              "file"),
#ifdef HAVE_SPREAD
    AF_OPTION("group", 'g', 0, AF_OPT_TYPE_STRING, &yaf_opt_spread_group,
              AF_OPTION_WRAP "Spread group name (comma seperated list)."
              AF_OPTION_WRAP "For groupby: comma separated"
              AF_OPTION_WRAP "group_name:value,[group_name:value,...]",
              "group-name"),
    AF_OPTION("groupby", 0, 0, AF_OPT_TYPE_STRING, &yaf_opt_spread_groupby,
              AF_OPTION_WRAP "<port, vlan, applabel, protocol, version>"
              AF_OPTION_WRAP "(Must be used with group and group must have"
              AF_OPTION_WRAP "values to groupby",
              "type"),
#endif /* ifdef HAVE_SPREAD */
    AF_OPTION("live", 'P', 0, AF_OPT_TYPE_STRING, &yaf_config.livetype,
              AF_OPTION_WRAP "Capture from interface in --in; type is"
              AF_OPTION_WRAP "pcap, dag, napatech, netronome, pfring, zc",
              "type"),
    AF_OPTION("filter", 'F', 0, AF_OPT_TYPE_STRING, &yaf_config.bpf_expr,
              AF_OPTION_WRAP "Set BPF filtering expression",
              "expression"),
    AF_OPTION("caplist", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_caplist_mode,
              AF_OPTION_WRAP "Read an ordered list of input files from"
              AF_OPTION_WRAP "the file specified in --in",
              NULL),
#if YAF_ENABLE_ZLIB
    AF_OPTION("decompress", 0, 0, AF_OPT_TYPE_STRING, &yaf_tmp_dir,
              AF_OPTION_WRAP "Set directory for decompressing files [$TMPDIR]",
              "dir"),
#endif
#ifdef HAVE_SPREAD
    AF_OPTION("ipfix", 0, 0, AF_OPT_TYPE_STRING, &yaf_opt_ipfix_transport,
              AF_OPTION_WRAP "Export via IPFIX (tcp, udp, sctp, spread) to"
              AF_OPTION_WRAP "collector process at address in --out",
              "protocol"),
#else /* ifdef HAVE_SPREAD */
    AF_OPTION("ipfix", 0, 0, AF_OPT_TYPE_STRING, &yaf_opt_ipfix_transport,
              AF_OPTION_WRAP "Export via IPFIX (tcp, udp, sctp) to collector"
              AF_OPTION_WRAP "process at address in --out",
              "protocol"),
#endif /* ifdef HAVE_SPREAD */
    AF_OPTION("rotate", 'R', 0, AF_OPT_TYPE_INT, &yaf_opt_rotate,
              AF_OPTION_WRAP "Write multiple IPFIX files rotating the output"
              AF_OPTION_WRAP "files every sec seconds [0(single file)]",
              "sec"),
    AF_OPTION("lock", 'k', 0, AF_OPT_TYPE_NONE, &yaf_config.lockmode,
              AF_OPTION_WRAP "Use exclusive .lock files on output for"
              AF_OPTION_WRAP "concurrency",
              NULL),
    AF_OPTION("daemonize", 'd', 0, AF_OPT_TYPE_NONE, &yaf_daemon,
              AF_OPTION_WRAP "Daemonize yaf",
              NULL),
    AF_OPTION("pidfile", 0, 0, AF_OPT_TYPE_STRING, &yaf_pidfile,
              AF_OPTION_WRAP "Set complete path to the process ID file",
              "path"),
    AF_OPTION("promisc-off", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_promisc,
              AF_OPTION_WRAP "Do not put the interface in promiscuous mode",
              NULL),
    AF_OPTION("noerror", 0, 0, AF_OPT_TYPE_NONE, &yaf_config.noerror,
              AF_OPTION_WRAP "Do not error out on single PCAP file issue"
              AF_OPTION_WRAP "with multiple inputs",
              NULL),
    AF_OPTION_END
};

static AirOptionEntry yaf_optent_dec[] = {
    AF_OPTION("no-frag", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_nofrag,
              AF_OPTION_WRAP "Disable IP fragment reassembly",
              NULL),
    AF_OPTION("max-frags", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_max_frags,
              AF_OPTION_WRAP "Set maximum size of fragment table [0(no limit)]",
              "fragments"),
    AF_OPTION("ip4-only", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ip4_mode,
              AF_OPTION_WRAP "Process only IPv4 packets",
              NULL),
    AF_OPTION("ip6-only", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ip6_mode,
              AF_OPTION_WRAP "Process only IPv6 packets",
              NULL),
    AF_OPTION("gre-decode", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_gre_mode,
              AF_OPTION_WRAP "Decode GRE encapsulated packets",
              NULL),
    AF_OPTION("vxlan-decode", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_vxlan_mode,
              AF_OPTION_WRAP "Decode VxLAN encapsulated packets on port 4789",
              NULL),
    AF_OPTION("vxlan-decode-ports", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_vxlan_ports_str,
              AF_OPTION_WRAP "Change ports where VxLAN decoding occurs [4789]",
              "port[,port...]"),
    AF_OPTION("geneve-decode", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_geneve_mode,
              AF_OPTION_WRAP "Decode Geneve encapsulated packets on port 6801",
              NULL),
    AF_OPTION("geneve-decode-ports", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_geneve_ports_str,
              AF_OPTION_WRAP "Change ports where Geneve decoding occurs [6081]",
              "port[,port...]"),
    AF_OPTION_END
};

static AirOptionEntry yaf_optent_flow[] = {
    AF_OPTION("idle-timeout", 'I', 0, AF_OPT_TYPE_INT, &yaf_opt_idle,
              AF_OPTION_WRAP "Set idle flow timeout [300(5min)]",
              "sec"),
    AF_OPTION("active-timeout", 'A', 0, AF_OPT_TYPE_INT, &yaf_opt_active,
              AF_OPTION_WRAP "Set active flow timeout [1800(30min)]",
              "sec"),
    AF_OPTION("max-flows", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_max_flows,
              AF_OPTION_WRAP "Set maximum size of flow table [0(no limit)]",
              "flows"),
    AF_OPTION("udp-temp-timeout", 0, 0, AF_OPT_TYPE_INT,
              &yaf_opt_udp_temp_timeout,
              AF_OPTION_WRAP "Set UDP template timeout period [600(10min)]",
              "sec"),
    AF_OPTION("force-read-all", 0, 0, AF_OPT_TYPE_NONE,
              &yaf_opt_force_read_all,
              AF_OPTION_WRAP "Force read of any out of sequence packets",
              NULL),
    AF_OPTION("no-vlan-in-key", 0, 0, AF_OPT_TYPE_NONE, &yaf_novlan_in_key,
              AF_OPTION_WRAP "Do not use the VLAN in the flow key hash"
              AF_OPTION_WRAP "calculation",
              NULL),
    AF_OPTION_END
};

static AirOptionEntry yaf_optent_exp[] = {
    AF_OPTION("no-output", 0, 0, AF_OPT_TYPE_NONE, &yaf_config.no_output,
              AF_OPTION_WRAP "Turn off IPFIX export",
              NULL),
    AF_OPTION("no-stats", 0, 0, AF_OPT_TYPE_NONE, &yaf_config.nostats,
              AF_OPTION_WRAP "Turn off stats option records IPFIX export",
              NULL),
    AF_OPTION("stats", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_stats,
              AF_OPTION_WRAP "Export yaf process statistics every sec seconds;"
              AF_OPTION_WRAP "if 0, disable stats export [300(5min)]",
              "sec"),
    AF_OPTION("no-tombstone", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_no_tombstone,
              AF_OPTION_WRAP "Turn off export of tombstone records",
              NULL),
    AF_OPTION("tombstone-configured-id", 0, 0, AF_OPT_TYPE_INT,
              &yaf_opt_configured_id,
              AF_OPTION_WRAP "Set tombstone record's 16 bit configured"
              AF_OPTION_WRAP "identifier [0]",
              "ident"),
    AF_OPTION("silk", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_silk_mode,
              AF_OPTION_WRAP "Clamp octets to 32 bits, note continued in"
              AF_OPTION_WRAP "flowEndReason, export TCP Fields within"
              AF_OPTION_WRAP "flow record instead of subTemplateMultiList",
              NULL),
    AF_OPTION("mac", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_mac_mode,
              AF_OPTION_WRAP "Export MAC-layer information",
              NULL),
    AF_OPTION("uniflow", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_uniflow_mode,
              AF_OPTION_WRAP "Write uniflows for compatibility",
              NULL),
    AF_OPTION("udp-uniflow", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_udp_uniflow_port,
              AF_OPTION_WRAP "Export a single UDP packet as a flow on the"
              AF_OPTION_WRAP "given port. Use 1 for all ports [0(disabled)]",
              "port"),
    AF_OPTION("force-ip6-export", 0, 0, AF_OPT_TYPE_NONE, &yaf_config.force_ip6,
              AF_OPTION_WRAP "Export all IPv4 addresses as IPv6 in ::ffff/96",
              NULL),
    AF_OPTION("observation-domain", 0, 0, AF_OPT_TYPE_INT, &yaf_config.odid,
              AF_OPTION_WRAP "Set observationDomainID on exported"
              AF_OPTION_WRAP "messages [0]",
              "odId"),
    AF_OPTION("flow-stats", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_flowstats_mode,
              AF_OPTION_WRAP "Export extra flow attributes and statistics",
              NULL),
    AF_OPTION("delta", 0, 0, AF_OPT_TYPE_NONE, &yaf_config.deltaMode,
              AF_OPTION_WRAP "Export packet and octet counts using delta"
              AF_OPTION_WRAP "information elements",
              NULL),
    AF_OPTION("ingress", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_ingress_int,
              AF_OPTION_WRAP "Set ingressInterface field in flow template [0]",
              "ingressId"),
    AF_OPTION("egress", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_egress_int,
              AF_OPTION_WRAP "Set egressInterface field in flow template [0]",
              "egressId"),
#if YAF_ENABLE_METADATA_EXPORT
    AF_OPTION("metadata-export", 0, 0, AF_OPT_TYPE_NONE,
              &yaf_config.tmpl_metadata,
              AF_OPTION_WRAP "Export template and information element"
              AF_OPTION_WRAP "metadata before data",
              NULL),
#endif /* if YAF_ENABLE_METADATA_EXPORT */
#if YAF_ENABLE_DAG_SEPARATE_INTERFACES || YAF_ENABLE_SEPARATE_INTERFACES
    AF_OPTION("export-interface", 0, 0, AF_OPT_TYPE_NONE,
              &yaf_config.exportInterface,
              AF_OPTION_WRAP "Export DAG, Napatech, or Netronome interface"
              AF_OPTION_WRAP "numbers in export records",
              NULL),
#endif /* if YAF_ENABLE_DAG_SEPARATE_INTERFACES ||
        * YAF_ENABLE_SEPARATE_INTERFACES */
    AF_OPTION("time-elements", 0, 0, AF_OPT_TYPE_STRING, &yaf_opt_time_elements,
              AF_OPTION_WRAP "Export flow timestamps in these elements [1,2]."
              AF_OPTION_WRAP "Choices:"
              AF_OPTION_WRAP "1. flowStartMilliseconds, flowEndMilliseconds"
              AF_OPTION_WRAP "2. flowStartMicroseconds, flowEndMicroseconds"
              AF_OPTION_WRAP "3. flowStartNanoseconds, flowEndNanoseconds",
              "choice[,choice...]"),
    AF_OPTION_END
};

static AirOptionEntry yaf_optent_ipfix[] = {
    AF_OPTION("ipfix-port", 0, 0, AF_OPT_TYPE_STRING,
              &(yaf_config.connspec.svc),
              AF_OPTION_WRAP "Select IPFIX export port [4739, 4740]",
              "port"),
    AF_OPTION("tls", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ipfix_tls,
              AF_OPTION_WRAP "Use TLS/DTLS to secure IPFIX export",
              NULL),
    AF_OPTION("tls-ca", 0, 0, AF_OPT_TYPE_STRING,
              &(yaf_config.connspec.ssl_ca_file),
              AF_OPTION_WRAP "Specify TLS Certificate Authority file",
              "cafile"),
    AF_OPTION("tls-cert", 0, 0, AF_OPT_TYPE_STRING,
              &(yaf_config.connspec.ssl_cert_file),
              AF_OPTION_WRAP "Specify TLS Certificate file",
              "certfile"),
    AF_OPTION("tls-key", 0, 0, AF_OPT_TYPE_STRING,
              &(yaf_config.connspec.ssl_key_file),
              AF_OPTION_WRAP "Specify TLS Private Key file",
              "keyfile"),
    AF_OPTION_END
};

static AirOptionEntry yaf_optent_pcap[] = {
    AF_OPTION("pcap", 'p', 0, AF_OPT_TYPE_STRING, &yaf_config.pcapdir,
              AF_OPTION_WRAP "Set Directory/File prefix to use for storing"
              AF_OPTION_WRAP "rolling pcap files",
              "dir"),
    AF_OPTION("pcap-per-flow", 0, 0, AF_OPT_TYPE_NONE,
              &yaf_config.pcap_per_flow,
              AF_OPTION_WRAP "Create a separate pcap file for each flow in"
              AF_OPTION_WRAP "the --pcap directory",
              NULL),
    AF_OPTION("max-pcap", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_max_pcap,
              AF_OPTION_WRAP "Set maximum size of a pcap file, in MB [25]",
              "MB"),
    AF_OPTION("pcap-timer", 0, 0, AF_OPT_TYPE_INT, &yaf_opt_pcap_timer,
              AF_OPTION_WRAP "Rotate pcap files after this time [300(5min)]",
              "sec"),
    AF_OPTION("pcap-meta-file", 0, 0, AF_OPT_TYPE_STRING, &yaf_pcap_meta_file,
              AF_OPTION_WRAP "Enable metadata file for rolling pcap output or"
              AF_OPTION_WRAP "indexing input pcap",
              "path"),
    AF_OPTION("index-pcap", 0, 0, AF_OPT_TYPE_NONE, &yaf_index_pcap,
              AF_OPTION_WRAP "Index the pcap with offset and lengths"
              AF_OPTION_WRAP "per packet",
              NULL),
    AF_OPTION("hash", 0, 0, AF_OPT_TYPE_INT64, &yaf_hash_search,
              AF_OPTION_WRAP "Create a PCAP file only for the given hash",
              "hash"),
    AF_OPTION("stime", 0, 0, AF_OPT_TYPE_STRING, &yaf_stime_search,
              AF_OPTION_WRAP "Create a PCAP file only for the given stime in"
              AF_OPTION_WRAP "epoch milliseconds (--hash must also be present)",
              "ms"),
    AF_OPTION_END
};


#if YAF_ENABLE_PAYLOAD
static AirOptionEntry yaf_optent_payload[] = {
    AF_OPTION("max-payload", 's', 0, AF_OPT_TYPE_STRING,
              &yaf_opt_max_payload_str,
              AF_OPTION_WRAP "Set max payload to capture per flow for each"
              AF_OPTION_WRAP "direction; use k suffix for kilobytes [0(none)]",
              "octets"),
    AF_OPTION("export-payload", 0, 0, AF_OPT_TYPE_NONE,
              &yaf_opt_payload_export_enable,
              AF_OPTION_WRAP "Enable payload export up to --max-export limit",
              NULL),
#if YAF_ENABLE_APPLABEL
    AF_OPTION("payload-applabel-select", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_payload_applabel_select_str,
              AF_OPTION_WRAP "Enable payload export but for only flows that"
              AF_OPTION_WRAP "match these silkApplabels",
              "appLabel[,appLabel...]"),
#endif  /* YAF_ENABLE_APPLABEL */
    AF_OPTION("max-export", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_payload_export_max_str,
              AF_OPTION_WRAP "Set maximum payload to export per flow direction"
              AF_OPTION_WRAP "when export-payload is active; use 'k' suffix"
              AF_OPTION_WRAP "for kilobytes [max-payload]",
              "octets"),
    AF_OPTION("udp-payload", 0, 0, AF_OPT_TYPE_NONE,
              &yaf_opt_udp_multipkt_payload,
              AF_OPTION_WRAP "Enable payload capture across all packets in a"
              AF_OPTION_WRAP "non-TCP flow instead of only the first packet",
              NULL),
#if YAF_ENABLE_ENTROPY
    AF_OPTION("entropy", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_entropy_mode,
              AF_OPTION_WRAP "Export Shannon entropy of captured payload",
              NULL),
#endif
#if YAF_ENABLE_APPLABEL
    AF_OPTION("applabel", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_applabel_mode,
              AF_OPTION_WRAP "Enable the packet inspection protocol"
              AF_OPTION_WRAP "application labeler engine",
              NULL),
    AF_OPTION("applabel-rules", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_applabel_rules,
              AF_OPTION_WRAP "Specify the location of the application labeler"
              AF_OPTION_WRAP "rules file",
              "file"),
    AF_OPTION("applabel-check-early", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_applabel_check_early_str,
              AF_OPTION_WRAP "Determine the appLabel when the flow has this"
              AF_OPTION_WRAP "amount of payload; only use this when also"
              AF_OPTION_WRAP "using --applabel-max-payload",
              "octets"),
    AF_OPTION("applabel-max-payload", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_applabel_max_payload_str,
              AF_OPTION_WRAP "For the given appLabel (first number) store only"
              AF_OPTION_WRAP "this much payload (second number). Use a comma"
              AF_OPTION_WRAP "separated list of 'appLabel=payload' pairs. An"
              AF_OPTION_WRAP "appLabel is 'rest' applies to remainder. This"
              AF_OPTION_WRAP "option requires use of --applabel-check-early",
              "app=pay[,app=pay...]"),
#endif /* if YAF_ENABLE_APPLABEL */
#if YAF_ENABLE_NDPI
    AF_OPTION("ndpi", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_ndpi,
              AF_OPTION_WRAP "Enable nDPI application labeling",
              NULL),
    AF_OPTION("ndpi-protocol-file", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_ndpi_proto_file,
              AF_OPTION_WRAP "Specify protocol file for sub-protocol"
              AF_OPTION_WRAP "and port-based protocol detection",
              "file"),
#endif /* if YAF_ENABLE_NDPI */
#if YAF_ENABLE_P0F
    AF_OPTION("p0f-fingerprints", 0, 0, AF_OPT_TYPE_STRING,
              &yaf_opt_p0f_fingerprints,
              AF_OPTION_WRAP "Specify the location of the p0f fingerprint"
              AF_OPTION_WRAP "file",
              "file"),
    AF_OPTION("p0fprint", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_p0fprint_mode,
              AF_OPTION_WRAP "Enable the p0f OS fingerprinter",
              NULL),
#endif /* if YAF_ENABLE_P0F */
#if YAF_ENABLE_FPEXPORT
    AF_OPTION("fpexport", 0, 0, AF_OPT_TYPE_NONE, &yaf_opt_fpExport_mode,
              AF_OPTION_WRAP "Enable export of handshake headers for"
              AF_OPTION_WRAP "external OS fingerprinters",
              NULL),
#endif /* if YAF_ENABLE_FPEXPORT */
    AF_OPTION_END
};
#endif /* if YAF_ENABLE_PAYLOAD */

#ifdef YAF_ENABLE_HOOKS
static AirOptionEntry yaf_optent_plugin[] = {
    AF_OPTION("plugin-name", 0, 0, AF_OPT_TYPE_STRING, &pluginName,
              AF_OPTION_WRAP "Load one or more yaf plugins",
              "libplugin_name[,libplugin_name...]"),
    AF_OPTION("plugin-opts", 0, 0, AF_OPT_TYPE_STRING, &pluginOpts,
              AF_OPTION_WRAP "Provide options to the plugin(s)",
              "\"plugin_opts[,plugin_opts...]\""),
    AF_OPTION("plugin-conf", 0, 0, AF_OPT_TYPE_STRING, &pluginConf,
              AF_OPTION_WRAP "Set configuration file(s) for the plugin(s)",
              "\"plugin_conf[,plugin_conf...]\""),
    AF_OPTION_END
};
#endif /* ifdef YAF_ENABLE_HOOKS */

/**
 * yfVersionString
 *
 * Print version info and info about how YAF was configured
 *
 */
static GString *
yfVersionString(
    const char  *verNumStr)
{
    GString *resultString;

    resultString = g_string_new(NULL);

    g_string_append_printf(resultString, "%s  Build Configuration:\n",
                           verNumStr);

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Timezone support:",
#if ENABLE_LOCALTIME
                           "local"
#else
                           "UTC"
#endif
                           );

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Fixbuf version:",
                           FIXBUF_VERSION);

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "DAG support:",
#if YAF_ENABLE_DAG
                           "YES"
#else
                           "NO"
#endif
                           );

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Napatech support:",
#if YAF_ENABLE_NAPATECH
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Netronome support:",
#if YAF_ENABLE_NETRONOME
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Bivio support:",
#if YAF_ENABLE_BIVIO
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "PFRING support:",
#if YAF_ENABLE_PFRING
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Compact IPv4 support:",
#if YAF_ENABLE_COMPACT_IP4
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Plugin support: ",
#if YAF_ENABLE_HOOKS
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Application Labeling:",
#if YAF_ENABLE_APPLABEL
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Payload Processing Support:",
#if YAF_ENABLE_PAYLOAD
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Entropy support:",
#if YAF_ENABLE_ENTROPY
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Fingerprint Export Support:",
#if YAF_ENABLE_FPEXPORT
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "P0F Support:",
#if YAF_ENABLE_P0F
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Spread Support:",
#if HAVE_SPREAD
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "MPLS Support:",
#if YAF_MPLS
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Non-IP Support:",
#if YAF_NONIP
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Separate Interface Support:",
#if YAF_ENABLE_SEPARATE_INTERFACES
                           "YES"
#elif YAF_ENABLE_DAG_SEPARATE_INTERFACES
                           "YES (Dag)"
#else
                           "NO"
#endif /* if YAF_ENABLE_SEPARATE_INTERFACES */
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "nDPI Support:",
#if YAF_ENABLE_NDPI
                           "YES"
#else
                           "NO"
#endif
                           );

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "IE Metadata Export:",
#if YAF_ENABLE_METADATA_EXPORT
                           "YES"
#else
                           "NO"
#endif
                           );

    return resultString;
}


#ifdef HAVE_SPREAD
static void
groups_from_list(
    char      *list,
    char    ***groups,
    uint16_t **spreadIndex,
    uint8_t   *numSpreadGroups)
{
    gchar  **sa = g_strsplit(list, ",", -1);
    int      n, g;
    gchar  **spread_split = NULL;
    gboolean catch_all_group = FALSE;

    n = g_strv_length(sa);
    g_debug("Adding Spread Groups: %s", list);

    *groups = g_new0(char *, n + 1);
    *spreadIndex = g_new0(uint16_t, n);

    if (n > 255) {
        g_debug("Spread Max Groups is 255: "
                "List will be contained to 255 Groups");
        n = 255;
    }
    *numSpreadGroups = n;

    g = 0;
    for (n = 0; n < *numSpreadGroups; ++n) {
        if ('\0' == sa[n][0]) {
            continue;
        }

        spread_split = g_strsplit(sa[n], ":", 3);
        if (spread_split[0] && *spread_split[0]) {
            const char *sp = spread_split[0];
            /* Remove leading white space */
            while (isspace(*sp)) {
                ++sp;
            }
            (*groups)[g] = g_strdup(sp);

            if (spread_split[1] && *(spread_split[1])) {
                (*spreadIndex)[g] = atoi(spread_split[1]);
            } else {
                (*spreadIndex)[g] = 0;
                catch_all_group = TRUE;
            }
            g++;
        }
        g_strfreev(spread_split);
    }
    *numSpreadGroups = g;

    if (!catch_all_group) {
        g_warning("NO CATCHALL SPREAD GROUP GIVEN - FLOWS WILL BE LOST");
    }

    g_strfreev(sa);
}
#endif /* HAVE_SPREAD */


/**
 * yfExit
 *
 * exit handler for YAF
 *
 */
static void
yfExit(
    void)
{
    if (yaf_pidfile) {
        unlink(yaf_pidfile);
    }
}


/**
 * yfDaemonize
 *
 * daemonize yaf.  An alternative to using airdaemon which has
 * it's issues.
 *
 */
static void
yfDaemonize(
    void)
{
    pid_t pid;
    int   rv = -1;
    char  str[256];
    int   fp;

    if (chdir("/") == -1) {
        rv = errno;
        g_warning("Cannot change directory: %s", strerror(rv));
        exit(-1);
    }

    if ((pid = fork()) == -1) {
        rv = errno;
        g_warning("Cannot fork for daemon: %s", strerror(rv));
        exit(-1);
    } else if (pid != 0) {
        g_debug("Forked child %ld.  Parent exiting", (long)pid);
        _exit(EXIT_SUCCESS);
    }

    setsid();

    umask(0022);

    rv = atexit(yfExit);
    if (rv == -1) {
        g_warning("Unable to register function with atexit(): %s",
                  strerror(rv));
        exit(-1);
    }

    /* Close out the standard file descriptors */
    close(STDIN_FILENO);

    if (yaf_pidfile) {
        fp = open(yaf_pidfile, O_RDWR | O_CREAT, 0640);
        if (fp < 0) {
            g_warning("Unable to open pid file %s", yaf_pidfile);
            exit(1);
        }
        sprintf(str, "%d\n", getpid());
        if (!write(fp, str, strlen(str))) {
            g_warning("Unable to write pid to file");
        }
    } else {
        g_debug("pid: %d", getpid());
    }
}


/*
 * Lua helper functions
 *
 */

/**
 *    Prints a warning if the top of the Lua stack is not equal `_top_`.
 */
#define yf_lua_checktop(_L_, _top_)                                     \
    if (lua_gettop(_L_) == (_top_)) { /* no-op */ } else {              \
        g_warning("Programmer error at %s:%d:"                          \
                  " Top of Lua stack is %d, expected %d",               \
                  __FILE__, __LINE__, lua_gettop(_L_), (_top_));        \
    }

/**
 *    Gets the value of the global `_key_` and, if it exists, sets `_ret_` to
 *    its numeric value.
 */
#define yf_lua_getnum(_key_, _ret_)             \
    if (lua_getglobal(L, _key_) != LUA_TNIL) {  \
        _ret_ = (int)lua_tonumber(L, -1);       \
    }                                           \
    lua_pop(L, 1);

/**
 *    Gets the value of the global `_key_` and, if it exists, sets `_ret_` to
 *    its string value.
 */
#define yf_lua_getstr(_key_, _ret_)             \
    if (lua_getglobal(L, _key_) != LUA_TNIL) {  \
        _ret_ = g_strdup(lua_tostring(L, -1));  \
    }                                           \
    lua_pop(L, 1);

/**
 *    Gets the value of the global `_key_` and, if it exists, sets `_ret_` to
 *    its boolean value.
 */
#define yf_lua_getbool(_key_, _ret_)            \
    if (lua_getglobal(L, _key_) != LUA_TNIL) {  \
        _ret_ = (int)lua_toboolean(L, -1);      \
    }                                           \
    lua_pop(L, 1);

/**
 *    Looks up `_key_` in the table at the top of the stack and sets `_val_`
 *    to its boolean value.
 */
#define yf_lua_checktablebool(_key_, _val_) \
    lua_pushstring(L, _key_);               \
    if (lua_gettable(L, -2) != LUA_TNIL) {  \
        _val_ = (int)lua_toboolean(L, -1);  \
    }                                       \
    lua_pop(L, 1);

/**
 *    Looks up `_key_` in the table at the top of the stack and sets `_val_`
 *    to its numeric value or raises an error if value is not a number.
 */
#define yf_lua_gettableint(_key_, _val_)                        \
    lua_pushstring(L, _key_);                                   \
    if (lua_gettable(L, -2) != LUA_TNIL) {                      \
        if (!lua_isnumber(L, -1)) {                             \
            air_opterr("Error in %s: %s must be a number",      \
                       yaf_config_file, _key_);                 \
        }                                                       \
        _val_ = (int)lua_tonumber(L, -1);                       \
    }                                                           \
    lua_pop(L, 1);

static int
yfLuaGetLen(
    lua_State  *L,
    int         index)
{
    int len = 0;

    lua_len(L, index);
    len = lua_tointeger(L, -1);
    lua_pop(L, 1);

    return len;
}

/*
 *    Looks up `key` in the table at the top of the stack and returns its
 *    value as a string.
 */
static char *
yfLuaGetStrField(
    lua_State   *L,
    const char  *key)
{
    const char *result;

    lua_pushstring(L, key);
    lua_gettable(L, -2);

    result = lua_tostring(L, -1);
    lua_pop(L, 1);

    return (char *)g_strdup(result);
}

/**
 *    Helper function for parsing a list of ports where `tabname` is the name
 *    of the table containing the ports.
 */
static void
yfLuaGetSaveTablePort(
    lua_State   *L,
    const char  *tabname,
    GArray      *ports_array)
{
    int ltype;

    ltype = lua_getglobal(L, tabname);
    if (LUA_TNIL != ltype) {
        if (LUA_TTABLE == ltype) {
            gboolean warned = FALSE;
            long     i, port;
            uint16_t p;
            int      len = yfLuaGetLen(L, -1);

            /* Add the ports to the array */
            for (i = 1; i <= len; ++i) {
                if (lua_rawgeti(L, -1, i) == LUA_TNUMBER) {
                    port = (long)lua_tonumber(L, -1);
                    if (port < 0 || port > UINT16_MAX) {
                        g_warning("Ignoring out-of-range entry %ld in %s",
                                  port, tabname);
                    } else {
                        p = port;
                        g_array_append_val(ports_array, p);
                    }
                } else if (!warned) {
                    warned = TRUE;
                    g_warning("Ignoring non-number entry in %s", tabname);
                }
                lua_pop(L, 1);
            }
            if (0 == ports_array->len) {
                air_opterr("Error in %s: Found no valid entries in %s",
                           yaf_config_file, tabname);
            }
        } else {
            air_opterr("Error in %s: %s is not a valid table."
                       " Should be in the form:"
                       " %s = { 4789, 6081, ...}",
                       yaf_config_file, tabname, tabname);
        }
    }
    /* Finished with the table (or nil) */
    lua_pop(L, 1);

    yaf_opt_remove_array_dups(ports_array);
}

/**
 *    Helper function for parsing time_elements.
 */
static void
yfLuaParseTimeElementValue(
    lua_State  *L,
    int         ltype,
    GString    *str)
{
    if (LUA_TSTRING == ltype) {
        const char *s = lua_tostring(L, -1);
        if (0 == strcmp(s, "milli") || 0 == strcmp(s, "1")) {
            g_string_append_printf(str, "%d,", YF_TIME_IE_MILLI);
        } else if (0 == strcmp(s, "micro") || 0 == strcmp(s, "2")) {
            g_string_append_printf(str, "%d,", YF_TIME_IE_MICRO);
        } else if (0 == strcmp(s, "nano") || 0 == strcmp(s, "3")) {
            g_string_append_printf(str, "%d,", YF_TIME_IE_NANO);
        } else {
            air_opterr("Error in %s: Invalid string in time_elements ('%s');"
                       " must be one of 'milli', 'micro', or 'nano'",
                      yaf_config_file, s);
        }
    } else if (LUA_TNUMBER == ltype) {
        int n = lua_tointeger(L, -1);
        if (n <= 0) {
            air_opterr("Error in %s: Invalid integer in time_elements (%d):"
                       " value cannot be less than 0",
                       yaf_config_file, n);
        }
        g_string_append_printf(str, "%d,", n);
    } else {
        air_opterr("Error in %s: Invalid object type in time_elements;"
                   " item is %s, expected string or integer",
                   yaf_config_file, lua_typename(L, ltype));
    }
}

#if YAF_ENABLE_APPLABEL
/**
 *    Parses the applabel_max_payload table from the yaf_config_file and fills
 *    the yaf_opt_applabel_max_payload global GArray, creating it if
 *    necessary.
 *
 *    @param L      Lua state
 *    @param ltype  Type of the applabel_max_payload variable
 *    @param t      Location of the applabel_max_payload in the Lua stack
 */
static void
yfLuaParseApplabelMaxPayload(
    lua_State  *L,
    int         ltype,
    const int   t)
{
    const char *option_name = "applabel_max_payload";
    const char *rest;
    long long   num;
    size_t      len;
    int         isnum;
    uint32_t    entry[2] = {0, 0};

    /* check that it is a table */
    if (LUA_TTABLE != ltype) {
        air_opterr("Error in %s: %s is not a valid table."
                   " Should be in the form: %s ="
                   " {[applabel] = octets, ..., rest = octets}",
                   yaf_config_file, option_name, option_name);
    }

    /* create the array if needed */
    if (NULL == yaf_opt_applabel_max_payload) {
        yaf_opt_applabel_max_payload =
            g_array_new(FALSE, TRUE, sizeof(uint32_t));
    }

    /* push a dummy first key that lua_next() can pop */
    lua_pushnil(L);
    while (lua_next(L, t) != 0) {
        /* 'key' is at index -2 and 'value' is at index -1 */

        /* get the octets value */
        num = lua_tointegerx(L, -1, &isnum);
        if (!isnum) {
            air_opterr("Error in %s: %s table values must be numbers"
                       " but found object type %s instead",
                       yaf_config_file, option_name,
                       lua_typename(L, lua_type(L, -1)));
        }
        if (num < 0 || num > (long long)UINT32_MAX) {
            air_opterr("Error in %s: %s table values must be 32-bit"
                       " non-negative numbers but found %lld instead",
                       yaf_config_file, option_name, num);
        }
        entry[1] = (uint32_t)num;

        /* get the key */
        /* push a copy of 'key' onto the stack so that we can safely retrieve
         * it without confusing lua_next() */
        lua_pushvalue(L, -2);
        if (lua_isinteger(L, -1)) {
            num = lua_tointeger(L, -1);
            if (num < 0 || num > UINT16_MAX) {
                air_opterr("Error in %s: %s table keys must be 16-bit"
                           " non-negative numbers or 'rest' but found"
                           " %lld instead",
                           yaf_config_file, option_name, num);
            }
            entry[0] = (uint32_t)num;
        } else {
            rest = lua_tolstring(L, -1, &len);
            if (NULL == rest) {
                air_opterr("Error in %s: %s table keys must be 16-bit"
                           " non-negative numbers or 'rest' but found"
                           " object type %s instead",
                           yaf_config_file, option_name,
                           lua_typename(L, lua_type(L, -1)));
            }
            if (len != 4 || 0 != (strcmp(rest, "rest"))) {
                air_opterr("Error in %s: %s table keys must be 16-bit"
                           " non-negative numbers or 'rest' but found"
                           " %s instead",
                           yaf_config_file, option_name, rest);
            }
            entry[0] = UINT32_MAX;
        }

        /* pop the copy of the key and the value from the stack; leave the
         * original key for next iteration */
        lua_pop(L, 2);

        /* add entry[] to the GArray, keeping the Garray sorted */
        guint  j;
        for (j = 0; j < yaf_opt_applabel_max_payload->len; j += 2) {
            uint32_t label = g_array_index(yaf_opt_applabel_max_payload,
                                           uint32_t, j);
            if (entry[0] < label) {
                g_array_insert_vals(yaf_opt_applabel_max_payload, j, entry, 2);
                break;
            }
            if (entry[0] == label) {
                /* should never happen in Lua */
                air_opterr("Duplicate entries for applabel %u appear in %s",
                           entry[0], option_name);
            }
        }
        if (j == yaf_opt_applabel_max_payload->len) {
            g_array_insert_vals(yaf_opt_applabel_max_payload, j, entry, 2);
        }
    }
    /* when no more entries, lua_next() removes the previous key */

    /* verify something was found */
    if (0 == yaf_opt_applabel_max_payload->len) {
        air_opterr("Error in %s: No valid appLabel = octets entries"
                   " found in table %s",
                   yaf_config_file, option_name);
    }
}
#endif  /* #if YAF_ENABLE_APPLABEL */

/**
 * yfLuaLoadConfig
 *
 *
 */
static void
yfLuaLoadConfig(
    void)
{
    lua_State *L = luaL_newstate();
    int        i, len, top;
    int        ltype;
    char      *str = NULL;
    GError    *err = NULL;

    luaopen_base(L);
    luaopen_io(L);
    luaopen_string(L);
    luaopen_math(L);

    if (luaL_loadfile(L, yaf_config_file)) {
        air_opterr("Error loading config file: %s", lua_tostring(L, -1));
    }

    if (lua_pcall(L, 0, 0, 0)) {
        air_opterr("can't run the config file: %s", lua_tostring(L, -1));
    }

    /* logging options */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "log");
    if (LUA_TNIL != ltype) {
        if (LUA_TTABLE != ltype) {
            air_opterr("Error in %s: log is not a valid table."
                       " Should be in the form: "
                       "log = {spec=\"filename\", level=\"debug\"}",
                       yaf_config_file);
        }
        str = yfLuaGetStrField(L, "spec");
        logc_set(str, NULL);
        g_free(str);
        str = yfLuaGetStrField(L, "level");
        logc_set(NULL, str);
        g_free(str);
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    if (!logc_setup(&err)) {
        air_opterr("%s", err->message);
    }

    /* input settings */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "input");
    if (LUA_TTABLE != ltype) {
        air_opterr("Error in %s: input is not a valid table. "
                   "Should be in the form: input = {inf=, type=}",
                   yaf_config_file);
    }

    yaf_config.livetype = yfLuaGetStrField(L, "type");
    yf_lua_checktablebool("force_read_all", yaf_opt_force_read_all);
#if YAF_ENABLE_DAG_SEPARATE_INTERFACES || YAF_ENABLE_SEPARATE_INTERFACES
    yf_lua_checktablebool("export_interface", yaf_config.exportInterface);
#endif

    if (yaf_config.livetype == NULL) {
        yaf_config.inspec = yfLuaGetStrField(L, "file");
    } else if (strncmp(yaf_config.livetype, "file", 4) == 0) {
        yaf_config.inspec = yfLuaGetStrField(L, "file");
        g_free(yaf_config.livetype);
        yaf_config.livetype = 0;
    } else if (strncmp(yaf_config.livetype, "caplist", 7) == 0) {
        yaf_config.inspec = yfLuaGetStrField(L, "file");
        yf_lua_checktablebool("noerror", yaf_config.noerror);
        yaf_opt_caplist_mode = TRUE;
        g_free(yaf_config.livetype);
        yaf_config.livetype = 0;
    } else {
        yaf_config.inspec = yfLuaGetStrField(L, "inf");
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    /* output settings */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "output");
    if (LUA_TTABLE != ltype) {
        air_opterr("Error in %s: output is not a valid table."
                   " Should be in the form: output = {host=, port=, protocol=}",
                   yaf_config_file);
    }

    str = yfLuaGetStrField(L, "file");
    if (str) {
        yaf_config.outspec = str;
        yf_lua_gettableint("rotate", yaf_opt_rotate);
        yf_lua_checktablebool("lock", yaf_config.lockmode);
    } else {
        yaf_opt_ipfix_transport = yfLuaGetStrField(L, "protocol");
        if (strcmp(yaf_opt_ipfix_transport, "spread") == 0) {
#ifdef HAVE_SPREAD
            yaf_config.outspec = yfLuaGetStrField(L, "daemon");
            yaf_config.ipfixSpreadTrans = TRUE;
            yaf_opt_spread_groupby = yfLuaGetStrField(L, "groupby");
            lua_pushstring(L, "groups");
            ltype = lua_gettable(L, -2);
            if (LUA_TNIL != ltype) {
                if (LUA_TTABLE != ltype) {
                    air_opterr("Error in %s: groups is not a valid table."
                               " Should be in the form:"
                               " groups={{name=\"NAME\"}}", yaf_config_file);
                }
                len = yfLuaGetLen(L, -1);
                yaf_config.numSpreadGroups = len;
                if (len) {
                    yaf_config.spreadparams.groups = g_new0( char *, len + 1);
                    yaf_config.spreadGroupIndex = g_new0(uint16_t, len);
                }
                for (i = 1; i <= len; i++) {
                    if (lua_rawgeti(L, -1, i) != LUA_TTABLE) {
                        air_opterr("Error in %s: group must be a valid table."
                                   " Should be in the form:"
                                   " {name=\"NAME\", [value=]}",
                                   yaf_config_file);
                    }
                    yaf_config.spreadparams.groups[i - 1] =
                        yfLuaGetStrField(L, "name");
                    yf_lua_gettableint("value",
                                       yaf_config.spreadGroupIndex[i - 1]);
                    lua_pop(L, 1);
                }
            }
            lua_pop(L, 1);
#else /* ifdef HAVE_SPREAD */
            air_opterr("Spread is not enabled. Configure --with-spread");
#endif /* ifdef HAVE_SPREAD */
        } else {
            yaf_config.outspec = yfLuaGetStrField(L, "host");
            yaf_config.connspec.svc = yfLuaGetStrField(L, "port");
            yf_lua_gettableint("udp_temp_timeout", yaf_opt_udp_temp_timeout);
        }
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    /* various top-level settings */
    yf_lua_getnum("stats", yaf_opt_stats);
    yf_lua_getbool("no_tombstone", yaf_opt_no_tombstone);
    yf_lua_getnum("tombstone_configured_id", yaf_opt_configured_id);
    yf_lua_getnum("ingress", yaf_opt_ingress_int);
    yf_lua_getnum("egress", yaf_opt_egress_int);
    yf_lua_getnum("obdomain", yaf_config.odid);
    yf_lua_getnum("maxflows", yaf_opt_max_flows);
    yf_lua_getnum("maxfrags", yaf_opt_max_frags);
    yf_lua_getnum("idle_timeout", yaf_opt_idle);
    yf_lua_getnum("active_timeout", yaf_opt_active);
    yf_lua_getnum("maxpayload", yaf_opt_max_payload);
    yf_lua_getnum("maxexport", yaf_opt_payload_export_max);
    yf_lua_getbool("export_payload", yaf_opt_payload_export_enable);
    yf_lua_getnum("udp_uniflow", yaf_opt_udp_uniflow_port);
    yf_lua_getbool("udp_payload", yaf_opt_udp_multipkt_payload);

#if YAF_ENABLE_APPLABEL
    /* enable payload export but only for these applabels */
    top = lua_gettop(L);
    yfLuaGetSaveTablePort(L, "export_payload_applabels",
                          yaf_opt_payload_applabel_select);
    yf_lua_checktop(L, top);
#endif  /* YAF_ENABLE_APPLABEL */

    /* decode options */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "decode");
    if (LUA_TNIL != ltype) {
        if (LUA_TTABLE != ltype) {
            air_opterr("Error in %s: decode is not a valid table."
                       " Should be in the "
                       "form: decode = {gre=true, ip4_only=true}",
                       yaf_config_file);
        }
        yf_lua_checktablebool("gre", yaf_opt_gre_mode);
        yf_lua_checktablebool("ip4_only", yaf_opt_ip4_mode);
        yf_lua_checktablebool("ip6_only", yaf_opt_ip6_mode);
        yf_lua_checktablebool("nofrag", yaf_opt_nofrag);
        yf_lua_checktablebool("vxlan", yaf_opt_vxlan_mode);
        yf_lua_checktablebool("geneve", yaf_opt_geneve_mode);
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    /* export options */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "export");
    if (LUA_TNIL != ltype) {
        if (LUA_TTABLE != ltype) {
            air_opterr("Error in %s: export is not a valid table."
                       " Should be in the form:"
                       " export = {silk=true, uniflow=true, mac=true}",
                       yaf_config_file);
        }
        yf_lua_checktablebool("silk", yaf_opt_silk_mode);
        yf_lua_checktablebool("uniflow", yaf_opt_uniflow_mode);
        yf_lua_checktablebool("force_ip6", yaf_config.force_ip6);
        yf_lua_checktablebool("flow_stats", yaf_opt_flowstats_mode);
        yf_lua_checktablebool("delta", yaf_config.deltaMode);
        yf_lua_checktablebool("mac", yaf_opt_mac_mode);
#if YAF_ENABLE_METADATA_EXPORT
        yf_lua_checktablebool("metadata", yaf_config.tmpl_metadata);
#endif
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    /* time-elements export option */
    top = lua_gettop(L);
    /* do not override value set from command line */
    if (NULL == yaf_opt_time_elements) {
        /* Fill yaf_opt_time_elements from Lua time_elements setting, then
         * parse that value in ycParseOptions(). */
        GString *gstr = NULL;

        ltype = lua_getglobal(L, "time_elements");
        if (LUA_TTABLE == ltype) {
            len = yfLuaGetLen(L, -1);
            gstr = g_string_sized_new(32);
            for (i = 1; i <= len; ++i) {
                ltype = lua_geti(L, -1, i);
                if (ltype != LUA_TNONE) {
                    yfLuaParseTimeElementValue(L, ltype, gstr);
                }
                lua_pop(L, 1);
            }
            if (0 == gstr->len) {
                air_opterr("Error in %s:"
                           " No valid values found in time_elements table",
                           yaf_config_file);
            } else {
                /* remove final , */
                g_string_truncate(gstr, gstr->len - 1);
            }
            yaf_opt_time_elements = g_string_free(gstr, FALSE);
        } else if (LUA_TNIL != ltype) {
            gstr = g_string_sized_new(32);
            yfLuaParseTimeElementValue(L, ltype, gstr);
            if (gstr->len) {
                /* remove final , */
                g_string_truncate(gstr, gstr->len - 1);
            }
            yaf_opt_time_elements = g_string_free(gstr, FALSE);
        }
        lua_pop(L, 1);
    }
    yf_lua_checktop(L, top);

    /* tls options */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "tls");
    if (LUA_TNIL != ltype) {
        if (LUA_TTABLE != ltype) {
            air_opterr("Error in %s: tls is not a valid table."
                       " Should be in the form: "
                       "tls = {ca=\"\", cert=\"\", key=\"\"}",
                       yaf_config_file);
        }
        yaf_opt_ipfix_tls = TRUE;
        yaf_config.connspec.ssl_ca_file = yfLuaGetStrField(L, "ca");
        yaf_config.connspec.ssl_cert_file = yfLuaGetStrField(L, "cert");
        yaf_config.connspec.ssl_key_file = yfLuaGetStrField(L, "key");
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    /*entropy options */
#if YAF_ENABLE_ENTROPY
    yf_lua_getbool("entropy", yaf_opt_entropy_mode);
#endif

    /* applabel options */
#if YAF_ENABLE_APPLABEL
    yf_lua_getbool("applabel", yaf_opt_applabel_mode);
    yf_lua_getstr("applabel_rules", yaf_opt_applabel_rules);
    yf_lua_getnum("applabel_check_early",
                  yaf_opt_applabel_check_early);

    /* applabel_max_payload */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "applabel_max_payload");
    if (LUA_TNIL != ltype) {
        yfLuaParseApplabelMaxPayload(L, ltype, lua_gettop(L));
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

#endif

#if YAF_ENABLE_NDPI
    yf_lua_getbool("ndpi", yaf_opt_ndpi);
    yf_lua_getstr("ndpi_proto_file", yaf_ndpi_proto_file);
#endif

    /* p0f options */
#if YAF_ENABLE_P0F
    yf_lua_getbool("p0fprint", yaf_opt_p0fprint_mode);
    yf_lua_getstr("p0f_fingerprints", yaf_opt_p0f_fingerprints);
#endif

    /* fpexport option */
#if YAF_ENABLE_FPEXPORT
    yf_lua_getbool("fpexport",  yaf_opt_fpExport_mode);
#endif

#if YAF_ENABLE_ZLIB
    yf_lua_getstr("decompress", yaf_tmp_dir);
#endif

    /* plugin options */
#if YAF_ENABLE_HOOKS
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "plugin");
    if (LUA_TNIL != ltype) {
        /* The plugin value must be a table */
        if (LUA_TTABLE != ltype) {
            air_opterr("Error in %s: plugin is not a valid table."
                       " Should be in the form:"
                       " plugin = {{name=\"NAME\", options=\"OPT\","
                       " conf=\"CONF\"}} where options and conf are optional",
                       yaf_config_file);
        }
        len = yfLuaGetLen(L, -1);
        for (i = 1; i <= len; i++) {
            lua_rawgeti(L, -1, i);
            if (lua_istable(L, -1)) {
                pluginName = yfLuaGetStrField(L, "name");
                pluginConf = yfLuaGetStrField(L, "conf");
                pluginOpts = yfLuaGetStrField(L, "options");
                if (!yfHookAddNewHook(
                        pluginName, pluginOpts, pluginConf, yfctx, &err))
                {
                    g_warning("Couldn't load requested plugin: %s",
                              err->message);
                }
                hooks_initialized = TRUE;
            }
            lua_pop(L, 1);
        }
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);
#endif /* if YAF_ENABLE_HOOKS */

    /* Use these ports to trigger VxLAN or Geneve decoding */
    top = lua_gettop(L);
    yfLuaGetSaveTablePort(L, "vxlan_ports", yaf_opt_vxlan_ports);
    yfLuaGetSaveTablePort(L, "geneve_ports", yaf_opt_geneve_ports);
    yf_lua_checktop(L, top);

    /* pcap options */
    top = lua_gettop(L);
    ltype = lua_getglobal(L, "pcap");
    if (LUA_TNIL != ltype) {
        if (LUA_TTABLE != ltype) {
            air_opterr("Error in %s: pcap is not a valid table."
                       " Should be in the form:"
                       " pcap = {path=\"\", meta=\"\", maxpcap=25}",
                       yaf_config_file);
        }

        yf_lua_gettableint("maxpcap", yaf_opt_max_pcap);
        yf_lua_gettableint("pcap_timer", yaf_opt_pcap_timer);
        yaf_pcap_meta_file = yfLuaGetStrField(L, "meta");
        yaf_config.pcapdir = yfLuaGetStrField(L, "path");
        /* pcap per flow and index pcap */
    }
    lua_pop(L, 1);
    yf_lua_checktop(L, top);

    /* pidfile */
    yf_lua_getstr("pidfile", yaf_pidfile);

    /* BPF filter */
    yf_lua_getstr("filter", yaf_config.bpf_expr);

    lua_close(L);
}


/**
 * yfParseOptions
 *
 * parses the command line options via calls to the Airframe
 * library functions
 *
 *
 *
 */
static void
yfParseOptions(
    int   *argc,
    char **argv[])
{
#define WARN_OPT_REQUIRES(opt1, opt2)                                   \
    g_warning("Ignoring --%s option which requires --%s", (opt1), (opt2))

#define WARN_OPT_CONFLICTS(opt1, opt2)                                  \
    g_warning("Ignoring --%s option which conflicts with --%s", (opt1), (opt2))

    AirOptionCtx *aoctx = NULL;
    GError       *err = NULL;
    GString      *versionString;
    GString      *commandString = g_string_sized_new(512);

    g_string_assign(commandString, "YAF invocation");
    for (int i = 0; i < *argc; ++i) {
        g_string_append_printf(commandString, " '%s'", (*argv)[i]);
    }

    aoctx = air_option_context_new("", argc, argv, yaf_optent_core);

    /* Initialize opt variables */
    yaf_opt_vxlan_ports = g_array_new(FALSE, TRUE, sizeof(uint16_t));
    yaf_opt_geneve_ports = g_array_new(FALSE, TRUE, sizeof(uint16_t));
#if YAF_ENABLE_APPLABEL
    yaf_opt_payload_applabel_select = g_array_new(FALSE, TRUE,
                                                  sizeof(uint16_t));
#endif

    air_option_context_add_group(
        aoctx, "decode", "Decoder Options:",
        AF_OPTION_WRAP "Show help for packet decoder options", yaf_optent_dec);
    air_option_context_add_group(
        aoctx, "flow", "Flow table Options:",
        AF_OPTION_WRAP "Show help for flow table options", yaf_optent_flow);
    air_option_context_add_group(
        aoctx, "export", "Export Options:",
        AF_OPTION_WRAP "Show help for export format options", yaf_optent_exp);
    air_option_context_add_group(
        aoctx, "ipfix", "IPFIX Options:",
        AF_OPTION_WRAP "Show help for IPFIX export options", yaf_optent_ipfix);
    air_option_context_add_group(
        aoctx, "pcap", "PCAP Options:",
        AF_OPTION_WRAP "Show help for PCAP export options", yaf_optent_pcap);
#if YAF_ENABLE_PAYLOAD
    air_option_context_add_group(
        aoctx, "payload", "Payload Options:",
        AF_OPTION_WRAP "Show help for payload options", yaf_optent_payload);
#endif /* if YAF_ENABLE_PAYLOAD */
#ifdef YAF_ENABLE_HOOKS
    air_option_context_add_group(
        aoctx, "plugin", "Plugin Options:",
        AF_OPTION_WRAP "Show help for plugin interface options",
        yaf_optent_plugin);
#endif /* ifdef YAF_ENABLE_HOOKS */
    privc_add_option_group(aoctx);

    versionString = yfVersionString(VERSION);

    logc_add_option_group(aoctx, "yaf", versionString->str);

    air_option_context_set_help_enabled(aoctx);

    air_option_context_parse(aoctx);

    if (*argc > 1) {
        GString *str = g_string_sized_new(256);
        g_string_printf(str, "Unrecognized argument%s: %s",
                        ((*argc > 2) ? "s" : ""), (*argv)[1]);
        for (int i = 2; i < *argc; ++i) {
            g_string_append_printf(str, " %s", (*argv)[i]);
        }
        g_warning("%s", str->str);
        g_string_free(str, TRUE);
    }

    if (yaf_config_file) {
        yfLuaLoadConfig();
    } else {
        /* set up logging and privilege drop */
        if (!logc_setup(&err)) {
            air_opterr("%s", err->message);
        }
    }

    if (!privc_setup(&err)) {
        air_opterr("%s", err->message);
    }

    /* For vxlan and Geneve decoding: free the array if disabled; otherwise
     * remove duplicates from the array or add the default port to it if
     * empty */
    if (!yaf_opt_vxlan_mode) {
        if (yaf_opt_vxlan_ports_str) {
            WARN_OPT_REQUIRES("vxlan-decode-ports", "vxlan-decode");
            g_free(yaf_opt_vxlan_ports_str);
        }
        g_array_free(yaf_opt_vxlan_ports, TRUE);
        yaf_opt_vxlan_ports = NULL;
    } else {
        if (yaf_opt_vxlan_ports_str) {
            g_array_set_size(yaf_opt_vxlan_ports, 0);
            yaf_opt_ports_str_2_array("vxlan-decode-ports",
                                      yaf_opt_vxlan_ports_str,
                                      yaf_opt_vxlan_ports);
            g_free(yaf_opt_vxlan_ports_str);
        }
        if (0 == yaf_opt_vxlan_ports->len) {
            uint16_t default_port = DEFAULT_VXLAN_PORT;
            g_array_append_val(yaf_opt_vxlan_ports, default_port);
        }
    }

    if (!yaf_opt_geneve_mode) {
        if (yaf_opt_geneve_ports_str) {
            WARN_OPT_REQUIRES("geneve-decode-ports", "geneve-decode");
            g_free(yaf_opt_geneve_ports_str);
        }
        g_array_free(yaf_opt_geneve_ports, TRUE);
        yaf_opt_geneve_ports = NULL;
    } else {
        if (yaf_opt_geneve_ports_str) {
            g_array_set_size(yaf_opt_geneve_ports, 0);
            yaf_opt_ports_str_2_array("geneve-decode-ports",
                                      yaf_opt_geneve_ports_str,
                                      yaf_opt_geneve_ports);
            g_free(yaf_opt_geneve_ports_str);
        }
        if (0 == yaf_opt_geneve_ports->len) {
            uint16_t default_port = DEFAULT_GENEVE_PORT;
            g_array_append_val(yaf_opt_geneve_ports, default_port);
        }
    }

#if YAF_ENABLE_APPLABEL
    if (yaf_opt_max_payload_str) {
        yaf_opt_max_payload =
            yaf_opt_parse_payload("max-payload", yaf_opt_max_payload_str,
                                  UINT32_MAX);
    }
    if (yaf_opt_applabel_check_early_str) {
        yaf_opt_applabel_check_early =
            yaf_opt_parse_payload("applabel-check-early",
                                  yaf_opt_applabel_check_early_str, UINT32_MAX);
    }
    if (FALSE == yaf_opt_applabel_mode || 0 == yaf_opt_max_payload) {
        const char *msg = "applabel and --max-payload";
        if (yaf_opt_applabel_mode) {
            WARN_OPT_REQUIRES("applabel", "max-paylad");
            g_warning("Application labeling engine will not operate");
            yaf_opt_applabel_mode = FALSE;
        }
        if (yaf_opt_applabel_rules) {
            WARN_OPT_REQUIRES("applabel-rules", msg);
            g_free(yaf_opt_applabel_rules);
            yaf_opt_applabel_rules = NULL;
        }
        if (yaf_opt_applabel_check_early) {
            WARN_OPT_REQUIRES("applabel-check-early", msg);
            yaf_opt_applabel_check_early = 0;
        }
        if (yaf_opt_payload_applabel_select_str
            || yaf_opt_payload_applabel_select->len)
        {
            WARN_OPT_REQUIRES("payload-applabel-select", msg);
            g_free(yaf_opt_payload_applabel_select_str);
            yaf_opt_payload_applabel_select_str = NULL;
            g_array_set_size(yaf_opt_payload_applabel_select, 0);
        }
        if (yaf_opt_applabel_max_payload_str) {
            WARN_OPT_REQUIRES("applabel-max-payload", msg);
            g_free(yaf_opt_applabel_max_payload_str);
            yaf_opt_applabel_max_payload_str = NULL;
        }
    } else if (!yfAppLabelInit(yaf_opt_applabel_rules, &err)) {
        if (NULL != err) {
            g_warning("Error initializing application labeler: %s",
                      err->message);
            g_warning("Application labeling engine will not operate");
            g_clear_error(&err);
        }
        yaf_opt_applabel_mode = FALSE;
        g_free(yaf_opt_applabel_rules);
        yaf_opt_applabel_check_early = 0;
        g_free(yaf_opt_payload_applabel_select_str);
        yaf_opt_payload_applabel_select_str = NULL;
        g_array_set_size(yaf_opt_payload_applabel_select, 0);
        g_free(yaf_opt_applabel_max_payload_str);
        yaf_opt_applabel_max_payload_str = NULL;
    }
    /* If yaf is running without appLabel, any appLabel-related option given
     * by the user has been turned off. */

    /* handle --payload-applabel-select and enable payload-export if
     * necessary */
    if (yaf_opt_payload_applabel_select_str) {
        g_array_set_size(yaf_opt_payload_applabel_select, 0);
        yaf_opt_ports_str_2_array("payload-applabel-select",
                                  yaf_opt_payload_applabel_select_str,
                                  yaf_opt_payload_applabel_select);
        g_free(yaf_opt_payload_applabel_select_str);
    }
    if (0 == yaf_opt_payload_applabel_select->len) {
        g_array_free(yaf_opt_payload_applabel_select, TRUE);
        yaf_opt_payload_applabel_select = NULL;
    } else {
        yaf_opt_payload_export_enable = TRUE;
        yaf_config.payload_export_applabels_size
            = yaf_opt_payload_applabel_select->len;
        yaf_config.payload_export_applabels
            = (uint16_t *)g_array_free(yaf_opt_payload_applabel_select, FALSE);
    }

    if (yaf_opt_applabel_check_early > yaf_opt_max_payload) {
        g_warning("--applabel-check-early exceeds --max-payload;"
                  " setting to max-payload (%d)", yaf_opt_max_payload);
        yaf_opt_applabel_check_early = yaf_opt_max_payload;
    }

    /* handle --applabel-max-payload */
    if (yaf_opt_applabel_max_payload_str) {
        if (yaf_opt_applabel_max_payload) {
            g_array_set_size(yaf_opt_applabel_max_payload, 0);
        } else {
            yaf_opt_applabel_max_payload
                = g_array_new(FALSE, TRUE, sizeof(uint32_t));
        }
        yaf_opt_parse_applabel_max_payload(yaf_opt_applabel_max_payload_str);
        g_free(yaf_opt_applabel_max_payload_str);
    }
    if (yaf_opt_applabel_max_payload) {
        if (0 == yaf_opt_applabel_max_payload->len) {
            g_array_free(yaf_opt_applabel_max_payload, TRUE);
            yaf_opt_applabel_max_payload = NULL;
        } else if (0 == yaf_opt_applabel_check_early) {
            WARN_OPT_REQUIRES("applabel-max-payload", "applabel-check-early");
            g_array_free(yaf_opt_applabel_max_payload, TRUE);
            yaf_opt_applabel_max_payload = NULL;
        }
    }
#endif /* if YAF_ENABLE_APPLABEL */

#if YAF_ENABLE_NDPI
    if (yaf_ndpi_proto_file && (FALSE == yaf_opt_ndpi)) {
        WARN_OPT_REQUIRES("ndpi-proto-file", "ndpi");
        g_warning("NDPI labeling will not operate");
    }
    if (TRUE == yaf_opt_ndpi) {
        if (yaf_opt_max_payload == 0) {
            WARN_OPT_REQUIRES("ndpi", "max-payload");
            g_warning("NDPI labeling will not operate");
            yaf_opt_ndpi = FALSE;
        }
    }
#endif /* if YAF_ENABLE_NDPI */

#if YAF_ENABLE_P0F
    if (yaf_opt_p0f_fingerprints && (FALSE == yaf_opt_p0fprint_mode)) {
        WARN_OPT_REQUIRES("p0f-fingerprints", "p0fprint");
        g_warning("p0f fingerprinting engine will not operate");
        yaf_opt_p0fprint_mode = FALSE;
    }
    if (TRUE == yaf_opt_p0fprint_mode) {
        if (yaf_opt_max_payload == 0) {
            WARN_OPT_REQUIRES("p0fprint", "max-payload");
            yaf_opt_p0fprint_mode = FALSE;
        } else if (!yfpLoadConfig(yaf_opt_p0f_fingerprints, &err)) {
            g_warning("Error loading p0f config file '%s': %s",
                      yaf_opt_p0f_fingerprints, err->message);
            g_warning("p0f fingerprinting engine will not operate");
            yaf_opt_p0fprint_mode = FALSE;
            g_clear_error(&err);
        }
    }
#endif /* if YAF_ENABLE_P0F */

#if YAF_ENABLE_FPEXPORT
    if (TRUE == yaf_opt_fpExport_mode) {
        if (yaf_opt_max_payload == 0) {
            WARN_OPT_REQUIRES("fpexport", "max-payload");
            yaf_opt_fpExport_mode = FALSE;
        }
    }
#endif /* if YAF_ENABLE_FPEXPORT */

#if YAF_ENABLE_ENTROPY
    if (TRUE == yaf_opt_entropy_mode) {
        if (yaf_opt_max_payload == 0) {
            WARN_OPT_REQUIRES("entropy", "max-payload");
            yaf_opt_entropy_mode = FALSE;
        }
    }
#endif /* if YAF_ENABLE_ENTROPY */

    if (TRUE == yaf_opt_udp_multipkt_payload) {
        if (yaf_opt_max_payload == 0) {
            WARN_OPT_REQUIRES("udp-payload", "payload");
            yaf_opt_udp_multipkt_payload = FALSE;
        }
    }

#ifdef YAF_ENABLE_HOOKS
    if (NULL != pluginName && !hooks_initialized) {
        pluginOptParse(&err);
    }
#endif

#if YAF_ENABLE_BIVIO
    /* export Interface numbers if BIVIO is enabled */
    yaf_config.exportInterface = TRUE;
#endif

    if (NULL == yaf_opt_time_elements) {
        yaf_config.time_elements = YF_TIME_IE__DEFAULT;
    } else {
        gchar      **token = g_strsplit(yaf_opt_time_elements, ",", -1);
        long         value;
        char        *ep;
        unsigned int i;

        yaf_config.time_elements = YF_TIME_IE__UNSET;
        for (i = 0; token[i] != NULL; ++i) {
            ep = token[i];
            errno = 0;
            value = strtol(token[i], &ep, 0);
            if (ep == token[i] || *ep != '\0' || errno != 0
                || value < YF_TIME_IE__FIRST || value > YF_TIME_IE__LAST)
            {
                air_opterr("Invalid time-element value");
            }
            yaf_config.time_elements |= yfRecordTimeIEBitSet(value);
        }
        if (YF_TIME_IE__UNSET == yaf_config.time_elements) {
            air_opterr("No value time-element values were found");
        }
    }

    /* process ip4mode and ip6mode */
    if (yaf_opt_ip4_mode && yaf_opt_ip6_mode) {
        g_warning("Cannot run in both ip4-only and ip6-only modes;"
                  " ignoring options");
        yaf_opt_ip4_mode = FALSE;
        yaf_opt_ip6_mode = FALSE;
    }

    if (yaf_opt_ip4_mode) {
        yaf_reqtype = YF_TYPE_IPv4;
    } else if (yaf_opt_ip6_mode) {
        yaf_reqtype = YF_TYPE_IPv6;
    } else {
        yaf_reqtype = YF_TYPE_IPANY;
    }

    /* process core library options */
    if (yaf_opt_payload_export_max_str) {
        yaf_opt_payload_export_max = yaf_opt_parse_payload(
            "max-export", yaf_opt_payload_export_max_str, UINT16_MAX);
    }
    if (yaf_opt_payload_export_enable || yaf_opt_payload_export_max) {
        if (0 == yaf_opt_payload_export_max) {
            yaf_opt_payload_export_max = yaf_opt_max_payload;
        }
        if (0 == yaf_opt_max_payload) {
            if (yaf_opt_payload_export_enable) {
                WARN_OPT_REQUIRES("export-payload", "max-payload");
            } else if (yaf_opt_payload_export_max) {
                WARN_OPT_REQUIRES("max-export", "max-payload");
            }
            g_warning("disabling payload export");
            yaf_opt_payload_export_enable = FALSE;
            yaf_opt_payload_export_max = 0;
        }
    }
    if (yaf_opt_payload_export_max) {
        if (yaf_opt_payload_export_max > yaf_opt_max_payload) {
            g_warning("--max-export exceeds --max-payload;"
                      " setting to max-payload (%d)", yaf_opt_max_payload);
            yaf_opt_payload_export_max = yaf_opt_max_payload;
        }
        yaf_config.export_payload = yaf_opt_payload_export_max;
    }

    /* Pre-process input options */
    if (yaf_config.livetype) {
        const struct yaf_live_type_st *livetype;

        /* can't use caplist with live */
        if (yaf_opt_caplist_mode) {
            air_opterr("Please choose only one of --live or --caplist");
        }

        /* allow "" as an alias for "pcap" */
        if ('\0' == yaf_config.livetype[0]) {
            yaf_config.livetype = g_strdup("pcap");
        }

        /* select live capture type */
        for (livetype = yaf_live_type; livetype->name; ++livetype) {
            if (strncmp(yaf_config.livetype, livetype->name,
                        strlen(livetype->name)) == 0)
            {
                break;
            }
        }
        if (NULL == livetype->name) {
            /* unsupported live capture type */
            air_opterr("Unsupported live capture type '%s'",
                       yaf_config.livetype);
        }

        /* Can only write to pcap when reading pcap */
        if (yaf_config.pcapdir && strcmp(livetype->name, "pcap") != 0) {
            g_warning("Ignoring --pcap option which is not valid for --live %s",
                      yaf_live_type->name);
            yaf_config.pcapdir = NULL;
        }

        /* Require an interface name for live input */
        if (!yaf_config.inspec) {
            air_opterr("--live requires interface name in --in");
        }
        yaf_liveopen_fn = livetype->open_fn;
        yaf_loop_fn = livetype->loop_fn;
        yaf_close_fn = livetype->close_fn;
    } else {
        /* Use pcap loop and close functions */
        yaf_loop_fn = (yfLoop_fn)yfCapMain;
        yaf_close_fn = (yfClose_fn)yfCapClose;

        /* Default to stdin for no input */
        if (!yaf_config.inspec || !strlen(yaf_config.inspec)) {
            yaf_config.inspec = C("-");
        }
    }

    /* set the live rotation delay */
    yfDiffTimeFromSeconds(&yaf_config.rotate_interval, yaf_opt_rotate);

    if (yaf_opt_stats == 0) {
        yaf_config.nostats = TRUE;
    } else {
        yaf_config.stats_interval = (double)yaf_opt_stats;
    }

    yaf_config.tombstone_configured_id = yaf_opt_configured_id;
    yaf_config.no_tombstone = yaf_opt_no_tombstone;
    yaf_config.layer2IdExportMode = yaf_opt_vxlan_mode || yaf_opt_geneve_mode;
    yaf_config.ingressInt = (uint32_t)yaf_opt_ingress_int;
    yaf_config.egressInt = (uint32_t)yaf_opt_egress_int;

    /* Pre-process output options */
    if (yaf_opt_ipfix_transport) {
        /* set default port */
        if (!yaf_config.connspec.svc) {
            yaf_config.connspec.svc = yaf_opt_ipfix_tls ? C("4740") : C("4739");
        }

        /* Require a hostname for IPFIX output */
        if (!yaf_config.outspec) {
            air_opterr("--ipfix requires hostname in --out");
        }

        /* set hostname */
        yaf_config.connspec.host = yaf_config.outspec;

        if ((*yaf_opt_ipfix_transport == (char)0) ||
            (strcmp(yaf_opt_ipfix_transport, "sctp") == 0))
        {
            if (yaf_opt_ipfix_tls) {
                yaf_config.connspec.transport = FB_DTLS_SCTP;
            } else {
                yaf_config.connspec.transport = FB_SCTP;
            }
        } else if (strcmp(yaf_opt_ipfix_transport, "tcp") == 0) {
            if (yaf_opt_ipfix_tls) {
                yaf_config.connspec.transport = FB_TLS_TCP;
            } else {
                yaf_config.connspec.transport = FB_TCP;
            }
        } else if (strcmp(yaf_opt_ipfix_transport, "udp") == 0) {
            if (yaf_opt_ipfix_tls) {
                yaf_config.connspec.transport = FB_DTLS_UDP;
            } else {
                yaf_config.connspec.transport = FB_UDP;
            }
            if (yaf_opt_udp_temp_timeout <= 0) {
                yaf_opt_udp_temp_timeout = 600;
            }
            /* divide the timeout by 3 to set the resend interval, where 3 is
             * recommended by RFC 5101.  currently YAF does not expire UDP
             * templates */
            yaf_opt_udp_temp_timeout /= 3;
            yfDiffTimeFromSeconds(&yaf_config.udp_tmpl_interval,
                                  yaf_opt_udp_temp_timeout);

#ifdef HAVE_SPREAD
        } else if (strcmp(yaf_opt_ipfix_transport, "spread") == 0) {
            yaf_config.spreadparams.daemon = yaf_config.outspec;
            if (0 == yaf_config.numSpreadGroups) {
                if (NULL == yaf_opt_spread_group) {
                    air_opterr("'--ipfix spread' requires at least one Spread "
                               "group in '--group'");
                }
                groups_from_list(yaf_opt_spread_group,
                                 &yaf_config.spreadparams.groups,
                                 &yaf_config.spreadGroupIndex,
                                 &yaf_config.numSpreadGroups);
            }
            yaf_config.ipfixSpreadTrans = TRUE;
            yaf_config.spreadGroupby = 0;
            if (0 == yaf_opt_spread_groupby) {
                if (yaf_config.spreadGroupIndex[0]) {
                    air_opterr("--groupby <value> not given - "
                               "No value to groupby");
                }
            } else {
                struct groupby_name_value_st {
                    const char *name;
                    uint8_t     value;
                } groupby_name_value[] = {
                    {"port",     YAF_SPREAD_GROUPBY_DESTPORT},
                    {"vlan",     YAF_SPREAD_GROUPBY_VLANID},
                    {"applabel", YAF_SPREAD_GROUPBY_APPLABEL},
                    {"protocol", YAF_SPREAD_GROUPBY_PROTOCOL},
                    {"version",  YAF_SPREAD_GROUPBY_IPVERSION},
                    {NULL,       0},
                };
                unsigned int i;

                /*if (!yaf_config.spreadGroupIndex[0]) {
                 *  air_opterr("Invalid groupby: Must have values to group by"
                 *             " in --group");
                 *             }*/
                for (i = 0; groupby_name_value[i].name != NULL; ++i) {
                    if (0 == strcasecmp(yaf_opt_spread_groupby,
                                        groupby_name_value[i].name))
                    {
                        yaf_config.spreadGroupby = groupby_name_value[i].value;
                        break;
                    }
                }
                if (0 == yaf_config.spreadGroupby) {
                    air_opterr("Unsupported groupby type %s",
                               yaf_opt_spread_groupby);
                }
                if (YAF_SPREAD_GROUPBY_APPLABEL == yaf_config.spreadGroupby
                    && !yaf_opt_applabel_mode)
                {
                    air_opterr("Spread cannot groupby applabel without "
                               "--applabel");
                }
            }
#endif /* HAVE_SPREAD */

        } else {
            air_opterr("Unsupported IPFIX transport protocol %s",
                       yaf_opt_ipfix_transport);
        }

        /* grab TLS password from environment */
        if (yaf_opt_ipfix_tls) {
            yaf_config.connspec.ssl_key_pass = getenv("YAF_TLS_PASS");
        }

        /* mark that a network connection is requested for this spec */
        yaf_config.ipfixNetTrans = TRUE;
    } else if (yaf_config.connspec.svc) {
        air_opterr("--ipfix-port requires --ipfix");
    } else {
        if (!yaf_config.outspec || !strlen(yaf_config.outspec)) {
            if (yaf_opt_rotate) {
                /* Require a path prefix for IPFIX output */
                air_opterr("--rotate requires prefix in --out");
            } else {
                /* Default to stdout for no output without rotation */
                if (!yaf_config.no_output) {
                    yaf_config.outspec = C("-");
                }
            }
        }
    }

    /* Check for stdin/stdout is terminal */
    if ((strlen(yaf_config.inspec) == 1) && yaf_config.inspec[0] == '-') {
        /* Don't open stdin if it's a terminal */
        if (isatty(fileno(stdin))) {
            air_opterr("Refusing to read from terminal on stdin");
        }
    }

    if (!yaf_config.no_output) {
        if ((strlen(yaf_config.outspec) == 1) && yaf_config.outspec[0] == '-') {
            /* Don't open stdout if it's a terminal */
            if (isatty(fileno(stdout))) {
                air_opterr("Refusing to write to terminal on stdout");
            }
        }
    } else {
        yfDiffTimeClear(&yaf_config.rotate_interval);
        if (yaf_config.outspec) {
            g_warning("Ignoring --out %s due to presence of --no-output.",
                      yaf_config.outspec);
        }
    }

    if (yaf_config.pcapdir) {
        if (yaf_config.pcap_per_flow) {
            if (yaf_opt_max_payload == 0) {
                air_opterr("--pcap-per-flow requires --max-payload");
            }
            if (!(g_file_test(yaf_config.pcapdir, G_FILE_TEST_IS_DIR))) {
                air_opterr("--pcap requires a valid directory when "
                           "using --pcap-per-flow");
            }
            if (yaf_index_pcap) {
                WARN_OPT_CONFLICTS("index-pcap", "pcap-per-flow");
                yaf_index_pcap = FALSE;
            }
            if (yaf_pcap_meta_file) {
                WARN_OPT_CONFLICTS("pcap-meta-file", "pcap-per-flow");
                yaf_pcap_meta_file = NULL;
            }
        }

        if (yaf_hash_search) {
            if (yaf_hash_search < 0 || yaf_hash_search > UINT32_MAX) {
                air_opterr(
                    "Invalid value for --hash; must be positive 32-bit number");
            }
            if (yaf_pcap_meta_file) {
                WARN_OPT_CONFLICTS("pcap-meta-file", "hash");
                yaf_pcap_meta_file = NULL;
            }
            yaf_config.pcap_per_flow = TRUE;
        } else {
            if (yaf_stime_search) {
                air_opterr("--stime requires --hash");
            }
        }
    } else {
        if (yaf_config.pcap_per_flow) {
            air_opterr("--pcap-per-flow requires --pcap");
        }
        if (yaf_hash_search) {
            air_opterr("--hash requires --pcap");
        }
        if (yaf_stime_search) {
            air_opterr("--stime requires --pcap and --hash");
        } if (yaf_opt_pcap_timer) {
            WARN_OPT_REQUIRES("pcap-timer", "pcap");
            yaf_opt_pcap_timer = 0;
        }
    }

    yaf_config.pcap_timer = yaf_opt_pcap_timer;
    if (yaf_opt_max_pcap) {
        yaf_config.max_pcap = yaf_opt_max_pcap * 1024 * 1024;
    } else {
        yaf_config.max_pcap = yaf_config.max_pcap * 1024 * 1024;
    }

    /* log how yaf was invoked */
    g_debug("%s", commandString->str);
    g_string_free(commandString, TRUE);
    g_string_free(versionString, TRUE);
    air_option_context_free(aoctx);

    if (yaf_opt_promisc) {
        yfSetPromiscMode(0);
    }

    if (yaf_daemon) {
        yfDaemonize();
    }
}


/**
 *    Parses `payload_str` as a payload value, which may include a trailing
 *    'k' or 'K' for kilobytes, and ensures the value is not greater than
 *    `max_value`.  `option_name` is used for reporting errors.  Frees
 *    `payload_str` after parsing.  Exits the application on error.
 */
static uint32_t
yaf_opt_parse_payload(
    const gchar  *option_name,
    gchar        *payload_str,
    long long     max_value)
{
    const long long  k = 1024;
    long long        num;
    char            *ep;

    ep = payload_str;
    errno = 0;
    num = strtoll(payload_str, &ep, 0);
    if (num >= 0 && num <= max_value && errno == 0 && ep != payload_str) {
        if ('\0' == *ep) {
            g_free(payload_str);
            return (uint32_t)num;
        }
        if (('k' == *ep || 'K' == *ep) && '\0' == *(ep + 1)
            && num <= ((max_value + 1) / k))
        {
            g_free(payload_str);
            return (uint32_t)MIN(num * k, max_value);
        }
    }
    air_opterr("Invalid %s '%s': Expected value in range 0-%lld",
               option_name, payload_str, max_value);
}


/**
 * @brief Parses the string containing comma separated numbers and appends the
 * values into the GArray.  The GArray must be created with an element_size
 * equal to sizeof(uint16_t).
 *
 * @param option_name The option that called this function for error reporting
 * @param ports_str A comma separated string of ports between 0 and 65535
 *        inclusive
 * @param ports_array The GArray to append the ports to
 */
static void
yaf_opt_ports_str_2_array(
    const gchar  *option_name,
    const gchar  *ports_str,
    GArray       *ports_array)
{
    gchar   **ports = g_strsplit(ports_str, ",", -1);
    char     *ep;
    long      port;

    /* Append the ports into the array */
    for (unsigned int i = 0; ports[i] != NULL; ++i) {
        ep = ports[i];
        errno = 0;
        port = strtol(ports[i], &ep, 0);
        if (port >= 0 && port <= UINT16_MAX && ep != ports[i] && 0 == errno) {
            uint16_t p = port;
            g_array_append_val(ports_array, p);
        } else {
            air_opterr("Invalid entry '%s' in %s", ports[i], option_name);
        }
    }
    g_strfreev(ports);

    yaf_opt_remove_array_dups(ports_array);
}

#if YAF_ENABLE_APPLABEL
/**
 *  OptionArgFunc callback function to parse the --applabel-max-payload
 *  option.
 *
 *  @param option_name The name of the option being parsed
 *  @param yaf_opt_payload_applabel_select The value to be parsed
 *  @param data User data added to the GOptionGroup ogroup (unused)
 *  @param error The return location for a recoverable error
 */
static void
yaf_opt_parse_applabel_max_payload(
    const gchar  *applabel_max_payload)
{
    const char     *option_name = "applabel-max-payload";
    const uint32_t  k = 1024;
    long long       mx;
    gchar         **item;
    char           *ep;
    long long       num;
    uint32_t        entry[2] = {0, 0};

    item = g_strsplit(applabel_max_payload, ",", -1);

    for (unsigned int i = 0; item[i] != NULL; ++i) {
        /* payload length is after '=' */
        char           *paylen = strchr(item[i], '=');

        if (NULL == paylen || paylen == item[i] || '\0' == *(paylen + 1)) {
            air_opterr("Invalid entry '%s' in %s:"
                       " Each entry should be appLabel=octets",
                       item[i], option_name);
        }
        *paylen = '\0';
        ++paylen;

        /* parse the applabel */
        mx = UINT16_MAX;
        ep = item[i];
        errno = 0;
        num = strtol(item[i], &ep, 0);
        if (num >= 0 && num <= UINT16_MAX
            && ep != item[i] && *ep == '\0' && 0 == errno)
        {
            entry[0] = num;
        } else if (ep == item[i] && 0 == strcmp("rest", item[i])) {
            entry[0] = UINT32_MAX;
        } else {
            air_opterr("Invalid applabel '%s' in %s:"
                       " Expected value in range 0-%lld or 'rest'",
                      item[i], option_name, mx);
        }

        /* parse the octets */
        ep = paylen;
        errno = 0;
        num = strtoll(paylen, &ep, 0);
        if ((errno != 0 || ep == paylen) && '.' != *paylen) {
            if (ERANGE == errno) {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Value overflows the parser",
                           paylen, item[i], option_name);
            }
            air_opterr("Invalid octets value '%s' for label %s in %s:"
                       " Expected a numeric value",
                       paylen, item[i], option_name);
        }
        if ('\0' == *ep) {
            mx = UINT32_MAX;
            if (num < 0 || num > mx) {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Expected value in range 0-%lld",
                           paylen, item[i], option_name, mx);
            }
            entry[1] = num;
        } else if ('k' == *ep || 'K' == *ep) {
            if ('\0' != *(ep + 1)) {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Unexpected text ('%s') follows '%c''",
                           paylen, item[i], option_name, ep + 1, *ep);
            }
            mx = ((long long)1 + UINT32_MAX) / k;
            if (num < 0 || num > mx) {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Expected whole number value in range 0-%lld"
                           " when using kilobytes",
                           paylen, item[i], option_name, mx);
            }
            entry[1] = MIN(num * k, UINT32_MAX);
        } else if ('%' == *ep) {
            if ('\0' != *(ep + 1)) {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Unexpected text ('%s') follows '%c'",
                           paylen, item[i], option_name, ep + 1, *ep);
            }
            mx = (long long)UINT32_MAX / yaf_opt_max_payload * 100;
            if (num < 0 || num > mx) {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Expected whole number in range 0-%lld"
                           " when using percent",
                           paylen, item[i], option_name, mx);
            }
            entry[1] = ((double)yaf_opt_max_payload / 100.0 * (double)num);
        } else if ('.' == *ep) {
            double mxd;
            double mult;
            errno = 0;
            mxd = (double)UINT32_MAX / (double)yaf_opt_max_payload;
            mult = strtod(paylen, &ep);
            if (mult >= 0 && mult <= mxd && ep != paylen && *ep == '\0'
                && 0 == errno)
            {
                entry[1] = (uint32_t)(mult * yaf_opt_max_payload);
            } else {
                air_opterr("Invalid octets value '%s' for label %s in %s:"
                           " Expected value in range 0-%.2f"
                           " for floating-point multipler",
                           paylen, item[i], option_name, mxd);
            }
        } else {
            air_opterr("Invalid octets value '%s' for label %s in %s:"
                       " Unexpected text '%s'",
                       paylen, item[i], option_name, ep);
        }

        /* find the location for the entry, keeping the GArray sorted */
        guint  j;
        for (j = 0; j < yaf_opt_applabel_max_payload->len; j += 2) {
            uint32_t label = g_array_index(yaf_opt_applabel_max_payload,
                                           uint32_t, j);
            if (entry[0] < label) {
                g_array_insert_vals(yaf_opt_applabel_max_payload, j, entry, 2);
                break;
            }
            if (entry[0] == label) {
                air_opterr("Duplicate entries for applabel %u appear in %s",
                           entry[0], option_name);
            }
        }
        if (j == yaf_opt_applabel_max_payload->len) {
            g_array_insert_vals(yaf_opt_applabel_max_payload, j, entry, 2);
        }
    }

    g_strfreev(item);

    if (0 == yaf_opt_applabel_max_payload->len) {
        air_opterr("No valid appLabel=octets entries found in %s '%s'",
                   option_name, applabel_max_payload);
    }
}
#endif  /* YAF_ENABLE_APPLABEL */


/**
 * @brief Remove duplicate uint16's from GArray in-place.
 *
 * @param g The GArray to edit
 */
static void
yaf_opt_remove_array_dups(
    GArray  *g)
{
    if (g->len <= 1) {
        return;
    }
    for (guint i = 0; i < (g->len - 1); ++i) {
        uint16_t a = g_array_index(g, uint16_t, i);
        guint j = i + 1;
        while (j < g->len) {
            uint16_t b = g_array_index(g, uint16_t, j);
            if (a == b) {
                g_array_remove_index(g, j);
            } else {
                j++;
            }
        }
    }
}

#ifdef YAF_ENABLE_HOOKS
/*
 * yfPluginLoad
 *
 * parses parameters for plugin loading and calls the hook add function to
 * load the plugins
 *
 */
static void
pluginOptParse(
    GError **err)
{
    char         *plugName, *endPlugName = NULL;
    char         *plugOpt, *endPlugOpt = NULL;
    char         *plugConf, *endPlugConf = NULL;
    char         *plugNameIndex, *plugOptIndex, *plugConfIndex;
    unsigned char plugNameAlloc = 0;
    unsigned char plugOptAlloc = 0;
    unsigned char plugConfAlloc = 0;

    plugNameIndex = pluginName;
    plugOptIndex = pluginOpts;
    plugConfIndex = pluginConf;

    while (NULL != plugNameIndex) {
        /* Plugin file */
        endPlugName = strchr(plugNameIndex, ',');
        if (NULL == endPlugName) {
            plugName = plugNameIndex;
        } else {
            plugName = g_new0(char, (endPlugName - plugNameIndex + 1));
            strncpy(plugName, plugNameIndex, (endPlugName - plugNameIndex));
            plugNameAlloc = 1;
        }

        /* Plugin options */
        if (NULL == plugOptIndex) {
            plugOpt = NULL;
        } else {
            endPlugOpt = strchr(plugOptIndex, ',');
            if (NULL == endPlugOpt) {
                plugOpt = plugOptIndex;
            } else if (plugOptIndex == endPlugOpt) {
                plugOpt = NULL;
            } else {
                plugOpt = g_new0(char, (endPlugOpt - plugOptIndex + 1));
                strncpy(plugOpt, plugOptIndex, (endPlugOpt - plugOptIndex));
                plugOptAlloc = 1;
            }
        }

        /* Plugin config */
        if (NULL == plugConfIndex) {
            plugConf = NULL;
        } else {
            endPlugConf = strchr(plugConfIndex, ',');
            if (NULL == endPlugConf) {
                plugConf = plugConfIndex;
            } else if (plugConfIndex == endPlugConf) {
                plugConf = NULL;
            } else {
                plugConf = g_new0(char, (endPlugConf - plugConfIndex + 1));
                strncpy(plugConf, plugConfIndex, (endPlugConf - plugConfIndex));
                plugConfAlloc = 1;
            }
        }

        /* Attempt to load/initialize the plugin */
        if (!yfHookAddNewHook(plugName, plugOpt, plugConf, yfctx, err)) {
            g_warning("couldn't load requested plugin: %s",
                      (*err)->message);
        }

        if (NULL != plugNameIndex) {
            if (NULL != endPlugName) {
                plugNameIndex = endPlugName + 1;
            } else {
                /* we're done anyway */
                break;
            }
        }
        if (NULL != plugOptIndex) {
            if (NULL != endPlugOpt) {
                plugOptIndex = endPlugOpt + 1;
            } else {
                plugOptIndex = NULL;
            }
        }

        if (NULL != plugConfIndex) {
            if (NULL != endPlugConf) {
                plugConfIndex = endPlugConf + 1;
            } else {
                plugConfIndex = NULL;
            }
        }

        if (0 != plugNameAlloc) {
            g_free(plugName);
            plugNameAlloc = 0;
        }
        if (0 != plugOptAlloc) {
            g_free(plugOpt);
            plugOptAlloc = 0;
        }
        if (0 != plugConfAlloc) {
            g_free(plugConf);
            plugConfAlloc = 0;
        }
    }
}
#endif /* ifdef YAF_ENABLE_HOOKS */


/**
 *
 *    Called by the signal handler.
 *
 *
 *
 */
static void
yfQuit(
    int   s)
{
    yaf_quit++;

    /* typically do not want to call logger from within sighandler */
    /* g_debug("yaf received %s", strsignal(s)); */
    if (0 == yaf_quit_signal) {
        yaf_quit_signal = s;
    }

#if YAF_ENABLE_PFRING
    yfPfRingBreakLoop(NULL);
#endif
}


/**
 *
 *    Installs the signal handler for SIGINT, SIGTERM.
 *
 *
 *
 */
static void
yfQuitInit(
    void)
{
    struct sigaction sa, osa;

    /* install quit flag handlers */
    sa.sa_handler = yfQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGINT, &sa, &osa)) {
        g_error("sigaction(SIGINT) failed: %s", strerror(errno));
    }

    sa.sa_handler = yfQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGTERM, &sa, &osa)) {
        g_error("sigaction(SIGTERM) failed: %s", strerror(errno));
    }
}


/**
 *
 *
 *
 *
 *
 */
int
main(
    int    argc,
    char  *argv[])
{
    GError     *err = NULL;
    yfContext_t ctx = YF_CTX_INIT;
    int         datalink;
    gboolean    loop_ok = TRUE;
    yfFlowTabConfig_t  flowtab_config;

    memset(&flowtab_config, 0, sizeof(flowtab_config));

    /* check structure alignment */
    yfAlignmentCheck();

    /* parse options */
    yfParseOptions(&argc, &argv);
    ctx.cfg = &yaf_config;

    /* record yaf start time */
    yfTimeNow(&ctx.yaf_start_time);

    /* Set up quit handler */
    yfQuitInit();

    /* open interface if we're doing live capture */
    if (yaf_liveopen_fn) {
        /* open interface */
        if (!(ctx.pktsrc = yaf_liveopen_fn(yaf_config.inspec,
                                           yaf_opt_max_payload + 96,
                                           &datalink, &err)))
        {
            g_warning("Cannot open interface %s: %s", yaf_config.inspec,
                      err->message);
            exit(1);
        }

        /* drop privilege */
        if (!privc_become(&err)) {
            if (g_error_matches(err, PRIVC_ERROR_DOMAIN, PRIVC_ERROR_NODROP)) {
                g_warning("Running as root in --live mode, "
                          "but not dropping privilege");
                g_clear_error(&err);
            } else {
                yaf_close_fn(ctx.pktsrc);
                g_warning("Cannot drop privilege: %s", err->message);
                exit(1);
            }
        }
    } else {
        if (yaf_opt_caplist_mode) {
            /* open input file list */
            if (!(ctx.pktsrc = yfCapOpenFileList(yaf_config.inspec, &datalink,
                                                 yaf_tmp_dir, &err)))
            {
                g_warning("Cannot open packet file list file %s: %s",
                          yaf_config.inspec, err->message);
                exit(1);
            }
            /* drop privilege */
            if (!privc_become(&err)) {
                if (g_error_matches(err, PRIVC_ERROR_DOMAIN,
                                    PRIVC_ERROR_NODROP))
                {
                    g_warning("Running as root in --caplist mode, "
                              "but not dropping privilege");
                    g_clear_error(&err);
                } else {
                    yaf_close_fn(ctx.pktsrc);
                    g_warning("Cannot drop privilege: %s", err->message);
                    exit(1);
                }
            }
        } else {
            /* open input file */
            if (!(ctx.pktsrc = yfCapOpenFile(yaf_config.inspec, &datalink,
                                             yaf_tmp_dir, &err)))
            {
                g_warning("Cannot open packet file %s: %s",
                          yaf_config.inspec, err->message);
                exit(1);
            }
        }
    }

    if (yaf_opt_mac_mode) {
        yaf_config.macmode = TRUE;
    }

    if (yaf_opt_flowstats_mode) {
        yaf_config.flowstatsmode = TRUE;
    }

    if (yaf_opt_silk_mode) {
        yaf_config.silkmode = TRUE;
    }

    /* Calculate packet buffer size */
    if (yaf_opt_max_payload) {
        /* 54 for Headers (14 for L2, 20 for IP, 20 for L4) */
        /* This was added bc we now capture starting at L2 up to max-payload
         * for possible PCAP capture */
        ctx.pbuflen = YF_PBUFLEN_BASE + yaf_opt_max_payload + 54;
    } else {
        ctx.pbuflen = YF_PBUFLEN_NOPAYLOAD;
    }

    /* Allocate a packet ring. */
    ctx.pbufring = rgaAlloc(ctx.pbuflen, 128);

    /* Set up decode context */
    ctx.dectx = yfDecodeCtxAlloc(datalink,
                                 yaf_reqtype,
                                 yaf_opt_gre_mode,
                                 yaf_opt_vxlan_ports,
                                 yaf_opt_geneve_ports);

    /* Set up flow table */
    flowtab_config.active_sec = yaf_opt_active;
    flowtab_config.idle_sec = yaf_opt_idle;
    flowtab_config.max_flows = yaf_opt_max_flows;
    flowtab_config.max_payload = yaf_opt_max_payload;
    flowtab_config.udp_uniflow_port = yaf_opt_udp_uniflow_port;

#if YAF_ENABLE_APPLABEL
    flowtab_config.applabel_check_early = yaf_opt_applabel_check_early;
    flowtab_config.applabel_mode = yaf_opt_applabel_mode;
#endif
    flowtab_config.entropy_mode = yaf_opt_entropy_mode;
    flowtab_config.p0f_mode = yaf_opt_p0fprint_mode;
    flowtab_config.force_read_all = yaf_opt_force_read_all;
    flowtab_config.fpexport_mode = yaf_opt_fpExport_mode;
    flowtab_config.mac_mode = yaf_opt_mac_mode;
    flowtab_config.no_vlan_in_key = yaf_novlan_in_key;
    flowtab_config.silk_mode = yaf_opt_silk_mode;
    flowtab_config.flowstats_mode = yaf_opt_flowstats_mode;
    flowtab_config.udp_multipkt_payload = yaf_opt_udp_multipkt_payload;
    flowtab_config.uniflow_mode = yaf_opt_uniflow_mode;

    flowtab_config.ndpi = yaf_opt_ndpi;
    flowtab_config.ndpi_proto_file = yaf_ndpi_proto_file;

    flowtab_config.pcap_dir = yaf_config.pcapdir;
    flowtab_config.pcap_index = yaf_index_pcap;
    flowtab_config.pcap_max = yaf_config.max_pcap;
    flowtab_config.pcap_meta_file = yaf_pcap_meta_file;
    flowtab_config.pcap_per_flow = yaf_config.pcap_per_flow;
    flowtab_config.pcap_search_flowkey = (uint32_t)yaf_hash_search;
    flowtab_config.pcap_search_stime = yaf_stime_search;

#if YAF_ENABLE_APPLABEL
    flowtab_config.applabel_max_paylen = yaf_opt_applabel_max_payload;
#endif

    /* Set up flow table */
    ctx.flowtab = yfFlowTabAlloc(&flowtab_config, yfctx);

#if YAF_ENABLE_APPLABEL
    /* Freed by yfFlowTabAlloc() */
    flowtab_config.applabel_max_paylen = NULL;
#endif

    /* Set up fragment table - ONLY IF USER SAYS */
    if (!yaf_opt_nofrag) {
        ctx.fragtab = yfFragTabAlloc(30000,
                                     yaf_opt_max_frags,
                                     yaf_opt_max_payload);
    }

    /* We have a packet source, an output stream,
    * and all the tables we need. Run with it. */

    g_debug("YAF version %s, pid %ld is ready", VERSION, (long)getpid());

    yfStatInit(&ctx);

    loop_ok = yaf_loop_fn(&ctx);

    if (yaf_quit_signal) {
        g_debug("yaf terminating on signal %s", strsignal(yaf_quit_signal));
    }

    yfStatComplete();

    /* Close packet source */
    yaf_close_fn(ctx.pktsrc);

    /* Clean up! */
    if (ctx.flowtab) {
        yfFlowTabFree(ctx.flowtab);
    }
    if (ctx.fragtab) {
        yfFragTabFree(ctx.fragtab);
    }
    if (ctx.dectx) {
        yfDecodeCtxFree(ctx.dectx);
    }
    if (ctx.pbufring) {
        rgaFree(ctx.pbufring);
    }
    g_free(yaf_config.payload_export_applabels);

    /* Print exit message */
    if (loop_ok) {
        g_debug("yaf terminating");
    } else {
        g_warning("yaf terminating on error: %s", ctx.err->message);
    }

    return loop_ok ? 0 : 1;
}
