#! /usr/bin/env perl

##  Copyright 2003-2025 Carnegie Mellon University
##  See license information in LICENSE.txt.

##  This script modifies the HTML man pages generated by pod2html.  It
##  performs minor clean up and creates links to other man pages.


use warnings;
use strict;
use File::Copy  qw//;
use File::Temp  qw//;

# name of this script
my $NAME = $0;
$NAME =~ s,.*/,,;

# name of the project
my $PROJECT = 'YAF';

# Other man pages to link to
my @man_pages = qw(
    airdaemon
    applabel
    filedaemon
    getFlowKeyHash
    yaf
    yaf.init
    yafMeta2Pcap
    yafdhcp
    yafdpi
    yafscii
    yafzcbalance
    );
our $man_re = '('.join('|', map { "\Q$_\E" } @man_pages).')';

# get and open input
our $old = $ARGV[0];
if (!$old || $old =~ /^-/) {
    die "Usage: $NAME <filename>\n\tClean HTML in the specified file\n";
}

open(OLD, '<', $old)
    or die "$NAME: Cannot open '$old' for reading: $!\n";

# open output
our ($fh, $new) = File::Temp::tempfile(UNLINK => 1)
    or die "$NAME: Unable to create a temporary file: $!\n";
*NEW = $fh;

do_manpage();

# close files and replace $old with $new
close OLD;
close NEW
    or die "$NAME: Cannot close '$new': $!\n";

File::Copy::copy $new, $old
    or die "$NAME: Cannot replace '$old' with '$new': $!\n";

exit;


sub do_manpage
{
    $/ = "";  # read one paragraph at a time
    while (<OLD>) {
        # Normalize bold/strong tags between pod2html versions
        s{(</?)strong>}{$1b>}g;

        # # Downgrade all <Hn> tags by one
        # s{(</?h)(\d)\b([^>]*>)}{$1.($2 +1 ).$3}gieo;

        # Get rid of any mailto: links
        s{<link[^>]+mailto:[^>]+>}{}i;

        # Remove all <hr>
        s{<hr\b[^>]*>}{}iog;

        # Add "$PROJECT - " in front of the page's title.
        s{(<title>)([^<]+)(</title>)}{$1$PROJECT &mdash; $2$3};

        # Make all headers that have names into anchors.  We do this
        # by changing every
        #    <hN id="...">...</hN>
        # to
        #    <hN id=...><a href...>...</a></hN>
        s{(<h\d id=\"([^\"]+)\">)(.+?)(</h\d>)}{$1<a href="#$2">$3</a>$4}iog;

        # # Change <a name> to <a id>
        # s{<a name="}{<a id="}iog;

        # Make links to other YAF man pages
        s{(<b>$man_re\(\d\)</b>)}{<a href="$2.html">$1</a>}og;

        # Make links to SiLK man pages
        s{(<b>(flowcap|packlogic-\w+|rw[-.\w]+|sensor\.conf\|silk\.conf)\(\d\)</b>)}
        {<a href="/silk/$2.html">$1</a>}xog;

        # Make links libfixbuf man pages
        s{(<b>(ipfix(?:Dump|2json))\(\d\)</b>)}
        {<a href="/fixbuf/$2.html">$1</a>}xog;

        # Make links to super_mediator man pages
        s{(<b>(super_[-.\w]+)\(\d\)</b>)}
        {<a href="/super_mediator1/$2.html">$1</a>}xog;

        print NEW;
    }
}
# do_manpage


__END__
@DISTRIBUTION_STATEMENT_BEGIN@
YAF 2.18

Copyright 2025 Carnegie Mellon University.

NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
INFRINGEMENT.

Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
contact permission@sei.cmu.edu for full terms.

[DISTRIBUTION STATEMENT A] This material has been approved for public
release and unlimited distribution.  Please see Copyright notice for
non-US Government use and distribution.

This Software includes and/or makes use of Third-Party Software each
subject to its own license.

DM25-1281
@DISTRIBUTION_STATEMENT_END@
