#!/bin/sh -e

usage() {
  cat <<__HERE__
$0 OUTPUT_FILE [ --program-name NAME ] INPUT_FILE ...

This script is only expected to be used via the GEN_STARTUP_FILE
definition in Makefile.am, which defines needed environment variables
for the build process and provides the name of the output file.
Makefile rules should look like this:

yaf_startup: Makefile \$(YAF_SYSTEMD_SNIPPETS)
	\$(GEN_STARTUP_FILE) \$(YAF_SYSTEMD_SNIPPETS)

With any '--program-name' option immediately following \$(GEN_STARTUP_FILE).

Given an output filename and a set of input snippets, composes an output
file by concatenating those snippets and performing @var@ substitions on
the result.

    OUTPUT_FILE
        The path to the output file to produce. This is used to derive the
        program name if '--program-name' is not provided by removing
        everything after the *first* dot '.', underscore '_', dash '-',
        or at sign '@' in the filename.  For example, "x/y/z/test-conf.txt"
        would produce "test" as the program name by default. "foo@.service"
        would use "foo".

    --program-name PROGRAM_NAME
        Optionally use 'PROGRAM_NAME' as the program name instead of
        deriving it from the output filename. If this is included, it
        must occur as the second argument.

    INPUT_FILE ...
        The path to input files to be included in the output.

ENVIRONMENT

    This program requires the following variables to be set in the
    environment for substitution purposes. Each is substituted for an
    @variable-name@ substitution of the same name.

        bindir
        libdir
        libexecdir
        localstatedir
        runstatedir
        prefix
        sbindir
        sysconfdir

    It will also use srcdir, if it is set, as an alternative source for
    input files.

    Note: Since autoconf on RHEL8 does not support runstatedir, this
    script substitutes "${localstatedir}/run" if needed.

EXAMPLES

    $0 test_startup ../snippet-part_1.sh snippet-middle_bit.sh ../snippet-part_2.sh

        Will produce a file named "test_startup" by concatenating the
        given snippets and performing variable substitution. The
        program_name variable will be substituted with "test".

    $0 config-for-thing.conf --program-name thing snippet-part_a.conf snippet-part_b.conf

        Will produce a file named "config-for-thing.conf" from the given
        snippets using "thing" as the program name.

__HERE__
}

if [ $# -lt 2 ] || [ "$2" = "--program-name" ] && [ $# -lt 4 ]; then
    usage
    exit 1
elif [ "$1" = "-h" ] || [ "$1" = "--help" ]; then
    usage
    exit 0
fi

# Required. The name of the output file.
OUTPUT_FILE="$1"
shift

# Optionally, --program-name X
if [ "$1" = "--program-name" ]; then
    program_name="$2"
    shift 2
else
    program_name="$( echo "$OUTPUT_FILE" | ${SED:-sed} -e 's|^.*/||' -e 's/[-._@].*$//' )"
fi

# Check whether OUTPUT_FILE includes a directory.  If so, ensure the
# directory exists.
if dir="$( expr "${OUTPUT_FILE}" : '\(.*\)/' 2>/dev/null )"; then
    if [ -n "${dir}" ] && [ ! -d "${dir}" ]; then
        mkdir -p -- "${dir}"
    fi
fi

# Removing any existing output files
rm -f "$OUTPUT_FILE" "$OUTPUT_FILE.tmp"

# Transform the program name if needed.
if [ -n "$transform" ]; then
    program_name="$( echo "$program_name" | ${SED:-sed} -e "$transform" )"
fi

# For each input file, see if it's in the working or separate srcdir,
# then cat it. All of this output goes through sed, and if there are any
# missing substitution variables (which should be provided in the
# environment, it will error out.
for f in "$@"; do
    if [ -f "$f" ]; then
        input_file="$f"
    elif [ -n "$srcdir" ] && [ -f "$srcdir/$f" ]; then
        input_file="$srcdir/$f"
    else
        echo "Missing input snippet: '$f'" 1>&2
        exit 1
    fi
    cat "${input_file}" \
    | ${SED:-sed} \
        -e "s|@input_file[@]|${input_file:?}|g" \
        -e "s|@bindir[@]|${bindir:?}|g" \
        -e "s|@libdir[@]|${libdir:?}|g" \
        -e "s|@libexecdir[@]|${libexecdir:?}|g" \
        -e "s|@localstatedir[@]|${localstatedir:?}|g" \
        -e "s|@runstatedir[@]|${runstatedir:-${localstatedir:?}/run}|g" \
        -e "s|@prefix[@]|${prefix:?}|g" \
        -e "s|@sbindir[@]|${sbindir:?}|g" \
        -e "s|@sysconfdir[@]|${sysconfdir:?}|g" \
        -e "s|@program_name[@]|$program_name|g" \
        -e "/__GEN_STARTUP_FILE_END__/{s/^.*//; q;}"
done \
> "$OUTPUT_FILE"

# Make sure to document new substitution variables added to the above
# sed invocation in the usage, and to add them to Makefile.am's
# GEN_STARTUP_FILE definition.


#####################################################################
##  Copyright 2025 Carnegie Mellon University
##  See license information in LICENSE.txt.
#####################################################################
##  @DISTRIBUTION_STATEMENT_BEGIN@
##  YAF 2.18
##
##  Copyright 2025 Carnegie Mellon University.
##
##  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
##  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
##  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
##  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
##  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
##  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
##  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
##  INFRINGEMENT.
##
##  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
##  contact permission@sei.cmu.edu for full terms.
##
##  [DISTRIBUTION STATEMENT A] This material has been approved for public
##  release and unlimited distribution.  Please see Copyright notice for
##  non-US Government use and distribution.
##
##  This Software includes and/or makes use of Third-Party Software each
##  subject to its own license.
##
##  DM25-1281
##  @DISTRIBUTION_STATEMENT_END@
#####################################################################
