/*
 *  Copyright 2006-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  yaftab.h
 *  YAF Active Flow Table
 *
 *  ------------------------------------------------------------------------
 *  Authors: Brian Trammell
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 2.18
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1281
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

/*
 * This is the documentation for the _old_ yaftab.h; it is no longer current,
 * and should not be read by anyone.
 *
 * Flow generation interface for YAF. This facility works by maintaining a
 * current flow table. Packets may be added to the active flows within this
 * table using the yfFlowPkt() call. Completed flows may be written to an
 * IPFIX message buffer using yfFlowFlush().
 *
 * The flow table is configured by a number of global variables.
 *
 * <tt>yaf_idle</tt> sets
 * the idle timeout in seconds. A flow that receives no packets for the idle
 * timeout is assumed to be complete. The idle timeout is set to 300 seconds
 * (five minutes) by default.
 *
 * <tt>yaf_active</tt> sets the active timeout in seconds.
 * The maximum duration of a flow is the active timeout; additional packets
 * for the same flow will be counted as part of a new flow. The active timeout
 * is set to 1800 seconds (half an hour) by default.
 *
 * <tt>yaf_flowlim</tt> sets the maximum size of the flow table; flows
 * exceeding
 * this limit will be expired in least-recent order, as if they were idle. The
 * flow limit defaults to zero, for no limit. Use this global to limit resource
 * usage by the flow table.
 *
 * <tt>yaf_paylen</tt> sets the number of bytes of payload to capture from the
 * start of each flow. The payload length defaults to zero, which disables
 * payload capture.
 *
 * <tt>yaf_uniflow</tt>, if TRUE, exports flows in uniflow mode, using the
 * record adjacency export method described in section 3 of
 * draft-ietf-ipfix-biflow. Defaults to FALSE.
 *
 * <tt>yaf_macmode</tt>, if TRUE, exports layer 2 information with each flow;
 * presently this is limited to VLAN tags but may be expanded to include the
 * MPLS stack and MAC addresses in the future. Defaults to FALSE.
 *
 * <tt>yaf_silkmode</tt>, if TRUE, enables SiLK compatibility mode. In this
 * mode, totalOctetCount and reverseTotalOctetCount are clamped to 32 bits.
 * Any packet that would cause either of these counters to overflow 32 bits
 * will force an active timeout. The high-order bit of the flowEndReason IE
 * is set on any flow created on a counter overflow, as above, or on an active
 * timeout. Defaults to FALSE.
 *
 * <tt>yaf_reqtype</tt> limits the flow table to collecting IPv4 or IPv6 flows
 * only. Set to YF_TYPE_IPv4 for IPv4 flows only, YF_TYPE_IPv6 for IPv6 flows
 * only, or YF_TYPE_IPANY (the default) to collect both IPv4 and IPv6 flows.
 *
 * This facility is used by YAF to assemble packets into flows.
 */

/**
 * @file
 *
 * Flow generation interface for YAF. [TODO - frontmatter]
 *
 * This facility is used by YAF to assemble packets into flows.
 */

#ifndef _YAF_TAB_H_
#define _YAF_TAB_H_

#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>


/**
 *    Size of yfFlowTabStats_t.stat_closed[] table for holding YAF_END_*
 *    reasons. (YAF_END_* are defined in yafcore.h.)  Values 1 to
 *    YAF_STAT_CLOSED_SIZE-1 get their own bins; Values YAF_STAT_CLOSED_SIZE
 *    and larger go into bin 0.
 */
#define YAF_STAT_CLOSED_SIZE  6


/**
 *  A flow table. Opaque. Create with yfFlowTabAlloc() and free with
 *  yfFlowTabFree().
 */
typedef struct yfFlowTab_st yfFlowTab_t;

/**
 *  Configuration settings used to initalize the flow table in
 *  yfFlowTabAlloc().
 */
typedef struct yfFlowTabConfig_st {
    /**
     *  Active timeout in seconds. The maximum duration of a flow is the
     *  active timeout; additional packets for the same flow will be counted
     *  as part of a new flow.
     */
    int32_t     active_sec;
    /**
     *  Idle timeout in seconds. A flow that receives no packets for the idle
     *  timeout is assumed to be complete.
     */
    int32_t     idle_sec;
    /**
     *  Maximum number of active flows. Flows exceeding this limit will be
     *  expired in least-recent order, as if they were idle.  Used to limit
     *  resource usage of a flow table. A value of 0 disables flow count
     *  limits.
     */
    uint32_t    max_flows;
    /**
     *  Maximum octets of payload to capture per flow direction.  Requires at
     *  least max_payload octets of payload to be available in each packet
     *  buffer passed to yfFlowPBuf().  A value of 0 disables payload capture
     *  and export.
     */
    uint32_t    max_payload;
    /**
     *  A non-zero value enables early appLabel detection.  The value is
     *  number of payload octets one side of a flow (FlowVal) has to have to
     *  trigger running the appLabel detector.
     */
    uint32_t    applabel_check_early;

    /**
     *  If not NULL, and ndpi is TRUE, use the provided protocol file to
     *  expand the sub-protocols list and port-based detection methods.
     */
    const char *ndpi_proto_file;

    /**
     *  Directory to put pcap-per-flow files
     */
    const char *pcap_dir;
    /**
     *  File for pcap meta output. Default is stdout
     */
    const char *pcap_meta_file;
    /**
     *  Maximum size [in bytes] of a pcap file before rotating.
     */
    uint64_t    pcap_max;
    /**
     *  The flow key hash to create a PCAP for.
     */
    uint32_t    pcap_search_flowkey;
    /**
     *  The start time to create a PCAP for.
     */
    const char *pcap_search_stime;

    /**
     *  Max payload length to store per appLabel.  Must be NULL if not in use.
     *  Array contains uint32_t added in pairs, where the first is the
     *  appLabel and the second is the payload-length.  An appLabel of
     *  UINT32_MAX indicates the payload to use for any appLabel that is not
     *  mentioned in array.  yfFlowTabAlloc() expects the array's appLabel
     *  entries to be unique and sorted (ascending) by appLabel.
     *
     *  yfFlowTabAlloc() frees the GAarray.
     */
    GArray     *applabel_max_paylen;

    /**
     *  If not 0, then this will enable exporting a single UDP packet with
     *  this src/dst port as a flow.
     */
    uint16_t    udp_uniflow_port;

    /**
     *  If TRUE, then the payload, (as limited by max_payload,) is sent
     *  through various plugins and code in order to determine which protocol
     *  is running on the flow by doing only payload inspection and exporting
     *  payload relevent information.
     */
    gboolean    applabel_mode;
    /**
     *  If TRUE, then a Shannon Entropy measurement is made over the captured
     *  payload (as limited by max_payload).  The entropy value is exported as
     *  two values one for forward payload and one for reverse payload.
     */
    gboolean    entropy_mode;
    /**
     *  If TRUE, then YAF will do some extra calculations on flows.
     */
    gboolean    flowstats_mode;
    /**
     *  If TRUE, then yaf will process files that are out of
     *  sequence.
     */
    gboolean    force_read_all;
    /**
     *  If TRUE, then this will enable exporting of full packet banners of the
     *  TCP negotiations for the first three packets (including IP and
     *  transport headers) for external fingerprinting
     */
    gboolean    fpexport_mode;
    /**
     *  If TRUE, collect and export source and destination Mac Addresses.
     */
    gboolean    mac_mode;
    /**
     *  If TRUE, enable nDPI application labeling with standard protocols.
     */
    gboolean    ndpi;
    /**
     *  If TRUE, this will remove the vlan in the calculation of the flow key
     *  hash.
     */
    gboolean    no_vlan_in_key;
    /**
     *  If TRUE, then this will enable passive OS finger printing using the
     *  p0f engine based mostly on TCP negotiation
     */
    gboolean    p0f_mode;
    /**
     *  If TRUE, print one line per packet we export. This
     *  will give offset and length into the pcap yaf writes.
     */
    gboolean    pcap_index;
    /**
     *  If TRUE, then pcap_dir will be set to the directory
     *  to place pcap-per-flow files.
     */
    gboolean    pcap_per_flow;
    /**
     *  If TRUE, clamp totalOctetCount and maxTotalOctetCount to 32 bits and
     *  force active timeout on overflow. Set high order bit in flowEndReason
     *  for each flow created on an overflow or active timeout. Breaks IPFIX
     *  interoperability; use for direct export to SiLK rwflowpack or flowcap.
     */
    gboolean    silk_mode;
    /**
     *  If TRUE, then this will enable capturing payload for all UDP packets
     *  in a flow (instead of just the first packet) up to `max_payload
     *  value`.
     */
    gboolean    udp_multipkt_payload;
    /**
     *  If TRUE, export biflows using record adjacency (two uniflows exported
     *  back-to-back. Use this for interoperability with IPFIX collectors that
     *  do not implement RFC 5103.
     */
    gboolean    uniflow_mode;

} yfFlowTabConfig_t;


/**
 *    Flow table statistics.  Unless otherwise noted, all values are relative
 *    to yaf's invocation.
 */
typedef struct yfFlowTabStats_st {
    /**
     *  Sum of lengths for all packets read.
     */
    uint64_t   stat_octets;
    /**
     *  Number of packets read.
     */
    uint64_t   stat_packets;
    /**
     *  Number of packets rejected because they were out of sequence.
     */
    uint64_t   stat_seqrej;
    /**
     *  Total number of flow records created.
     */
    uint64_t   stat_flows;
    /**
     *  Subset of `stat_flows` that were uniflow (had no reverse data).
     */
    uint64_t   stat_uniflows;
    /**
     *  Number of flow records created.
     */
    uint64_t   stat_opened;
    /**
     *  Number of flow records closed for various YAF_END_* reasons.
     */
    uint64_t   stat_closed[YAF_STAT_CLOSED_SIZE];
    /**
     *  Peak number of active flows ever held in the flow table.
     */
    uint32_t   stat_peak;
    /**
     *  Number of flow table flush events.
     */
    uint32_t   stat_flush;
    /**
     *  Current number of active flow records.
     */
    uint32_t   stat_count;
#if YAF_MPLS
    /**
     *  Maximum number of MPLS nodes.
     */
    uint32_t   max_mpls_labels;
    /**
     *  Current number of active MPLS labels.
     */
    uint32_t   stat_mpls_labels;
#endif
} yfFlowTabStats_t;

/**
 * yfFlowTabAlloc
 *
 * Allocate a flow table.
 *
 * @param ftconfig  The configuration settings to use for the table.
 * @param hfctx     The plugin hooks context variable (NULL if plugins not
 *                  enabled)
 *
 * @return a new flow table.
 */
yfFlowTab_t *
yfFlowTabAlloc(
    const yfFlowTabConfig_t  *ftconfig,
    void                    **hfctx);

/**
 * Free a previously allocated flow table. Discards any outstanding active
 * flows without closing or flushing them; use yfFlowTabFlush() with `close`
 * set to TRUE before yfFlowTabFree() to do this.
 *
 * @param flowtab a flow table allocated by yfFlowTabAlloc()
 */
void
yfFlowTabFree(
    yfFlowTab_t  *flowtab);


/**
 * Update the Pcap Filename in the Flowtab for pcap meta data output
 *
 * @param flowtab pointer to flow table
 * @param new_file_name the filename of the next pcap file to write to
 */
void
yfUpdateRollingPcapFile(
    yfFlowTab_t  *flowtab,
    char         *new_file_name);

/**
 * Fills `tabstats` with a copy of the current statistics for the flow table.
 *
 * @param flowtab flow table
 * @param tabstats destination for the flow table's statistics
 */
void
yfFlowTabGetStats(
    const yfFlowTab_t  *flowtab,
    yfFlowTabStats_t   *tabstats);

/**
 * Add a decoded packet buffer to a given flow table. Adds the packet to
 * the flow to which it belongs, creating a new flow if necessary. Causes
 * the flow to which it belongs to time out if it is longer than the active
 * timeout.  Closes the flow if the flow closure conditions (TCP RST, TCP FIN
 * four-way teardown) are met.
 *
 * @param flowtab   flow table to add the packet to
 * @param pbuflen   size of the packet buffer pbuf
 * @param pbuf      packet buffer containing decoded packet to add.
 */
void
yfFlowPBuf(
    yfFlowTab_t  *flowtab,
    size_t        pbuflen,
    yfPBuf_t     *pbuf);

/**
 * Flush closed flows in the given flow table to the given IPFIX Message
 * Buffer. Causes any idle flows to time out, removing them from the active
 * flow table; also enforces the flow table's resource limit. If close is
 * TRUE, additionally closes all active flows and flushes as well.
 *
 * @param yfContext YAF thread context structure, holds pointers for the
 *                  flowtable from which to flush flows and the fbuf, the
 *                  destination to which the flows should be flushed
 * @param close     close all active flows before flushing
 * @param err       An error description pointer; must not be NULL.
 * @return TRUE on success, FALSE otherwise.
 */
gboolean
yfFlowTabFlush(
    void      *yfContext,
    gboolean   close,
    GError   **err);

/**
 * Get the current packet clock from a flow table.
 *
 * @param flowtab a flow table
 * @param output variable where the function stores the current packet clock
 */
void
yfFlowTabCurrentTime(
    const yfFlowTab_t  *flowtab,
    yfTime_t           *yftime);

/**
 * Print flow table statistics to the log.
 *
 * @param flowtab flow table to dump stats for
 * @param timer a GTimer containing the runtime
 *              (for packet and flow rate logging). May be NULL to suppress
 *              rate logging.
 */
uint64_t
yfFlowTabDumpStats(
    yfFlowTab_t  *flowtab,
    GTimer       *timer);

#endif /* ifndef _YAF_TAB_H_ */
