#! /usr/bin/env python
## ------------------------------------------------------------------------
## nmsg_to_pipeline.py
## sample IPFIX exporter using pyfixbuf.
## Takes a CSV file that has format <domain name>,<type>,<ttl>,<ip>
## ------------------------------------------------------------------------
## Copyright (C) 2017-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## Authors: Emily Sarneso
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt

from __future__ import print_function
import sys
import pyfixbuf
import pyfixbuf.cert
import csv

# Test that the argument number is correct
if (len(sys.argv) < 3 or sys.argv[1] in ['-h', '--help']):
    print("Must supply an IPFIX file to write to.")
    print("Usage: nmsg_to_pipeline.py nmsg_csv_file.txt <ipfix file or domain/ip> <port_number>")
    sys.exit()

# Create the information model with the standard IPFIX elements
infomodel = pyfixbuf.InfoModel()

# Add YAF's IPFIX elements
pyfixbuf.cert.add_elements_to_model(infomodel)

# Create the "outer" template
tmpl = pyfixbuf.Template(infomodel)

# Add elements we want in our template
a_list = [
    pyfixbuf.InfoElementSpec("dnsA"),
    pyfixbuf.InfoElementSpec("dnsName"),
    pyfixbuf.InfoElementSpec("dnsTTL"),
    pyfixbuf.InfoElementSpec("dnsRRType")]

# Add elements to our template
tmpl.add_spec_list(a_list)

aaaa_list = [
    pyfixbuf.InfoElementSpec("dnsAAAA"),
    pyfixbuf.InfoElementSpec("dnsName"),
    pyfixbuf.InfoElementSpec("dnsTTL"),
    pyfixbuf.InfoElementSpec("dnsRRType")]

tmplaaaa = pyfixbuf.Template(infomodel)
tmplaaaa.add_spec_list(aaaa_list)

# Create the exporter
exporter = pyfixbuf.Exporter()

# Create the IPFIX file to write to or open a network socket
if (len(sys.argv) == 3):
    exporter.init_file(sys.argv[2])
else:
    exporter.init_net(hostname=sys.argv[2], port=sys.argv[3], transport='tcp')

# Create the session
session = pyfixbuf.Session(infomodel)

# Create internal and external templates since this is an exporter
session.add_internal_template(tmpl, 999)
session.add_external_template(tmpl, 999)

session.add_internal_template(tmplaaaa, 1000)
session.add_external_template(tmplaaaa, 1000)

# Create the records to fill for export
rec = pyfixbuf.Record(infomodel, tmpl)

reca = pyfixbuf.Record(infomodel, tmplaaaa)

# Create the buffer for exporter
buf = pyfixbuf.Buffer(rec)

# Make the buffer an export buffer
buf.init_export(session, exporter)

# Set the internal template on the buffer
buf.set_internal_template(999)

# Export the templates to the file
session.export_templates()

# Set the export template
buf.set_export_template(999)

# Open NMSG CSV file
if sys.argv[1] == '-':
    f = sys.stdin
else:
    f = open(sys.argv[1], 'r')

csv.field_size_limit(sys.maxsize)

c = csv.reader(f, delimiter=',')

count = 0

for row in c:
    if (row[1] == "A(1)" or row[1] == "1"):
        try:
            rec['dnsA'] = row[3]
        except:
            sys.stderr.write("row[3] is '%s'\n" % row[3])
        rec['dnsName'] = row[0]
        rec['dnsTTL'] = int(row[2])
        rec['dnsRRType'] = 1

        buf.set_internal_template(999)
        buf.set_export_template(999)
        buf.append(rec)

        count += 1

        #some records have more than 1 IPv4Address on a line
        if (len(row) > 4):
            k = len(row) - 4
            while (k):
                rec['dnsA'] = row[3+k]
                rec['dnsName'] = row[0]
                rec['dnsTTL'] = int(row[2])
                rec['dnsRRType'] = 1
                buf.append(rec)

                count += 1
                k -= 1
    elif (row[1] == "AAAA(28)" or row[1] == "28"):
        try:
            reca['dnsAAAA'] = row[3]
        except:
            sys.stderr.write("row[3] is '%s'" % row[3])
        reca['dnsName'] = row[0]
        reca['dnsTTL'] = int(row[2])
        reca['dnsRRType'] = 28

        buf.set_internal_template(1000)
        buf.set_export_template(1000)
        buf.append(reca)

        count += 1

        if (len(row) > 4):
            k = len(row) - 4
            while (k):
                reca['dnsAAAA'] = row[3+k]
                reca['dnsName'] = row[0]
                reca['dnsTTL'] = int(row[2])
                reca['dnsRRType'] = 28
                buf.append(reca)

                count += 1
                k -= 1

buf.emit()

sys.stderr.write("FINISHED sending %d records\n" % count)

f.close()
