#! /usr/bin/env python
## ------------------------------------------------------------------------
## sample_collector.py
## sample IPFIX collector using pyfixbuf.
## Reads an IPFIX file and writes to a text file.
## ------------------------------------------------------------------------
## Copyright (C) 2013-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## Authors: Emily Sarneso
## See license information in LICENSE-OPENSOURCE.txt

from __future__ import print_function
import os
import sys
import pyfixbuf as p
import pyfixbuf.cert
from time_utils import make_datetime

# Test that the argument number is correct

if (len(sys.argv) != 3 or sys.argv[1] in ['-h', '--help']):
    print("Must supply an IPFIX file to read and a text file to write.")
    print("Usage: sample_collector.py file.ipfix file.txt")
    sys.exit()

# Create an InfoModel

infomodel = p.InfoModel()

# Add YAF to the infomodel
pyfixbuf.cert.add_elements_to_model(infomodel)

# Create a Template for Data Flow Records

tmpl = p.Template(infomodel)

# Create a Template for Stats (statistic) Records

stats_tmpl = p.Template(infomodel)

# Specify the elements for the data template

data_list = [
    p.InfoElementSpec("flowStartMilliseconds"),
    p.InfoElementSpec("flowEndMilliseconds"),
    p.InfoElementSpec("octetTotalCount"),
    p.InfoElementSpec("reverseOctetTotalCount"),
    p.InfoElementSpec("packetTotalCount"),
    p.InfoElementSpec("reversePacketTotalCount"),
    p.InfoElementSpec("sourceIPv4Address"),
    p.InfoElementSpec("destinationIPv4Address"),
    p.InfoElementSpec("sourceTransportPort"),
    p.InfoElementSpec("destinationTransportPort"),
    p.InfoElementSpec("flowAttributes"),
    p.InfoElementSpec("reverseFlowAttributes"),
    p.InfoElementSpec("protocolIdentifier"),
    p.InfoElementSpec("flowEndReason"),
    p.InfoElementSpec("silkAppLabel"),
    p.InfoElementSpec("subTemplateMultiList")]

# Specify the elements for the stats template (this is only a subset
# of stats that YAF exports)

stats_list = [
    p.InfoElementSpec("exportedFlowRecordTotalCount"),
    p.InfoElementSpec("packetTotalCount"),
    p.InfoElementSpec("droppedPacketTotalCount"),
    p.InfoElementSpec("ignoredPacketTotalCount")]

# Add the lists to their respective template

tmpl.add_spec_list(data_list)
stats_tmpl.add_spec_list(stats_list)

# Create a collector

collector = p.Collector()

# Give the collector the input file to read from

collector.init_file(sys.argv[1])

# Create a session

session = p.Session(infomodel)

# Add the data template to the session

session.add_internal_template(tmpl, 999)

# Add the stats template

session.add_internal_template(stats_tmpl, 911)

# Create Records for the main template and stats template

rec = p.Record(infomodel, tmpl)

statsrec = p.Record(infomodel, stats_tmpl)

# Create a buffer to read from

buf = p.Buffer()

# Add the session and collector the buffer

buf.init_collection(session, collector)

# Tell the buffer to process any Info Element Description Records

buf.auto_insert()

# Set the internal template to the data template

buf.set_internal_template(999)

# Open the output file

if sys.argv[2] == "-":
    outFile = sys.stdout
else:
    outFile = open(sys.argv[2], "w")

flowcount = 0

noprintlist=["paddingOctets", "subTemplateList", "subTemplateMultiList"]

# Now we are ready to get the elements from the buffer!
for data in buf:
    flowcount += 1
    for field in data.iterfields():
        if field.ie.type == p.DataType.MILLISECONDS:
            outFile.write(field.name + ": " +
                          str(make_datetime(field.value/1000)) + "\n")
        elif field.name not in noprintlist:
            outFile.write(field.name + ": " + str(field.value) + "\n")

    stml = data["subTemplateMultiList"]
    for entry in stml:
        if "tcpSequenceNumber" in entry:
            for record in entry:
                for field in record.iterfields():
                    outFile.write(field.name + ": " + str(field.value) + "\n")
        elif entry.template_id == 0xCE00:
            for record in entry:
                stl = record[0]
                for dnsrec in stl:
                    for field in dnsrec.iterfields():
                        if (field.name not in noprintlist):
                            outFile.write(field.name + ": " + str(field.value) + '\n')

            stl.clear()

    stml.clear()
    outFile.write("----------------------------\n")

    # check to see if the next template is a stats template
    try:
        tmpl_next = buf.next_template()
    except StopIteration:
        tmpl_next = None

    if ( tmpl_next and tmpl_next.scope ):
        outFile.write("STATS Record:\n")
        # set the interal template to the stats template
        buf.set_internal_template(911)
        # get the record
        stats = buf.next_record(statsrec)
        if (stats != None):
            # print all the items in stats
            for fields in stats.iterfields():
                outFile.write(field.name + ": " + str(field.value) + "\n")
        # set the template back to the data template
        buf.set_internal_template(999)

sys.stderr.write("Finished. Processed " + str(flowcount) + " flows.\n")

