#! /usr/bin/env python
## ------------------------------------------------------------------------
## Copyright (C) 2018-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt
## ------------------------------------------------------------------------

from __future__ import print_function
import argparse
import sys
import os
import shared_utils
import dnsQNameFinder_utils
from string import punctuation


class Args(object):
    pass


if __name__ == '__main__':
    shared_utils.print_warning_message()

    in_files = []
    bad_dnsQNames = []
    args = Args()

    # Create command-line argument parser and add arguments
    parser = argparse.ArgumentParser(prog='dnsQNameFinder.py',
                                     description='descripton: Process some IPFIX file(s)')
    parser.add_argument('--input', '--in', nargs=1, required=True,
                        help='input file or directory (no *)')

    ip_group = parser.add_mutually_exclusive_group(required=True)
    ip_group.add_argument('--dnsqfile', nargs=1,
                          help='text file of DNS query names to filter records by')
    ip_group.add_argument('--dnsq', nargs=1,
                          help='single DNS query name to filter records by')

    output_group = parser.add_mutually_exclusive_group(required=True)
    output_group.add_argument('--outfields', nargs='+',
                              help="Fields to print from filtered records")
    output_group.add_argument('--outfile', nargs=1,
                              help="IPFIX file to output filtered records to")
    output_group.add_argument('--count', action='store_true',
                              help="Print total count of filtered records")

    parser.add_argument('--delim', nargs=1, default='|',
                        help="field delimiter for printed fields")

    # Add all command-line arguments to instance of class Args
    parser.parse_args(namespace=args)

    # handle input directory vs single input file
    if os.path.isdir(args.input[0]):
        for filename in os.listdir(args.input[0]):
            in_files.append(args.input[0] + '/' + filename)
    else:
        in_files.append(args.input[0])

    # handle single dnsName vs text file
    if args.dnsq:
        bad_dnsQNames.append(args.dnsq[0].strip(punctuation))
    else:
        dnsq_file = open(args.dnsqfile[0], 'r')
        for line in dnsq_file:
            bad_dnsQNames.append(line.strip(punctuation))
        dnsq_file.close()

    # handle counting or outputting field values to console or records to ipfix file
    if args.outfields:
        # output filtered records to console
        dnsQNameFinder_utils.filter_to_screen(in_files, bad_dnsQNames, args.outfields, args.delim[0])
    elif args.count:
        # output total count of filtered records
        print('Counting filtered records...')
        count = dnsQNameFinder_utils.count_records(in_files, bad_dnsQNames)
    else:
        # ourput filtered records to IPFIX file
        print('Filtering to %s...' % args.outfile[0], file=sys.stderr)
        dnsQNameFinder_utils.filter_to_ipfix(in_files, bad_dnsQNames, args.outfile[0])
