## ------------------------------------------------------------------------
## Copyright 2008-2022 by Carnegie Mellon University
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt
## ------------------------------------------------------------------------

from datetime import date, datetime, timedelta, tzinfo
import re

try:
    long(3)
except:
    long = int


# Note: When we format dates in ISO format, we do not include the Z or
# +00:00 to designate that it's UTC.  We assume that any date without
# an explicit timezone is UTC.  Internally, we work with datetime
# objects that include the UTC timezone field to be sure that they are
# correctly handled by other things.

TD_ZERO = timedelta(0)

class tzinfo_UTC(tzinfo):
    def utcoffset(self, dt):
        return TD_ZERO
    def tzname(self, dt):
        return "Z"
    def dst(self, dt):
        return TD_ZERO
    def __repr__(self):
        return "tzinfo_UTC()"

utc = tzinfo_UTC()

re_iso_datetime = re.compile(r"""
       ^ (?P<year>\d\d\d\d)
       - (?P<mon>\d\d)
       - (?P<day>\d\d)
(?: [ T] (?P<hour>\d\d)
(?:    : (?P<min>\d\d)
(?:    : (?P<sec>\d\d)
(?:   \. (?P<fsec>\d+) )? )? )? )?
(?:      (?P<tz> Z | [+-] \d\d (?: : \d\d )? ) )? $
""", re.VERBOSE)

re_silk_datetime = re.compile(r"""
        ^ (?P<year>\d\d?\d?\d?)
        / (?P<mon>\d\d?)
        / (?P<day>\d\d?)
(?: [ :T] (?P<hour>\d\d?)
(?:     : (?P<min>\d\d?)
(?:     : (?P<sec>\d\d?)
(?:    \. (?P<fsec>\d+) )? )? )? )? $
""", re.VERBOSE)

def make_datetime(value, utc_only=True):
    """

    Produces a :class:`datetime.datetime` object from a number
    (seconds from UNIX epoch), a string (in ISO format, SiLK format),
    or a :class:`datetime.datetime` object.  If
    *utc_only* is ``True``, coerces the result to be in the UTC time
    zone.

    """
    v = value
    if isinstance(v, (int, long, float)):
        # seconds from epoch: return datetime
        v = datetime.utcfromtimestamp(v)
        return v.replace(tzinfo=utc)
    if isinstance(v, basestring):
        # string representation: parse to datetime, then proceed
        vs = v.strip()
        m = (re_iso_datetime.match(vs) or
             re_silk_datetime.match(vs) or
             re_old_silk_datetime.match(vs))
        if not m:
            raise ValueError("Could not parse %s as a datetime" % repr(v))
        dt_year = int(m.group('year'))
        dt_mon = int(m.group('mon'))
        dt_day = int(m.group('day'))
        dt_hour = int(m.group('hour') or 0)
        dt_min = int(m.group('min') or 0)
        dt_sec = int(m.group('sec') or 0)
        dt_usec = int(((m.group('fsec') or '0') + '00000')[:6])
        try:
            dt_tz_str = m.group('tz')
        except:
            dt_tz_str = 'Z'
        if dt_tz_str == '' or dt_tz_str is None:
            dt_tz = None
        elif dt_tz_str == 'Z':
            dt_tz = utc
        else:
            if len(dt_tz_str) == 3:
                dt_tz_offset = int(dt_tz_str[1:3]) * 60
            else:
                dt_tz_offset = int(dt_tz_str[1:3]) * 60 + int(dt_tz_str[4:6])
            if dt_tz_str[0] == '-':
                dt_tz_offset = -dt_tz_offset
            dt_tz = tzinfo_fixed(dt_tz_offset)
        v = datetime(dt_year, dt_mon, dt_day, dt_hour, dt_min, dt_sec,
                     dt_usec, dt_tz)
    if isinstance(v, datetime):
        # datetime object
        if not utc_only:
            # allowing non-UTC times, so return it
            return v
        else:
            # normalize to UTC
            if v.tzinfo == None:
                # if it's a datetime with no timezone, assume UTC
                return v.replace(tzinfo=utc)
            else:
                # otherwise, convert it to UTC
                return v.astimezone(utc)
    value_error = ValueError("can't interpret %s as a datetime" % repr(value))
    raise value_error
