#!/usr/bin/env python
# Copyright 2013-2022 by Carnegie Mellon University
# See license information in LICENSE-OPENSOURCE.txt

import os, os.path
import sys

PYFIXBUF_NAME = "pyfixbuf"
PYFIXBUF_VERSION = "0.9.0"

FIXBUF_VERSION_MIN = "2.2"
FIXBUF_VERSION_MAX = "3.90.0"

pyversion = ((0x020600f0 <= sys.hexversion < 0x03000000)
             or (0x030300f0 <= sys.hexversion < 0x04000000))

if (not pyversion):
    sys.stderr.write("Unsupported version of Python. 2.6, 2.7, or 3.3+ required.\n")
    sys.exit(1)

# Add the 'buildtools' directory to PYTHONPATH for this invocation and
# any sub-processes
buildtools = os.path.join(os.getcwd(), "buildtools")
sys.path = [buildtools] + sys.path
if 'PYTHONPATH' in os.environ:
    os.environ['PYTHONPATH'] = buildtools + ':' + os.environ['PYTHONPATH']
else:
    os.environ['PYTHONPATH'] = buildtools

# Import fbufdist from the buildtools directory
import fbufdist as dist

# Warn if ipaddress is not available
try:
    import ipaddress
except ImportError:
    sys.stderr.write("setup.py: WARNING: Unable to load ipaddress module needed to use pyfixbuf.  Continuing...")


dist.set_name(PYFIXBUF_NAME)
dist.set_version(PYFIXBUF_VERSION)

dist.set_title("Python fixbuf")
dist.set_description("""
A set of Python bindings for the fixbuf IPFIX protocol library.
""")

dist.set_maintainer("NetSA Group <netsa-help@cert.org>")
dist.set_url("http://tools.netsa.cert.org/pyfixbuf/index.html")
dist.set_license("LGPL")
dist.set_copyright("2011-2022 Carnegie Mellon University")

# Compares two version strings of the form "1.2.3" and returns <0, ==0, >0 if
# `a_str` is less than, equal to, or greater than `b_str`.  Missing values are
# trated at 0, allowing "1.2" to equal "1.2.0".  Attempts to compare component
# values as `int`, but compares as strings if that fails.  One failing is that
# "1.2.0-alpha" compares as greater than "1.2.0".
#
def version_cmp(a_str, b_str):
    aa = a_str.split('.')
    bb = b_str.split('.')
    while aa or bb:
        if len(aa):
            a = aa.pop(0)
        else:
            a = '0'
        if len(bb):
            b = bb.pop(0)
        else:
            b = '0'
        if a != b:
            try:
                return int(a) - int(b)
            except:
                if a < b:
                    return -1
                return 1
    return 0

# Helper function to use pkg-config to get extra switches, etc.
def pkgconfig(*pkgs):
    from fbufdist.shell import PipelineException, run_collect, command
    import shlex
    try:
        (output, error) = run_collect(command("pkg-config", "--cflags", *pkgs))
        if not isinstance(output, str):
            output = output.decode('ascii')
        compile_args = shlex.split(output)
        (output, error) = run_collect(command("pkg-config", "--libs", *pkgs))
        if not isinstance(output, str):
            output = output.decode('ascii')
        link_args = shlex.split(output)
        (output, error) = run_collect(command("pkg-config", "--modversion", *pkgs))
        if not isinstance(output, str):
            output = output.decode('ascii')
        version_args = shlex.split(output)
        # version_list is something like ('2', '.', '3', '.', '4')
        if ((version_cmp(version_args[0], FIXBUF_VERSION_MIN) < 0) or
            (version_cmp(version_args[0], FIXBUF_VERSION_MAX) >= 0)):
            import sys
            sys.stderr.write("FATAL: Found libfixbuf version: %s\n" %
                             version_args[0])
            sys.stderr.write("FATAL: pyfixbuf requires: %s <= libfixbuf < %s\n"
                             % (FIXBUF_VERSION_MIN, FIXBUF_VERSION_MAX))
            sys.exit(-1)
        return dict(extra_compile_args=compile_args,
                    extra_link_args=link_args)
    except PipelineException as ex:
        import sys
        sys.stderr.write("setup.py: unable to find required libraries:\n")
        sys.stderr.write("\n".join(ex.get_message().split("\n")[2:]))
        sys.stderr.write("\n")
        sys.exit(-1)

dist.add_package("pyfixbuf")
dist.add_package("pyfixbuf.cert")
dist.add_package("pyfixbuf.test")

dist.add_module_py("pyfixbuf.yaflists")
dist.add_version_file("src/pyfixbuf/version.py", "pyfixbuf_version = \"%s\"\n")

dist.add_module_ext("pyfixbuf/_pyfixbuf", ["pyfixbuf/_pyfixbuf.c"],
                    **pkgconfig('libfixbuf'))

dist.add_package_data("pyfixbuf.cert", "cert_ipfix.xml")
dist.add_package_data("pyfixbuf.test", "sampleipfix.ipfix")
dist.add_extra_files("LGPL.txt")
dist.add_extra_files("samples/README.md")
dist.add_extra_files("samples/*.py")
dist.add_extra_files("samples/expected/*.py.out")
dist.add_unit_test_module("pyfixbuf.test")

# Files to replace netsa.dist, needed during build
dist.add_extra_files("buildtools/fbufdist/*.py")
dist.add_extra_files("buildtools/fbufdist/*.py.in")
dist.add_extra_files("buildtools/fbufdist/tools_web/layout.html")
dist.add_extra_files("buildtools/fbufdist/tools_web/theme.conf")
dist.add_extra_files("buildtools/fbufdist/tools_web/static/tools.css_t")

dist.execute()
