#########################################################################
# Copyright 2013-2022 by Carnegie Mellon University
#
# @OPENSOURCE_HEADER_START@
# Use of the Network Situational Awareness Python support library and
# related source code is subject to the terms of the following licenses:
#
# GNU Lesser Public License (LGPL) Rights pursuant to Version 2.1, February 1999
# Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
#
# NO WARRANTY
#
# ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
# PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
# PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
# "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
# KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
# LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
# MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
# OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
# SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
# TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
# WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
# LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
# CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
# CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
# DELIVERABLES UNDER THIS LICENSE.
#
# Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
# Mellon University, its trustees, officers, employees, and agents from
# all claims or demands made against them (and any related losses,
# expenses, or attorney's fees) arising out of, or relating to Licensee's
# and/or its sub licensees' negligent use or willful misuse of or
# negligent conduct or willful misconduct regarding the Software,
# facilities, or other rights or assistance granted by Carnegie Mellon
# University under this License, including, but not limited to, any
# claims of product liability, personal injury, death, damage to
# property, or violation of any laws or regulations.
#
# Carnegie Mellon University Software Engineering Institute authored
# documents are sponsored by the U.S. Department of Defense under
# Contract FA8721-05-C-0003. Carnegie Mellon University retains
# copyrights in all material produced under this contract. The U.S.
# Government retains a non-exclusive, royalty-free license to publish or
# reproduce these documents, or allow others to do so, for U.S.
# Government purposes only pursuant to the copyright license under the
# contract clause at 252.227.7013.
#
# @OPENSOURCE_HEADER_END@
#
#######################################################################

from __future__ import absolute_import
import warnings
from . import _pyfixbuf
from . import version
import binascii
import ipaddress
import itertools
import re
import struct
import sys
import collections

__all__ = ['BL', 'Buffer', 'Collector', 'DataType',
           'Exporter', 'InfoElement', 'InfoElementSpec', 'InfoModel',
           'Listener', 'Record', 'STL', 'STML', 'Semantic',
           'Session', 'Template', 'Units',
           'CERT_PEN', 'VARLEN',
           'BASICLIST', 'SUBTEMPLATELIST', 'SUBTEMPLATEMULTILIST',
           '__version__']

CERT_PEN = 6871
VARLEN = 65535
__version__ = version.pyfixbuf_version

if sys.hexversion >= 0x03000000:
    long = int
    zip_longest = itertools.zip_longest
else:
    # Allow Python3 syntax to invoke Python2 methods
    range = xrange
    zip = itertools.izip
    zip_longest = itertools.izip_longest


class Enum(object):
    def __new__(cls):
        raise Exception("Class %s may not be instantiated" % cls.__name__)

    @classmethod
    def get_name(cls, value):
        """
        Returns the string name associated with *value* in this enumeration.

        Raises :exc:`KeyError` if *value* is not known to this enumeration.
        Raises :exc:`TypeError` if *value* is not an instance of an
        :class:`int`.
        """
        if isinstance(value, int):
            return cls._names[value]
        raise TypeError("Integer argument required")

    @classmethod
    def to_string(cls, value):
        """
        Returns a string that includes the class name and the name associated
        with *value* in this enumeration; e.g., `DataType(UINT8)` or
        `Units(PACKETS)`.

        Returns a string even when *value* is not known to this enumeration.
        """
        try:
            return "%s(%s)" % (cls.__name__, cls._names[value])
        except:
            return "%s(%s)" % (cls.__name__, str(value))

    @classmethod
    def by_name(cls, name):
        """
        Returns the value associated with the name *name* in this enumeration.

        Raises :exc:`KeyError` if this enumeration does not have a value
        asociated with *name*.  Raises :exc:`TypeError` if *name* is not a
        :class:`str`.
        """
        if name in cls._names:
            return cls.__dict__[name]
        if isinstance(name, str):
            raise  KeyError("Invalid %s '%s'" % (cls.__name__, str(name)))
        raise TypeError("String argument required")

    @classmethod
    def _init(cls):
        for i,name in enumerate(cls._names):
            setattr(cls, name, i)


def enum(name, seq, **named):
    named['_names'] = seq
    enums = dict(list(zip(seq, list(range(len(seq))))), **named)
    return type(name, (Enum,), enums)

class DataType(Enum):
    """
    An enumeration containing the DataTypes supported by pyfixbuf.

    In the following table, the RLE column indicates whether the type supports
    reduced length encoding.

    .. list-table::
      :header-rows: 1
      :widths: 20, 8, 8, 4, 20

      * - Type
        - Integer Value
        - Length
        - RLE
        - Python Return Type
      * - DataType.OCTET_ARRAY
        - 0
        - VARLEN
        - Yes
        - bytearray
      * - DataType.UINT8
        - 1
        - 1
        - No
        - int
      * - DataType.UINT16
        - 2
        - 2
        - Yes
        - long
      * - DataType.UINT32
        - 3
        - 4
        - Yes
        - long
      * - DataType.UINT64
        - 4
        - 8
        - Yes
        - long
      * - DataType.INT8
        - 5
        - 1
        - No
        - long
      * - DataType.INT16
        - 6
        - 2
        - Yes
        - long
      * - DataType.INT32
        - 7
        - 4
        - Yes
        - long
      * - DataType.INT64
        - 8
        - 8
        - Yes
        - long
      * - DataType.FLOAT32
        - 9
        - 4
        - No
        - float
      * - DataType.FLOAT64
        - 10
        - 8
        - Yes
        - float
      * - DataType.BOOL
        - 11
        - 1
        - No
        - bool
      * - DataType.MAC_ADDR
        - 12
        - 6
        - No
        - string
      * - DataType.STRING
        - 13
        - VARLEN
        - No
        - string
      * - DataType.SECONDS
        - 14
        - 4
        - No
        - long
      * - DataType.MILLISECONDS
        - 15
        - 8
        - No
        - long
      * - DataType.MICROSECONDS
        - 16
        - 8
        - No
        - long
      * - DataType.NANOSECONDS
        - 17
        - 8
        - No
        - long
      * - DataType.IP4ADDR
        - 18
        - 4
        - No
        - ipaddress.IPv4Address
      * - DataType.IP6ADDR
        - 19
        - 16
        - No
        - ipaddress.IPv6Address
      * - DataType.BASIC_LIST
        - 20
        - VARLEN
        - No
        - pyfixbuf.BL
      * - DataType.SUB_TMPL_LIST
        - 21
        - VARLEN
        - No
        - pyfixbuf.STL
      * - DataType.SUB_TMPL_MULTI_LIST
        - 22
        - VARLEN
        - No
        - pyfixbuf.STML
    """
    _names = ("OCTET_ARRAY", "UINT8", "UINT16", "UINT32", "UINT64", "INT8",
              "INT16", "INT32", "INT64", "FLOAT32", "FLOAT64", "BOOL",
              "MAC_ADDR", "STRING", "SECONDS", "MILLISECONDS",
              "MICROSECONDS", "NANOSECONDS", "IP4ADDR", "IP6ADDR",
              "BASIC_LIST", "SUB_TMPL_LIST", "SUB_TMPL_MULTI_LIST")

    @staticmethod
    def check_type(dt, value):
        """
        Returns ``True`` if a field whose :class:`InfoElement`'s DataType is
        `dt` may be set to `value` given it the Pythonic type of `value`.
        """
        if type(value) is int:
            if dt in [DataType.BOOL, DataType.UINT8, DataType.UINT16,
                      DataType.UINT32, DataType.UINT64,
                      DataType.INT8, DataType.INT16,
                      DataType.INT32, DataType.INT64,
                      DataType.SECONDS, DataType.MILLISECONDS,
                      DataType.MICROSECONDS, DataType.NANOSECONDS]:
                return True
        if type(value) is long:
            if dt in [DataType.UINT32, DataType.UINT64,
                      DataType.INT32, DataType.INT64,
                      DataType.SECONDS, DataType.MICROSECONDS,
                      DataType.NANOSECONDS, DataType.MILLISECONDS]:
                return True
        if type(value) is str:
            if (dt in [DataType.OCTET_ARRAY, DataType.STRING] or (dt==VARLEN)):
                return True
        if type(value) is bytearray:
            if dt in [DataType.OCTET_ARRAY, DataType.MAC_ADDR]:
                return True
        if type(value) is float and dt in [DataType.FLOAT32, DataType.FLOAT64]:
            return True
        if type(value) is bool and dt in [DataType.BOOL]:
            return True
        if type(value) is ipaddress.IPv4Address and dt in [DataType.IP4ADDR]:
            return True
        if type(value) is ipaddress.IPv6Address and dt in [DataType.IP46DDR]:
            return True
        return False

    @staticmethod
    def get_length(dt):
        """Returns the standard length of a DataType"""
        if dt in [DataType.OCTET_ARRAY, DataType.STRING, DataType.BASIC_LIST,
                  DataType.SUB_TMPL_LIST, DataType.SUB_TMPL_MULTI_LIST]:
            return VARLEN
        elif dt in [DataType.UINT8, DataType.INT8, DataType.BOOL]:
            return 1
        elif dt in [DataType.UINT16, DataType.INT16]:
            return 2
        elif dt in [DataType.UINT32, DataType.INT32, DataType.SECONDS,
                    DataType.FLOAT32, DataType.IP4ADDR]:
            return 4
        elif (dt == DataType.MAC_ADDR):
            return 6
        elif dt in [DataType.UINT64, DataType.INT64, DataType.FLOAT64,
                    DataType.MILLISECONDS, DataType.MICROSECONDS,
                    DataType.NANOSECONDS]:
            return 8
        elif (dt == DataType.IP6ADDR):
            return 16
        else:
            return VARLEN

    @staticmethod
    def refine_type_for_length(dt, len):
        """Chooses a DataType given a DataType and a length"""
        # FIXME: Move this function onto DataType class
        signed = False
        float = False
        if dt in [DataType.INT8,DataType.INT16,DataType.INT32,DataType.INT64]:
            signed = True
        elif dt in [DataType.FLOAT32, DataType.FLOAT64]:
            float = True
        if (len == 1):
            if signed:
                return DataType.INT8
            else:
                return DataType.UINT8
        elif (len == 2):
            if signed:
                return DataType.INT16
            else:
                return DataType.UINT16
        elif (len <= 4):
            if signed:
                return DataType.INT32
            elif float:
                return DataType.FLOAT32
            else:
                return DataType.UINT32
        elif (len <= 8):
            if signed:
                return DataType.INT64
            elif float:
                return DataType.FLOAT64
            else:
                return DataType.UINT64

    @staticmethod
    def supports_RLE(dt):
        """Returns ``True`` if the DataType supports reduced length encoding"""
        return dt not in [DataType.BOOL, DataType.UINT8, DataType.INT8,
                          DataType.IP4ADDR, DataType.IP6ADDR,
                          DataType.SECONDS, DataType.MILLISECONDS,
                          DataType.MICROSECONDS, DataType.NANOSECONDS,
                          DataType.MAC_ADDR, DataType.FLOAT32,
                          DataType.BASIC_LIST, DataType.SUB_TMPL_LIST,
                          DataType.SUB_TMPL_MULTI_LIST]
DataType._init()


class Units(Enum):
    """
    An enumeration containing the Units supported by pyfixbuf.

    ============================   =============
    Units                          Integer Value
    ============================   =============
    Units.NONE                     0
    Units.BITS                     1
    Units.OCTETS                   2
    Units.PACKETS                  3
    Units.FLOWS                    4
    Units.SECONDS                  5
    Units.MILLISECONDS             6
    Units.MICROSECONDS             7
    Units.NANOSECONDS              8
    Units.WORDS                    9
    Units.MESSAGES                 10
    Units.HOPS                     11
    Units.ENTRIES                  12
    Units.FRAMES                   13
    Units.PORTS                    14
    UNITS.INFERRED                 15
    ============================   =============
    """
    _names = ("NONE", "BITS", "OCTETS", "PACKETS", "FLOWS", "SECONDS",
              "MILLISECONDS", "MICROSECONDS", "NANOSECONDS", "WORDS",
              "MESSAGES", "HOPS", "ENTRIES", "FRAMES", "PORTS", "INFERRED")
Units._init()


class Semantic(Enum):
    """
    An enumeration containg the available Semantic values for Information
    Elements (:class:`InfoElement`).  This is not for the semantics of
    structured data items.

    ============================   =============
    Semantic                       Integer Value
    ============================   =============
    Semantic.DEFAULT               0
    Semantic.QUANTITY              1
    Semantic.TOTALCOUNTER          2
    Semantic.DELTACOUNTER          3
    Semantic.IDENTIFIER            4
    Semantic.FLAGS                 5
    Semantic.LIST                  6
    Semantic.SNMPCOUNTER           7
    Semantic.SNMPGAUGE             8
    ============================   =============
    """
    _names = ("DEFAULT", "QUANTITY", "TOTALCOUNTER", "DELTACOUNTER",
              "IDENTIFIER", "FLAGS", "LIST", "SNMPCOUNTER", "SNMPGAUGE")
Semantic._init()


BASICLIST = DataType.BASIC_LIST
SUBTEMPLATELIST = DataType.SUB_TMPL_LIST
SUBTEMPLATEMULTILIST = DataType.SUB_TMPL_MULTI_LIST


def assert_type(typ, arg):
    if not isinstance(arg, typ):
        raise TypeError("Expected %s but got %s instead" %
                        (typ.__name__, type(arg).__name__))

def yield_record_subrecords(record, tmpl_id):
    assert_type(Record, record)
    if tmpl_id == 0:
        yield record
    elif tmpl_id == record.template.template_id:
        yield record
    for infoelem in record:
        if (type(infoelem) == STL) or (type(infoelem) == STML):
            for sub_record in infoelem.iter_records(tmpl_id):
                yield sub_record


# class InfoElement(object):
#    defined entirely in C
InfoElement = _pyfixbuf.InfoElement

# class InfoElementSpec(object):
#    defined entirely in C
InfoElementSpec = _pyfixbuf.InfoElementSpec


class _SessionCallbacks(object):
    def __init__(self, session):
        self.session = session
        self.callbacks = []

    def add(self, callback):
        if not callable(callback):
            raise ValueError("Callbacks must be callable")
        self.callbacks.append(callback)

    def __call__(self, template):
        ctx = None
        for cb in self.callbacks:
            nctx = cb(self.session, Template._wrap(template), ctx)
            if nctx is not None:
                ctx = nctx
        return ctx

class _IgnoreList(object):
    def __init__(self, values):
        self.list = values

    def __call__(self, session, template, ctx):
        if template.template_id in self.list:
            session.add_template_pair(template.template_id, 0)


class Session(_pyfixbuf.fbSessionBase):
    """
    Creates an empty :class:`!Session` given an :class:`InfoModel`.
    """
    def __init__(self, model):
        assert_type(InfoModel, model)
        self.model = model
        _pyfixbuf.fbSessionBase.__init__(self, model._unwrap())
        self.internal_templates = dict()
        self.external_templates = dict()
        self._callbacks = _SessionCallbacks(self)
        _pyfixbuf.fbSessionBase.setTemplateCallback(self, self._callbacks)
        self._ignorelist = None

    def _add_template(self, template, template_id, internal):
        """
        Adds the given Template to the Session.  Adds only as internal
        template when `internal` is True, only as external when `internal` is
        False, and as both when `internal` is None.
        """
        assert_type(Template, template)
        if template_id != 0:
            assert_type(int, template_id)
            if (template_id <= 0xff or template_id > 0xffff):
                raise Exception("Invalid ID: " + str(template_id) +
                                " Template ID must be between 256 and 65535")
        template._finalize()
        if internal is not True:
            # `internal` is either False or None; add external
            template_id = _pyfixbuf.fbSessionBase.addTemplate(
                self, template._unwrap(), template_id, False)
            self.external_templates[template_id] = template
        if internal is not False:
            # `internal` is either True or None; add internal
            template_id = _pyfixbuf.fbSessionBase.addTemplate(
                self, template._unwrap(), template_id, True)
            self.internal_templates[template_id] = template
        return template_id

    def add_template(self, template, template_id=0):
        """
        Adds the given :class:`Template` *template* to the
        :class:`!Session` with the optional *template_id*.  This template
        may be used as both an internal and an external template.  Use
        :meth:`add_internal_template` or :meth:`add_external_template` to be
        more selective on template usage.

        If a *template_id* is not given or 0, libfixbuf automatically chooses
        an unused template ID. *template_id* is used for representing both the
        internal and external template.  The valid range for *template_id* is
        256 to 65535 (0x100 to 0xffff).

        When *template_id* is specified and it is already in use by the
        :class:`!Session`, *template* replaces those :class:`!Template`\ (s).

        Returns the template ID of the added template.
        """
        return self._add_template(template, template_id, None)

    def add_internal_template(self, template, template_id=0):
        """
        Adds the given *template* as an internal template to the session with
        the optionally given *template_id*. An internal template determines
        how the data is presented when transcoded.  See also
        :meth:`add_template`.

        If *template_id* is not set or 0, libfixbuf will automatically
        choose an unused value.

        Returns the template ID of the added :class:`!Template`.
        """
        return self._add_template(template, template_id, True)

    def add_external_template(self, template, template_id=0):
        """
        Adds the given *template* as an external template to the session with
        the optionally given *template_id*.  An external template is used when
        exporting IPFIX data.  (libfixbuf automatically creates external
        templates when collecting IPFIX.)  See also :meth:`add_template`.

        If *template_id* is not set or 0, libfixbuf will automatically
        choose an unused value.

        Returns the template ID of the added :class:`!Template`.
        """
        return self._add_template(template, template_id, False)

    def get_template(self, template_id, internal=False):
        """
        Returns the :class:`Template` with the given *template_id*.  By
        default, the external template is returned.  Set *internal* to
        ``True`` to retrieve the internal template with the given
        *template_id*.  The returned :class:`!Template` may not be modified.

        Raises an Exception if a :class:`!Template` with the given
        *template_id* does not exist on the :class:`!Session`.
        """
        template = _pyfixbuf.fbSessionBase.getTemplate(self, template_id,
                                                       internal)
        if (template == None):
            if internal:
                raise Exception("Internal template with ID " + str(template_id)
                                + " does not belong to this Session.")
            else:
                raise Exception("External template with ID " + str(template_id)
                                + " has not been added to this Session.")
        new_template = Template._wrap(template)
        return new_template

    def decode_only(self, id_list):
        """
        When decoding, causes this :class:`!Session` to ignore a record in a
        subTemplateList (:class:`STL`) or subTemplateMultiList (:class:`STML`)
        **unless** the record has a template ID in the given *id_list* of
        ``int``\ s.  The method is only used when the :class:`!Session` is
        bound to a :class:`Collector` or :class:`Listener`.

        This method has no effect on records that are not in a subTemplateList
        or subTemplateMultiList.  See also :meth:`ignore_templates`.
        """
        for item in id_list:
            if not isinstance(item, int):
                raise ValueError("Invalid Template ID: " + str(item) +
                                 ". Template ID in List must be an Integer");
            _pyfixbuf.fbSessionBase.addTemplatePair(self, item, item)

    def ignore_templates(self, id_list):
        """
        When decoding, causes this :class:`!Session` to ignore a record in a
        subTemplateList (:class:`STL`) or subTemplateMultiList (:class:`STML`)
        **when** the record has a template ID in the given *id_list* of
        ``int``\ s.  The method is only used when the :class:`!Session` is
        bound to a :class:`Collector` or :class:`Listener`.

        This method has no effect on records that are not in a subTemplateList
        or subTemplateMultiList.  See also :meth:`decode_only`.
        """
        for item in id_list:
            if not isinstance(item, int):
                raise ValueError("Invalid Template ID: " + str(item) +
                                 ". Template ID in List must be an Integer");
        if self._ignorelist is None:
            self._ignorelist = _IgnoreList(id_list)
        else:
            self._ignorelist.list = id_list
        self.add_template_callback(self._ignorelist)

    def add_template_pair(self, external_template_id, internal_template_id):
        """
        Specifies how to transcode data within structured data (a list).

        That is, tells a :class:`Collector` or :class:`Listener` that data in
        a :class:`STML` or :class:`STL` whose :class:`Template` is
        *external_template_id* is to be transcoded into the :class:`!Template`
        given by *internal_template_id*.

        By default, libfixbuf transcodes each entry in the :class:`STML` or
        :class:`STL` with the external template it received, requiring the
        collector to free or clear any memory allocated for the list
        elements. The collector can change this behavior by adding a "template
        pair."  For each entry in the STL or STML, if the entry has the given
        *external_template_id*, it will use the given *internal_template_id*
        to transcode the record.  The *internal_template_id* must reference an
        internal template that was previously added to the :class:`!Session`.
        If *internal_template_id* is 0, the entry will not be transcoded and
        will be ignored by libfixbuf.

        Once a template pair has been added - the default is to ONLY decode
        entries that have an external_template_id in this template pair table.
        Therefore, any entries in a STML or STL that reference a template id
        not in this table will be dropped.
        """
        if (internal_template_id in self.internal_templates or
            internal_template_id == 0):
            _pyfixbuf.fbSessionBase.addTemplatePair(self, external_template_id,
                                                    internal_template_id)
        else:
            raise Exception("An internal template with id: " +
                            str(internal_template_id) +
                            " has not been added to this session")

    def add_template_callback(self, callback):
        """
        Adds the callable *callback* to be called whenever a new external
        template is read from a :class:`Collector` or :class:`Listener`.

        Multiple callbacks may be added to a :class:`!Session`.  The callbacks
        may specify a **context** object that is stored with the
        :class:`Template`.

        After setting the callback(s), when a new template is read the
        callbacks will be called in the order they are added, and each may
        modify the context object.  After all callbacks are called, the
        :class:`!Template`'s context object will be set to the result.  A
        :class:`!Template`'s context may be retrieved via
        :meth:`Template.get_context`.

        The *callback* must be a callable and it will be called with three
        arguments: *session*, *template*, and *context*.  *session* is the
        :class:`!Session` from which the callback was added.  *template* is
        the new :class:`!Template` and will have its `template_id` property
        set to the external template ID.

        The *context* argument is the current object that is to be stored as
        the Template's context.  The callback should either return a new
        object to use as the context or ``None``.  A return value of ``None``
        indicates that the context object should not change.

        To avoid undefined behavior, the callback functions should be added
        before the :class:`!Session` is bound to a :class:`Buffer`.
        """
        self._callbacks.add(callback)


class InfoModel(object):
    """
    An IPFIX Information Model stores all of the Information Elements
    (:class:`InfoElement`) that can be collected or exported by a
    Collecting or Exporting Process.

    The constructor creates a new Information Model and adds the default
    IANA-managed Information Elements.
    """
    def __init__(self):
        # Instead of having the Python InfoModel and Template classes inherit
        # from the CPython Base classes, these classes include their Base
        # classes in the '_parent' member.  See additional information in the
        # Template class.
        self._parent = _pyfixbuf.fbInfoModelBase()

    @classmethod
    def _wrap(cls, base):
        """Wraps a InfoModelBase object as an InfoModel."""
        obj = cls.__new__(cls)
        obj._parent = base
        return obj

    def _unwrap(self):
        """Returns the InfoModelBase object."""
        return self._parent

    def add_element(self, element):
        """Adds the given :class:`InfoElement` to the
        :class:`InfoModel`.
        """
        self._unwrap().addElement(element)

    def add_element_list(self, elements):
        """Adds each of the :class:`InfoElement` objects in *elements* to the
        :class:`InfoModel`."""
        if isinstance(elements, str):
            raise TypeError("Expected a non-str iterable")
        for item in elements:
            if item != None:
                self.add_element(item)

    def get_element_length(self, *args, **kwds):
        """
        Returns the in-memory size of an :class:`InfoElement`.

        Specifically, searches the :class:`!InfoModel` for an
        :class:`!InfoElement` named *name* and if found, returns the length of
        a value of that :class:`!InfoElement` when the value is part of an
        in-memory :class:`!Record`.  If the :class:`!InfoElement` is of
        variable length or a list, returns the size of the data structure that
        holds the value within libfixbuf.

        However, if *type* is given and is one of ``BASICLIST``,
        ``SUBTEMPLATELIST``, or ``SUBTEMPLATEMULTILIST``, ignores *name* and
        returns the length of the list data structure within libfixbuf.  Any
        other integer value for *type* is ignored.

        Raises :exc:`TypeError` if *name* is not a :class:`str` or if *type*
        is given and is not an :class:`!int`.  Raises :exc:`KeyError` if
        *name* is not ignored and not the name of a known
        :class:`!InfoElement`.
        """
        return self._unwrap().getElementLength(*args, **kwds)

    def get_element_type(self, name):
        """
        Returns the type of the Information Element as defined in the
        :class:`InfoModel` given the :class:`InfoElement` *name*.
        """
        return self._unwrap().getElementTrueType(name)

    def get_element(self, *args, **kwds):
        """
        Returns the :class:`InfoElement` given the *name* or *id* and *ent*.
        Raises :exc:`KeyError` when the element is not found.

        Example:

        >>> ie = model.get_element('protocolIdentifier')
        >>> ie.name
        'protocolIdentifier'
        >>> ie = model.get_element(id = 4)
        >>> ie.name
        'protocolIdentifier'
        >>> ie = m.get_element(id = 4, ent = CERT_PEN)
        Traceback ...
        KeyError: 'No element 6871/4'
        """
        return self._unwrap().getElement(*args, **kwds)

    def add_options_element(self, rec):
        """
        Adds the information element contained in the Options :class:`Record`.
        Use this method for incoming Options Records that contain Information
        Element Type Information.
        """
        if (not isinstance(rec, dict)):
            rec.as_dict()
        dt = rec["informationElementDataType"]
        len = DataType.get_length(dt)
        return self.add_element(
            InfoElement(rec["informationElementName"],
                        rec["privateEnterpriseNumber"],
                        rec["informationElementId"], len, type=dt,
                        min=rec["informationElementRangeBegin"],
                        max=rec["informationElementRangeEnd"],
                        units=rec["informationElementUnits"],
                        semantic=rec["informationElementSemantics"],
                        description=rec["informationElementDescription"]))

    def read_from_xml_file(self, filename):
        """
        Adds to this :class:`InfoModel` the Information Elements found in the
        XML file in `filename`.  See also :meth:`read_from_xml_data`.
        """
        file = open(filename, "r")
        data = file.read()
        file.close()
        return self._unwrap().readXMLData(data)

    def read_from_xml_data(self, xml_data):
        """
        Adds to this :class:`InfoModel` the Information Elements found in the
        XML data stored in `xml_data`, which is an object that supports the
        buffer call interface.  See also :meth:`read_from_xml_file`.
        """
        return self._unwrap().readXMLData(xml_data)


class Exporter(_pyfixbuf.fbExporterBase):
    """
    Creates an empty :class:`Exporter`.  Initialize the exporter using
    :meth:`init_file` or :meth:`init_net`.
    """
    def __init__(self):
        self.initialized = 0
        _pyfixbuf.fbExporterBase.__init__(self)

    def init_file(self, filename):
        """
        Initializes the :class:`Exporter` to write to the given *filename*.
        """
        self.initialized = 1
        _pyfixbuf.fbExporterBase.allocFile(self, filename)

    def init_net(self, hostname, transport="tcp", port=4739):
        """
        Initializes the :class:`Exporter` to write to the given *hostname*,
        *port* over the given *transport*.

        Given *hostname* may be a hostname or IP address.

        Acceptable values for *transport* are ``"tcp"`` and ``"udp"``. Default
        is ``"tcp"``.

        Given *port* must be greater than 1024. Default is 4739.
        """
        if not isinstance(port, int):
            port = int(port)
        if (port < 1024 or port == 0):
            raise Exception("Invalid Port. Port must be greater than 1024.")
        self.initialized = 1
        _pyfixbuf.fbExporterBase.allocNet(self, host=hostname,
                                          transport=transport, port=str(port))


class Collector(_pyfixbuf.fbCollectorBase):
    """
    Creates an uninitialized :class:`Collector`.  An IPFIX Collector manages
    the file it is reading from.
    Initialize the collector using :meth:`init_file`.
    """
    def __init__(self):
        self.initialized = 0
        _pyfixbuf.fbCollectorBase.__init__(self)

    def init_file(self, filename):
        """
        Initialize the :class:`Collector` to read from the given *filename*.
        *filename* should be the path to a valid IPFIX File or the string
        ``"-"`` to read from the standard input.
        """
        self.initialized = 1
        return _pyfixbuf.fbCollectorBase.allocCollectorFile(self, filename)


class Template(object):
    """
    Creates a new Template using the given *model*, an :class:`InfoModel`.
    An IPFIX Template
    is an ordered list of the Information Elements that are to be
    collected or exported.  For export, the order of Information Elements
    in the Templates determines how the data will be exported.

    If *type* is given, an Information Element Type Information Options
    Template will be created.  The appropriate elements will automatically
    be added to the template and the scope will be sent.  See :rfc:`5610`
    for more information.

    Once a Template has been added to a :class:`Session`, it cannot be
    altered.

    A :class:`Template` can be accessed like a dictionary or a list to
    retrieve a specific :class:`InfoElementSpec`.
    """
    def __init__(self, model, type=False):
        # Instead of having the Python InfoModel and Template classes inherit
        # from the CPython Base classes, these classes include their Base
        # classes in the '_parent' member.  This solves problems involving the
        # C layer having to return Python objects.  'Template._wrap(base)'
        # transforms a TemplateBase object 'base' to a Template object.
        # t._unwrap() allows the wrapper 't' to get access to the TemplateBase
        # object.
        assert_type(InfoModel, model)
        self._parent = _pyfixbuf.fbTemplateBase(model._unwrap(), type)
        self.infomodel = model
        # an array of InfoElementSpecs describing this template
        self._specs = []
        # an array of canonical InfoElements describing this template
        self._elements = []
        self._scope = self._unwrap().scope or None
        if type:
            self._build_element_list()

    @classmethod
    def _wrap(cls, base):
        """Wraps a TemplateBase object as a Template."""
        obj = cls.__new__(cls)
        obj._parent = base
        obj.infomodel = InfoModel._wrap(base.infomodel)
        obj._elements = []
        obj._specs = []
        obj._build_element_list()
        obj._scope = base.scope or None
        return obj

    def _unwrap(self):
        """Returns the TemplateBase object."""
        return self._parent

    def _finalize(self):
        """Called before adding to a session."""
        if (not self.read_only and self._unwrap().scope == 0 and
            self.scope is not None):
            self._unwrap().scope = self._scope

    @property
    def read_only(self):
        """
        Returns ``True`` if this :class:`Template` has been added to a
        :class:`Session`.
        """
        return self._unwrap().read_only

    @property
    def template_id(self):
        """Returns the template ID for this :class:`Template`."""
        return self._unwrap().template_id

    @property
    def type(self):
        """
        Returns ``True`` if this :class:`Template` is an Information Element
        Type Information Options Template.
        """
        return self._unwrap().type

    @property
    def scope(self):
        """
        Returns the number of scoped elements in this :class:`Template`.  A
        scope of ``None`` means that this :class:`Template` is not an Options
        template.
        """
        return self._scope

    @scope.setter
    def scope(self, value):
        """
        Sets the scope for this :class:`Template`.  ``None`` means no scope,
        and an argument of zero means that all elements currently in the
        :class:`Template` will be made part of the scope.
        """
        if self.read_only:
            raise AttributeError(
                "Scope cannot be changed; Template is owned by a Session")
        if value is not None and (value < 0 or value > len(self)):
            raise AttributeError(
                "Too few or too many scoped elements")
        if value == 0:
            value = len(self)
        self._scope = value

    def copy(self):
        """Returns a copy of this :class:`Template`."""
        copy = Template(self.infomodel)
        copy.add_spec_list(iter(self))
        copy._scope = self.scope
        return copy

    def _add_spec(self, spec):
        """Adds the given :class:`InfoElementSpec` *spec* to the Template."""
        # A helper method for add_spec(), add_spec_list(), add_element()
        assert_type(InfoElementSpec, spec)
        ie = self.infomodel.get_element(spec.name)
        if spec.length and spec.length != ie.length and ie.length != VARLEN:
            if (spec.length > self.infomodel.get_element_length(spec.name)):
                raise Exception("InfoElementSpec " + spec.name + " uses "
                                "invalid reduced-length.  Must be smaller than "
                                "the default InfoElement length.")
            if (not(DataType.supports_RLE(ie.type))):
                raise Exception("Data Type of InfoElement " + spec.name +
                                " does not support reduced-length encoding.")
        self._unwrap().addSpec(spec)
        self._specs.append(spec)
        self._elements.append(ie)

    def add_spec(self, spec):
        """Appends a given :class:`InfoElementSpec` *spec* to the Template.

        Once the Template has been added to a :class:`Session`, it
        cannot be altered.
        """
        if (self.read_only):
            raise Exception("Invalid add: Template has already been added "
                            "to the session.")
        self._add_spec(spec)

    def add_spec_list(self, specs):
        """Appends each of the :class:`InfoElementSpec` items in *specs* to
        the :class:`Template`.

        Once the :class:`Template` has been added to the :class:`Session`,
        it can not be altered.
        """
        if (self.read_only):
            raise Exception("Invalid add: Template has already been added"
                            " to the session.")
        if isinstance(specs, str):
            raise TypeError("Expected a non-str iterable")
        for spec in specs:
            self._add_spec(spec)

    def add_element(self, name):
        """Appends an Information Element with the given *name* to this
        :class:`Template`.  This function may be used as an alternative to
        :meth:`add_spec`.

        This function creates an :class:`InfoElementSpec` with the given
        element *name* and default length and adds it to the Template.

        Once the Template has been added to the :class:`Session`, it
        can not be altered.
        """
        if (self.read_only):
            raise Exception("Invalid add: Template has already been added "
                            "to the session.")
        self._add_spec(InfoElementSpec(name))

    def get_indexed_ie(self, index):
        """
        Returns the :class:`InfoElement` at the given positional *index* in
        the :class:`Template`.  Unlike the :meth:`__getitem__` method which
        returns the :class:`InfoElementSpec`, this method
        returns the :class:`InfoElement` at a particular index.
        """
        return self._elements[index]

    def __contains__(self, element):
        """
        Implements the ``in`` operator for :class:`Template` instances: Tests
        whether *element* is in the :class:`Template`.

        If *element* is an :class:`int`, return ``True`` if it is non-negative
        and less than the length of the :class:`Template`.

        If *element* is a :class:`str`, return ``True`` if an Information
        Element with the name *element* is included in the :class:`Template`.

        If *element* is an :class:`InfoElement` or :class:`InfoElementSpec`,
        return ``True`` if the element exists in the :class:`Template`,
        ``False`` otherwise.

        Examples:

        >>> t = Template(InfoModel())
        >>> t.add_element("protocolIdentifier")
        >>> "protocolIdentifier" in t
        True
        >>> InfoElementSpec("protocolIdentifier") in t
        True
        """
        if isinstance(element, int):
            return element >= 0 and element < len(self)
        return self._unwrap().containsElement(element)

    def __len__(self):
        """
        Implements the built-in `len()` method for :class:`Template`
        instances: Returns the number of :class:`InfoElementSpec` objects in
        the :class:`Template`.

        Examples:

        >>> t = Template(InfoModel())
        >>> t.add_element("protocolIdentifier")
        >>> len(t)
        1
        """
        return self._unwrap().elementCount()

    def __getitem__(self, key):
        """
        Implements the evaluation of ``template[key]`` for :class:`Template`
        instances: Returns the :class:`InfoElementSpec` represented by *key*.

        If *key* is an :class:`int`, it is treated as a positional index and
        an :exc:`IndexError` exception is raised when it is out of range.

        If *key* is a :class:`str`, it is treated as the name of the
        :class:`InfoElementSpec` to find and :exc:`KeyError` Exception is
        raised when *key* is not the name of an :class:`InfoElementSpec` on
        the :class:`Template`.  If an :class:`InfoElementSpec` is repeated in
        the :class:`Template`, querying by name always returns the first
        element.

        Any other type for *key* raises a :exc:`TypeError`.

        Examples:

        >>> t = Template(InfoModel())
        >>> t.add_element("protocolIdentifier")
        >>> t[0]
        <pyfixbuf.InfoElementSpec object at 0x107a0fc00>
        >>> t["protocolIdentifier"]
        <pyfixbuf.InfoElementSpec object at 0x107a0fc00>
        """
        if (isinstance(key, str)):
            for spec in self._specs:
                if spec.name == key:
                    return spec
            raise KeyError("No InfoElementSpec with name " + key)
        elif (isinstance(key, int)):
            if key < len(self._elements):
                return self._specs[key]
            raise IndexError("No element at index " + str(key))
        else:
            raise TypeError("Expected a str or an int")

    def __iter__(self):
        """
        Implements the method to return an Iterator over the
        :class:`InfoElementSpec` objects in this :class:`Template`.  See also
        :meth:`ie_iter`.
        """
        for spec in self._specs:
            yield spec

    def ie_iter(self):
        """
        Returns an Iterator over the :class:`InfoElement` objects in this
        :class:`Template`.  See also :meth:`__iter__`.

        Example:

        >>> for ie in template:
        ...     print(ie.name, DataType.get_name(ie.type))
        ...
        """
        for ie in self._elements:
            yield ie

    def get_context(self):
        """
        Returns the :class:`Template`'s context object as set by the callables
        registered with the :meth:`Session.add_template_callback` function.
        """
        return self._unwrap().getContext()

    def _build_element_list(self):
        """
        Updates the _elements list with the IEs from the base object.
        """
        for i in range(len(self)):
            self._elements.append(self._unwrap().getIndexedIE(i))
            self._specs.append(self._unwrap().getIndexedSpec(i))


class Record(_pyfixbuf.fbRecordBase):
    """
    Creates an empty Record given an :class:`InfoModel`, *model*, and
    optionally a *template* or a *record*.

    The :class:`!Record` is returned from a collection :class:`Buffer` or is
    added to an exporting :class:`Buffer`.

    When adding elements to a :class:`!Record`, the :class:`!Record` should
    match a :class:`Template`.  If the process is collecting, the
    :class:`!Record` should match the Internal Template.  For an Exporting
    process, the :class:`!Record` should match the External Template, and
    there should be one :class:`!Record` for each External Template.  A
    :class:`!Record` can not contain more Information Elements than it's
    associated *template*.  Information Elements should be added to the
    :class:`!Record` in the same order as the :class:`Template`.

    If a *template* is given to the constructor, all Information Elements that
    exist in the *template* are added to the :class:`!Record` in the same
    order as they exist in the Template.  The *record* argument is ignored.

    If a *record* is given, all Information Elements that exist in the
    *record* are added to the :class:`!Record` in the same order as they exist
    in the *record*.

    One element must exist in the :class:`!Record` before exporting any data.

    A :class:`!Record` maintains internal dictionaries for the elements that
    it contains.  For this reason, if a template contains more than 1 of the
    same Information Element, elements must be added using the
    :meth:`add_element` method in order to give alternate key names to
    elements that are the same.

    A :class:`!Record` may also be accessed similar to a list.
    """

    Field = collections.namedtuple('Field', 'name instance ie length value')

    _Elem = collections.namedtuple('_Elem', 'key ie wire_len offset mem_len')
    #
    # The _Elem class maintains internal information about each field
    #
    #  key:      a (name, instance) pair, where instance is 0 for the first
    #            occurance of 'name', 1 for the sencond, etc
    #  ie:       the canonical InfoElement
    #  wire_len: the length of this value in a string---VARLEN for lists and
    #            variable length strings, octetArrays
    #  offset:   the offset of this value in the byte array
    #  mem_len:  the length of this value in memory, sizeof(struct) for lists
    #            and variable length strings, octetArrays

    def __init__(self, model, template=None, record=None):
        assert_type(InfoModel, model)
        # References to other pyfixbuf classes
        self.model = model
        self.template = None
        self.session = None
        # Information about each field, a list of _Elem objects, in the order
        # in which they were appended
        self._fields = []
        # A mapping from (key -> _Elem), where each _Elem appears two or
        # maybe three times: once keyed by its numeric position, once keyed by
        # a (key_name,instance_count) tuple, and when "instance_count" is
        # zero, again by just the key_name.
        self._field_dict = dict()
        # Information about basicLists, keyed by _Elem
        self._basic_list = dict()
        # Once STML/STL elements are initialzed, no more elements may be added
        self.list_init = False
        # Current byte offset in C structure
        offset = 0
        if template:
            assert_type(Template, template)
            self.template = template
            for spec in template:
                ie = self.model.get_element(spec.name)
                if (spec.length and spec.length != VARLEN):
                    wire_len = spec.length
                    mem_len = wire_len
                else:
                    wire_len = ie.length
                    mem_len = model.get_element_length(spec.name)
                self._add_field(spec.name, ie, wire_len, offset, mem_len)
                offset += mem_len
                if ie.type == DataType.BASIC_LIST:
                    field = self._fields[-1]
                    self._basic_list[field] = None
        elif isinstance(record, Record):
            self.template = record.template
            for field in record._fields:
                # namedtuples are immutatable, so add it directly
                self._fields.append(field)
                offset += field.mem_len
            for key,field in record._field_dict.items():
                self._field_dict[key] = field
            for field,value in record._basic_list.items():
                self._basic_list[field] = value
        _pyfixbuf.fbRecordBase.__init__(self, offset)
        self.length = offset

    def _get_field_offset(self, key):
        """
        Returns the byte offset of the field *key* in the data array for this
        :class:`!Record`.
        """
        return self._field_dict[key].offset

    def _add_field(self, key_name, ie, wire_len, offset, mem_len):
        """
        Creates a _Elem() and adds it to this :class:`!Record`.
        """
        # location of the new field
        pos = len(self._fields)
        # make a unique key
        key = (key_name, 0)
        while key in self._field_dict:
            key = (key_name, key[1]+1)
        # create the field and add to record
        field = Record._Elem(key, ie, wire_len, offset, mem_len)
        self._fields.append(field)
        # key the field two or maybe three ways
        self._field_dict[pos] = field
        self._field_dict[key] = field
        if key[1] == 0:
            self._field_dict[key[0]] = field

    def add_element(self, key_name, type=0, element_name=None, length=0):
        """
        Appends a field named *key_name* to this :class:`!Record`, where
        *key_name* will be used to get or set the value of an Information
        Element.

        The :class:`InfoElement` may be specified in three ways: The
        *key_name* may match the name of an :class:`!InfoElement` in the
        :class:`InfoModel`, the *element_name* may name the
        :class:`!InfoElement`, or the *type* may be specified as
        ``BASICLIST``, ``SUBTEMPLATELIST``, ``SUBTEMPLATEMULTILIST``, or
        ``VARLEN`` for octetArray.

        When the :class:`!Record` contains duplicate Information Elements, the
        *key_name* should be unique to ease reference to the elements and the
        :class:`InfoElement` should be specified using the *element_name* or
        *type* parameters.

        The *length* parameter specifies the reduced-length encoding length
        for the Information Element, similar to the
        :attr:`~InfoElementSpec.length` attribute of :class:`InfoElementSpec`.
        This may only be applied to certain data types and must be smaller
        than the default length.  When 0, the default length specified by the
        :class:`InfoModel` is used.

        When *type* is ``BASICLIST``, the *element_name* parameter is used as
        the content type for the elements in the basicList.  See
        :meth:`init_basic_list` and the :class:`BL` constructor.

        Elements must be added in the same order as they exist in the
        template.

        Examples::

        >>> my_rec = pyfixbuf.Record(model)
        >>> my_rec.add_element("sourceTransportPort")
        >>> my_rec.add_element("sourceTransportPort2", 0, "sourceTransportPort")
        >>> my_rec.add_element("basicList")
        >>> my_rec.add_element("basicList2", BASICLIST)
        >>> my_rec.add_element("octetTotalCount", length=4)

        In the above example, an empty :class:`!Record` was created.
        The corresponding template
        to the above :class:`!Record` would look something like:

        >>> tmpl = Template(model)
        >>> tmpl.add_spec_list([
        ...     pyfixbuf.InfoElementSpec("sourceTransportPort"),
        ...     pyfixbuf.InfoElementSpec("sourceTransportPort"),
        ...     pyfixbuf.InfoElementSpec("basicList"),
        ...     pyfixbuf.InfoElementSpec("basicList"),
        ...     pyfixbuf.InfoElementSpec("octetTotalCount", 4)])

        As you can see, we have two sourceTransportPort elements and
        two basicList elements.  A basicList is a list
        of one or more of the same Information Element.
        The Information Element in the basicList does not have to be
        initialized until data is added to the :class:`!Record`.

        Since we have two sourceTransportPort fields, we must give a
        *key_name* to one of the elements, in this case, sourceTransport2.
        Since sourceTransportPort2 is not a defined Information Element in the
        Information Model, the *element_name* must be given to the method.

        Similarly, in order to access the dictionary of elements
        in the :class:`!Record`, we had to give the second basicList a
        *key_name*, basicList2. Since basicList2 is not a defined Information
        Element, it needs to be given the *type*, ``BASICLIST``.
        Since *type* is not 0, it does not need an *element_name*.
        """
        if self.list_init:
            raise Exception("Information Elements may not be added to the "
                            "Record once the Record's lists have been "
                            "initialized.")
        if self.template:
            raise Exception("Information Elements may not be added to the "
                            "Record once the Record's Template has been set.")
        assert_type(str, key_name)
        if element_name == None:
            element_name = key_name
        if type == BASICLIST:
            ie = self.model.get_element("basicList")
        elif type == SUBTEMPLATELIST:
            ie = self.model.get_element("subTemplateList")
        elif type == SUBTEMPLATEMULTILIST:
            ie = self.model.get_element("subTemplateMultiList")
        elif type != 0 and type != VARLEN:
            raise Exception("Invalid Type " + str(type))
        else:
            try:
                ie = self.model.get_element(element_name)
            except Exception:
                raise Exception("Information Element " + element_name +
                                " does Not Exist in Information Model")
        try:
            mem_len = self.model.get_element_length(element_name, type)
        except Exception:
            raise Exception("Information Element " + element_name +
                            " does Not Exist in Information Model")
        # use modified length if requested and type is not a list/varlen
        wire_len = ie.length
        if (ie.type in [BASICLIST, SUBTEMPLATELIST, SUBTEMPLATEMULTILIST] or
            length == 0 or length == VARLEN or type == VARLEN):
            mem_len = self.model.get_element_length(element_name, type)
        elif not DataType.supports_RLE(ie.type):
            raise Exception("Information Element " + element_name +
                            " does not support reduced-length encoding.")
        elif length > wire_len:
            raise Exception("Length value must be smaller than default "
                            "Information Element Length (default: " +
                            ie.length + ").")
        else:
            type = DataType.refine_type_for_length(ie.type, length)
            wire_len = length
            mem_len = wire_len
        self._add_field(key_name, ie, wire_len, self.length, mem_len)
        self.length += mem_len
        if ie.type == DataType.BASIC_LIST:
            field = self._fields[-1]
            try:
                ie = self.model.get_element(element_name)
                self._basic_list[field] = element_name
            except KeyError:
                self._basic_list[field] = None

    def add_element_list(self, name_list):
        """
        Treats each string in *name_list* as the name of an
        :class:`InfoElement` and appends that information element to the
        :class:`!Record`.  See the above method :meth:`add_element`.
        """
        if isinstance(name_list, str):
            raise TypeError("Expected a non-str iterable")
        for name in name_list:
            self.add_element(name)

    def clear_all_lists(self):
        """
        Clears all the lists in the top level of the :class:`!Record`.

        Any nested lists must be accessed and cleared manually.
        """
        for field in self._fields:
            ty = field.ie.type
            if ty == DataType.BASIC_LIST:
                self.clear_basic_list(field.key)
            elif ty == DataType.SUB_TMPL_LIST:
                stl = self._get_value(field)
                stl.clear()
            elif ty == DataType.SUB_TMPL_MULTI_LIST:
                stml = self._get_value(field)
                stml._clear_stml(self, self._off_dict[key])

    def clear(self):
        """
        Clears any memory allocated for the :class:`!Record`.
        """
        _pyfixbuf.fbRecordBase.clear(self)

    def init_basic_list(self, basic_list_key, count=0, element_name=None):
        """Initializes a basicList for export with the given *basic_list_key*
        name to a list of  *count* elements.  If a name is not given to
        the *element_name* keyword, it assumes
        the *basic_list_key* is a valid Information Element Name.

        Examples::

            >>> my_rec.add_element("bL", BASICLIST, "octetTotalCount")
            >>> my_rec.add_element("basicList")
            >>> my_rec.add_element("basicList2", BASICLIST)
            >>> my_rec.init_basic_list("bL", 4)
            >>> my_rec.init_basic_list("basicList", 3, "destinationTransportPort")
            >>> my_rec.init_basic_list("basicList2", 2, "souceIPv4Address")

        In the above example, we have initialized three basicLists.
        The first initializes a basicList of octetTotalCounts by
        adding the element as as basicList to the record.  Later we
        initialize the basicList to 4 items.  The second does the
        initialization of the type, destintationTransportPort, when
        calling :meth:`init_basic_list` as opposed to the first, which
        is done when the basicList is added to the record. The third,
        basicList2, is initialized to two sourceIPv4Addresses.

        It is perfectly acceptable to initialize a list to 0 elements.
        All basicLists in the :class:`!Record` must be initialized before
        appending the :class:`!Record` to the :class:`Buffer`.

        A basicList may be initialized via this method, or by using the
        :class:`BL` and setting the basicList element in the
        :class:`!Record` to the :class:`BL`.
        """
        field = self._field_dict[basic_list_key]
        if field.ie.type != DataType.BASIC_LIST:
            raise Exception("The type of field %s is not BASICLIST" %
                            str(basic_list_key))
        bl = self._basic_list[field]
        if isinstance(bl, BL):
            return bl
        if element_name:
            ie = element_name
        elif bl is None:
            raise Exception("Field " + basic_list_key + " must be"
                            " initialized to an information element.")
        elif bl == "basicList":
            raise Exception("Nested basicLists are not supported")
        else:
            ie = self._basic_list[field]
        self._basic_list[field] = BL(self.model, ie, count)

    def clear_basic_list(self, basic_list_key):
        """
        Clears the basicList on this :class:`!Record` identified by
        *basic_list_key*, freeing any memory allocated for the list.  This
        should be called after the :class:`!Record` has been appended to the
        :class:`Buffer`.  Does nothing if the type of the field identified by
        *basic_list_key* is not :class:`DataType`\ .BASIC_LIST, but does
        raises :exc:`KeyError` if *basic_list_key* is not known on this
        :class:`!Record`.
        """
        field = self._field_dict[basic_list_key]
        if field.ie.type == DataType.BASIC_LIST:
            _pyfixbuf.fbRecordBase.basicListClear(self, field.offset)

    def __getitem__(self, key):
        """
        Implements the evaluation of ``record[key]`` for :class:`!Record`
        instances: Returns the value of the element with the given *key*.

        If *key* is a string, it may the name of an :class:`InfoElement` or
        the *key_name* specified to the :meth:`add_element` method.  *key* may
        also be an integer corresponding to the positional index in the
        :class:`!Record`.

        Raises an :exc:`Exception` when *key* is not in the :class:`!Record`.
        Use :meth:`get` for a similar function that does not raise an
        Exception.  See also :meth:`get_field`.

        The return type depends on the Information Element type which was
        defined when initializing the :class:`InfoElement`.

        ============================   =============
        Element Type                   Return Type
        ============================   =============
        UINT*, INT*                    long
        FLOAT*                         float
        MILLISECONDS, MICROSECONDS     long
        NANOSECONDS, SECONDS           long
        OCTET_ARRAY                    bytearray
        BASICLIST                      :class:`BL`
        STRING                         string
        IP (v4 or v6)                  ipaddress object
        MAC_ADDR                       MAC Address String xx:xx:xx:xx:xx:xx
        SUBTEMPLATELIST                :class:`STL`
        SUBTEMPLATEMULTILIST           :class:`STML`
        Default (Undefined Type)       bytearray
        ============================   =============

        Example:

        >>> rec
        <pyfixbuf.Record object at 0x10d0f49f0>
        >>> rec['protocolIdentifier']
        6
        >>> rec[1]
        80
        >>> rec.template[1].name
        'sourceTransportPort'
        """
        field = self._field_dict.get(key)
        if field is None:
            if isinstance(key, int):
                ex = IndexError
            else:
                ex = KeyError
            raise ex("Key %s does not exist on this Record" % str(key))
        return self._get_value(field)

    def __setitem__(self, key, value):
        """
        Implements assignment to ``record[key]`` for :class:`!Record`
        instances: Sets the value of the element having the given *key* to
        *value*.

        If *key* is a string, it may the name of an :class:`InfoElement` or
        the *key_name* specified to the :meth:`add_element` method.  *key* may
        also be an integer corresponding to the positional index in the
        :class:`!Record`.
        """
        field = self._field_dict.get(key)
        if field is None:
            if isinstance(key, int):
                ex = IndexError
            else:
                ex = KeyError
            raise ex("Key %s does not exist on this Record" % str(key))
        return self._set_value(field, value)

    def _get_value(self, field):
        """
        Returns the value for a field.  For internal use.
        """
        #assert_type(Record._Elem, field)
        ty = field.ie.type
        if ty == DataType.BASIC_LIST:
            bl = BL(self.model, self._basic_list[field])
            _pyfixbuf.fbRecordBase.getBL(self, bl, field.offset)
            bl._fill_bl_list()
            return bl
        if ty == DataType.SUB_TMPL_MULTI_LIST:
            stml = STML(self, field.key)
            stml.session = self.session
            return stml
        if ty == DataType.SUB_TMPL_LIST:
            stl = STL(self, field.key)
            stl.session = self.session
            return stl
        data = _pyfixbuf.fbRecordBase.getOffset(self, field.offset,
                                                field.mem_len, ty,
                                                (VARLEN == field.wire_len))
        if ty == DataType.IP4ADDR:
            return ipaddress.IPv4Address(data)
        if ty == DataType.IP6ADDR:
            return ipaddress.IPv6Address(bytes(data))
        if ty == DataType.MAC_ADDR:
            return bytes_to_macaddr(data)
        return data

    def _set_value(self, field, value):
        """
        Sets a field to a value.  For internal use.
        """
        #assert_type(Record._Elem, field)
        ty = field.ie.type
        if ty == DataType.BASIC_LIST:
            if isinstance(value, BL):
                bl = value
            elif self._basic_list[field] == None:
                raise Exception("Basic List \"" + str(field.key) + "\"  has not"
                                " been initialized, call init_basic_list()")
            else:
                if not isinstance(value, list):
                    value = [value]
                if not isinstance(self._basic_list[field], BL):
                    self._basic_list[field] = BL(
                        self.model, self._basic_list[field], len(value))
                self._basic_list[field].copy(value)
                bl = self._basic_list[field]
            _pyfixbuf.fbRecordBase.setOffset(self, bl, field.offset,
                                             field.mem_len, ty, 0)
            return
        # Handle anything that is not a basicList
        if ty == DataType.IP4ADDR:
            value = ip4addr_to_int(value)
        elif ty == DataType.IP6ADDR:
            value = ip6addr_to_bytes(value)
        elif ty == DataType.MAC_ADDR:
            value = macaddr_to_bytes(value)
        elif ty == DataType.SUB_TMPL_LIST:
            self.list_init = True
            if isinstance(value, STL):
                value.inrec = self
            elif isinstance(value, list):
                template = None
                for rec in value:
                    if not isinstance(rec, Record):
                        raise Exception("Value must be a list of Records.")
                    if template == None and rec.template != None:
                        template = rec.template
                if (template == None):
                    raise Exception("At least one Record in list must "
                                    "be associated with template.")
                stl = STL(self, field.key)
                stl.entry_init(value[0], template, len(value))
                for i,rec in enumerate(value):
                    stl[i] = rec
                value = stl
        elif ty == DataType.SUB_TMPL_MULTI_LIST:
            self.list_init = True
            if isinstance(value, STML):
                value._put_stml_rec(self, field.offset)
            elif isinstance(value, list):
                stml = create_stml_from_list(value)
                value = stml
            else:
                raise Exception("Value is type of " + str(type(value)) +
                                " but key " + str(field.key) + " has type " +
                                DataType.to_string(ty))
        elif not DataType.check_type(ty, value):
            raise Exception("Value is type of " + str(type(value)) +
                            " but key " + str(field.key) + " has type " +
                            DataType.to_string(ty))
        _pyfixbuf.fbRecordBase.setOffset(self, value, field.offset,
                                         field.mem_len, ty,
                                         (VARLEN == field.wire_len))

    def get(self, key, default=None):
        """
        Returns ``record[key]`` if *key* exists on this :class:`!Record`;
        otherwise returns *default*.
        """
        field = self._field_dict.get(key)
        if field is None:
            return default
        return self._get_value(field)

    def get_field(self, key):
        """
        Returns a :class:`Record.Field` object for *key* on this
        :class:`!Record`.  Raises :exc:`KeyError` when *key* is not present.
        """
        field = self._field_dict.get(key)
        return Record.Field(field.key[0], field.key[1], field.ie,
                            field.wire_len, self._get_value(field))

    def copy(self, other):
        """
        Copies all the matching elements from the *other* :class:`!Record` to
        this :class:`!Record`.
        """
        assert_type(Record, other)
        for field in self._fields:
            other_field = other._field_dict.get(field.key)
            if other_field is not None:
                self._set_value(field, other._get_value(other_field))

    def is_list(self, key):
        """
        Returns ``True`` or ``False`` depending on the type of the given
        *key*.  Raises :exc:`KeyError` when *key* is not on this
        :class:`!Record`.
        """
        return (self._field_dict[key].ie.type in
                [BASICLIST, SUBTEMPLATELIST, SUBTEMPLATEMULTILIST])

    def get_field_type(self, key):
        """
        Returns the :class:`DataType` for the given *key*.  Raises
        :exc:`KeyError` when *key* is not on this :class:`!Record`.
        """
        return self._field_dict[key].ie.type

    def get_stl_list_entry(self, key):
        """
        Gets the subTemplateList from this :class:`!Record` with the given
        *key* and returns a newly allocated :class:`STL`.  Returns ``None`` if
        the type of *key* is not :class:`DataType`\ .SUB_TMPL_LIST.  Raises
        :exc:`KeyError` if *key* is not a known key on this :class:`!Record`.

        A :class:`STL` may also be accessed by using :meth:`__getitem__`.
        """
        field = self._field_dict[key]
        if field.ie.type == DataType.SUB_TMPL_LIST:
            return self._get_value(field)

    def get_stml_list_entry(self, key):
        """
        Gets the subTemplateMultiList from this :class:`!Record` with the
        given *key* and returns a newly allocated :class:`STML`.  Returns
        ``None`` if the type of *key* is not
        :class:`DataType`\.SUB_TMPL_MULTI_LIST.  Raises :exc:`KeyError` if
        *key* is not a known key on this :class:`!Record`.

        A :class:`STML` may also be retrieved by using :meth:`__getitem__`.
        """
        field = self._field_dict[key]
        if field.ie.type == DataType.SUB_TMPL_MULTI_LIST:
            return self._get_value(field)

    def as_dict(self):
        """
        Returns a :class:`dict` that represents the :class:`!Record`.

        The keys of the dictionary are normally strings, but if the
        :class:`!Record`'s :class:`Template` contains duplicate a
        :class:`InfoElement`, the key for the **second** such element is a
        couple containing the name and the :class:`int` 1, the third would be
        the name and the :class:`int` 2, et cetera.
        """
        recdict = dict()
        for field in self._fields:
            if field.key[1] == 0:
                recdict[field.key[0]] = self._get_value(field)
            else:
                recdict[field.key] = self._get_value(field)
        return recdict

    def __len__(self):
        """
        Implements the built-in `len()` method for :class:`!Record` instances:
        Returns the number of elements in the :class:`!Record`.
        """
        return len(self._fields)

    def __contains__(self, element):
        """
        Implements the ``in`` operator for :class:`!Record` instances: Tests
        whether :meth:`add_element` was called with a `key_name` that is equal
        to *element*.  If the :class:`!Record` was initialized with a
        :class:`Template`, tests whether the :class:`Template` included an
        :class:`InfoElement` having the name *element*.

        Example:

        >>> rec
        <pyfixbuf.Record object at 0x10d0f49f0>
        >>> 'protocolIdentifier' in rec
        True
        """
        return element in self._field_dict

    def __iter__(self):
        """
        Implements the method to return an Iterator over the values in the
        :class:`!Record`.  See also :meth:`iterfields`.

        Example:

        >>> for value in record:
        ...     print(value)
        """
        for field in self._fields:
            yield self._get_value(field)

    def iterfields(self):
        """
        Returns an Iterator over the :class:`!Record`'s values where each
        iteration returns a :class:`Record.Field` object.  To get a
        :class:`Record.Field` for a single field, use :meth:`get_field`.

        Example:

        >>> for field in record.iterfields():
        ...     print(field.ie.name, DataType.get_name(field.ie.type), field.value)
        ...
        """
        for field in self._fields:
            yield Record.Field(field.key[0], field.key[1], field.ie,
                               field.wire_len, self._get_value(field))

    def matches_template(self, template, exact=False):
        """
        Returns ``True`` if this :class:`!Record` matches :class:`Template`
        using the checks specified by :meth:`check_template`.  Returns
        ``False`` otherwise.
        """
        assert_type(Template, template)
        try:
            self.check_template(template, exact)
            return True
        except Exception:
            return False

    def check_template(self, template, exact=False):
        """
        Checks whether :class:`Template` has exactly the same
        :class:`InfoElement`\s as this :class:`!Record` in the same order with
        the same reduced-length encodings.  When *exact* is ``False``, the
        :class:`Template` may have additional Elements.  Raises an
        :exc:`Exception` if there is a mismatch; returns no result otherwise.
        """
        assert_type(Template, template)
        for field, t_ie, spec in zip_longest(
                self._fields, template.ie_iter(), template):
            if field is None or t_ie is None:
                if not exact and field is None:
                    return
                raise Exception("The Record and Template contain a"
                                " different number of elements")
            if (t_ie.id != field.ie.id or
                t_ie.enterprise_number != field.ie.enterprise_number):
                raise Exception("The Record and Template do not contain the"
                                " same elements")
            if spec.length != field.wire_len:
                raise Exception("The element lengths in the Record and"
                                " Template differ")

    def set_template(self, template):
        """
        If this :class:`!Record` was not initialized with a :class:`Template`,
        this method may be used to set the :class:`!Record`'s
        :class:`Template`.  A :class:`!Record` must have a :class:`Template`
        associated with it when assigning a :class:`!Record` to a
        subTemplateList element.

        Examples:

        >>> tmpl = pyfixbuf.Template(model)
        >>> tmpl.add_spec_list([
        ...     pyfixbuf.InfoElementSpec("sourceTransportPort"),
        ...     pyfixbuf.InfoElementSpec("destinationTransportPort")]
        >>> my_rec = pyfixbuf.Record(model)
        >>> my_rec.add_element("sourceTransportPort", "destinationTransportPort")
        >>> my_rec["sourceTransportPort"] = 13
        >>> my_rec["destinationTransportPort"] = 15
        >>> my_rec.set_template(tmpl)
        >>> other_rec["subTemplateList"] = [my_rec]
        """
        self.check_template(template, True)
        self.template = template

    def count(self, element_name):
        """Counts the occurrences of the *element_name* in the
        :class:`!Record`.

        Examples:

        >>> rec.add_element_list(["basicList", "basicList", "basicList"])
        >>> rec.count("basicList")
        3
        >>> rec.count("sourceTransportPort")
        0
        """
        for field in reversed(self._fields):
            if field.key[0] == element_name:
                return 1 + field.key[1]
        return 0


class Buffer(_pyfixbuf.fBufBase):
    """
    Creates an uninitialized :class:`!Buffer`.

    The :class:`!Buffer` must be initialized for collection or export using
    :meth:`init_collection` or :meth:`init_export` prior to use.

    If *auto* is ``True`` and the :class:`!Buffer` is initialized for
    collection, a :class:`Template` is automatically generated from the
    external template that is read from the :class:`Collector`, the internal
    template on the :class:`!Buffer` is set to that template, and a matching
    :class:`Record` is created to match that :class:`!Template`.

    Example:

    >>> collector = Collector()
    >>> collector.init_file(sys.argv[1])
    >>> session = Session(infomodel)
    >>> buffer = Buffer(auto=True)
    >>> buffer.init_collection(session, collector)
    >>> for record in buffer:
    >>>     print(record.as_dict())

    If *record* is specified, it is set as the :class:`!Buffer`\ 's cached
    :class:`!Record` for use during collection.  The :meth:`set_record` method
    may also be used to set the cached :class:`!Record`.

    There is no requirement to set a cached :class:`!Record` on the
    :class:`!Buffer` since the :class:`!Buffer` creates one as needed.  The
    cached :class:`!Record` is used during collection when its
    :class:`!Template` matches the :class:`!Buffer`\ 's current internal
    :class:`!Template`.  Calling :meth:`set_internal_template` may clear the
    cached :class:`!Record`.

    For export, the user may use :meth:`set_internal_template` and
    :meth:`set_export_template` to specify the format of the record, and call
    :meth:`append` to append a :class:`!Record` to the :class:`!Buffer`.

    """
    def __init__(self, record=None, auto=False):
        self.session = None
        self.model = None
        # The ID of the current internal and external templates
        self.int_tmpl = None
        self.ext_tmpl = None
        #set 1 for collection, 2 for export
        self.mode = 0
        self.auto = auto
        if record:
            self.set_record(record)
        else:
            self.rec = None
        _pyfixbuf.fBufBase.__init__(self)

    def init_collection(self, session, collector):
        """
        Initializes the :class:`!Buffer` for collection given the
        :class:`Session`, *session*, and :class:`Collector`, *collector*.
        """
        assert_type(Session, session)
        assert_type(Collector, collector)
        self.session = session
        self.model = session.model
        self.mode = 1
        if (collector.initialized == 0):
            raise Exception("Collector has not been initialized for file "
                            "collection")
        return _pyfixbuf.fBufBase.allocForCollection(self, session, collector)

    def init_export(self, session, exporter):
        """
        Initializes the :class:`!Buffer` for Export given the
        :class:`Session`, *session*, and :class:`Exporter`, *exporter*.
        """
        assert_type(Session, session)
        assert_type(Exporter, exporter)
        self.session = session
        if (exporter.initialized == 0):
            raise Exception("Exporter has not been initialized for file "
                            " or transport.")
        self.mode = 2
        return _pyfixbuf.fBufBase.allocForExport(self, session, exporter)

    def _init_listener(self, session):
        """
        Internal function to set session on buffer.
        """
        self.session = session
        self.model = session.model
        self.mode = 1

    def set_internal_template(self, template_id):
        """
        Sets the internal :class:`Template` to the one whose ID is
        *template_id*.  The :class:`!Buffer` must have an internal template
        set on it before collecting or exporting.  Causes the
        :class:`!Buffer` to discard the cached :class:`Record` specified in
        the call to :meth:`set_record` or to the constructor if its
        :class:`!Template` does not match that specified by *template_id*.
        """
        if (self.session == None):
            raise Exception("Buffer needs to be initialized for collection "
                            "or export before setting templates.")
        tmpl = self.session.internal_templates.get(template_id)
        if not tmpl:
            assert_type(int, template_id)
            if template_id <= 0xff or template_id > 0xffff:
                raise Exception("Invalid ID: " + str(template_id) +
                                " Template ID must be between 256 and 65535")
            raise Exception("Template %s has not been added to Session's "
                            "Internal Templates." % str(template_id))
        self.int_tmpl = template_id
        if self.rec and not self.rec.matches_template(tmpl, True):
            self.rec = None
        _pyfixbuf.fBufBase.setInternalTemplate(self, template_id)

    def set_export_template(self, template_id):
        """
        Sets the external :class:`Template` to the one whose ID is
        *template_id*.  The :class:`!Buffer` must have an export template set
        before appending (:meth:`append`) a :class:`Record` to the
        :class:`!Buffer`.  The export :class:`!Template` describes how fixbuf
        will write the given :class:`Record` to the output stream.
        """
        if (self.session == None):
            raise Exception("Buffer needs to be initialized for collection "
                            "or export before setting templates.")
        if (self.session.external_templates.get(template_id) == None):
            assert_type(int, template_id)
            if template_id <= 0xff or template_id > 0xffff:
                raise Exception("Invalid ID: " + str(template_id) +
                                " Template ID must be between 256 and 65535")
            raise Exception("Template %s has not been added to Session's "
                            "Internal Templates." % str(template_id))
        self.ext_tmpl = template_id
        _pyfixbuf.fBufBase.setExportTemplate(self, template_id)

    def __iter__(self):
        """
        Implements the method to return an Iterator over the :class:`Record`
        objects in this :class:`!Buffer` initialized for collection.

        Example:

        >>> for record in buffer:
        ...     print(record.as_dict())
        """
        return self

    def next(self):
        """
        Returns the next :class:`Record` from this :class:`!Buffer`
        initialized for collection.  Raises a :exc:`StopIteration` Exception
        when there are no more :class:`Record` objects.  Raises
        :exc:`Exception` if an internal template has not been set on the
        :class:`!Buffer`.
        """
        return self.__next__()

    def __next__(self):
        """
        Returns the next :class:`Record` from this :class:`!Buffer`
        initialized for collection.  Raises a :exc:`StopIteration` Exception
        when there are no more :class:`Record` objects.  Raises
        :exc:`Exception` if an internal template has not been set on the
        :class:`!Buffer`.
        """
        if (self.auto):
            self._auto_generate_template()
        if self.int_tmpl == None:
            raise Exception("No Internal Template Set on Buffer")
        if not self.rec:
            tmpl = self.session.get_template(self.int_tmpl, True)
            self.rec = Record(self.model, tmpl)
        if (self.rec.length == 0):
            raise Exception("No Information Elements in Record")
        _pyfixbuf.fBufBase.nextRecord(self, self.rec)
        self.rec.session = self.session
        return self.rec

    def next_record(self, record=None):
        """
        Gets the next record on this :class:`!Buffer` in the form of the given
        :class:`Record`, *record*.  Returns ``None`` if the :class:`!Buffer`
        is empty.  Raises an exception if an internal template is not set on
        the :class:`!Buffer` or if *record* does not match the internal
        template.
        """
        if (self.auto):
            try:
                self._auto_generate_template()
            except StopIteration:
                return None

        if self.int_tmpl == None:
            raise Exception("No internal template set on buffer")
        tmpl = self.session.get_template(self.int_tmpl, True)
        if record == None:
            record = Record(self.model, tmpl)
        else:
            # Ensure the internal_template matches record.
            assert_type(Record, record)
            record.check_template(tmpl, True)
        if (record.length == 0):
            raise Exception("No Information Elements in Record")
        try:
            _pyfixbuf.fBufBase.nextRecord(self, record)
            record.session = self.session
            return record
        except StopIteration:
            return None

    def set_record(self, record):
        """
        Sets the cached :class:`Record` on this :class:`!Buffer` to *record*.
        If *record* is associated with a :class:`Template` (and this
        :class:`!Buffer` is initialized for collection or export), calls
        :meth:`set_internal_template` with the template_id of that
        :class:`!Template`.
        """
        assert_type(Record, record)
        self.rec = record
        if self.session and record.template:
            if (record.template.template_id):
                self.set_internal_template(record.template.template_id)

    def next_template(self):
        """
        Retrieves the external :class:`Template` that **will** be used to read
        the **next** :class:`Record` from this :class:`!Buffer` initialized
        for collection.  If no next :class:`!Record` is available, raises
        :exc:`StopIteration`.

        If :meth:`ignore_options` is `True`, skips options records and returns
        the :class:`!Template` for the next non-options :class:`!Record`.

        See also :meth:`get_template`.
        """
        template = _pyfixbuf.fBufBase.nextTemplate(self)
        new_template = Template._wrap(template)
        return new_template

    def get_template(self):
        """
        Retrieves the external :class:`Template` that **was** used to read the
        **current** :class:`Record` from this :class:`!Buffer` initialized for
        collection.  If no :class:`!Record` has been read, returns ``None``.

        See also :meth:`next_template`.
        """
        template = _pyfixbuf.fBufBase.getTemplate(self)
        new_template = Template._wrap(template)
        return new_template

    def append(self, *args):
        """
        Appends the first argument, a :class:`Record`, to this
        :class:`!Buffer` initialized for export.  If a second argument, an
        :class:`int` representing a length, is given, appends only the first
        *length* number of bytes to the :class:`!Buffer`.

        An internal and external :class:`Template` must be set on the
        :class:`!Buffer` prior to appending a :class:`!Record`.
        """
        if self.ext_tmpl == None:
            raise Exception("No External Template Set on Buffer")
        if self.int_tmpl == None:
            raise Exception("No Internal Template Set on Buffer")
        return _pyfixbuf.fBufBase.append(self, *args)

    def write_ie_options_record(self, name, template):
        """
        Appends an Information Element Type Information Record (:rfc:`5610`)
        to this :class:`!Buffer` initialized for export.  An Options Record is
        appended with information about the Information Element with the given
        *name*.  *template* is the Information Element Type Options Template
        that was created by giving ``type=True`` to the :class:`Template`
        constructor.
        """
        inttid = self.int_tmpl
        exttid = self.ext_tmpl
        if (self.mode == 0):
            raise Exception("This buffer has not been initialized for export.")
        elif (self.mode == 1):
            raise Exception("This buffer is for collection only. It cannot "
                            "write options records.")
        if (not(template.scope)):
            raise Exception("Given Template is not an Options Template.")
        if (template.template_id not in self.session.internal_templates):
            tid = self.session.add_template(template)
        _pyfixbuf.fBufBase.writeOptionsRecord(self,template.infomodel._unwrap(),
                                              name, template.template_id)
        #set templates back to where they were
        if inttid:
            self.set_internal_template(inttid)
        if exttid:
            self.set_export_template(exttid)

    def auto_insert(self):
        """
        Tells this :class:`!Buffer` intialized for collection to watch the
        stream for Information Element Option Records (:rfc:`5610`) and
        automatically create and insert an :class:`InfoElement` to the
        :class:`InfoModel` for each one seen.  Information elements that have
        an enterprise number of 0 are ignored.
        """
        _pyfixbuf.fBufBase.setAutoInsert(self)


    def ignore_options(self, ignore):
        """
        Tells this :class:`!Buffer` initialized for collection how to handle
        Options Records.

        If *ignore* is set to ``True``, the :class:`!Buffer` ignores Options
        Templates and Records.  By default, *ignore* is ``False``, and the
        :class:`!Buffer` returns Options Records that the application must
        handle.

        The application may use :meth:`next_template` to retrieve the
        :class:`Template` and determine if it is an Options Template.
        """
        if (ignore < 0):
            raise Exception("Invalid value given to ignore_options.  "
                            "Should be True/False")
        _pyfixbuf.fBufBase.ignoreOptions(self, ignore)


    def _auto_generate_template(self):
        """
        If *auto* is set to True, the buffer will auto-generate internal
        templates from the external templates it receives and set them
        on the buffer before decoding the next IPFIX message.  The
        :class:`Record` returned from the :class:`!Buffer` will match
        the external template retrieved from the exporting message.

        Raises :exc:`StopIteration` if no more data is available.
        """
        if (self.session == None):
            raise Exception("Buffer must be initialized for collection "
                            "before template can be received.")
        if (self.mode != 1):
            raise Exception("Auto generation for templates only applies "
                            "to Buffers in collection mode.")
        # allow this to raise StopIteration if no data
        tmpl_next = self.next_template()
        if tmpl_next.template_id != self.int_tmpl:
            self.rec = None
        try:
            tmpl = self.session.get_template(tmpl_next.template_id, True)
        except:
            tmpl = self.session.get_template(tmpl_next.template_id)
            self.session.add_internal_template(tmpl, tmpl_next.template_id)
        self.set_internal_template(tmpl_next.template_id)


class STML(_pyfixbuf.fbSTMLBase):
    """A :class:`!STML` object represents a subTemplateMultiList.

    If a :class:`Record` object, *record*, and *key_name*, a string,
    are provided, the :class:`!STML` object with *key_name* will be
    initialized in the given :class:`Record`.  It is only necessary to
    initialize and give a *type_count* if the subTemplateMultiList
    will be exported.  All subTemplateMultiLists in an exported
    :class:`Record` must be initialized.  It is acceptable to
    initialize an STML to 0 list entries.

    A :class:`!STML` must be initialized with *record* and *key_name*
    OR a *type_count*.  This object can be used to set a
    subTemplateMultiList element in a :class:`Record`.

    The subTemplateMultiList is initialized to ``None`` unless it is
    given a *type_count*, in which case it will intialize the list and
    allocate memory in the given record.

    *type_count* is the number of different templates that the
    :class:`!STML` will contain.  For example, if you plan to have an
    STML with entries of type Template ID 999 and 888, *type_count*
    would be 2.  *type_count* would also be 2 even if both instances
    will use Template ID 999.

    Examples::

    >>> stml = my_rec["subTemplateMultiList"]   # sufficient for collection
    >>> stml = pyfixbuf.STML(rec, "subTemplateMultiList", 3)  # STML with 3 entries for export
    >>> stml = pyfixbuf.STML(type_count=2)
    >>> stml = [record1, record2]
    >>> stml2 = pyfixbuf.STML(type_count=3)
    >>> stml2[0] = [record1, record2]
    >>> stml2[1][0] = record3
    >>> stml2[2].entry_init(record3, tmpl3, 0) #all entries must be init'd - even to 0.
    >>> rec["subTemplateMultiList"] = stml
    """
    def __init__(self, record=None, key_name=None, type_count=-1):
        self.info_model = None
        self.offset = 0
        self.entries = []
        # The record that owns this STML
        self.rec = None
        self.session = None
        if record is None and type_count >= 0:
            _pyfixbuf.fbSTMLBase.__init__(self, None, 0, type_count)
        elif record is not None and key_name is not None:
            assert_type(Record, record)
            if (record.get_field_type(key_name) != SUBTEMPLATEMULTILIST):
                raise Exception("DataType of " + str(key_name) +
                                " is not SUBTEMPLATEMULTILIST")
            self.offset = record._get_field_offset(key_name)
            _pyfixbuf.fbSTMLBase.__init__(self, record, self.offset)
            record.list_init = True
            self.rec = record
            self.info_model = record.model
            self.session = record.session
        else:
            raise Exception("STML must be intialized with either"
                            " a type_count or a Record and key_name")

    def __iter__(self):
        """
        Implements the method to return an Iterator over the
        :class:`STMLEntry` objects in the :class:`!STML`.

        Example:

        >>> for entry in stml:
        ...     for record in entry:
        ...         print(record.as_dict())
        """
        return self

    def next(self):
        """Returns the next :class:`STMLEntry` in the :class:`!STML`.
        """
        return self.__next__()

    def __next__(self):
        """Returns the next :class:`STMLEntry` in the :class:`!STML`.
        """
        _pyfixbuf.fbSTMLBase.getNextEntry(self, self.rec, self.offset)
        stmlentry = STMLEntry(self)
        return stmlentry

    def clear(self):
        """
        Clears the entries in the subTemplateMultiList and frees any
        memory allocated.
        """
        _pyfixbuf.fbSTMLBase.clear(self, self.rec, self.offset)

    def _clear_stml(self, record, offset):
        _pyfixbuf.fbSTMLBase.clear(self, record, offset)

    def _put_stml_rec(self, record, offset):
        self.rec = record
        self.offset = offset

    def __len__(self):
        """
        Implements the built-in `len()` method for :class:`!STML` instances:
        Returns the number of :class:`STMLEntry` objects in the
        :class:`!STML`.
        """
        return self.count

    def __contains__(self, name):
        """
        Implements the ``in`` operator for :class:`!STML` instances: Tests
        whether the :class:`Template` used by the first :class:`STMLEntry` in
        this :class:`!STML` contains an :class:`InfoElement` having the name
        *name*.
        """
        _pyfixbuf.fbSTMLBase.getFirstEntry(self, self.rec, self.offset)
        stmlentry = STMLEntry(self)
        if name in stmlentry:
            rv = True
        else:
            rv = False
        _pyfixbuf.fbSTMLBase.rewind(self)
        return rv

    def __getitem__(self, index):
        """
        Implements the evaluation of ``stml[index]`` for :class:`!STML`
        instances: Returns the :class:`STMLEntry` at position *index*.

        Examples::

        >>> entry = stml[0]
        >>> stml[0].entry_init[record, template, 3]
        """
        if (not(isinstance(index, int))):
            raise TypeError
        if (len(self.entries) == 0):
            for i in range(len(self)):
                _pyfixbuf.fbSTMLBase.getIndex(self, i)
                stmlentry = STMLEntry(self)
                self.entries.append(stmlentry)
        return self.entries[index]

    def __setitem__(self, index, value):
        """
        Implements assignment to ``stml[index]`` for :class:`!STML` instances:
        Sets the :class:`STMLEntry` at position *index* in the :class:`!STML`
        to the list of :class:`Record` objects in *value*.  *value* must be a
        list. All :class:`Records` in the list should have the same
        :class:Template.

        Examples:

        >>> stml[0] = [rec1, rec2, rec3, rec4]
        """
        if (not(isinstance(value, list))):
            raise TypeError
        if (len(value) == 0):
            raise Exception("List must not be empty. Use entry_init to "
                            "initialze STMLEntry to empty list.")
        entry = self[index]
        if (value[0].template):
            entry.entry_init(value[0], value[0].template, len(value))
        else:
            raise Exception("Records in list must have Template "
                            "associated with them, or STMLEntry must "
                            "be initialized with entry_init()")
        for i,rec in enumerate(value):
            entry[i] = rec

    def iter_records(self, tmpl_id=0):
        """
        Returns an Iterator over the :class:`!STML`'s records, including
        records in any nested :class:`!STML` or :class:`STL`. If a template ID
        is passed, returns an iterator over all the :class:`Record` objects
        with a template ID matching the passed value.

        Example:

        >>> for record in stml.iter_records():
        ...     print(record.template.template_id)
        ...
        """
        for entry in self:
            for record in entry:
                for sub_rec in yield_record_subrecords(record, tmpl_id):
                    yield sub_rec


class STMLEntry(_pyfixbuf.fbSTMLEntryBase):
    """
    Creates an empty :class:`!STMLEntry` and associates it to the
    given :class:`STML`, *stml*.  There should be one
    :class:`!STMLEntry` for each different :class:`Template` in the
    :class:`STML`.

    Each :class:`!STMLEntry` should be initialized using :meth:`entry_init`
    to associate a :class:`Record` and :class:`Template` with the entry.
    """
    def __init__(self, stml):
        assert_type(STML, stml)
        _pyfixbuf.fbSTMLEntryBase.__init__(self, stml)
        self.info_model = stml.info_model
        self.stml = stml
        self.rec = None
        self.initialized = False
        self.items = 0
        self._exp_template = None
        # this variable keeps track if getNextDatPtr or getNextEntry was called
        # to set the record to the appropriate data buffer for __getitem__
        self.recset = False

    def entry_init(self, record, template, count=0):
        """
        Initializes the :class:`!STMLEntry` to the given :class:`Record`,
        *record*, :class:`Template`, *template*, and *count* instances of the
        *record* it will contain.

        This should only be used for exporting a subTemplateMultiList.
        Entries in the :class:`STML` must all be initialized, even if it
        is initialized to 0.  This method is not necessary if a :class:`Record`
        has a template associated with it. The application can simply set
        the :class:`!STMLEntry` to a list of :class:`Record` objects and the
        :class:`!STMLEntry` will automatically be initialized.

        Raises an :exc:`Exception` when `template` has a template_id of 0. You
        should add the `template` to a :class:`Session` before using it for
        the :class:`!STMLEntry`.

        Examples::

        >>> stml = pyfixbuf.STML(my_rec, "subTemplateMultiList", 1)
        >>> stml[0].entry_init(my_rec, template, 2)
        >>> my_rec["sourceTransportPort"] = 3
        >>> stml[0][0] = my_rec
        >>> my_rec["sourceTransportPort"] = 5
        >>> stml[0][1] = my_rec
        """
        assert_type(Record, record)
        assert_type(Template, template)
        if template.template_id == 0:
            raise Exception("Template has a template_id of 0")
        record.check_template(template, True)
        self.set_record(record)
        _pyfixbuf.fbSTMLEntryBase.entryInit(self, record, template._unwrap(),
                                            template.template_id, count)
        self.initialized = True
        self.items = count
        self.info_model = record.model
        self._exp_template = template

    def set_record(self, record):
        """
        Sets the :class:`Record` on the :class:`!STMLEntry` to *record* in
        order to access its elements.
        """
        assert_type(Record, record)
        tid = self.template_id
        if (tid and self.stml.session):
            tmpl = self.stml.session.get_template(tid, False)
            record.check_template(tmpl)
        elif self._exp_template:
            record.check_template(self._exp_template)
        self.rec = record

    def set_template(self, template):
        """
        Assigns a :class:`Template` to the :class:`!STMLEntry`.  The
        :class:`Template` must be valid.

        This method may be used as an alternative to :meth:`entry_init`. This
        is only required if the :class:`Record` that will be assigned to the
        :class:`!STMLEntry` was not created with a :class:`Template`.  Using
        this method instead of :meth:`entry_init` results in only allocating
        one item for the :class:`!STMLEntry`.

        Raises an :exc:`Exception` when `template` has a template_id of 0. You
        should add the `template` to a :class:`Session` before using it for
        the :class:`!STMLEntry`.
        """
        assert_type(Template, template)
        if template.template_id == 0:
            raise Exception("Template has a template_id of 0")
        _pyfixbuf.fbSTMLEntryBase.entryInit(self, None, template._unwrap(),
                                            template.template_id, 1)
        self.initialized = True
        self.items = 1
        self.info_model = template.infomodel
        self._exp_template = template

    def __iter__(self):
        """
        Implements the method to return an Iterator over the :class:`Record`
        objects in the :class:`!STMLEntry`.

        Example:

        >>> for entry in stml:
        ...     for record in entry:
        ...         print(record.as_dict())
        """
        return self

    def next(self):
        """Retrieves the next :class:`Record` in the :class:`!STMLEntry`.
        """
        return self.__next__()

    def __next__(self):
        """Retrieves the next :class:`Record` in the :class:`!STMLEntry`.

        If a :class:`Record` has not been associated with this
        :class:`!STMLEntry` a :class:`Record` will be auto generated
        using the current :class:`Template` set on this
        :class:`!STMLEntry`.
        """
        if (self.rec == None):
            tid = self.template_id
            tmpl = self.stml.session.get_template(tid)
            self.rec = Record(self.info_model, tmpl)
            self.rec.session = self.stml.session
            #raise Exception("No record set on STML Entry")
        _pyfixbuf.fbSTMLEntryBase.getNextRecord(self, self.rec)
        self.recset = True
        return self.rec

    def __contains__(self, name):
        """
        Implements the ``in`` operator for :class:`!STMLEntry` instances:
        Tests whether the :class:`Template` associated with this
        :class:`!STMLEntry` contains an :class:`InfoElement` having the name
        *name*.

        Alternatively, you can access the :class:`Template` ID associated with
        this :class:`!STMLEntry` to determine the type of :class:`Record` that
        should be used to access the elements.
        """
        return _pyfixbuf.fbSTMLEntryBase.containsElement(
            self, self.info_model._unwrap(), name)

    def __len__(self):
        """
        Implements the built-in `len()` method for :class:`!STMLEntry`
        instances: Returns the number of :class:`Record` objects in the
        :class:`!STMLEntry`.
        """
        return self.count

    def __getitem__(self, item):
        """
        Implements the evaluation of ``stmlEntry[item]`` for
        :class:`!STMLEntry` instances:

        If *item* is an :class:`int`, returns the :class:`Record` at that
        positional index.

        If *item* is a :class:`str`, finds the :class:`InfoElement` with the
        name *item* in the :class:`!STMLEntry`'s :class:`Template` and returns
        the value for that element in the first :class:`Record` in the
        :class:`!STMLEntry`.

        Returns ``None`` if *item* has an unexpected type.
        """
        if (self.rec == None):
            if (self.info_model):
                tid = self.template_id
                tmpl = self.stml.session.get_template(tid)
                self.rec = Record(self.info_model, tmpl)
            else:
                raise Exception("No record set on STMLEntry")
        elif (self.info_model == None):
            self.info_model = self.rec.model

        newRecord = Record(self.info_model, record=self.rec)
        newRecord.session=self.stml.session
        if (isinstance(item, str) or isinstance(item, tuple)):
            _pyfixbuf.fbSTMLEntryBase.getIndexedEntry(self, newRecord, 0)
            return newRecord[item]
        elif (isinstance(item, int)):
            _pyfixbuf.fbSTMLEntryBase.getIndexedEntry(self, newRecord, item)
            self.recset=True
            return newRecord

    def __setitem__(self, index, value):
        """
        Implements assignment to ``stmlentry[index]`` for :class:`!STMLEntry`
        instances: Sets the :class:`Record` at position *index* in this
        :class:`!STMLEntry` to *value*.

        If this :class:`!STMLEntry` has not been previously initialized, it is
        initialized with the :class:`Record`'s :class:`Template` and a length
        of 1.
        """
        assert_type(Record, value)
        if (self.initialized == False):
            if (self.items == 0):
                if (value.template):
                    self.entry_init(value, value.template, 1)
            else:
               raise Exception(
                   "STMLEntry must be initialized for some number of items.")
        if self.stml.session and self.template_id:
            tmpl = self.stml.session.get_template(tid)
            value.check_template(tmpl)
        elif self._exp_template:
            value.check_template(self._exp_template)
        self.rec = value
        _pyfixbuf.fbSTMLEntryBase.setIndexedEntry(self, index, value)


class STL(_pyfixbuf.fbSTLBase):
    """
    A :class:`!STL` represents a subTemplateList.

    If *record*, a :class:`Record` object, and *key_name*, a string, are
    provided, the subTemplateList for *key_name* in the given *record* is
    initialized, otherwise a generic :class:`!STL` is initialized.  Eventually
    a :class:`Template` must be associated with the :class:`!STL` for
    encoding.

    For decoding, a :class:`Record` must be associated with the :class:`!STL`.
    """
    def __init__(self, record=None, key_name=None):
        self.initialized = False
        self.info_model = None
        self.session = None
        # The record that gets filled and matches the STL
        self.rec = None
        # The template to use for export
        self._exp_template = None
        # The record that owns this STL
        self.inrec = None
        if (record != None and key_name != None):
            assert_type(Record, record)
            if (record.get_field_type(key_name) != SUBTEMPLATELIST):
                raise Exception("DataType of " + str(key_name) +
                                " is not SUBTEMPLATELIST")
            _pyfixbuf.fbSTLBase.__init__(self, record,
                                         record._get_field_offset(key_name))
            record.list_init = True
            self.inrec = record
            self.info_model = record.model
            self.session = record.session
        else:
            _pyfixbuf.fbSTLBase.__init__(self)

    def set_record(self, record):
        """
        Sets the :class:`Record` on this :class:`!STL` to *record*.
        """
        assert_type(Record, record)
        tid = self.template_id
        if (tid and self.session):
            tmpl = self.session.get_template(tid, False)
            record.check_template(tmpl)
        elif self._exp_template:
            value.check_template(self._exp_template)
        self.rec = record

    def __contains__(self, name):
        """
        Implements the ``in`` operator for :class:`!STL` instances: Tests
        whether the :class:`Template` associated with this :class:`!STL`
        contains an :class:`InfoElement` having the name *name*.
        """
        return _pyfixbuf.fbSTLBase.containsElement(
            self, self.info_model._unwrap(), name)

    def entry_init(self, record, template, count=0):
        """
        Initializes the :class:`!STL` to *count* entries of the given
        :class:`Record` and :class:`Template`.

        This method should only be used to export a :class:`!STL`.

        Each :class:`!STL` should be initialized before appending
        the :class:`Record` to the :class:`Buffer` even if it
        is initialized to 0.

        Raises an :exc:`Exception` when `template` has a template_id of 0. You
        should add the `template` to a :class:`Session` before using it for
        the :class:`!STL`.

        The record that contains the :class:`!STL` should not be modified
        after calling entry_init().
        """
        assert_type(Record, record)
        assert_type(Template, template)
        if template.template_id == 0:
            raise Exception("Template has a template_id of 0.")
        record.check_template(template, True)
        self.set_record(record)
        _pyfixbuf.fbSTLBase.entryInit(
            self, template._unwrap(), template.template_id, count)
        self.initialized = True
        self.info_model = record.model
        self._exp_template = template

    def __iter__(self):
        """
        Implements the method to return an Iterator over the :class:`Record`
        objects in the :class:`!STL`.

        Example:

        >>> for record in stl:
        ...     print(record.as_dict())
        """
        return self

    def next(self):
        """Returns the next :class:`Record` in the :class:`!STL`"""
        return self.__next__()

    def __next__(self):
        """Returns the next :class:`Record` in the :class:`!STL`"""
        if self.rec == None:
            if self.session:
                tid = self.template_id
                tmpl = self.session.get_template(tid)
                self.rec = Record(self.info_model, tmpl)
                self.rec.session = self.session
            else:
                raise Exception("No Record or Session set for STL")
        _pyfixbuf.fbSTLBase.getNext(self, self.rec)
        return self.rec

    def clear(self):
        """
        Clears all entries in the list.  Nested elements should be accessed
        and freed before calling this method.  Frees any memory previously
        allocated for the list.
        """
        _pyfixbuf.fbSTLBase.clear(self)

    def __len__(self):
        """
        Implements the built-in `len()` method for :class:`!STL` instances:
        Returns the number of :class:`Record` objects in the :class:`!STL`.
        """
        return self.count

    def __getitem__(self, item):
        """
        Implements the evaluation of ``stl[item]`` for :class:`!STL`
        instances:

        If *item* is an :class:`int`, returns the :class:`Record` at that
        positional index.

        If *item* is a :class:`str`, finds the :class:`InfoElement` with the
        name *item* in the :class:`!STL`'s :class:`Template` and returns the
        value for that element in the most recently accessed :class:`Record`
        in the :class:`!STL`.

        Returns ``None`` if *item* has an unexpected type.
        """
        if self.rec == None:
            if self.session:
                tid = self.template_id
                tmpl = self.session.get_template(tid)
                self.rec = Record(self.info_model, tmpl)
                self.rec.session = self.session
            else:
                raise Exception("No Record or Session set for STL")
        if (isinstance(item, str) or isinstance(item, tuple)):
            return self.rec[item]
        elif (isinstance(item, int)):
            newRecord = Record(self.rec.model, record=self.rec)
            newRecord.session=self.session
            _pyfixbuf.fbSTLBase.getIndexedEntry(self, newRecord, item)
            return newRecord

    def __setitem__(self, index, value):
        """
        Implements assignment to ``stl[index]`` for :class:`!STL` instances:
        Sets the :class:`Record` at position *index* in this :class:`!STL` to
        *value*.

        If this :class:`!STL` was not previously initialized via
        :meth:`entry_init`, it is initialized with the given :class:`Record`'s
        :class:`Template` and a count of 1.
        """
        assert_type(Record, value)
        if (self.initialized == False):
            if (value.template):
                self.entry_init(value, value.template, 1)
            else:
                raise Exception(
                    "STL has not been initialized. Use entry_init().")
        if self.session and self.template_id:
            tmpl = self.session.get_template(tid)
            value.check_template(tmpl)
        elif self._exp_template:
            value.check_template(self._exp_template)
        self.rec = value
        _pyfixbuf.fbSTLBase.setIndexedEntry(self, index, value)

    def iter_records(self, tmpl_id=0):
        """
        Returns an Iterator over the :class:`!STL`\ 's records, including
        records in any nested :class:`STML` or :class:`!STL`. If a template
        ID is passed, returns an iterator over all the :class:`Record` objects
        with a template ID matching the passed value.

        Example:

        >>> for record in stl.iter_records():
        ...     print(record.template.template_id)
        ...
        """
        for record in self:
            for sub_rec in yield_record_subrecords(record, tmpl_id):
                yield sub_rec


class BL(_pyfixbuf.fbBLBase):
    """
    A :class:`!BL` represents a basicList.

    A basicList is a list of zero or more instances of an Information Element.

    A basicList can be initialized through a :class:`Record` via
    :meth:`~Record.init_basic_list` or by creating a :class:`!BL` object.

    The constructor requires an :class:`InfoModel` *model*, and a
    :class:`InfoElementSpec`, :class:`InfoElement`, or string *element*.
    Additionally, it takes an optional integer *count* which represents
    the number of elements in the list, and an optional integer
    *semantic* to express the relationship among the list items.

    All basicLists in a :class:`!Record` must be initialized (even to 0)
    before appending a :class:`!Record` to a :class:`Buffer`.

    Examples::

    >>> rec.add_element("basicList", BASICLIST)
    >>> rec.add_element("basicList2", BASICLIST)
    >>> bl = BL(model, "sourceTransportPort", 2)
    >>> bl[0] = 80
    >>> bl[1] = 23
    >>> rec["basicList"] = bl
    >>> rec.init_basic_list("basicList2", 4, "octetTotalCount")
    >>> rec["basicList2"] = [99, 101, 104, 23]
    """
    def __init__(self, model, element, count=0, semantic=0):
        assert_type(InfoModel, model)
        self.model = model
        self.list = []
        assert_type(InfoModel, model)
        if (isinstance(element, InfoElement)):
            _pyfixbuf.fbBLBase.__init__(self, element, count, semantic)
        elif (isinstance(element, str)):
            ie = model.get_element(element)
            _pyfixbuf.fbBLBase.__init__(self, ie, count, semantic)
        elif (isinstance(element, InfoElementSpec)):
            ie = model.get_element(element.name)
            _pyfixbuf.fbBLBase.__init__(self, ie, count, semantic)
        else:
            _pyfixbuf.fbBLBase.__init__(self)

    def _fill_bl_list(self):
        if (len(self) and len(self.list) == 0):
            blist = _pyfixbuf.fbBLBase.getitems(self)
            ty = self.element.type
            tlist = []
            if ty == DataType.IP4ADDR:
                for item in blist:
                    tlist.append(ipaddress.IPv4Address(item))
            elif ty == DataType.IP6ADDR:
                for item in blist:
                    tlist.append(ipaddress.IPv6Address(bytes(item)))
            elif ty == DataType.MAC_ADDR:
                for item in blist:
                    tlist.append(bytes_to_macaddr(item))
            else:
                self.list = blist
            if (len(tlist)):
                self.list = tlist

    def __len__(self):
        """
        Implements the built-in `len()` method for :class:`!BL` instances:
        Returns the number of entries in the basicList.

        Example:

        >>> bl = BL(model, "sourceTransportPort", 5)
        >>> len(bl)
        5
        """
        return self.count

    def __iter__(self):
        """
        Implements the method to return an Iterator over the items in the
        :class:`!BL`.

        Example:

        >>> bl = record['basicList']
        >>> bl.element.name
        'httpContentType'
        >>> for v in bl:
        ...     print(v)
        ...
        'text/xml'
        'text/plain'
        """
        self._fill_bl_list()
        for item in self.list:
            yield item

    def __getitem__(self, index):
        """
        Implements the evaluation of ``bl[index]`` for :class:`!BL` instances:
        Returns the value at position *index* in the basicList.

        Example:

        >>> bl = record['basicList']
        >>> bl.element.name
        'httpContentType'
        >>> print(bl[0])
        'text/xml'
        """
        self._fill_bl_list()
        return self.list[index]

    def __setitem__(self, index, value):
        """
        Implements assignment to ``bl[index]`` for :class:`!BL` instances:
        Sets the value at position *index* in the basicList to *value*.
        """
        if (index > len(self)):
            raise IndexError("Index %d is out of range" % index)
        if (self.element == None):
            raise Exception("BL must be initialized with InfoElement"
                            " before setting items.")
        ty = self.element.type
        if ty == DataType.IP6ADDR:
            value = ip6addr_to_bytes(value)
        if ty == DataType.IP4ADDR:
            value = ip4addr_to_int(value)
        if ty == DataType.MAC_ADDR:
            value = macaddr_to_bytes(value)
        _pyfixbuf.fbBLBase.setitems(self, index, value)

    def copy(self, other):
        """
        Copies the items in the list *other* to this :class:`!BL`, stopping
        when *other* is exhausted or the length of the :class:`!BL` is
        reached.

        Raises :exc:`TypeError` when *other* does not support iteration.
        """
        for i,value in zip(range(len(self)), other):
            self[i] = value

    def __contains__(self, item):
        """
        Implements the ``in`` operator for :class:`!BL` instances: Tests
        whether this :class:`!BL` contains an element whose value is *item*.

        Example:

        >>> bl = record['basicList']
        >>> bl.element.name
        'httpContentType'
        >>> 'text/plain' in bl
        True
        """
        self._fill_bl_list()
        return item in self.list

    def __str__(self):
        self._fill_bl_list()
        return str(self.list)

    def __eq__(self, other):
        """
        Determines whether *other* is equal to this :class:`!BL`.

        Returns ``True`` when *other* and this basicList contain the same
        elements in the same order.  Returns ``False`` otherwise.  Raises
        :exc:`TypeError` when *other* does not support iteration.

        Example:

        >>> bl = record['basicList']
        >>> bl.element.name
        'httpContentType'
        >>> bl == ['text/xml', 'text/plain']
        True
        """
        if (other == None):
            return False
        self._fill_bl_list()
        for x,y in zip_longest(iter(self.list), other):
            if x != y or x is None:
                return False
        return True


class Listener(_pyfixbuf.fbListenerBase):
    """
    Creates a :class:`!Listener` given the *session*, *hostname*,
    *transport*, and *port*.

    *session* must be a valid instance of :class:`Session`.

    *hostname* is a string containing the address to bind to, a hostname
    or an IP Address.  If *hostname* is ``None``, all addresses are used.

    *transport* is the transport protocol; it may contain ``"tcp"`` (the
    default) ``"udp"``, or ``"sctp"``.  (Using ``"sctp"`` raises an exception
    unless libfixbuf has been compiled with SCTP support.)

    *port* is the port number to listen on, and it must not be less than
    1024.  The default is 4739.

    Examples::

    >>> listener = Listener(session, hostname="localhost", port=18000)
    """
    def __init__(self, session, hostname, transport="tcp", port=4739):
        assert_type(Session, session)
        self.session = session
        if not isinstance(port, int):
            port = int(port)
        if (port < 1024 or port == 0):
            raise Exception("Invalid Port: Port should be greater than 1024")
        _pyfixbuf.fbListenerBase.__init__(self)
        _pyfixbuf.fbListenerBase.allocListener(self, transport, hostname,
                                               str(port), session)

    def wait(self, record=None):
        """
        Waits for a connection on the set host and port.  Once a connection is
        made, returns a newly allocated :class:`Buffer`.

        If a *record* is given to :meth:`wait` then the returned
        :class:`Buffer` will already be associated with a :class:`Record`.

        If no *record* is given, you must use :meth:`~Buffer.set_record`
        on the :class:`!Buffer` before accessing the elements.

        After receiving the :class:`!Buffer` you must set the internal
        template on the returned :class:`!Buffer` using
        :meth:`~Buffer.set_internal_template` before accessing the data.

        Examples::

        >>> buf = listener.wait()
        >>> buf.set_record(my_rec)
        >>> buf.set_internal_template(999)
        >>> for data in buf:
        >>> ...
        """
        # We should update wait() to have an `auto` argument.  Better: We
        # should change wait() to pass whatever args it is given to
        # Buffer.__init__().
        try:
            buf = Buffer(record)
            buf._init_listener(self.session)
            _pyfixbuf.fbListenerBase.listenerWait(self, buf, self.session)
            return buf
        except (KeyboardInterrupt, SystemExit):
            raise Exception("Stopped By User")


###################
# Misc Functions
###################

# Takes a user's value for an IPv4 address and converts it to int
def ip4addr_to_int(x):
    if isinstance(x, ipaddress.IPv4Address):
        x = int(x)
    elif isinstance(x, (int, long)):
        if x < 0 or x > 0xffffffff:
            raise Exception("Integer IP4ADDR value falls outside valid range: "
                            + str(x))
    elif isinstance(x, bytearray):
        if len(x) != 4:
            raise Exception(
                "Bytearray IP4ADDR value has length %d, expected 4: %s" %
                (len(x), str(x)))
        x = int(ipaddress.IPv4Address(bytes(x)))
    elif isinstance(b"python2.7", str):
        # python 2.x
        if isinstance(x, (str, unicode, bytes)):
            x = int(ipaddress.IPv4Address(x.decode()))
    elif isinstance(x, (str, bytes)):
        x = int(ipaddress.IPv4Address(x))
    return x

# Takes a user's value for an IPv6 address and converts it to bytearray
def ip6addr_to_bytes(x):
    if isinstance(x, ipaddress.IPv6Address):
        x = x.packed
    elif isinstance(x, (int, long)):
        if x < 0:
            raise Exception("Integer IP6ADDR is negative: " + str(x))
        x = ipaddress.IPv6Address(x).packed
    elif isinstance(x, bytearray):
        if len(x) != 16:
            raise Exception(
                "Bytearray IP6ADDR value has length %d, expected 16: %s" %
                (len(x), str(x)))
        x = bytes(x)
    elif isinstance(b"python2.7", str):
        # python 2.x
        if isinstance(x, (str, unicode, bytes)):
            x = ipaddress.IPv6Address(x.decode()).packed
    elif isinstance(x, (str, bytes)):
        x = ipaddress.IPv6Address(x).packed
    return bytearray(x)

re_mac_address = re.compile(r"\A" + ":".join(["[0-9a-f][0-9a-f]"] * 6) + r"\Z",
                            re.IGNORECASE)

# Takes a user's value for a MAC address and converts it to bytearray
def macaddr_to_bytes(value):
    if isinstance(value, bytearray):
        if len(value) != 6:
            raise Exception("Bytearray MacAddress value has incorrect "
                            "length (6 expected, got %d): %s" %
                            (len(value), str(value)))
    elif isinstance(value, (int, long)):
        if value < 0 or value > 0xffffffffffff:
            raise Exception("Integer MAC_ADDR value falls outside "
                            "valid range: %s" % str(value))
        value = bytearray(binascii.a2b_hex('%012x' % value))
    elif isinstance(b"python2.7", str):
        # python 2.x
        if isinstance(value, (str, unicode, bytes)):
            if not re_mac_address.match(value):
                raise Exception("String MAC_ADDR value has wrong form:" + value)
            value = bytearray(binascii.a2b_hex(value.replace(':', '')))
    elif isinstance(value, bytes):
        if len(value) != 6:
            raise Exception("Bytes MacAddress value has incorrect "
                            "length (6 expected, got %d): %s" %
                            (len(value), str(value)))
        value = bytearray(value)
    elif isinstance(value, str):
        if not re_mac_address.match(value):
            raise Exception("String MAC_ADDR value has wrong form:" + value)
        value = bytearray(binascii.a2b_hex(value.replace(':', '')))
    return value

# Prints the "xx:xx:xx:xx:xx:xx" format for a MAC addr bytearray
def bytes_to_macaddr(packed):
    return ':'.join('%02x' % b for b in packed)

def create_stml_from_list(item_list):
    listlen = len(item_list)
    tid = dict()
    tid2list=[]
    difflist = 0
    for i,item in enumerate(item_list):
        assert_type(Record, item)
        template = item.template
        if template:
            if template not in tid:
                newlist = []
                newlist.append(item)
                tid[template] = difflist
                tid2list.append(newlist)
                difflist += 1
            else:
                elist = tid2list[tid[template]]
                elist.append(item)
                tid2list[tid[template]] = elist
        else:
            raise Exception("No template associated with item " + str(i+1))
    stml = STML(type_count=len(tid))
    for i,lt in enumerate(tid2list):
        stml[i] = lt
    return stml


FILTER = ''.join([(len(repr(chr(x)))==3) and chr(x) or '.' for x in range(256)])
def pyfix_hex_dump(src, length=8):
    newstr=[]
    for i in range(0, len(src), length):
        s = src[i:i+length]
        hex = ' '.join(["%02X" % ord(x) for x in s])
        printchar = s.translate(FILTER)
        newstr.append("%04X   %-*s   %s\n" % (i, length*3, hex, printchar))
    return ''.join(newstr)


# Local Variables:
# indent-tabs-mode:nil
# fill-column:78
# End:
