#! /usr/bin/env python
## ------------------------------------------------------------------------
## sample_exporter_collector_blind.py
##
## sample IPFIX collector & exporter (aka mediator) using pyfixbuf.
## created to be used with the sample_exporter.py script.
## Processes the ipfix file created by sample_exporter.py and writes
## to a text file.  This is different from sample_exporter_collector.py
## in that it has no idea what to expect.
## ------------------------------------------------------------------------
## Copyright (C) 2013-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## Authors: Emily Sarneso
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt

from __future__ import print_function
import os
import sys
import pyfixbuf
import pyfixbuf.cert
from pyfixbuf import DataType

# Test that the argument number is correct
if (len(sys.argv) < 2 or sys.argv[1] in ['-h', '--help']):
    print("Must supply an IPFIX file to read and (optional) text file "
          "to write to.")
    sys.exit()


#create the information model with the standard IPFIX elements
infomodel = pyfixbuf.InfoModel()

# add YAF's IPFIX elements
pyfixbuf.cert.add_elements_to_model(infomodel)

# create the collector
collector = pyfixbuf.Collector()

# create the IPFIX file to read from
collector.init_file(sys.argv[1])

# create the session
session = pyfixbuf.Session(infomodel)

# create the buffer for the collector, set auto to True to
# automatically generate templates
buf = pyfixbuf.Buffer(auto=True)

# make the buffer an import buffer
buf.init_collection(session, collector)

# set auto insert on buffer in case we receive any Info Element Options Recs
buf.auto_insert()

if (len(sys.argv) > 2 and sys.argv[2] != "-"):
    outFile = open(sys.argv[2], "w")
else:
    outFile = sys.stdout

flowcount = 0

dontprintlist = [DataType.BASIC_LIST, DataType.SUB_TMPL_LIST,
                 DataType.SUB_TMPL_MULTI_LIST, DataType.OCTET_ARRAY]

def do_record(rec):
    for field in rec.iterfields():
        ty = field.ie.type
        if ty not in dontprintlist:
            outFile.write("%s: %s\n" % (field.name, str(field.value)))
        elif ty == DataType.BASIC_LIST:
            do_bl(field)
        elif ty == DataType.SUB_TMPL_LIST:
            do_stl(rec, field)
        elif ty == DataType.SUB_TMPL_MULTI_LIST:
            do_stml(rec, field)
        elif field.name == "paddingOctets":
            outFile.write("%s: (len %d)\n" % (field.name, len(field.value)))
        else:
            # handle octet array that is not padding
            outFile.write("%s: %s\n" % (field.name, str(field.value)))

def do_bl(field):
    bl = field.value
    name = bl.element.name
    outFile.write("%s:\n--- BL (ie = %s, count = %d) ---\n" %
                  (field.name, name, len(bl)))
    count = 0
    for item in bl:
        outFile.write("%s[%d]: %s\n" % (name, count, str(item)))
        count += 1
    bl.clear()

def do_stl(parent, field):
    #stl = parent[(field.name, field.instance)]
    stl = field.value
    outFile.write("%s:\n--- STL ---\n" % field.name)
    recs = 0
    for record in stl:
        outFile.write("-- REC %d --\n" % recs)
        do_record(record)
        recs += 1
    stl.clear()

def do_stml(parent, field):
    #stml = parent[(field.name, field.instance)]
    stml = field.value
    outFile.write("%s:\n" % field.name)
    subs = 0
    for entry in stml:
        outFile.write("--- STML %d ---\n" % subs)
        recs = 0
        for record in entry:
            outFile.write("-- ENTRY %d --\n" % recs)
            do_record(record)
            recs += 1
        subs += 1
    stml.clear()


for data in buf:
    do_record(data)
    outFile.write("------------------------------\n")
    flowcount += 1

sys.stderr.write("Processed %d flows\n" % flowcount)
