#! /usr/bin/env python
## ------------------------------------------------------------------------
## yaf_mediator_new.py
##
## Example of a true yaf mediator.  Can act as an IPFIX listener or file
## reader.  This will print to stdout or the provided text file.
## This script can handle any field that YAF exports.
## This script is different than yaf_mediator.py in that it doesn't define
## any of the templates or records before reading.
## ------------------------------------------------------------------------
## Copyright (C) 2014-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## Authors: Emily Sarneso
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt
## ------------------------------------------------------------------------

from __future__ import print_function
import sys
import os
import pyfixbuf
import pyfixbuf.cert
from time_utils import make_datetime

# Test that the argument number is correct
if (len(sys.argv) < 2 or sys.argv[1] in ['-h', '--help']):
    print("Example Usage: yaf_mediator_new.py hostname [port] [protocol] [outfile]")
    print("OR: yaf_mediator_new.py infile.ipfix [outfile]")
    sys.exit()

infomodel = pyfixbuf.InfoModel()
pyfixbuf.cert.add_elements_to_model(infomodel)

tmpl = pyfixbuf.Template(infomodel)

statstmpl = pyfixbuf.Template(infomodel)

exportertmpl = pyfixbuf.Template(infomodel)

data_list = [
    pyfixbuf.InfoElementSpec("flowStartMilliseconds"),
    pyfixbuf.InfoElementSpec("flowEndMilliseconds"),
    pyfixbuf.InfoElementSpec("octetTotalCount"),
    pyfixbuf.InfoElementSpec("reverseOctetTotalCount"),
    pyfixbuf.InfoElementSpec("packetTotalCount"),
    pyfixbuf.InfoElementSpec("reversePacketTotalCount"),
    pyfixbuf.InfoElementSpec("sourceIPv4Address"),
    pyfixbuf.InfoElementSpec("destinationIPv4Address"),
    pyfixbuf.InfoElementSpec("sourceIPv6Address"),
    pyfixbuf.InfoElementSpec("destinationIPv6Address"),
    pyfixbuf.InfoElementSpec("sourceTransportPort"),
    pyfixbuf.InfoElementSpec("destinationTransportPort"),
    pyfixbuf.InfoElementSpec("flowAttributes"),
    pyfixbuf.InfoElementSpec("reverseFlowAttributes"),
    pyfixbuf.InfoElementSpec("protocolIdentifier"),
    pyfixbuf.InfoElementSpec("initialTCPFlags"),
    pyfixbuf.InfoElementSpec("reverseInitialTCPFlags"),
    pyfixbuf.InfoElementSpec("unionTCPFlags"),
    pyfixbuf.InfoElementSpec("reverseUnionTCPFlags"),
    pyfixbuf.InfoElementSpec("tcpSequenceNumber"),
    pyfixbuf.InfoElementSpec("vlanId"),
    pyfixbuf.InfoElementSpec("flowEndReason"),
    pyfixbuf.InfoElementSpec("silkAppLabel"),
    pyfixbuf.InfoElementSpec("subTemplateMultiList")]

stats_list = [
    pyfixbuf.InfoElementSpec("exportedFlowRecordTotalCount"),
    pyfixbuf.InfoElementSpec("packetTotalCount"),
    pyfixbuf.InfoElementSpec("droppedPacketTotalCount"),
    pyfixbuf.InfoElementSpec("ignoredPacketTotalCount")]


# add the elements we want in collection template
tmpl.add_spec_list(data_list)

# add stats elements we want from the incoming stats template
statstmpl.add_spec_list(stats_list)

# create the sessions for the collector & exporter
session = pyfixbuf.Session(infomodel)

# add the appropriate templates to each session
session.add_internal_template(tmpl, template_id=876)

session.add_internal_template(statstmpl, template_id=900)

collector = None
listener = None
transport = "tcp"
port = "18000"

# get the file from the command line arguments to read
if (sys.argv[1] == "-" or os.path.isfile(sys.argv[1])):
    # create the collector
    collector = pyfixbuf.Collector()
    collector.init_file(sys.argv[1])
    if len(sys.argv) > 2:
        outFile = open(sys.argv[2], 'w')
    else:
        outFile = sys.stdout
else:
    if (len(sys.argv) > 3):
        transport = sys.argv[3]
    if (len(sys.argv) > 2):
        port = sys.argv[2]
    if (len(sys.argv) > 4):
        outFile = open(sys.argv[4], 'w')
    else:
        outFile = sys.stdout
    listener = pyfixbuf.Listener(session, hostname=sys.argv[1],
                                 transport=transport.lower(),
                                 port=port)


# create the input record from the input template since there are no
# duplicate elements
rec = pyfixbuf.Record(infomodel, tmpl)

# create buffers for input and export
if (collector):
    buf = pyfixbuf.Buffer(rec)

    buf.init_collection(session, collector)

# set the internal template on the input buffer
    buf.set_internal_template(876)

flowcount = 0

if (listener):
    buf = listener.wait()
    buf.set_record(rec)
    buf.set_internal_template(876)

while True:
    try:
        data = next(buf)
    except StopIteration:
        if (not(listener)):
            break
        else:
            buf = listener.wait()
            buf.set_record(rec)
            buf.set_internal_template(876)
            continue

    flowcount += 1

    outFile.write("----flow: %d ----\n" % flowcount)

    for field in data.iterfields():
        if field.ie.type == pyfixbuf.DataType.MILLISECONDS:
            outFile.write(field.name + ": " +
                          str(make_datetime(field.value/1000)) + "\n")
        elif field.ie.type != pyfixbuf.SUBTEMPLATEMULTILIST and field.value:
            outFile.write(field.name + ": " + str(field.value) + "\n")
    # get items from STML
    stml = data["subTemplateMultiList"]
    for entry in stml:
        if "payload" in entry:
            # this is the payload template from YAF
            result = pyfixbuf.pyfix_hex_dump(entry["payload"], length=16)
            outFile.write(result + "\n")
            if "reversePayload" in entry:
                outFile.write("reverse payload: ")
                result = pyfixbuf.pyfix_hex_dump(entry["reversePayload"], length=16)
                outFile.write(result + "\n")
        elif "tcpSequenceNumber" in entry:
            for record in entry:
                for field in record.iterfields():
                    outFile.write(field.name + ": " + str(field.value) + "\n")
        elif "sourceMacAddress" in entry:
            outFile.write("sourceMacAddress: " + entry["sourceMacAddress"] + "\n")
            outFile.write("destinationMacAddress: " + entry["destinationMacAddress"] + "\n")
        elif entry.template_id == 0xC600 or entry.template_id == 0xC610:
            # this is the HTTP Template from YAF
            for http in entry:
                i = 0
                while i < len(http):
                    # http[i] is a BASICLIST - could iterate through it
                    if (len(http[i])):
                        outFile.write("\t" + http[i].element.name + ": " + str(http[i]) + "\n")
                    i += 1
                http.clear_all_lists()
        elif entry.template_id == 0xCA0A:
            numcert = 0
            for ssl in entry:
                for field in ssl.iterfields():
                    if (field.ie.type not in (
                            pyfixbuf.SUBTEMPLATELIST, pyfixbuf.BASICLIST)):
                        outFile.write("\t" + field.name + ": " +
                                      str(field.value) + "\n")
                bl = ssl["basicList"]
                outFile.write("\t" + str(bl) + "\n")
                stl = ssl["subTemplateList"]
                for cert in stl:
                    numcert += 1
                    outFile.write("\t == cert: %d ==\n" % numcert)
                    issuer = cert[0]
                    subject = cert[1]
                    outFile.write("\tsslcertvaliditynotbefore: " + cert["sslCertValidityNotBefore"] + "\n")
                    outFile.write("\tsslcertvaliditynotafter: " + cert["sslCertValidityNotAfter"] + "\n")
                    for issue in issuer:
                        outFile.write("\t" + str(issue["sslObjectType"]) + ": " + issue["sslObjectValue"] + "\n")
                    issue.clear()
                    for sub in subject:
                        outFile.write( "\t" +  str(sub["sslObjectType"]) + ": " + sub["sslObjectValue"] + "\n")
                    subject.clear()
                stl.clear()
        elif entry.template_id == 0xCE00:
            # this is the DNS template from YAF
            for record in entry:
                stl = record["subTemplateList"]
                for dnsrec in stl:
                    for field in dnsrec.iterfields():
                        if field.ie.type != pyfixbuf.SUBTEMPLATELIST:
                            outFile.write("\t" +  field.name + ": "
                                          + str(field.value) + "\n")
                        else:
                            stl2 = dnsrec["subTemplateList"]
                            if (dnsrec['dnsRRType'] == 1):
                                for arec in stl2:
                                    outFile.write("\tA record: " + arec["sourceIPv4Address"] + "\n")
                            elif (dnsrec['dnsRRType'] == 2):
                                for ns in stl2:
                                    outFile.write("\tNS Record: " + ns["dnsNSDName"] + "\n")
                            elif (dnsrec['dnsRRType'] == 5):
                                for cn in stl2:
                                    outFile.write("\tCNAME rec: " + cn['dnsCNAME'] + "\n")
                            elif (dnsrec['dnsRRType'] == 12):
                                for cn in stl2:
                                    outFile.write("\tPTR rec: " + cn['dnsPTRDName'] + "\n")
                            elif (dnsrec['dnsRRType'] == 15):
                                for mx in stl2:
                                    for f in mx.iterfields():
                                        outFile.write("\t" +  f.name + ": " + str(f.value) + "\n")
                            elif (dnsrec['dnsRRType'] == 16):
                                for cn in stl2:
                                    outFile.write( "\tTXT rec: " +cn['dnsTXTData'] + "\n")
                            elif (dnsrec['dnsRRType'] == 28):
                                for cn in stl2:
                                    outFile.write( "\tAAAA rec: " + cn['sourceIPv6Address'] + "\n")
                            elif (dnsrec['dnsRRType'] == 6):
                                for soa in stl2:
                                    for field in soa.iterfields():
                                        outFile.write("\t" +  field.name + ": " + str(field.value) + "\n")
                            stl2.clear()
                    outFile.write("\t----\n")
                stl.clear()
        elif entry.template_id == 0xC200:
             # this is IRC
             for irc in entry:
                 for field in irc.iterfields():
                     outFile.write( "\t" +  field.name + ": " + str(field.value) + "\n")
                 irc.clear_all_lists()
        elif entry.template_id == 0xC300:
            # pop3
            for pop in entry:
                for field in pop.iterfields():
                    outFile.write( "\t" +  field.name + ": " + str(field.value) + "\n")
                pop.clear_all_lists()
        elif entry.template_id == 0xC500:
            #slp
            for slp in entry:
                for field in slp.iterfields():
                    outFile.write( "\t" +  field.name + ": " + str(field.value) + "\n")
        elif entry.template_id == 0xC700:
            #ftp
            for ftp in entry:
                for field in ftp.iterfields():
                    outFile.write( "\t" + str(field.name) + ": " + str(field.value) + "\n")
                ftp.clear_all_lists()
        elif entry.template_id == 0xC800:
            #imap
            for imap in entry:
                for field in imap.iterfields():
                    outFile.write( "\t" + str(field.name) + ": " + str(field.value) + "\n")
                imap.clear_all_lists()
        elif entry.template_id == 0xC900:
            #rtsp
            for rtsp in entry:
                for field in rtsp.iterfields():
                    outFile.write( "\t" + field.name + ": " + str(field.value) + "\n")
                rtsp.clear_all_lists()
        elif entry.template_id == 0xCA00:
            #sip
            for sip in entry:
                for fields in sip.iterfields():
                    outFile.write( "\t" + field.name + ": " + str(field.value) + "\n")
                sip.clear_all_lists()
        elif entry.template_id == 0xCB00:
            #smtp
            for smtp in entry:
                for fields in smtp.iterfields():
                    outFile.write( "\t" + str(field.name) + ": " + str(field.value) + "\n")
                smtp.clear_all_lists()
        elif entry.template_id == 0xCC00:
            #ssh
            for ssh in entry:
                for field in ssh.iterfields():
                    outFile.write( "\t" + field.name + ": " + str(field.value) + "\n")
                ssh.clear_all_lists()
        elif "tftpFilename" in entry:
            for tftp in entry:
                for field in tftp.iterfields():
                    outFile.write( "\t" + field.name + ": " + str(field.value) + "\n")
        elif entry.template_id == 0xC202:
            stl = entry["subTemplateList"]
            for dnp in stl:
                outFile.write("\t DNP3 srcAddress: %d\n" % dnp["dnp3SourceAddress"])
                outFile.write("\t DNP3 dstAddress: %d\n" % dnp["dnp3DestinationAddress"])
                outFile.write("\t DNP3 function: %d\n" % dnp["dnp3Function"])
                outFile.write("\t DNP3 object: \n")
                result = pyfixbuf.pyfix_hex_dump(dnp["dnp3ObjectData"], length=16)
                outFile.write( result + "\n")
            stl.clear()
        elif entry.template_id == 0xC204:
            md = entry["modbusData"]
            for i in md:
                outFile.write("modbusData:\n")
                result = pyfixbuf.pyfix_hex_dump(i, length=16)
                outFile.write( result + "\n")
            md.clear()
        elif entry.template_id == 0xC205:
            for enip in entry["enipData"]:
                outFile.write("Ethernet/IP Data:" + "\n")
                result = pyfixbuf.pyfix_hex_dump(enip, length=16)
                outFile.write(result + "\n")
            entry["enipData"].clear()
        elif entry.template_id == 0xC206:
            for rtp in entry:
                for field in rtp.iterfields():
                    outFile.write("\t" + field.name + ": " + str(field.value) + "\n")
        elif entry.template_id == 0xCD00:
            for nntp in entry:
                for field in nntp.iterfields():
                    outFile.write("\t" + field.name + ": " + str(field.value) + "\n")
        elif entry.template_id == 0xCE0C:
            stl = entry["subTemplateList"]
            stl.set_record(mysqlinner)
            outFile.write("\t MySQL user name: %s\n" % entry["mysqlUsername"])
            for my in stl:
                outFile.write("\t MySQL code: %d: %s\n" % (my["mysqlCommandCode"], my["mysqlCommandText"]))
            stl.clear()

    # clear the stml
    stml.clear()

    # check to see if the next template is a stats template
    try:
        tmpl_next = buf.next_template()
    except StopIteration:
        tmpl_next = None

    if ( tmpl_next and tmpl_next.scope ):
        # set the interal template to the stats template
        buf.set_internal_template(900)
        # get the record
        stats = buf.next_record()
        #if (stats != None):
        # print all the items in stats
        for field in stats.iterfields():
            outFile.write(field.name + ": " + str(field.value) + "\n")
            # print all the items in stats
        # set the template back to the data template
        buf.set_internal_template(876)


sys.stdout.write("Processed " + str(flowcount) + " flows \n")
