/**
 ** _pyfixbuf.c
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2006-2022 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell, Dan Ruef, Emily Ecoff
 ** ------------------------------------------------------------------------
 ** @OPENSOURCE_HEADER_START@
 ** Use of the libfixbuf system and related source code is subject to the terms
 ** of the following licenses:
 **
 ** GNU Lesser GPL (LGPL) Rights pursuant to Version 2.1, February 1999
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
 **
 ** NO WARRANTY
 **
 ** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
 ** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
 ** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
 ** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
 ** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
 ** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
 ** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
 ** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
 ** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
 ** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
 ** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
 ** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
 ** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
 ** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
 ** DELIVERABLES UNDER THIS LICENSE.
 **
 ** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
 ** Mellon University, its trustees, officers, employees, and agents from
 ** all claims or demands made against them (and any related losses,
 ** expenses, or attorney's fees) arising out of, or relating to Licensee's
 ** and/or its sub licensees' negligent use or willful misuse of or
 ** negligent conduct or willful misconduct regarding the Software,
 ** facilities, or other rights or assistance granted by Carnegie Mellon
 ** University under this License, including, but not limited to, any
 ** claims of product liability, personal injury, death, damage to
 ** property, or violation of any laws or regulations.
 **
 ** Carnegie Mellon University Software Engineering Institute authored
 ** documents are sponsored by the U.S. Department of Defense under
 ** Contract FA8721-05-C-0003. Carnegie Mellon University retains
 ** copyrights in all material produced under this contract. The U.S.
 ** Government retains a non-exclusive, royalty-free license to publish or
 ** reproduce these documents, or allow others to do so, for U.S.
 ** Government purposes only pursuant to the copyright license under the
 ** contract clause at 252.227.7013.
 ** Government purposes only pursuant to the copyright license under the
 ** contract clause at 252.227.7013.
 **
 ** @OPENSOURCE_HEADER_END@
 ** ------------------------------------------------------------------------
 */
#define PY_SSIZE_T_CLEAN 1
#include <Python.h>

#include <stdint.h>
#include <fixbuf/public.h>
#include <glib.h>

#if FIXBUF_CHECK_VERSION(2, 99, 0)
/* define macros to mimic the fixbuf-2.4 API */
#define fbInfoModelTypeInfoRecord(_r)                   \
    fbTemplateIsMetadata(_r, FB_TMPL_IS_META_ELEMENT)
#define fbSessionAddTemplate(_a, _b, _c, _d, _e)        \
    fbSessionAddTemplate(_a, _b, _c, _d, NULL, _e)
#else
/* define macros to mimic the fixbuf-3.0.0 API */
#define fbTemplateGetFieldByPosition    fbTemplateGetIndexedIE
#define fbTemplateField_t               fbInfoElement_t
#define fbTemplateFieldGetIE(_tf)       ((_tf)->ref.canon)
#define fbTemplateFieldGetPEN(_tf)      ((_tf)->ent)
#define fbTemplateFieldGetId(_tf)       ((_tf)->num)
#define fbTemplateFieldGetLen(_tf)      ((_tf)->len)
#define fbTemplateFieldGetName(_tf)     ((_tf)->ref.canon->ref.name)
#define fbInfoElementGetName(_ie)       ((_ie)->ref.name)
#define fbBasicListGetElementLength(_bl) (fbBasicListGetInfoElement(_bl)->len)
#if !FIXBUF_CHECK_VERSION(2, 3, 0)
/* add support for fixbuf-2.2 */
#define fbBasicListCountElements(_L)            ((_L)->numElements)
#define fbSubTemplateListCountElements(_L)      ((_L)->numElements)
#define fbSubTemplateMultiListCountElements(_L) ((_L)->numElements)
#endif  /* #if >= fixbuf-2.3.0 */
#endif  /* #else of #if >= fixbuf-2.99.0 */


/* quiet warnings about unused parameters */
#define FBPY_UNUSED_PARAM(_x)  (void)_x;

#define GET_GLOBALS(m) (&fixbufpy_globals)
#define GLOBALS GET_GLOBALS(PyState_FindModule(&fixbufpy_globals))
#define MAX_NAME 200
#define MAX_ELEMENTS 100
#define IPV6_TYPE 6

#if PY_MAJOR_VERSION >= 3
#  define IS_BYTE(o)                (PyByteArray_Check(o))
#  define IS_INT(o)                 (PyLong_Check(o) && !PyBool_Check(o))
#  define IS_STRING(o)              (PyUnicode_Check(o))

#  define PyInt_FromLong(o)         PyLong_FromLong(o)
#  define PyInt_AsLong(o)           PyLong_AsLong(o)
#  define PyNumber_Int(o)           PyNumber_Long(o)

#  define PyString_AsString(o)      PyUnicode_AsUTF8(o)
#  define PyString_FromStringAndSize(o1, o2) PyUnicode_FromStringAndSize(o1, o2)

#  define fbpyUnicode_Format(o1, o2)  PyUnicode_Format(o1, o2)
#  define fbpyUnicode_FromFormat      PyUnicode_FromFormat
#  define fbpyUnicode_FromString(o)   PyUnicode_FromString(o)

#else
#define IS_BYTE(o)   (PyByteArray_Check(o))
#define IS_INT(o)    ((PyInt_Check(o) && !PyBool_Check(o)) || PyLong_Check(o))
#define IS_STRING(o) (PyUnicode_Check(o) || PyString_Check(o))

#  define fbpyUnicode_Format(o1, o2)  PyString_Format(o1, o2)
#  define fbpyUnicode_FromFormat      PyString_FromFormat
#  define fbpyUnicode_FromString(o)   PyString_FromString(o)
#endif  /* PY_MAJOR_VERSION >= 3 */

#ifndef Py_TYPE
#define Py_TYPE(ob) (((PyObject*)(ob))->ob_type)
#endif

#define LONG_AS_UNSIGNED_LONGLONG(o)            \
    (PyLong_Check(o) ?                          \
     PyLong_AsUnsignedLongLong(o) :             \
     PyLong_AsUnsignedLong(o))

/* type of fbInfoModelBase */
typedef struct fixbufPyInfoModel_st {
    PyObject_HEAD
    fbInfoModel_t *infoModel;
} fixbufPyInfoModel;

static PyTypeObject fixbufPyInfoModelType;

#define fixbufPyInfoModel_Check(op)                     \
    PyObject_TypeCheck(op, &fixbufPyInfoModelType)

/* type of InfoElement (there is no InfoElementBase) */
typedef struct fixbufPyInfoElement_st {
    PyObject_HEAD
    fbInfoElement_t *infoElement;
    const fbInfoElement_t *ptr;
    char infoElementName[MAX_NAME];
    char description[500];
} fixbufPyInfoElement;

static PyTypeObject fixbufPyInfoElementType;

#define fixbufPyInfoElement_Check(op)                   \
    PyObject_TypeCheck(op, &fixbufPyInfoElementType)

/* type of fbSessionBase */
typedef struct fixbufPySession_st {
    PyObject_HEAD
    fbSession_t *session;
    fixbufPyInfoModel *model;
    PyObject *template_callback;
} fixbufPySession;

static PyTypeObject fixbufPySessionType;

#define fixbufPySession_Check(op)                       \
    PyObject_TypeCheck(op, &fixbufPySessionType)

/* type of fBufBase */
typedef struct fixbufPyfBuf_st {
    PyObject_HEAD
    fixbufPySession *session;
    fBuf_t *fbuf;
    int ignore_opttmpl;
} fixbufPyfBuf;

static PyTypeObject fixbufPyfBufType;

#define fixbufPyfBuf_Check(op)                  \
    PyObject_TypeCheck(op, &fixbufPyfBufType)

/* type of fbExporterBase */
typedef struct fixbufPyExporter_st {
    PyObject_HEAD
    fbExporter_t *exporter;
} fixbufPyExporter;

static PyTypeObject fixbufPyExporterType;

#define fixbufPyExporter_Check(op)                      \
    PyObject_TypeCheck(op, &fixbufPyExporterType)

/* type of fbCollectorBase */
typedef struct fixbufPyCollector_st {
    PyObject_HEAD
    fbCollector_t *collector;
} fixbufPyCollector;

static PyTypeObject fixbufPyCollectorType;

#define fixbufPyCollector_Check(op)                     \
    PyObject_TypeCheck(op, &fixbufPyCollectorType)

/* type of fbTemplateBase */
typedef struct fixbufPyTemplate_st {
    PyObject_HEAD
    fbTemplate_t *template;
    fixbufPyInfoModel *model;
    PyObject *owner;
    uint16_t template_id;
} fixbufPyTemplate;

static PyTypeObject fixbufPyTemplateType;

#define fixbufPyTemplate_Check(op)                      \
    PyObject_TypeCheck(op, &fixbufPyTemplateType)

/* type of InfoElementSpec (there is no InfoElementSpecBase) */
typedef struct fixbufPyInfoElementSpec_st {
    PyObject_HEAD
    fbInfoElementSpec_t *spec;
    char infoElementName[MAX_NAME];
} fixbufPyInfoElementSpec;

static PyTypeObject fixbufPyInfoElementSpecType;

#define fixbufPyInfoElementSpec_Check(op)                       \
    PyObject_TypeCheck(op, &fixbufPyInfoElementSpecType)

/* type of fbRecordBase */
typedef struct fixbufPyRecord_st {
    PyObject_HEAD
    uint8_t *rec;
    size_t reclen;
    gboolean memalloc;
} fixbufPyRecord;

static PyTypeObject fixbufPyRecordType;

#define fixbufPyRecord_Check(op)                \
    PyObject_TypeCheck(op, &fixbufPyRecordType)

/* type of fbSTMLBase */
typedef struct fixbufPySTML_st {
    PyObject_HEAD
    fbSubTemplateMultiList_t *stml;
    fbSubTemplateMultiListEntry_t *entry;
    gboolean stml_alloc;
} fixbufPySTML;

static PyTypeObject fixbufPySTMLType;

#define fixbufPySTML_Check(op)                  \
    PyObject_TypeCheck(op, &fixbufPySTMLType)

/* type of fbSTMLEntryBase */
typedef struct fixbufPySTMLEntry_st {
    PyObject_HEAD
    fbSubTemplateMultiListEntry_t *entry;
} fixbufPySTMLEntry;

static PyTypeObject fixbufPySTMLEntryType;

#define fixbufPySTMLEntry_Check(op)                     \
    PyObject_TypeCheck(op, &fixbufPySTMLEntryType)

/* type of fbSTMLBase */
typedef struct fixbufPySTL_st {
    PyObject_HEAD
    fbSubTemplateList_t *stl;
    gboolean stl_alloc;
} fixbufPySTL;

static PyTypeObject fixbufPySTLType;

#define fixbufPySTL_Check(op)                   \
    PyObject_TypeCheck(op, &fixbufPySTLType)

/* type of fbListenerBase */
typedef struct fixbufPyListener_st {
    PyObject_HEAD
    fbConnSpec_t conn;
    fbListener_t *listener;
} fixbufPyListener;

static PyTypeObject fixbufPyListenerType;

#define fixbufPyListener_Check(op)                      \
    PyObject_TypeCheck(op, &fixbufPyListenerType)

/* type of fbBLBase */
typedef struct fixbufPyBL_st {
    PyObject_HEAD
    fbBasicList_t *bl;
    gboolean bl_alloc;
    gboolean init;
} fixbufPyBL;

static PyTypeObject fixbufPyBLType;

#define fixbufPyBL_Check(op)                    \
    PyObject_TypeCheck(op, &fixbufPyBLType)


static void obj_dealloc(
    PyObject   *obj)
{
    Py_TYPE(obj)->tp_free(obj);
}

/**
 * fbInfoElement
 *
 **/

/* implements InfoElement.__new__ (tp_new) */
static PyObject *fixbufPyInfoElement_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    fixbufPyInfoElement *self;

    FBPY_UNUSED_PARAM(args);
    FBPY_UNUSED_PARAM(kwds);

    self = (fixbufPyInfoElement *)type->tp_alloc(type, 0);
    if (self != NULL) {
        self->infoElement = PyMem_Malloc(sizeof(fbInfoElement_t));
        if (self->infoElement == NULL) {
            Py_XDECREF((PyObject *)self);
            return PyErr_NoMemory();
        }
        memset(self->infoElement, 0, sizeof(fbInfoElement_t));
    }

    return (PyObject *)self;
}

/* implements InfoElement destructor (tp_dealloc) */
static void fixbufPyInfoElement_dealloc(
    fixbufPyInfoElement *obj)
{
    if (obj->infoElement) {
        PyMem_Free(obj->infoElement);
    }
    Py_TYPE(obj)->tp_free((PyObject *)obj);
}

/* implements InfoElement.__init__ (tp_init) */
static int fixbufPyInfoElement_init(
    fixbufPyInfoElement *self,
    PyObject            *args,
    PyObject            *kwds)
{
    static char *kwlist[] = {"name", "enterprise_number", "id", "length",
                             "reversible", "endian", "type", "min", "max",
                             "units", "semantic", "description", NULL};

    char         *name;
    char         *description = NULL;
    PY_LONG_LONG  ent;
    int           num;        /* id */
    int           len         = 0;
    int           reversible  = 0;
    int           endian      = 0;
    int           fb_flags    = 0;
    int           type        = 0;
    int           units       = 0;
    int           semantic    = 0;
    unsigned PY_LONG_LONG min = 0;
    unsigned PY_LONG_LONG max = 0;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "sLi|iiiiKKiis", kwlist,
                                     &name, &ent, &num, &len,
                                     &reversible, &endian,
                                     &type, &min, &max, &units, &semantic,
                                     &description))
    {
        return -1;
    }

    if (self->infoElement == NULL) {
        return -1;
    }

    if (ent < 0 || ent > (PY_LONG_LONG)UINT32_MAX) {
        PyErr_Format(PyExc_OverflowError,
                     "%s must be an unsigned 32-bit integer", kwlist[1]);
        return -1;
    }
    if (num < 0 || num > (int)UINT16_MAX) {
        PyErr_Format(PyExc_OverflowError,
                     "%s must be an unsigned 16-bit integer", kwlist[2]);
        return -1;
    }
    if (len < 0 || len > (int)UINT16_MAX) {
        PyErr_Format(PyExc_OverflowError,
                     "%s must be an unsigned 16-bit integer", kwlist[3]);
        return -1;
    }

    if (reversible) {
        fb_flags |= FB_IE_F_REVERSIBLE;
    }

    if (endian) {
        fb_flags |= FB_IE_F_ENDIAN;
    }

    if (len == 0) {
        /* if length is not given, use length for that type */
        switch (type) {
          case FB_UINT_8:
          case FB_INT_8:
          case FB_BOOL:
            len = 1;
            break;
          case FB_UINT_16:
          case FB_INT_16:
            len = 2;
            break;
          case FB_UINT_32:
          case FB_INT_32:
          case FB_FLOAT_32:
          case FB_DT_SEC:
          case FB_IP4_ADDR:
            len = 4;
            break;
          case FB_MAC_ADDR:
            len = 6;
            break;
          case FB_UINT_64:
          case FB_INT_64:
          case FB_FLOAT_64:
          case FB_DT_MILSEC:
          case FB_DT_MICROSEC:
          case FB_DT_NANOSEC:
            len = 8;
            break;
          case FB_IP6_ADDR:
            len = 16;
            break;
          default:
            /* lists, string, octet array */
            len = FB_IE_VARLEN;
            break;
        }
    }

    switch (semantic) {
      case 0:
        break;
      case 1:
        fb_flags |= FB_IE_QUANTITY;
        break;
      case 2:
        fb_flags |= FB_IE_TOTALCOUNTER;
        break;
      case 3:
        fb_flags |= FB_IE_DELTACOUNTER;
        break;
      case 4:
        fb_flags |= FB_IE_IDENTIFIER;
        break;
      case 5:
        fb_flags |= FB_IE_FLAGS;
        break;
      case 6:
        fb_flags |= FB_IE_LIST;
        break;
      case 7:
        fb_flags |= FB_IE_SNMPCOUNTER;
        break;
      case 8:
        fb_flags |= FB_IE_SNMPGAUGE;
        break;
      default:
        fprintf(stderr, "Invalid or Unrecognized Semantic Value %d.\n",
                semantic);
        break;
    }

    switch (units) {
      case 0:
        break;
      case 1:
        fb_flags |= FB_UNITS_BITS;
        break;
      case 2:
        fb_flags |= FB_UNITS_OCTETS;
        break;
      case 3:
        fb_flags |= FB_UNITS_PACKETS;
        break;
      case 4:
        fb_flags |= FB_UNITS_FLOWS;
        break;
      case 5:
        fb_flags |= FB_UNITS_SECONDS;
        break;
      case 6:
        fb_flags |= FB_UNITS_MILLISECONDS;
        break;
      case 7:
        fb_flags |= FB_UNITS_MICROSECONDS;
        break;
      case 8:
        fb_flags |= FB_UNITS_NANOSECONDS;
        break;
      case 9:
        fb_flags |= FB_UNITS_WORDS;
        break;
      case 10:
        fb_flags |= FB_UNITS_MESSAGES;
        break;
      case 11:
        fb_flags |= FB_UNITS_HOPS;
        break;
      case 12:
        fb_flags |= FB_UNITS_ENTRIES;
        break;
      case 13:
        fb_flags |= FB_UNITS_FRAMES;
        break;
      case 14:
        fb_flags |= FB_UNITS_PORTS;
        break;
      case 15:
        fb_flags |= FB_UNITS_INFERRED;
        break;
      default:
        fprintf(stderr, "Invalid or Unrecognized UNITS %d.\n", units);
        break;
    }

    memset(self->infoElement, 0, sizeof(fbInfoElement_t));

    strncpy(self->infoElementName, name, sizeof(self->infoElementName)-1);
    self->infoElementName[sizeof(self->infoElementName)-1] = '\0';
#if !FIXBUF_CHECK_VERSION(2, 99, 0)
    self->infoElement->ref.name = self->infoElementName;
#else
    self->infoElement->name = self->infoElementName;
#endif
    self->infoElement->ent = (uint32_t)ent;
    self->infoElement->num = (uint16_t)num;
    self->infoElement->len = (uint16_t)len;
    self->infoElement->flags = fb_flags;
    self->infoElement->type = type;
    self->infoElement->min = (uint64_t)min;
    self->infoElement->max = (uint64_t)max;

    if (description) {
        strncpy(self->description, description, sizeof(self->description));
        self->description[sizeof(self->description)-1] = '\0';
        self->infoElement->description = self->description;
    }

    return 0;
}

#define INFOELEMENT_HASH(elem)   ((((long)(elem)->ent) << 16) | (elem)->num)

/* implements hash() for InfoElement (tp_hash) */
static long fixbufPyInfoElement_hash(
    PyObject  *obj)
{
    return INFOELEMENT_HASH(((fixbufPyInfoElement *)obj)->infoElement);
}

/* implements comparison for InfoElement (tp_richcompare) */
static PyObject *fixbufPyInfoElement_richcompare(
    PyObject *self,
    PyObject *obj,
    int       cmp)
{
    long rv;

    if (!fixbufPyInfoElement_Check(obj)) {
        /* set ourself to be larger than any other type */
        return PyBool_FromLong(cmp == Py_GT || cmp == Py_GE || cmp == Py_NE);
    }

    rv = (INFOELEMENT_HASH(((fixbufPyInfoElement *)self)->infoElement)
          - INFOELEMENT_HASH(((fixbufPyInfoElement *)obj)->infoElement));
    if (rv < 0) {
        return PyBool_FromLong(cmp == Py_LT || cmp == Py_LE || cmp == Py_NE);
    }
    if (rv > 0) {
        return PyBool_FromLong(cmp == Py_GT || cmp == Py_GE || cmp == Py_NE);
    }
    return PyBool_FromLong(cmp == Py_EQ || cmp == Py_LE || cmp == Py_GE);
}

/* implements InfoElement.length getter */
static PyObject *fixbufPyInfoElement_getlength(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->infoElement->len);
}

/* implements InfoElement.name getter */
static PyObject *fixbufPyInfoElement_getname(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (fbInfoElementGetName(obj->infoElement)) {
        return fbpyUnicode_FromString(fbInfoElementGetName(obj->infoElement));
    }
    Py_RETURN_NONE;
}

/* implements InfoElement.id getter */
static PyObject *fixbufPyInfoElement_getid(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->infoElement->num);
}

/* implements InfoElement.description getter */
static PyObject *fixbufPyInfoElement_getdescription(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (obj->infoElement->description) {
        return fbpyUnicode_FromString(obj->infoElement->description);
    }
    Py_RETURN_NONE;
}

/* implements InfoElement.ent getter */
static PyObject *fixbufPyInfoElement_getent(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->infoElement->ent);
}

/* implements InfoElement.type getter */
static PyObject* fixbufPyInfoElement_gettype(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->infoElement->type);
}

/* implements InfoElement.min getter */
static PyObject* fixbufPyInfoElement_getmin(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->infoElement->min);
}

/* implements InfoElement.max getter */
static PyObject* fixbufPyInfoElement_getmax(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->infoElement->max);
}

/* implements InfoElement.unit getter */
static PyObject *fixbufPyInfoElement_getunits(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(FB_IE_UNITS(obj->infoElement->flags));
}

/* implements InfoElement.semantic getter */
static PyObject *fixbufPyInfoElement_getsemantic(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(FB_IE_SEMANTIC(obj->infoElement->flags));
}

/* implements InfoElement.reversible getter */
static PyObject *fixbufPyInfoElement_getreversible(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyBool_FromLong(obj->infoElement->flags & FB_IE_F_REVERSIBLE);
}

/* implements InfoElement.endian getter */
static PyObject *fixbufPyInfoElement_getendian(
    fixbufPyInfoElement *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyBool_FromLong(obj->infoElement->flags & FB_IE_F_ENDIAN);
}

/* helper function for InfoElement.as_dict method */
static int add_to_dict(PyObject *dict, const char *key, PyObject *value)
{
    int rv;
    if (value == Py_None) {
        Py_DECREF(value);
        return 0;
    }
    rv = PyDict_SetItemString(dict, key, value);
    Py_DECREF(value);
    return rv;
}

/* implements InfoElement.as_dict method */
static PyObject *fixbufPyInfoElement_as_dict(
    fixbufPyInfoElement *obj)
{
    PyObject *dict = PyDict_New();
    add_to_dict(dict, "name", fixbufPyInfoElement_getname(obj, NULL));
    add_to_dict(dict, "enterprise_number",
                fixbufPyInfoElement_getent(obj, NULL));
    add_to_dict(dict, "id", fixbufPyInfoElement_getid(obj, NULL));
    add_to_dict(dict, "length", fixbufPyInfoElement_getlength(obj, NULL));
    add_to_dict(dict, "reversible",
                fixbufPyInfoElement_getreversible(obj, NULL));
    add_to_dict(dict, "endian", fixbufPyInfoElement_getendian(obj, NULL));
    add_to_dict(dict, "type", fixbufPyInfoElement_gettype(obj, NULL));
    add_to_dict(dict, "min", fixbufPyInfoElement_getmin(obj, NULL));
    add_to_dict(dict, "max", fixbufPyInfoElement_getmax(obj, NULL));
    add_to_dict(dict, "units", fixbufPyInfoElement_getunits(obj, NULL));
    add_to_dict(dict, "semantic",
                fixbufPyInfoElement_getsemantic(obj, NULL));
    add_to_dict(dict, "description",
                fixbufPyInfoElement_getdescription(obj, NULL));
    return dict;
}

/* implements InfoElement.__str__ (tp_str) */
static PyObject *fixbufPyInfoElement_str(
    fixbufPyInfoElement *obj)
{
    PyObject *dict = fixbufPyInfoElement_as_dict(obj);
    PyObject *retval = PyObject_Str(dict);
    Py_DECREF(dict);
    return retval;
}

/* implements InfoElement.__repr__ (tp_repr) */
static PyObject *fixbufPyInfoElement_repr(
    fixbufPyInfoElement *obj)
{
    PyObject *args = Py_BuildValue(
        "sN", fixbufPyInfoElementType.tp_name,
        fixbufPyInfoElement_as_dict(obj));
    PyObject *fmt = fbpyUnicode_FromString("%s(%r)");
    PyObject *retval = fbpyUnicode_Format(fmt, args);
    Py_DECREF(fmt);
    Py_DECREF(args);
    return retval;
}


/**
 * InfoElement GetSetters
 *
 *
 */

/*  InfoElement GetSetters  */
static PyGetSetDef fixbufPyInfoElement_getsetters[] = {
    {"length", (getter)fixbufPyInfoElement_getlength, NULL,
     "Info Element Length", NULL},
    {"id", (getter)fixbufPyInfoElement_getid, NULL,
     "Info Element ID", NULL},
    {"name", (getter)fixbufPyInfoElement_getname, NULL,
     "Info Element Name", NULL},
    {"_templatename", (getter)fixbufPyInfoElement_getname, NULL,
     "Info Element Name (deprecated since 0.9.0).", NULL},
    {"enterprise_number", (getter)fixbufPyInfoElement_getent, NULL,
     "Info Element Enterprise Number", NULL},
    {"ent", (getter)fixbufPyInfoElement_getent, NULL,
     "Alias for 'enterprise_number'", NULL},
    {"type", (getter)fixbufPyInfoElement_gettype, NULL,
     "Info Element Data Type", NULL},
    {"min", (getter)fixbufPyInfoElement_getmin, NULL,
     "Info Element Minimum Value", NULL},
    {"max", (getter)fixbufPyInfoElement_getmax, NULL,
     "Info Element Maximum Value", NULL},
    {"semantic", (getter)fixbufPyInfoElement_getsemantic, NULL,
     "Info Element Semantic Value", NULL},
    {"units", (getter)fixbufPyInfoElement_getunits, NULL,
     "Info Element Units", NULL},
    {"description", (getter)fixbufPyInfoElement_getdescription, NULL,
     "Info Element Human-Readable Description", NULL},
    {"reversible", (getter)fixbufPyInfoElement_getreversible, NULL,
     "Info Element Reversible Flag True/False", NULL},
    {"endian", (getter)fixbufPyInfoElement_getendian, NULL,
     "Info Element Endian Flag True/False", NULL},
   {NULL, NULL, NULL, NULL, NULL}
};

static PyMethodDef fixbufPyInfoElement_methods[] = {
    {"as_dict", (PyCFunction)fixbufPyInfoElement_as_dict, METH_NOARGS,
     ("Return the InfoElement as a dictionary")},
    {NULL, NULL, 0, NULL}
};

static PyTypeObject fixbufPyInfoElementType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.InfoElement",     /*tp_name*/
    sizeof(fixbufPyInfoElement), /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyInfoElement_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)fixbufPyInfoElement_repr, /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    fixbufPyInfoElement_hash,   /*tp_hash */
    0,                          /*tp_call*/
    (reprfunc)fixbufPyInfoElement_str, /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,         /*tp_flags*/
    ("Represents an Information Element, a type of data to be stored and\n"
     "transfered via IPFIX."
     " Constructor requires a *name*, *enterprise_number*,\n"
     "and *id*, and takes optional *length* (default VARLEN), *reversible*\n"
     "flag (False), *endian* flag (False), *datatype* (DataType.OCTET_ARRAY),\n"
     "*units* (Units.NONE), *min* (0), *max* (0),\n"
     "*semantic* (Semantic.DEFAULT), and *description* (None).\n"
     ),                         /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    fixbufPyInfoElement_richcompare, /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyInfoElement_methods, /* tp_methods */
    0,                          /* tp_members */
    fixbufPyInfoElement_getsetters, /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyInfoElement_init, /* tp_init */
    0,                          /* tp_alloc */
    fixbufPyInfoElement_new,    /* tp_new */
    0                           /* tp_free */
};

/**
 *
 * fbInfoModel
 *
 **/

/* implements fbInfoModelBase.addElement method used by
 * InfoModel.add_element */
static PyObject *fixbufPyInfoModel_addElement(
    fixbufPyInfoModel   *self,
    PyObject *args)
{
    fixbufPyInfoElement *ie;

    if (!PyArg_ParseTuple(args, "O!", &fixbufPyInfoElementType, &ie)) {
        return NULL;
    }

    if (!fixbufPyInfoElement_Check(ie)) {
        PyErr_SetString(PyExc_ValueError, "Expected pyfixbuf.fbInfoElement");
        return NULL;
    }

    fbInfoModelAddElement(self->infoModel, ie->infoElement);

    Py_RETURN_NONE;
}

/* implements fbInfoModelBase.getElementLength method used by
 * InfoModel.get_element_length */
static PyObject *fixbufPyInfoModel_getElementLength(
    fixbufPyInfoModel *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"name", "type", NULL};
    char *name = NULL;
    const fbInfoElement_t *ie = NULL;
    long len = 0;
    long type = -1;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "s|l", kwlist, &name, &type)) {
        return NULL;
    }

    if (name == NULL) {
        PyErr_SetString(PyExc_AttributeError, "Need a valid info element name");
        return NULL;
    }

    if (type == FB_BASIC_LIST) {
        len = sizeof(fbBasicList_t);
    } else if (type == FB_SUB_TMPL_LIST) {
        len = sizeof(fbSubTemplateList_t);
    } else if (type == FB_SUB_TMPL_MULTI_LIST) {
        len = sizeof(fbSubTemplateMultiList_t);
    } else {
        ie = fbInfoModelGetElementByName(self->infoModel, name);
        if (ie == NULL) {
            PyErr_Format(PyExc_KeyError,
                         "Information Element %s does not exist", name);
            return NULL;
        }

        len = ie->len;

        /* return the length of the element */
        if (len == FB_IE_VARLEN) {
            if (ie->type == FB_BASIC_LIST) {
                len = sizeof(fbBasicList_t);
            } else if (ie->type == FB_SUB_TMPL_LIST) {
                len = sizeof(fbSubTemplateList_t);
            } else if (ie->type == FB_SUB_TMPL_MULTI_LIST) {
                len = sizeof(fbSubTemplateMultiList_t);
            } else {
                len = sizeof(fbVarfield_t);
            }
        }
    }

    return PyInt_FromLong(len);
}

/* implements fbInfoModelBase.is_varfield method */
static PyObject *fixbufPyInfoModel_is_varfield(
    fixbufPyInfoModel *self,
    PyObject *args)
{
    char *element_name = NULL;
    const fbInfoElement_t *ie;

    if (!PyArg_ParseTuple(args, "s", &element_name)) {
        PyErr_SetString(PyExc_AttributeError, "element name");
        return NULL;
    }

    ie = fbInfoModelGetElementByName(self->infoModel, element_name);
    if (ie == NULL) {
        PyErr_Format(PyExc_KeyError,
                     "Information Element %s does not exist", element_name);
        return NULL;
    }

    return PyBool_FromLong(ie->len == FB_IE_VARLEN);
}


/* implements fbInfoModelBase.getElementTrueType method used by
 * InfoModel.get_element_type */
static PyObject *fixbufPyInfoModel_getElementTrueType(
    fixbufPyInfoModel *self,
    PyObject *args)
{
    char *element_name = NULL;
    const fbInfoElement_t *ie;

    if (!PyArg_ParseTuple(args, "s", &element_name)) {
        PyErr_SetString(PyExc_AttributeError, "element name");
        return NULL;
    }

    ie = fbInfoModelGetElementByName(self->infoModel, element_name);
    if (!ie) {
        PyErr_Format(PyExc_KeyError,
                     "Information Element %s does not exist", element_name);
        return NULL;
    }
    return PyInt_FromLong(ie->type);
}


/* implements fbInfoModelBase.getElement method used by
 * InfoElement.get_element */
static PyObject *fixbufPyInfoModel_getElement(
    fixbufPyInfoModel *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"name", "id", "ent", NULL};
    char *name = NULL;
    fixbufPyInfoElement *element = NULL;
    const fbInfoElement_t *ie = NULL;
    PY_LONG_LONG ent = 0;
    /* value -43 is an arbitrary "was this value given" flag */
    long id = -43;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|slL", kwlist,
                                     &name, &id, &ent))
    {
        return NULL;
    }

    if (name) {
        ie = fbInfoModelGetElementByName(self->infoModel, name);
        if (!ie) {
            return PyErr_Format(PyExc_KeyError, "No element named %s", name);
        }
    } else if (id != -43) {
        if (id < 0 || id > (long)UINT16_MAX) {
            PyErr_Format(PyExc_OverflowError,
                         "%s must be an unsigned 16-bit integer", kwlist[1]);
            return NULL;
        }
        if (ent < 0 || ent > (PY_LONG_LONG)UINT32_MAX) {
            PyErr_Format(PyExc_OverflowError,
                         "%s must be an unsigned 32-bit integer", kwlist[2]);
            return NULL;
        }
        ie = fbInfoModelGetElementByID(self->infoModel,
                                       (uint16_t)id, (uint32_t)ent);
        if (!ie) {
            if (ent) {
                return PyErr_Format(PyExc_KeyError,
                                    "No element %" PRId64 "/%ld",
                                    (int64_t)ent, id);
            }
            return PyErr_Format(PyExc_KeyError, "No element %ld", id);
        }
    } else {
        PyErr_SetString(PyExc_AttributeError,
                        "Expected either name or ID.");
        return NULL;
    }

    element = ((fixbufPyInfoElement *)fixbufPyInfoElementType.tp_new(
                   &fixbufPyInfoElementType, NULL, NULL));
    if (element == NULL) {
        return PyErr_NoMemory();
    }

    if (ie) {
        memcpy(element->infoElement, ie, sizeof(fbInfoElement_t));
    }

    /* keep orig ptr so it can be passed to fbbasiclistinit */
    element->ptr = ie;

    /*Py_INCREF(element);*/

    return (PyObject *)element;
}

/* implements fbInfoModelBase.readXMLData method used by
 * InfoModel.read_from_xml_file and InfoModel.read_from_xml_data */
static PyObject *fixbufPyInfoModel_readXMLData(
    fixbufPyInfoModel *self,
    PyObject *args)
{
    char *buf;
    Py_ssize_t len;
    GError *err = NULL;

    if (!PyArg_ParseTuple(args, "s#", &buf, &len)) {
        return NULL;
    }
    if (!fbInfoModelReadXMLData(self->infoModel, buf, len, &err)) {
        PyErr_Format(PyExc_RuntimeError,
                     "Fixbuf Error reading XML data: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }
    Py_RETURN_NONE;
}

static PyMethodDef fixbufPyInfoModel_methods[] = {
    {"addElement", (PyCFunction)fixbufPyInfoModel_addElement, METH_VARARGS,
     ("Add the info element to the info model")},
    {"getElementLength", (PyCFunction)fixbufPyInfoModel_getElementLength,
     METH_KEYWORDS | METH_VARARGS, ("Return the Info Element Length")},
    {"getElementTrueType", (PyCFunction)fixbufPyInfoModel_getElementTrueType,
     METH_VARARGS,
     ("Return the true Info Element Type as defined in the Info Model")},
    {"is_varfield", (PyCFunction)fixbufPyInfoModel_is_varfield,
     METH_VARARGS, ("Find out if the element is a variable length field")},
    {"getElement", (PyCFunction)fixbufPyInfoModel_getElement,
     METH_KEYWORDS | METH_VARARGS,
     ("Return Information Element with given name or id")},
    {"readXMLData", (PyCFunction)fixbufPyInfoModel_readXMLData,
     METH_VARARGS, ("Read XML data containing information elements")},
    {NULL, NULL, 0, NULL}
};


/* implements fbInfoModelBase.__init__ (tp_init) */
static int fixbufPyInfoModel_init(
    fixbufPyInfoModel *self,
    PyObject *args,
    PyObject *kwds)
{
    FBPY_UNUSED_PARAM(args);
    FBPY_UNUSED_PARAM(kwds);
    if (self != NULL) {
        self->infoModel = fbInfoModelAlloc();
    }

    return 0;
}

/* implements fbInfoModelBase destructor (tp_dealloc) */
static void fixbufPyInfoModel_dealloc(
    fixbufPyInfoModel *obj)
{
    if (obj->infoModel) {
        fbInfoModelFree(obj->infoModel);
    }

    Py_TYPE(obj)->tp_free((PyObject *)obj);
}

static PyTypeObject fixbufPyInfoModelType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbInfoModelBase", /*tp_name*/
    sizeof(fixbufPyInfoModel),  /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyInfoModel_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,         /*tp_flags*/
    "fbInfoModel",              /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyInfoModel_methods,  /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyInfoModel_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /*tp_new */
    0                           /* tp_free */
};


/**
 * fBuf
 *
 *
 *  The base type for the Buffer class.
 **/

/* implements fBufBase.__init__ (tp_init) */
static int fixbufPyfBuf_init(
    fixbufPyfBuf        *self,
    PyObject            *args,
    PyObject            *kwds)
{
    FBPY_UNUSED_PARAM(args);
    FBPY_UNUSED_PARAM(kwds);

    self->ignore_opttmpl = 0;

    return 0;
}

/* implements fBufBase.free method (also used by destructor) */
static PyObject *fixbufPyfBuf_free(
    fixbufPyfBuf *self)
{
    if (self->fbuf) {
        fBufFree(self->fbuf);
        self->fbuf = NULL;
    }

    Py_RETURN_NONE;
}

/* implements fBufBase destructor (tp_dealloc) */
static void fixbufPyfBuf_dealloc(
    fixbufPyfBuf *self)
{
    PyObject *none = fixbufPyfBuf_free(self);
    Py_XDECREF(none);
    Py_XDECREF(self->session);
    obj_dealloc((PyObject *)self);
}

/* implements fBufBase.allocForCollection method */
static PyObject *fixbufPyfBuf_allocForCollection(
    fixbufPyfBuf *self,
    PyObject *args)
{
    fixbufPySession *session = NULL;
    fixbufPyCollector *collector = NULL;
    int rv;

    rv = PyArg_ParseTuple(args, "OO",
                          (PyObject *)&session, (PyObject *)&collector);
    if (!rv) {
        return NULL;
    }

    if (!fixbufPyfBuf_Check(self)) {
        PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.fBuf");
        return NULL;
    }

    if (!fixbufPySession_Check(session)) {
        PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.fbSession");
        return NULL;
    }

    if (!fixbufPyCollector_Check(collector)) {
        PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.fbCollector");
        return NULL;
    }

    self->fbuf = fBufAllocForCollection(session->session,
                                        collector->collector);
    if (!self->fbuf) {
        PyErr_SetString(PyExc_ValueError,
                        "Error allocating fBuf for Collection");
        return NULL;
    }

    Py_XDECREF(self->session);
    self->session = session;
    Py_INCREF(self->session);

    Py_RETURN_NONE;
}

/* implements fBufBase.allocForExport method */
static PyObject *fixbufPyfBuf_allocForExport(
    fixbufPyfBuf *self,
    PyObject *args)
{
    fixbufPySession *session = NULL;
    fixbufPyExporter *exporter = NULL;
    int rv;

    rv = PyArg_ParseTuple(args, "OO",
                          (PyObject *)&session, (PyObject *)&exporter);
    if (!rv) {
        return NULL;
    }

    if (!fixbufPyfBuf_Check(self)) {
        PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.fBuf");
        return NULL;
    }

    if (!fixbufPySession_Check(session)) {
        PyErr_SetString(PyExc_TypeError, "Expected valid pyfixbuf.fbSession");
        return NULL;
    }

    if (!fixbufPyExporter_Check(exporter)) {
        PyErr_SetString(PyExc_TypeError, "Expected valid pyfixbuf.fbExporter");
        return NULL;
    }

    self->fbuf = fBufAllocForExport(((fixbufPySession *)session)->session,
                                    ((fixbufPyExporter *)exporter)->exporter);
    if (!self->fbuf) {
        PyErr_SetString(PyExc_ValueError, "Error Allocating fbuf for Export");
        return NULL;
    }

    Py_XDECREF(self->session);
    self->session = session;
    Py_INCREF(self->session);

    Py_RETURN_NONE;
}

/* implements fBufBase.setInternalTemplate method */
static PyObject *fixbufPyfBuf_setInternalTemplate(
    fixbufPyfBuf *self,
    PyObject *args)
{
    int tid;
    GError *err = NULL;

    if (!PyArg_ParseTuple(args, "i", &tid)) {
        PyErr_SetString(PyExc_AttributeError, "Expected Template ID");
        return NULL;
    }

    if (!self->fbuf) {
        Py_RETURN_FALSE;
    }

    if (!fBufSetInternalTemplate(self->fbuf, tid, &err)) {
        PyErr_Format(PyExc_RuntimeError,
                     "Fixbuf Error Setting Internal Template on Buffer: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_TRUE;
}


/* implements fBufBase.setExportTemplate method */
static PyObject *fixbufPyfBuf_setExportTemplate(
    fixbufPyfBuf *self,
    PyObject *args)
{
    int tid;
    GError *err = NULL;

    if (!PyArg_ParseTuple(args, "i", &tid)) {
        PyErr_SetString(PyExc_AttributeError, "Expected Template ID");
        return NULL;
    }

    if (!self->fbuf) {
        Py_RETURN_FALSE;
    }

    if (!fBufSetExportTemplate(self->fbuf, tid, &err)) {
        PyErr_Format(PyExc_RuntimeError,
                     "Fixbuf Error Setting External Template on Buffer: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_TRUE;
}

/* implements fBufBase.nextRecord method used by Buffer.next_record */
static PyObject *fixbufPyfBuf_nextRecord(
    fixbufPyfBuf *self,
    PyObject *args)
{
    GError *err = NULL;
    fixbufPyRecord *fixrec = NULL;
    fbTemplate_t *tmpl = NULL;
    uint16_t tid = 0;
    size_t rec_len = 0;

    if (!PyArg_ParseTuple(args, "O", (PyObject *)&fixrec)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(fixrec)) {
        PyErr_SetString(PyExc_AttributeError, "Expected pyfixbuf.Record");
        return NULL;
    }

    if (!self->fbuf) {
        PyErr_SetString(PyExc_StopIteration, "End of File - NULL Buffer");
        return NULL;
    }

    if (fixrec->rec == NULL) {
        fixrec->rec = PyMem_Malloc(fixrec->reclen);
        if (fixrec->rec == NULL) {
            Py_XDECREF((PyObject *)self);
            return PyErr_NoMemory();
        }
        memset(fixrec->rec, 0, fixrec->reclen);
        fixrec->memalloc = TRUE;
    }

    if (self->ignore_opttmpl) {
        for (;;) {
            tmpl = fBufNextCollectionTemplate(self->fbuf, &tid, &err);
            if (NULL == tmpl) {
                goto err;
            }
            if (PyErr_Occurred()) {
                return NULL;
            }
            if (!fbTemplateGetOptionsScope(tmpl)) {
                break;
            }
            /* read and ignore this options record */
            rec_len = fixrec->reclen;
            if (!fBufNext(self->fbuf, fixrec->rec, &rec_len, &err)) {
                goto err;
            }
        }
    }

    rec_len = fixrec->reclen;
    if (!fBufNext(self->fbuf, fixrec->rec, &rec_len, &err)) {
        goto err;
    }
    if (PyErr_Occurred()) {
        return NULL;
    }
    Py_RETURN_NONE;

  err:
    if (g_error_matches(err, FB_ERROR_DOMAIN, FB_ERROR_EOF)) {
        g_clear_error(&err);
        fBufFree(self->fbuf);
        self->fbuf = NULL;
        if (!PyErr_Occurred()) {
            PyErr_SetString(PyExc_StopIteration, "End of File");
        }
    } else {
        if (!PyErr_Occurred()) {
            PyErr_Format(PyExc_RuntimeError, "Error retrieving fBuf: %s",
                         err->message);
        }
        g_clear_error(&err);
    }
    return NULL;
}

/* implements fBufBase.append */
static PyObject *fixbufPyfBuf_append(
    fixbufPyfBuf *self,
    PyObject *args)
{
    int optlen = 0;
    GError *err = NULL;
    fixbufPyRecord *fixrec = NULL;
    size_t len;
    int rv;

    if (!PyArg_ParseTuple(args, "O|i", (PyObject *)&fixrec, &optlen)) {
        PyErr_SetString(PyExc_AttributeError,
                        "Expected Record and Optional Record Length");
        return NULL;
    }

    if (!fixbufPyRecord_Check(fixrec)) {
        return NULL;
    }
    if (!self->fbuf) {
        PyErr_SetString(PyExc_AttributeError, "Must call init_export() first");
        return NULL;
    }

    if (optlen == 0) {
        len = fixrec->reclen;
    } else if (optlen > 0) {
        len = (size_t)optlen;
    } else {
        PyErr_SetString(PyExc_AttributeError,
                        "Optional record length may not be negative");
        return NULL;
    }

    if (fixrec->rec == NULL) {
        PyErr_SetString(PyExc_RuntimeError, "No Record Available to Append");
        return NULL;
    }

    rv = fBufAppend(self->fbuf, fixrec->rec, len, &err);
    if (rv == FALSE) {
        PyErr_Format(PyExc_RuntimeError, "Error appending Buffer: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_NONE;
}

/* implements fBufBase.emit method */
static PyObject *fixbufPyfBuf_emit(
    fixbufPyfBuf *self)
{
    GError *err = NULL;

    if (!self->fbuf) {
        PyErr_SetString(PyExc_AttributeError, "Must call init_export() first");
        return NULL;
    }
    if (!fBufEmit(self->fbuf, &err)) {
        PyErr_Format(PyExc_RuntimeError, "Error emiting Buffer: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_NONE;
}

/* implements fBufBase.nextTemplate method used by Buffer.next_template */
static PyObject *fixbufPyfBuf_nextCollectionTemplate(
    fixbufPyfBuf *self,
    PyObject *args)
{
    uint16_t tid;
    GError *err = NULL;
    fixbufPyTemplate *tmpl = NULL;
    fbTemplate_t *t;

    FBPY_UNUSED_PARAM(args);

    if (!self->fbuf) {
        PyErr_SetString(PyExc_AttributeError, "Buffer is NULL");
        return NULL;
    }

    t = fBufNextCollectionTemplate(self->fbuf, &tid, &err);
    if (t == NULL) {
        if (!PyErr_Occurred()) {
            PyErr_Format(PyExc_StopIteration, "%s", err->message);
        }
        g_clear_error(&err);
        return NULL;
    }
    if (self->ignore_opttmpl && fbTemplateGetOptionsScope(t)) {
        /*
         *  skip records.  this is more difficult than it should be.
         *
         *  create a template, add it to the session, use it as the internal
         *  template to read data until a non-options template is found,
         *  remove the template from the session.  and handle any errors.
         */
        fbInfoElementSpec_t skip_spec = {"protocolIdentifier", 1, 0};
        fbTemplate_t *skip_tmpl;
        uint16_t skip_tid;
        uint8_t skip_rec[8];
        size_t skip_len;

        skip_tmpl = fbTemplateAlloc(self->session->model->infoModel);
        if (!fbTemplateAppendSpec(skip_tmpl, &skip_spec, 0, &err)) {
            /* should never fail */
            fbTemplateFreeUnused(skip_tmpl);
            PyErr_Format(PyExc_RuntimeError, "Error creating template: %s",
                         err->message);
            g_clear_error(&err);
            return NULL;
        }
        skip_tid = fbSessionAddTemplate(self->session->session,
                                        TRUE, FB_TID_AUTO, skip_tmpl, &err);
        if (0 == skip_tid) {
            /* should only error if table is full; in this case, we could use
             * *ANY* tid as the internal template */
            fbTemplateFreeUnused(skip_tmpl);
            PyErr_Format(PyExc_RuntimeError, "Cannot add template to session: "
                         "%s", err->message);
            g_clear_error(&err);
            return NULL;
        }
        if (!fBufSetInternalTemplate(self->fbuf, skip_tid, &err)) {
            fbSessionRemoveTemplate(
                self->session->session, TRUE, skip_tid, NULL);
            PyErr_Format(PyExc_RuntimeError,
                         "Cannot find newly added template: %s", err->message);
            g_clear_error(&err);
            fbSessionRemoveTemplate(
                self->session->session, TRUE, skip_tid, NULL);
            return NULL;
        }
        do {
            skip_len = sizeof(skip_rec);
            if (!fBufNext(self->fbuf, skip_rec, &skip_len, &err)) {
                t = NULL;
                break;
            }
            t = fBufNextCollectionTemplate(self->fbuf, &tid, &err);
        } while (t && fbTemplateGetOptionsScope(t));

        fbSessionRemoveTemplate(self->session->session, TRUE, skip_tid, NULL);

        if (t == NULL) {
            if (!PyErr_Occurred()) {
                PyErr_Format(PyExc_StopIteration, "%s", err->message);
            }
            g_clear_error(&err);
            return NULL;
        }
    }

    tmpl = (fixbufPyTemplate *)fixbufPyTemplateType.tp_new(
        &fixbufPyTemplateType, NULL, NULL);
    if (tmpl == NULL) {
        return PyErr_NoMemory();
    }
    tmpl->template = t;
    tmpl->template_id = tid;
    tmpl->owner = (PyObject *)self->session;
    Py_XINCREF(tmpl->owner);
    tmpl->model = self->session->model;
    Py_XINCREF(tmpl->model);

    return (PyObject *)tmpl;
}


/* implements fBufBase.getTemplate method used by Buffer.get_template */
static PyObject *fixbufPyfBuf_getCollectionTemplate(
    fixbufPyfBuf *self,
    PyObject     *args)
{
    uint16_t tid;
    fixbufPyTemplate *tmpl = NULL;
    fbTemplate_t *t;

    FBPY_UNUSED_PARAM(args);

    if (!self->fbuf) {
        PyErr_SetString(PyExc_AttributeError, "Buffer is NULL");
        return NULL;
    }

    t = fBufGetCollectionTemplate(self->fbuf, &tid);
    if (t == NULL) {
        Py_RETURN_NONE;
    }
    tmpl = (fixbufPyTemplate *)fixbufPyTemplateType.tp_new(
        &fixbufPyTemplateType, NULL, NULL);
    if (tmpl == NULL) {
        return PyErr_NoMemory();
    }
    tmpl->template = t;
    tmpl->template_id = tid;
    tmpl->owner = (PyObject *)self->session;
    Py_INCREF(tmpl->owner);
    tmpl->model = ((fixbufPySession *)self->session)->model;
    Py_XINCREF(tmpl->model);

    return (PyObject *)tmpl;
}


/* implements fBufBase.writeOptionsRecord method */
static PyObject *fixbufPyfBuf_writeOptionsRecord(
    fixbufPyfBuf *self,
    PyObject *args)
{
    GError *err = NULL;
    fixbufPyInfoModel *pymodel = NULL;
    fbInfoModel_t *model = NULL;
    int itid = 0;
    int etid = INT_MAX;
    char *name;

    if (!PyArg_ParseTuple(args, "O!si|i", &fixbufPyInfoModelType,
                          &pymodel, &name, &itid, &etid))
    {
        return NULL;
    }

    if (!fixbufPyInfoModel_Check(pymodel)) {
        PyErr_SetString(PyExc_ValueError, "Need an InfoModel");
        return NULL;
    }

    if (itid <= 0 || itid > UINT16_MAX) {
        PyErr_SetString(PyExc_AttributeError, "Need a valid Template ID");
        return NULL;
    }
    if (etid <= 0 || etid > UINT16_MAX) {
        if (etid == INT_MAX) {
            etid = itid;
        } else {
            PyErr_SetString(PyExc_AttributeError,
                            "Need a valid external template ID");
            return NULL;
        }
    }

    if (name == NULL) {
        PyErr_SetString(PyExc_AttributeError,"Need a valid info element name");
        return NULL;
    }

    if (!self->fbuf) {
        PyErr_SetString(PyExc_AttributeError, "Buffer is NULL");
        return NULL;
    }

    model = pymodel->infoModel;

    if (fbInfoElementWriteOptionsRecord(
            self->fbuf, fbInfoModelGetElementByName(model, name),
            itid, etid, &err))
    {
        Py_RETURN_NONE;
    } else {
        PyErr_Format(PyExc_RuntimeError, "Unable to append Options Record: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }
}

/* implements fBufBase.setAutoInsert method */
static PyObject *fixbufPyfBuf_setAutoInsert(
    fixbufPyfBuf *self)
{
    GError *err = NULL;

    if (!self->fbuf) {
        PyErr_SetString(PyExc_AttributeError,
                        "Buffer has not been initialized");
        return NULL;
    }

    if (fBufSetAutomaticInsert(self->fbuf, &err)) {
        Py_RETURN_TRUE;
    } else {
        PyErr_Format(PyExc_RuntimeError, "Unable to set Buffer in Auto"
                     " Insert Mode: %s", err->message);
        g_clear_error(&err);
        return NULL;
    }
}

/* implements fBufBase.ignoreOptions method */
static PyObject *fixbufPyfBuf_ignoreOptions(
    fixbufPyfBuf *self,
    PyObject *args)
{
    int ignore;

    if (!PyArg_ParseTuple(args, "i", &ignore)) {
        return NULL;
    }

    self->ignore_opttmpl = ignore;

    Py_RETURN_NONE;
}


static PyMethodDef fixbufPyfBuf_methods[] = {
    {"allocForCollection", (PyCFunction)fixbufPyfBuf_allocForCollection,
     METH_VARARGS, ("Allocate an fbuf for collection")},
    {"allocForExport", (PyCFunction)fixbufPyfBuf_allocForExport, METH_VARARGS,
     ("Allocate an fbuf for export")},
    {"setInternalTemplate", (PyCFunction)fixbufPyfBuf_setInternalTemplate,
     METH_VARARGS, ("Set an Internal Template on the fBuf for export")},
    {"setExportTemplate", (PyCFunction)fixbufPyfBuf_setExportTemplate,
     METH_VARARGS, ("Set an External Template on a buffer.  The external "
    "tmpl describes the record that will be written to the IPFIX message")},
    {"nextRecord", (PyCFunction)fixbufPyfBuf_nextRecord, METH_VARARGS,
     ("Get the next record")},
    {"nextTemplate", (PyCFunction)fixbufPyfBuf_nextCollectionTemplate,
     METH_VARARGS, ("Retrieve the external tmpl that will be used to read "
     "the next record from the buffer")},
    {"getTemplate", (PyCFunction)fixbufPyfBuf_getCollectionTemplate,
     METH_VARARGS, ("Retrieve the external template that was used to read "
                    "the last record from the buffer")},
    {"append", (PyCFunction)fixbufPyfBuf_append, METH_VARARGS,
     ("Append the record to the buffer")},
    {"emit", (PyCFunction)fixbufPyfBuf_emit, METH_NOARGS, ("Emit the buffer")},
    {"free", (PyCFunction)fixbufPyfBuf_free, METH_NOARGS, ("Free the buffer")},
    {"writeOptionsRecord", (PyCFunction)fixbufPyfBuf_writeOptionsRecord,
     METH_VARARGS, ("Write an Info Element Type Information Options"
                    " Record for the given IE name")},
    {"setAutoInsert", (PyCFunction)fixbufPyfBuf_setAutoInsert, METH_NOARGS,
     ("Set the Buffer in Auto Insert Mode.")},
    {"ignoreOptions", (PyCFunction)fixbufPyfBuf_ignoreOptions, METH_VARARGS,
     ("Set the Buffer to ignore Options Templates and Records.")},
    {NULL, NULL, 0, NULL}
};


static PyTypeObject fixbufPyfBufType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fBufBase",        /*tp_name*/
    sizeof(fixbufPyfBuf),       /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyfBuf_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fBuf",                     /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyfBuf_methods,       /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyfBuf_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbRecord
 *
 *
 */

/* implements fbRecordBase.__init__ (tp_init) */
static int fixbufPyRecord_init(
    fixbufPyRecord *self,
    PyObject *args,
    PyObject *kwds)
{
    int len;

    FBPY_UNUSED_PARAM(kwds);

    if (!PyArg_ParseTuple(args, "i", &len)) {
        PyErr_SetString(PyExc_AttributeError, "Expected Record Length");
        return -1;
    }
    if (self != NULL) {
        self->rec = NULL;
        self->reclen = (size_t)len;
        self->memalloc = FALSE;
    }

    return 0;
}

/* implements fbRecordBase destructor (tp_dealloc) */
static void fixbufPyRecord_dealloc(
    fixbufPyRecord *self)
{
    if (self->memalloc) {
        PyMem_Free(self->rec);
    }
    Py_TYPE(self)->tp_free((PyObject *)self);
}


/*
 *    Helper function for Records and BasicLists to set a field's
 *    value by storing the Python object 'value' into the Record's or
 *    List's backing store ('bytearray').  'type' and 'len' are the
 *    data-type and octet-length of the field.  For variable sized
 *    fields, 'len' is sizeof(fbVarfield_t) and 'is_varlen' is true.
 */
static PyObject *pyfixbuf_set_value(
    int             type,
    uint8_t        *bytearray,
    int             len,
    PyObject       *value,
    int             is_varlen)
{
    int is_negative = -1;
    char *string;
    union val_un {
        uint64_t  u;
        int64_t   s;
        uint8_t   c[sizeof(uint64_t)];
    } val;

    if (IS_INT(value)) {
        PyObject *zero = PyLong_FromLong(0);
        is_negative = PyObject_RichCompareBool(value, zero, Py_LT);
        Py_DECREF(zero);

        if (is_negative) {
            val.s = (int64_t)(PyLong_Check(value) ?
                              PyLong_AsLongLong(value)
                              : PyLong_AsLong(value));
        } else {
            val.u = LONG_AS_UNSIGNED_LONGLONG(value);
        }
        if (PyErr_Occurred()) {
            PyErr_SetString(PyExc_ValueError,
                            "Value is not of long type.  "
                            "Try to convert using long()");
            return NULL;
        }
    }

    switch (type) {
      case FB_IP6_ADDR:
        if ( !IS_BYTE(value) ) {
            PyErr_SetString(PyExc_TypeError,
                            "Value is not a bytearay type");
            return NULL;
        } else if (len != 16) {
            PyErr_Format(PyExc_ValueError,
                         "IPv6 Addresss has unexpected length %d", len);
            return NULL;
        } else {
            string = PyByteArray_AS_STRING(value);
            memcpy(bytearray, string, len);
        }
        break;

      case FB_BOOL:
        if (is_negative == -1) {
            PyErr_SetString(PyExc_TypeError,
                            "Value is not an integer type. "
                            "Try to convert using int()");
            return NULL;
        } else {
            *bytearray = (val.u != 0);
        }
        break;

      case FB_UINT_8:
      case FB_UINT_16:
      case FB_UINT_32:
      case FB_UINT_64:
        if (is_negative != 0) {
            if (is_negative == 1) {
                PyErr_SetString(PyExc_ValueError,
                                "Value is negative");
            } else {
                PyErr_SetString(PyExc_TypeError,
                                "Value is not an integer type. "
                                "Try to convert using int()");
            }
            return NULL;
        } else {
            if (len == 8) {
                memcpy(bytearray, &val.u, len);
            } else if (len > 8) {
                PyErr_Format(PyExc_ValueError,
                             "Invalid length for an integer type: %d", len);
                return NULL;
            } else if ((val.u & (UINT64_MAX & (UINT64_MAX << (8 * len)))) != 0)
            {
                PyErr_Format(PyExc_ValueError,
                             "Value out of range for %d-byte integer",
                             len);
                return NULL;
            } else {
#if    G_BYTE_ORDER == G_LITTLE_ENDIAN
                memcpy(bytearray, val.c, len);
#elif  G_BYTE_ORDER == G_BIG_ENDIAN
                memcpy(bytearray, &val.c[8 - len], len);
#else
                PyErr_SetString(PyExc_ValueError,
                                "Unsupported byte order.");
                return NULL;
#endif
            }
        }
        break;

      case FB_INT_8:
      case FB_INT_16:
      case FB_INT_32:
      case FB_INT_64:
        if (is_negative == -1) {
            PyErr_SetString(PyExc_TypeError,
                            "Value is not an integer type. "
                            "Try to convert using int()");
            return NULL;
        } else {
            if (len == 8) {
                memcpy(bytearray, &val.s, len);
            } else if (len > 8) {
                PyErr_Format(PyExc_ValueError,
                             "Invalid length for an integer type: %d", len);
                return NULL;
            } else if (is_negative == 0 &&
                       ((val.s & (UINT64_MAX & (UINT64_MAX<<(8 * len)))) != 0))
            {
                PyErr_Format(PyExc_ValueError,
                             "Value out of range for %d-byte integer",
                             len);
                return NULL;
            } else if (is_negative == 1 &&
                       ((val.s & (UINT64_MAX & (UINT64_MAX << (8 * len)))) !=
                        (UINT64_MAX & (UINT64_MAX << (8 * len)))))
            {
                PyErr_Format(PyExc_ValueError,
                             "Value out of range for %d-byte integer",
                             len);
                return NULL;
            } else {
#if    G_BYTE_ORDER == G_LITTLE_ENDIAN
                memcpy(bytearray, val.c, len);
#elif  G_BYTE_ORDER == G_BIG_ENDIAN
                memcpy(bytearray, &val.c[8 - len], len);
#else
                PyErr_SetString(PyExc_ValueError,
                                "Unsupported byte order.");
                return NULL;
#endif
            }
        }
        break;

      case FB_FLOAT_32:
      case FB_FLOAT_64:
        if ( !IS_INT(value) && !PyFloat_Check(value) ) {
            PyErr_SetString(PyExc_ValueError,
                            "Value is not a numeric type. "
                            "Try to convert using float()");
            return NULL;
        } else {
            double d;
#if PY_MAJOR_VERSION < 3
            if (PyInt_Check(value)) {
                d = (double)PyInt_AsLong(value);
            } else
#endif
            if (PyLong_Check(value)) {
                d = (double)PyLong_AsLongLong(value);
            } else {
                d = PyFloat_AsDouble(value);
            }
            if (PyErr_Occurred()) {
                PyErr_SetString(PyExc_ValueError,
                                "Value is not of float type.  "
                                "Try to convert using float()");
                return NULL;
            }
            if (len == sizeof(float)) {
                float f;
                if (d < -FLT_MAX || d > FLT_MAX) {
                    PyErr_SetString(PyExc_ValueError,
                                    "Value out of range for a 4-byte float");
                    return NULL;
                }
                f = (float)d;
                memcpy(bytearray, &f, sizeof(float));
            } else if (len == sizeof(double)) {
                memcpy(bytearray, &d, sizeof(double));
            } else {
                PyErr_Format(PyExc_ValueError,
                             "Unsupported floating point length %d", len);
                return NULL;
            }
        }
        break;

      case FB_IP4_ADDR:
      case FB_DT_SEC:
        if (is_negative != 0) {
            if (is_negative == 1) {
                PyErr_SetString(PyExc_ValueError,
                                "Value is negative");
            } else {
                PyErr_SetString(PyExc_TypeError,
                                "Value is not an integer type. "
                                "Try to convert using int()");
            }
            return NULL;
        } else {
            uint32_t val32;
            if (val.u > UINT32_MAX) {
                PyErr_SetString(PyExc_ValueError,
                                "Value out of range for IPv4 Address");
                return NULL;
            }
            val32 = (uint32_t)val.u;
            memcpy(bytearray, &val32, sizeof(val32));
        }
        break;

      case FB_DT_MILSEC:
      case FB_DT_MICROSEC:
      case FB_DT_NANOSEC:
        if (is_negative != 0) {
            if (is_negative == 1) {
                PyErr_SetString(PyExc_ValueError,
                                "Value is negative");
            } else {
                PyErr_SetString(PyExc_TypeError,
                                "Value is not a long type. "
                                "Try to convert using long()");
            }
            return NULL;
        } else {
            memcpy(bytearray, &val.u, sizeof(val.u));
        }
        break;

      case FB_BASIC_LIST:
        if (!fixbufPyBL_Check(value)) {
            PyErr_SetString(PyExc_AttributeError,
                            "Value must be a valid BL");
            return NULL;
        }
        memcpy(bytearray, ((fixbufPyBL *)(value))->bl,
               sizeof(fbBasicList_t));
        if (((fixbufPyBL *)(value))->bl_alloc) {
            PyMem_Free(((fixbufPyBL *)(value))->bl);
            ((fixbufPyBL *)(value))->bl_alloc = FALSE;
        }
        ((fixbufPyBL *)(value))->bl = (fbBasicList_t *)bytearray;
        break;

      case FB_SUB_TMPL_LIST:
        if (!fixbufPySTL_Check(value)) {
            PyErr_SetString(PyExc_AttributeError,
                            "Value must be a STL");
            return NULL;
        }
        if (bytearray == (uint8_t *)((fixbufPySTL *)(value))->stl) {
            /* don't copy over ourself; this can occur when
             * initializing a record's STL from a list of Records */
            break;
        }
        memcpy(bytearray, ((fixbufPySTL *)(value))->stl,
               sizeof(fbSubTemplateList_t));
        if (((fixbufPySTL *)(value))->stl_alloc) {
            /* free fbsubtemplatelist*/
            PyMem_Free(((fixbufPySTL *)(value))->stl);
            ((fixbufPySTL *)(value))->stl_alloc = FALSE;
        }
        ((fixbufPySTL *)(value))->stl = (fbSubTemplateList_t*)bytearray;
        break;

      case FB_SUB_TMPL_MULTI_LIST:
        if (!fixbufPySTML_Check(value)) {
            PyErr_SetString(PyExc_AttributeError,
                            "Value must be a STML");
            return NULL;
        }
        memcpy(bytearray, ((fixbufPySTML *)(value))->stml,
               sizeof(fbSubTemplateMultiList_t));
        if (((fixbufPySTML *)(value))->stml_alloc) {
            /* free fbsubtemplatemultilist*/
            PyMem_Free(((fixbufPySTML *)(value))->stml);
            ((fixbufPySTML *)(value))->stml_alloc = FALSE;
        }
        ((fixbufPySTML *)(value))->stml = (fbSubTemplateMultiList_t*)bytearray;
        break;

      case FB_MAC_ADDR:
        if ( !IS_BYTE(value) ) {
            PyErr_SetString(PyExc_TypeError,
                            "Value is not a bytearray type");
            return NULL;
        } else if (len != 6) {
            PyErr_Format(PyExc_ValueError,
                         "MAC_ADDR has unexpected length %d", len);
            return NULL;
        } else {
            string = PyByteArray_AS_STRING(value);
            memcpy(bytearray, string, len);
        }
        break;

      case FB_STRING:
        {
            /* FIXME: The bytes of the PyString are never stored
             * anywhere; the value in the Record becomes gibberish if
             * the PyString is garbage collected before the Record is
             * appended to a buffer. */
            Py_ssize_t sz = INT_MAX;
#if PY_MAJOR_VERSION >= 3
            if (!PyUnicode_Check(value)) {
                PyErr_SetString(PyExc_TypeError,
                                "Value is not a string type.");
                return NULL;
            } else {
                string = (char *)PyUnicode_AsUTF8AndSize(value, &sz);
                if (NULL == string) {
                    return NULL;
                }
            }
#else  /* #if PY_MAJOR_VERSION >= 3 */
            if (!PyUnicode_Check(value) && !PyString_Check(value)) {
                PyErr_SetString(PyExc_TypeError,
                                "Value is not a string type.");
                return NULL;
            } else {
                if (PyString_AsStringAndSize(value, &string, &sz) == -1) {
                    return NULL;
                }
            }
#endif  /* #else of #if PY_MAJOR_VERSION >= 3 */
            if (sz > (is_varlen ? FB_IE_VARLEN : len)) {
                PyErr_Format(
                    PyExc_ValueError,
                    "String's length (%ld) is longer than maximum allowed %d",
                    (long)sz, (is_varlen ? FB_IE_VARLEN : len));
                return NULL;
            }
            if (is_varlen) {
                fbVarfield_t varfield;
                assert(len == sizeof(fbVarfield_t));
                varfield.buf = (uint8_t*)string;
                varfield.len = sz;
                memcpy(bytearray, &varfield, sizeof(fbVarfield_t));
            } else {
                memcpy(bytearray, string, sz);
                memset(bytearray + sz, 0, (len - sz));
            }
        }
        break;

      case FB_OCTET_ARRAY:
      default:
        {
            Py_ssize_t sz;
            if (PyByteArray_Check(value)) {
                string = PyByteArray_AS_STRING(value);
                sz = PyByteArray_GET_SIZE(value);
            } else if (PyBytes_Check(value)) {
                string = PyBytes_AS_STRING(value);
                sz = PyBytes_GET_SIZE(value);
            } else {
                PyErr_SetString(PyExc_TypeError,
                                "Value is not a bytes or bytearray type.");
                return NULL;
            }
            if (sz > (is_varlen ? FB_IE_VARLEN : len)) {
                PyErr_Format(
                    PyExc_ValueError,
                    "Bytes' length (%ld) is longer than maximum allowed %d",
                    (long)sz, (is_varlen ? FB_IE_VARLEN : len));
                return NULL;
            }
            if (is_varlen) {
                fbVarfield_t varfield;
                assert(len == sizeof(fbVarfield_t));
                varfield.buf = (uint8_t*)string;
                varfield.len = sz;
                memcpy(bytearray, &varfield, sizeof(fbVarfield_t));
            } else {
                memcpy(bytearray, string, sz);
                memset(bytearray + sz, 0, (len - sz));
            }
        }
        break;
    }

    if (PyErr_Occurred()) {
        return NULL;
    }

    Py_RETURN_NONE;
}

/* implements fbRecordBase.setOffset method */
static PyObject *fixbufPyRecord_setOffset(
    fixbufPyRecord *self,
    PyObject *args)
{
    PyObject *value = NULL;
    int offset;
    int len;
    int type;
    int is_varlen;

    if (!PyArg_ParseTuple(args, "Oiiii",
                          &value, &offset, &len, &type, &is_varlen))
    {
        return NULL;
    }

    if (!self->rec) {
        self->rec = PyMem_Malloc(self->reclen);
        if (!self->rec) {
            Py_XDECREF((PyObject *)self);
            return PyErr_NoMemory();
        }
        /* initialize memory to 0 */
        memset(self->rec, 0, self->reclen);
        self->memalloc = TRUE;
    }

    return pyfixbuf_set_value(type, self->rec + offset, len, value, is_varlen);
}


/* implements fbRecordBase.basicListClear method */
static PyObject *fixbufPyRecord_basicListClear(
    fixbufPyRecord *self,
    PyObject *args)
{
    int offset;

    if (!PyArg_ParseTuple(args, "i", &offset)) {
        return NULL;
    }

    if (!self->rec) {
        PyErr_SetString(PyExc_AttributeError, "No fixbuf Record to Clear");
        return NULL;
    }

    fbBasicListClear((fbBasicList_t *)(self->rec + offset));

    Py_RETURN_NONE;
}


/* implements fbRecordBase.getBL method */
static PyObject *fixbufPyRecord_getBL(
    fixbufPyRecord *self,
    PyObject *args)
{
    fixbufPyBL *bl;
    int offset;

    if (!PyArg_ParseTuple(args, "O!i", &fixbufPyBLType, &bl, &offset)) {
        return NULL;
    }

    if (!self->rec) {
        offset = 0;
        /* No record - return 0 */
        return PyInt_FromLong(offset);
    }

    memcpy(bl->bl, (self->rec + offset), sizeof(fbBasicList_t));

    bl->init = TRUE;

    Py_RETURN_NONE;
}

/*
 *    Helper function for Records and BasicLists to get a field's
 *    value retrieving the value from the Record's or List's backing
 *    store ('bytearray') and converting it to a Python object.
 *    'type' and 'len' are the data-type and octet-length of the
 *    field.  For a variable-length field, 'len' is
 *    sizeof(fbVarfield_t) and 'is_varfield' is true.
 */
static PyObject *pyfixbuf_get_value(
    long             type,
    const uint8_t   *bytearray,
    int              len,
    int              is_varlen)
{
    switch (type) {
      case FB_UINT_64:
      case FB_DT_MILSEC:
      case FB_DT_MICROSEC:
      case FB_DT_NANOSEC:
        {
            uint64_t grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyLong_FromUnsignedLongLong(grab);
            }
            if (type != FB_UINT_64) {
                break;
            }
        }
        /* FALLTHROUGH */
      case FB_UINT_32:
      case FB_IP4_ADDR:
      case FB_DT_SEC:
        {
            uint32_t grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyLong_FromUnsignedLongLong(grab);
            }
            if (type != FB_UINT_32 && type != FB_UINT_64) {
                break;
            }
        }
        /* FALLTHROUGH */
      case FB_UINT_16:
        {
            uint16_t grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyInt_FromLong(grab);
            }
        }
        /* FALLTHROUGH */
      case FB_UINT_8:
      case FB_BOOL:
        {
            uint8_t grab;
            if (len == sizeof(grab)) {
                grab = *bytearray;
                return PyInt_FromLong(grab);
            }
            if (type == FB_BOOL) {
                break;
            }
        }
        if (len < 8) {
            uint64_t grab = 0;
#if    G_BYTE_ORDER == G_LITTLE_ENDIAN
            memcpy(&grab, bytearray, len);
#elif  G_BYTE_ORDER == G_BIG_ENDIAN
            memcpy(((uint8_t *)&grab) + (8 - len), bytearray, len);
#else
            PyErr_SetString(PyExc_AttributeError, "Unsupported byte order");
            return NULL;
#endif
            return PyLong_FromUnsignedLongLong(grab);
        }
        break;
      case FB_INT_64:
        {
            int64_t grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyLong_FromLongLong(grab);
            }
        }
        /* FALLTHROUGH */
      case FB_INT_32:
        {
            int32_t grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyLong_FromLongLong(grab);
            }
        }
        /* FALLTHROUGH */
      case FB_INT_16:
        {
            int16_t grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyInt_FromLong(grab);
            }
        }
        /* FALLTHROUGH */
      case FB_INT_8:
        {
            int8_t grab;
            if (len == sizeof(grab)) {
                grab = *bytearray;
                return PyInt_FromLong(grab);
            }
        }
        if (len < 8) {
            const uint8_t *sign_byte;
            int64_t grab;
#if    G_BYTE_ORDER == G_LITTLE_ENDIAN
            sign_byte = bytearray + (len - 1);
            memset(&grab, ((0x80 & *sign_byte) ? 0xff : 0), sizeof(grab));
            memcpy(&grab, bytearray, len);
#elif  G_BYTE_ORDER == G_BIG_ENDIAN
            sign_byte = bytearray;
            memset(&grab, ((0x80 & *sign_byte) ? 0xff : 0), sizeof(grab));
            memcpy(((uint8_t *)&grab) + (8 - len), bytearray, len);
#else
            PyErr_SetString(PyExc_AttributeError, "Unsupported byte order");
            return NULL;
#endif
            return PyLong_FromLongLong(grab);
        }
        break;
      case FB_FLOAT_64:
        {
            double grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyFloat_FromDouble(grab);
            }
        }
        /* FALLTHROUGH */
      case FB_FLOAT_32:
        {
            float grab;
            if (len == sizeof(grab)) {
                memcpy(&grab, bytearray, sizeof(grab));
                return PyFloat_FromDouble(grab);
            }
            break;
        }
      case FB_IP6_ADDR:
        {
            char grab[16];
            if (len == sizeof(grab)) {
                memcpy(grab, bytearray, sizeof(grab));
                return PyByteArray_FromStringAndSize(grab, len);
            }
            break;
        }
      case FB_STRING:
        {
            if (is_varlen) {
                /* variable length field */
                fbVarfield_t grab;
                assert(sizeof(grab) == len);
                memcpy(&grab, bytearray, sizeof(grab));
                return PyUnicode_DecodeUTF8((char *)grab.buf, grab.len,
                                            "ignore");
            } else {
                char grab[UINT16_MAX];
                memcpy(grab, bytearray, len);
                return PyUnicode_DecodeUTF8((char *)grab, len, "ignore");
            }
        }
      case FB_MAC_ADDR:
        if (len != 6) {
            break;
        }
        /* FALLTHROUGH */
      case FB_OCTET_ARRAY:
      default:
        {
            if (is_varlen) {
                fbVarfield_t grab;
                assert(sizeof(grab) == len);
                memcpy(&grab, bytearray, sizeof(grab));
                return PyByteArray_FromStringAndSize((char*)grab.buf,grab.len);
            } else {
                char grab[UINT16_MAX];
                memcpy(grab, bytearray, len);
                return PyByteArray_FromStringAndSize(grab, len);
            }
        }
    }

    PyErr_Format(PyExc_ValueError,
                 "Invalid length for type: %d", len);
    return NULL;
}


/* implements fbRecordBase.getOffset method */
static PyObject *fixbufPyRecord_getOffset(
    fixbufPyRecord *self,
    PyObject *args)
{
    int len;
    int offset;
    long type = -1;
    int is_varlen = 0;

    if (!PyArg_ParseTuple(args, "ii|li", &offset, &len, &type, &is_varlen)) {
        return NULL;
    }

    if (!self->rec) {
        /* No record - return 0 */
        switch (type) {
          case FB_IP6_ADDR:
            return PyByteArray_FromStringAndSize(
                "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0", 16);
          case FB_MAC_ADDR:
            return PyByteArray_FromStringAndSize("\0\0\0\0\0\0", 6);
          case FB_OCTET_ARRAY:
            return PyByteArray_FromStringAndSize("", 0);
          case FB_STRING:
            return fbpyUnicode_FromString("");
          case FB_IE_VARLEN:
            fprintf(stderr, "VARLEN is not a valid type\n");
          default:
            return PyInt_FromLong(0);
        }
    }

    if (len < 0 || len > UINT16_MAX) {
        /* invalid length */
        Py_RETURN_NONE;
    }

    return pyfixbuf_get_value(type, self->rec + offset, len, is_varlen);
}


/* implements fbRecordBase.length getter */
static PyObject *fixbufPyRecord_getlen(
    fixbufPyRecord *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(self->reclen);
}

/* implements fbRecordBase.length setter */
static int fixbufPyRecord_setlen(
    fixbufPyRecord *self,
    PyObject *value,
    void *cbdata)
{
    PyObject *pyint_val;
    size_t new_len;

    FBPY_UNUSED_PARAM(cbdata);

    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_AttributeError, "Expected Positive Number");
        return -1;
    }

    pyint_val = PyNumber_Int(value);
    if (pyint_val == NULL) {
        PyErr_SetString(PyExc_AttributeError, "Expected Positive Number");
        return -1;
    }

    new_len = PyLong_AsLong(pyint_val);
    Py_XDECREF(pyint_val);

    if (self->memalloc) {
        if (new_len > self->reclen) {
            /* need to realloc */
            self->rec = PyMem_Realloc(self->rec, new_len);
            if (!self->rec) {
                return -1;
            }
            memset(self->rec + self->reclen, 0, new_len - self->reclen);
        }
    }

    self->reclen = new_len;

    return 0;
}

/* implements fbRecordBase.clear method */
static PyObject *fixbufPyRecord_clear(
    fixbufPyRecord *self)
{
    if (self->memalloc == FALSE) {
        self->rec = NULL;
    } else {
        memset(self->rec, 0, self->reclen);
    }

    Py_RETURN_NONE;
}


static PyMethodDef fixbufPyRecord_methods[] = {
    {"getOffset", (PyCFunction)fixbufPyRecord_getOffset,
     METH_VARARGS, ("Get offset into buffer")},
    {"setOffset", (PyCFunction)fixbufPyRecord_setOffset,
     METH_VARARGS, ("Set offset into buffer")},
    {"getBL", (PyCFunction)fixbufPyRecord_getBL,
     METH_VARARGS, ("Get basiclist items")},
    /* {"getBasicListItems", (PyCFunction)fixbufPyRecordGetBasicListItems, */
    /*  METH_VARARGS, ("Get basiclist items")}, */
    {"clear", (PyCFunction)fixbufPyRecord_clear, METH_NOARGS,
     ("Set Record to NULL")},
    {"basicListClear", (PyCFunction)fixbufPyRecord_basicListClear,
     METH_VARARGS, ("Clear given basiclist")},
    {NULL, NULL, 0, NULL}
};


static PyGetSetDef fixbufPyRecord_getsetters[] = {
    {"length", (getter)fixbufPyRecord_getlen, (setter)fixbufPyRecord_setlen,
     "Length of record in memory, where lists and variable fields are"
     " represented by data structures.", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};


static PyTypeObject fixbufPyRecordType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbRecordBase",    /*tp_name*/
    sizeof(fixbufPyRecord),     /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyRecord_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "Record",                   /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyRecord_methods,     /* tp_methods */
    0,                          /* tp_members */
    fixbufPyRecord_getsetters,  /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyRecord_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbTemplate
 *
 *
 */

/* implements fbTemplateBase.__init__ (tp_init) */
static int fixbufPyTemplate_init(
    fixbufPyTemplate *self,
    PyObject *args,
    PyObject *kwds)
{
    fixbufPyInfoModel *model = NULL;
    PyObject *type = NULL;
    static char *kwlist[] = {"model", "type", NULL};
    GError *err = NULL;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O!|O", kwlist,
                                     &fixbufPyInfoModelType, &model, &type))
    {
        return -1;
    }

    if (!fixbufPyInfoModel_Check(model)) {
        PyErr_SetString(PyExc_ValueError, "Need an InfoModel");
        return -1;
    }

    if (type && PyObject_IsTrue(type)) {
        self->template =
            fbInfoElementAllocTypeTemplate(model->infoModel, &err);
        if (!self->template) {
            PyErr_Format(PyExc_RuntimeError,
                         "Could not create Information Type Template: %s",
                         err->message);
            g_clear_error(&err);
            return -1;
        }
    } else {
        self->template = fbTemplateAlloc(model->infoModel);
    }

    self->model = model;
    Py_INCREF(self->model);
    self->owner = NULL;
    self->template_id = 0;

    return 0;
}

/* implements fbTemplateBase destructor (tp_dealloc) */
static void fixbufPyTemplate_dealloc(
    fixbufPyTemplate *self)
{
    if (!self->owner && self->template) {
        fbTemplateFreeUnused(self->template);
    }
    Py_XDECREF(self->model);
    Py_XDECREF(self->owner);
    Py_TYPE(self)->tp_free((PyObject *)self);
}

/* implements fbTemplateBase.addSpec method used by _add_spec() */
static PyObject *fixbufPyTemplate_addSpec(
    fixbufPyTemplate   *self,
    fixbufPyInfoElementSpec *spec)
{
    GError *err = NULL;

    if (self->owner) {
        PyErr_SetString(PyExc_RuntimeError,
                        ("Template has already been added to a session"
                         " and may not be modifed"));
        return NULL;
    }

    if (!fixbufPyInfoElementSpec_Check(spec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Info Element Spec");
        return NULL;
    }

    if (!self->template) {
        PyErr_SetString(PyExc_AttributeError, "NULL Template");
        return NULL;
    }

    if (spec->spec == NULL) {
        PyErr_SetString(PyExc_AttributeError, "NULL InfoElementSpec");
        return NULL;
    }

    /* set spec length */
    if (0 == spec->spec->len_override) {
        const fbInfoElement_t *ie = fbInfoModelGetElementByName(
            self->model->infoModel, spec->spec->name);
        if (ie) {
            spec->spec->len_override = ie->len;
        }
    }

    if (!fbTemplateAppendSpec(self->template, spec->spec, 0, &err)) {
        PyErr_Format(PyExc_RuntimeError,
                     "Error adding info element spec to template: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    return PyInt_FromLong(fbTemplateCountElements(self->template) - 1);
}

/* implements fbTemplateBase.infomodel getter */
static PyObject *fixbufPyTemplate_getinfomodel(
    fixbufPyTemplate *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    Py_INCREF(self->model);
    return (PyObject *)self->model;
}

/* implements fbTemplateBase.scope getter */
static PyObject *fixbufPyTemplate_getscope(
    fixbufPyTemplate *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (!self->template) {
        Py_RETURN_FALSE;
    }
    return PyInt_FromLong(fbTemplateGetOptionsScope(self->template));
}

/* implements fbTemplateBase.scope setter */
static int fixbufPyTemplate_setscope(
    fixbufPyTemplate *self,
    PyObject *value,
    void *cbdata)
{
    long scope_count;
    int num_elements;
    PyObject *pyint_val;

    FBPY_UNUSED_PARAM(cbdata);

    if (self->owner) {
        PyErr_SetString(PyExc_AttributeError,
                        ("Template has already been added to a session"
                         " and may not be modifed"));
        return -1;
    }

    scope_count = fbTemplateGetOptionsScope(self->template);
    if (scope_count != 0) {
        PyErr_SetString(PyExc_AttributeError,
                        "Scope can only be set once.");
        return -1;
    }

    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_AttributeError, "Expected Scope Count");
        return -1;
    }

    pyint_val = PyNumber_Int(value);
    if (pyint_val == NULL) {
        PyErr_SetString(PyExc_AttributeError, "Expected Scope Count");
        return -1;
    }

    scope_count = PyLong_AsLong(pyint_val);
    Py_XDECREF(pyint_val);

    num_elements = fbTemplateCountElements(self->template);

    if (num_elements < scope_count) {
        PyErr_SetString(PyExc_RuntimeError, "Scope count must be greater"
                        " than or equal to the number of elements in the "
                        "template");
        return -1;
    }

    fbTemplateSetOptionsScope(self->template, scope_count);
    return 0;
}

/* implements fbTemplateBase.read_only getter */
static PyObject *fixbufPyTemplate_getreadonly(
    fixbufPyTemplate *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyBool_FromLong(self->owner != NULL);
}

/* implements fbTemplateBase.template_id getter */
static PyObject *fixbufPyTemplate_gettid(
    fixbufPyTemplate *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(self->template_id);
}

/* implements fbTemplateBase.template_id getter */
static PyObject *fixbufPyTemplate_gettype(
    fixbufPyTemplate *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyBool_FromLong(fbInfoModelTypeInfoRecord(self->template));
}

/* implements fbTemplateBase.elementCount method used by __len__() */
static PyObject *fixbufPyTemplate_elementCount(
    fixbufPyTemplate *self)
{
    if (!self->template) {
        return PyInt_FromLong(0);
    }
    return PyInt_FromLong(fbTemplateCountElements(self->template));
}

/* implements fbTemplateBase.containsElement method used by __contains__() */
static PyObject *fixbufPyTemplate_containsElement(
    fixbufPyTemplate *self,
    PyObject *arg)
{
    char *name = NULL;
    fbInfoElementSpec_t spec;

    if (fixbufPyInfoElementSpec_Check(arg)) {
        return PyBool_FromLong(
            fbTemplateContainsElementByName(
                self->template, ((fixbufPyInfoElementSpec *)arg)->spec));
    } else if (fixbufPyInfoElement_Check(arg)) {
        return PyBool_FromLong(
            fbTemplateContainsElement(
                self->template, ((fixbufPyInfoElement *)arg)->infoElement));
    } else if (IS_STRING(arg)) {
        name = (char *)PyString_AsString(arg);
        if (name) {
            spec.name = name;
            return PyBool_FromLong(
                fbTemplateContainsElementByName(self->template, &spec));
        }
    }

    PyErr_SetString(PyExc_AttributeError,
                    "Expected a name, InfoElement, or InfoElementSpec");
    return NULL;
}

/* implements fbTemplateBase.getIndexedIE method used by
 * Template._build_element_list() */
static PyObject *fixbufPyTemplate_getIndexedIE(
    fixbufPyTemplate *self,
    PyObject *args)
{
    long index = 0;
    fixbufPyInfoElement *element = NULL;
    const fbTemplateField_t *field = NULL;
    const fbInfoElement_t *ie;

    if (!PyArg_ParseTuple(args, "l", &index)) {
        return NULL;
    }

    if (index >= 0 && index <= (long)UINT16_MAX) {
        field = fbTemplateGetFieldByPosition(self->template, (uint16_t)index);
    }
    if (!field) {
        PyErr_SetString(PyExc_IndexError, "Index Out of Bounds");
        return NULL;
    }

    element = (fixbufPyInfoElement *)
        fixbufPyInfoElementType.tp_new(&fixbufPyInfoElementType, NULL, NULL);
    if (element == NULL) {
        return PyErr_NoMemory();
    }

    ie = fbTemplateFieldGetIE(field);
    memcpy(element->infoElement, ie, sizeof(fbInfoElement_t));

    return (PyObject *)element;
}

/* implements fbTemplateBase.getIndexedSpec method used by
 * Template._build_element_list() */
static PyObject *fixbufPyTemplate_getIndexedSpec(
    fixbufPyTemplate *self,
    PyObject *args)
{
    long index = 0;
    fixbufPyInfoElementSpec *spec = NULL;
    const fbTemplateField_t *field = NULL;

    if (!PyArg_ParseTuple(args, "l", &index)) {
        return NULL;
    }

    if (index >= 0 && index <= (long)UINT16_MAX) {
        field = fbTemplateGetFieldByPosition(self->template, (uint16_t)index);
    }
    if (!field) {
        PyErr_SetString(PyExc_IndexError, "Index Out of Bounds");
        return NULL;
    }

    spec = (fixbufPyInfoElementSpec *)fixbufPyInfoElementSpecType.tp_new(
        &fixbufPyInfoElementSpecType, NULL, NULL);
    if (spec == NULL) {
        return PyErr_NoMemory();
    }

    memcpy(spec->infoElementName, fbTemplateFieldGetName(field),
           sizeof(spec->infoElementName));
    spec->spec->name = spec->infoElementName;
    spec->spec->len_override = fbTemplateFieldGetLen(field);

    return (PyObject *)spec;
}

/* implements fbTemplateBase.getContext method used by
 * Template.get_context */
static PyObject *fixbufPyTemplate_getContext(
    fixbufPyTemplate *self)
{
    PyObject *ctx = (PyObject *)fbTemplateGetContext(self->template);
    if (ctx) {
        Py_INCREF(ctx);
        return ctx;
    }
    Py_RETURN_NONE;
}


static PyGetSetDef fixbufPyTemplate_getsetters[] = {
    {"template_id", (getter)fixbufPyTemplate_gettid, NULL, "Template ID",NULL},
    {"scope", (getter)fixbufPyTemplate_getscope,
     (setter)fixbufPyTemplate_setscope, "Scope", NULL},
    {"type", (getter)fixbufPyTemplate_gettype, NULL,
     "True if this Template is used to describe Information Elements", NULL},
    {"read_only", (getter)fixbufPyTemplate_getreadonly, NULL,
     "True if this Template has been added to a Session", NULL},
    {"infomodel", (getter)fixbufPyTemplate_getinfomodel, NULL,
     "Return the infomodel", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyMethodDef fixbufPyTemplate_methods[] = {
    {"addSpec", (PyCFunction)fixbufPyTemplate_addSpec, METH_O,
     ("Adds an Information Element Spec to the template"
      " and returns the index of the element")},
    {"elementCount", (PyCFunction)fixbufPyTemplate_elementCount, METH_NOARGS,
     ("Returns number of elements in template")},
    {"containsElement", (PyCFunction)fixbufPyTemplate_containsElement, METH_O,
     ("Determines if an Information Element exists in the Template."
      " Accepts an InfoElement, InfoElementSpec, or a string.")},
    {"getIndexedIE", (PyCFunction)fixbufPyTemplate_getIndexedIE, METH_VARARGS,
     ("Gets the Information Element at the given index in the Template.")},
    {"getIndexedSpec", (PyCFunction)fixbufPyTemplate_getIndexedSpec,
     METH_VARARGS,
     ("Gets the Information Element Spec at the given index in the Template.")},
    {"getContext", (PyCFunction)fixbufPyTemplate_getContext, METH_NOARGS,
     ("Return the context object as set by"
      " the new template callback function")},
    {NULL, NULL, 0, NULL}
};

static PyTypeObject fixbufPyTemplateType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbTemplateBase",  /*tp_name*/
    sizeof(fixbufPyTemplate),   /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyTemplate_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,         /*tp_flags*/
    "fbTemplate",               /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyTemplate_methods,   /* tp_methods */
    0,                          /* tp_members */
    fixbufPyTemplate_getsetters,/* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyTemplate_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbSession
 *
 */

/* implements fbSessionBase.__init__ (tp_init) */
static int fixbufPySession_init(
    fixbufPySession *self,
    PyObject *args,
    PyObject *kwds)
{
    fixbufPyInfoModel *info = NULL;
    static char *kwlist[] = {"model", NULL};

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O!", kwlist,
                                     &fixbufPyInfoModelType, &info))
    {
        PyErr_SetString(PyExc_TypeError, "Expected InfoModel Object");
        return -1;
    }

    if (!fixbufPyInfoModel_Check(info)) {
        PyErr_SetString(PyExc_ValueError, "Expects an InfoModel");
        return -1;
    }

    self->session = fbSessionAlloc(info->infoModel);

    /* Keep a reference to the infomodel for the lifetime of this object */
    self->model = info;
    Py_INCREF(self->model);

    return 0;
}

/* implements fbSessionBase destructor (tp_dealloc) */
static void fixbufPySession_dealloc(
    fixbufPySession *self)
{
    Py_XDECREF(self->model);
    Py_XDECREF(self->template_callback);
    obj_dealloc((PyObject *)self);
}


/* implements fbSessionBase.addTemplate method used by
 * Session._add_template */
static PyObject *fixbufPySession_addTemplate(
    fixbufPySession *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"template", "template_id", "internal", NULL};
    fixbufPyTemplate *tmpl = NULL;
    GError *err = NULL;
    int internal = 1;
    int tid = 0;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O|ii", kwlist,
                                     (PyObject *)&tmpl, &tid, &internal))
    {
        return NULL;
    }
    if (!fixbufPyTemplate_Check(tmpl)) {
        PyErr_SetString(PyExc_TypeError, "Expected Template");
        return NULL;
    }

    if (tid != FB_TID_AUTO) {
        /* do not add the template if its already on the session,
         * since doing so could accidentally free the fbTemplate. */
        if (fbSessionGetTemplate(self->session, internal, tid, NULL)
            == tmpl->template)
        {
            assert((PyObject *)self == tmpl->owner);
            assert(tid == tmpl->template_id);
            return PyInt_FromLong(tmpl->template_id);
        }
    }

    tmpl->template_id = fbSessionAddTemplate(self->session, internal, tid,
                                             tmpl->template, &err);
    if ( tmpl->template_id == 0 ) {
        PyErr_Format(PyExc_RuntimeError,
                     "Error adding template to the session: %s",
                     err->message);
        g_clear_error(&err);
    }
    tmpl->owner = (PyObject *)self;
    Py_INCREF(tmpl->owner);

    return PyInt_FromLong(tmpl->template_id);
}

/* implements fbSessionBase.getTemplate method used by
 * Session.get_template */
static PyObject *fixbufPySession_getTemplate(
    fixbufPySession *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"template_id", "internal", NULL};
    fixbufPyTemplate *tmpl = NULL;
    GError *err = NULL;
    int tid;
    int internal = 0;
    fbTemplate_t *t;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "i|i", kwlist,
                                     &tid, &internal))
    {
        return NULL;
    }

    t = fbSessionGetTemplate(self->session, internal, tid, &err);
    if (t == NULL) {
        Py_RETURN_NONE;
    }
    tmpl = (fixbufPyTemplate *)fixbufPyTemplateType.tp_new(
        &fixbufPyTemplateType, NULL, NULL);
    if (tmpl == NULL) {
        return PyErr_NoMemory();
    }
    tmpl->template = t;
    tmpl->owner = (PyObject *)self;
    Py_INCREF(tmpl->owner);
    tmpl->template_id = tid;
    tmpl->model = self->model;
    Py_XINCREF(tmpl->model);

    return (PyObject *)tmpl;
}

/*
 *    Function called by libfixbuf when the reference count of the
 *    fbTemplate_t goes to 0 to free the template's context pointer.
 *
 *    The signature of this function is fbTemplateCtxFree_fn.
 */
static void session_template_ctx_free(
    void *tmpl_ctx,
    void *app_ctx)
{
    (void)app_ctx;
    Py_XDECREF(tmpl_ctx);
}

/*
 *    Function called by libfixbuf when a new template is seen.
 *
 *    The signature must match fbNewTemplateCallback_fn.
 */
static void session_template_callback(
    fbSession_t           *session,
    uint16_t               tid,
    fbTemplate_t          *tmpl,
    void                  *session_ctx,
    void                 **tmpl_ctx,
    fbTemplateCtxFree_fn  *free_fn)
{
    fixbufPySession *self = (fixbufPySession *)session_ctx;
    fixbufPyTemplate *pytmpl;
    PyObject *new_ctx;

    FBPY_UNUSED_PARAM(session);

    if (PyErr_Occurred()) {
        /* This might be the case if a previous callback operation
         * threw an exception. */
        return;
    }

    if (!self->template_callback) {
        return;
    }

    pytmpl = (fixbufPyTemplate *)fixbufPyTemplateType.tp_new(
        &fixbufPyTemplateType, NULL, NULL);
    if (pytmpl == NULL) {
        return;
    }
    pytmpl->template = tmpl;
    pytmpl->template_id = tid;
    pytmpl->owner = (PyObject *)self;
    Py_INCREF(pytmpl->owner);
    pytmpl->model = self->model;
    Py_XINCREF(pytmpl->model);

    new_ctx = PyObject_CallFunctionObjArgs(
        self->template_callback, pytmpl, NULL);

    Py_DECREF(pytmpl);
    if (PyErr_Occurred()) {
        return;
    }
    *tmpl_ctx = new_ctx;
    *free_fn = session_template_ctx_free;
}


/* implements fbSessionBase.setTemplateCallback method used by
 * Session.add_template_callback */
static PyObject *fixbufPySession_setTemplateCallback(
    fixbufPySession *self,
    PyObject *value)
{
    if (!PyCallable_Check(value)) {
        PyErr_SetString(PyExc_ValueError, "Callback must be a callable");
        return NULL;
    }
    if (self->template_callback) {
        Py_DECREF(self->template_callback);
    } else {
        fbSessionAddNewTemplateCallback(
            self->session, session_template_callback, self);
    }
    Py_INCREF(value);
    self->template_callback = value;
    Py_RETURN_NONE;
}

/* implements fbSessionBase.domain setter */
static int fixbufPySession_setdomain(
    fixbufPySession *self,
    PyObject *value,
    void *cbdata)
{
    unsigned long domain;

    FBPY_UNUSED_PARAM(cbdata);

    if (!IS_INT(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    domain = PyLong_AsUnsignedLong(value);
    if (PyErr_Occurred() || domain > UINT32_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "Domain value must be an unsigned 32-bit integer");
        return -1;
    }

    fbSessionSetDomain(self->session, (uint32_t)domain);

    return 0;
}

/* implements fbSessionBase.domain getter */
static PyObject *fixbufPySession_getdomain(
    fixbufPySession *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyLong_FromUnsignedLong(fbSessionGetDomain(self->session));
}

/* implements fbSessionBase.export_templates method */
static PyObject *fixbufPySession_exportTemplates(
    fixbufPySession *self)
{
    GError *err = NULL;

    if (!fbSessionExportTemplates(self->session, &err)) {
        PyErr_Format(PyExc_RuntimeError, "Error exporting templates: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_NONE;
}


/* implements fbSessionBase.addTemplatePair method used by
 * Session.add_template_pair */
static PyObject *fixbufPySession_addTemplatePair(
    fixbufPySession *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"external_template_id", "internal_template_id",
                             NULL};
    int ext_tid;
    int int_tid;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "ii", kwlist,
                                     &ext_tid, &int_tid))
    {
        return NULL;
    }

    if (self->session != NULL) {
        fbSessionAddTemplatePair(self->session, ext_tid, int_tid);
   }

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPySession_methods[] = {
    {"addTemplate", (PyCFunction)fixbufPySession_addTemplate,
     METH_KEYWORDS | METH_VARARGS, ("Add a template to the session")},
    {"getTemplate", (PyCFunction)fixbufPySession_getTemplate,
     METH_KEYWORDS | METH_VARARGS,
     ("Get template in session with template_id")},
    {"export_templates", (PyCFunction)fixbufPySession_exportTemplates,
     METH_NOARGS, ("Export templates associated with a session")},
    {"addTemplatePair", (PyCFunction)fixbufPySession_addTemplatePair,
     METH_KEYWORDS | METH_VARARGS,
     ("add an external-internal pair to the session in order "
      "to decode certain templates appropriately")},
    {"setTemplateCallback", (PyCFunction)fixbufPySession_setTemplateCallback,
     METH_O, ("Set the new template callback function")},
    {NULL, NULL, 0, NULL}
};

static PyGetSetDef fixbufPySession_getsetters[] = {
    {"domain", (getter)fixbufPySession_getdomain,
     (setter)fixbufPySession_setdomain,
     "Current Observation Domain on Session", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyTypeObject fixbufPySessionType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbSessionBase",   /*tp_name*/
    sizeof(fixbufPySession),    /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPySession_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbSession",                /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPySession_methods,    /* tp_methods */
    0,                          /* tp_members */
    fixbufPySession_getsetters, /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPySession_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbCollector
 *
 */

/* implements fbCollectorBase.allocCollectorFile method used by
 * Collector.init_file */
static PyObject *fixbufPyCollector_allocFile(
    fixbufPyCollector *self,
    PyObject *args)
{
    char *filename = NULL;
    GError *err = NULL;

    if (!PyArg_ParseTuple(args, "et", Py_FileSystemDefaultEncoding, &filename)
        || NULL == filename)
    {
        PyErr_SetString(PyExc_AttributeError, "filename");
        return NULL;
    }

    self->collector = fbCollectorAllocFile(NULL, filename, &err);
    if (!self->collector) {
        PyErr_Format(PyExc_RuntimeError,
                     "Error allocating file collector: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPyCollector_methods[] = {
    {"allocCollectorFile", (PyCFunction)fixbufPyCollector_allocFile,
     METH_VARARGS,
     ("Allocate a collector for an IPFIX file given input filename")},
    {NULL, NULL, 0, NULL}
};

static PyTypeObject fixbufPyCollectorType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbCollectorBase", /*tp_name*/
    sizeof(fixbufPyCollector),  /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    obj_dealloc,                /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbCollector",              /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyCollector_methods,  /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbListener
 *
 */

/* implements fbListenerBase.allocListener method used by
 * Listener.__init__ */
static PyObject *fixbufPyListenerAlloc(
    fixbufPyListener *self,
    PyObject         *args,
    PyObject         *kwds)
{
    static char *kwlist[] = {"transport", "host", "port", "session", NULL};
    char *transport;
    char *host;
    char *port;
    fixbufPySession *session;
    GError *err = NULL;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "szsO", kwlist,
                                     &transport, &host, &port, &session))
    {
        return NULL;
    }

    if (!fixbufPySession_Check(session)) {
        PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.fbSession");
        return NULL;
    }

    if (strcmp(transport, "tcp") == 0) {
        self->conn.transport = FB_TCP;
    } else if (strcmp(transport, "udp") == 0) {
        self->conn.transport = FB_UDP;
    } else if (strcmp(transport, "sctp") == 0) {
        self->conn.transport = FB_SCTP;
    } else {
        PyErr_Format(PyExc_RuntimeError,
                     "%s is not a supported transport",
                     transport);
        return NULL;
    }

    self->conn.svc = port;
    self->conn.host = host;
    self->conn.ssl_ca_file = NULL;
    self->conn.ssl_cert_file = NULL;
    self->conn.ssl_key_file = NULL;
    self->conn.ssl_key_pass = NULL;
    self->conn.vai = NULL;
    self->conn.vssl_ctx = NULL;

    self->listener = fbListenerAlloc(&(self->conn), session->session,
                                     NULL, NULL, &err);
    if (!self->listener) {
        PyErr_Format(PyExc_MemoryError, "Error allocating listener: %s",
                     err->message);
        g_clear_error(&err);
        return NULL;
    }

    Py_RETURN_NONE;
}


/* implements fbListenerBase.listenerWait method used by Listener.wait */
static PyObject *fixbufPyListener_wait(
    fixbufPyListener *self,
    PyObject         *args)
{
    fixbufPyfBuf *buf = NULL;
    fixbufPySession *session = NULL;
    fBuf_t *ret_buf = NULL;
    GError *err = NULL;

    if (!PyArg_ParseTuple(args, "OO", &buf, &session)) {
        return NULL;
    }

    if (!fixbufPyfBuf_Check(buf)) {
        PyErr_SetString(PyExc_TypeError, "Expected Buffer");
        return NULL;
    }

    if (!fixbufPySession_Check(session)) {
        PyErr_SetString(PyExc_TypeError, "Expected Session");
        return NULL;
    }

    if (!self->listener) {
        PyErr_SetString(PyExc_TypeError, "Invalid Listener");
        return NULL;
    }

    ret_buf = fbListenerWait(self->listener, &err);

    if (PyErr_CheckSignals()) {
        return NULL;
    }

    if (ret_buf == NULL) {
        PyErr_Format(PyExc_RuntimeError, "Error: %s", err->message);
        g_clear_error(&err);
        return NULL;
    }

    buf->fbuf = ret_buf;

    Py_XDECREF(buf->session);
    buf->session = session;
    Py_INCREF(buf->session);

    session->session = fBufGetSession(ret_buf);

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPyListener_methods[] = {
    {"allocListener", (PyCFunction)fixbufPyListenerAlloc,
     METH_KEYWORDS | METH_VARARGS,
     ("Allocate a listener for a given host and port")},
    {"listenerWait", (PyCFunction)fixbufPyListener_wait, METH_VARARGS,
     ("Listen on the interface")},
    {NULL, NULL, 0, NULL}
};

static PyTypeObject fixbufPyListenerType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbListenerBase",  /*tp_name*/
    sizeof(fixbufPyListener),   /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    obj_dealloc,                /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbListener",               /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyListener_methods,   /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbExporter
 *
 */

/* implements fbExporterBase.allocFile method used by Exporter.init_file */
static PyObject *fixbufPyExporter_allocFile(
    fixbufPyExporter *self,
    PyObject *args)
{
    char *filename = NULL;

    if (!PyArg_ParseTuple(args, "et", Py_FileSystemDefaultEncoding, &filename)
        || NULL == filename)
    {
        PyErr_SetString(PyExc_AttributeError, "filename");
        return NULL;
    }

    self->exporter = fbExporterAllocFile(filename);
    if (!self->exporter) {
        PyErr_SetString(PyExc_MemoryError, "Error allocating file exporter");
        return NULL;
    }

    Py_RETURN_NONE;
}


/* implements fbExporterBase.allocNet method used by Exporter.init_net */
static PyObject *fixbufPyExporter_allocNet(
    fixbufPyExporter *self,
    PyObject         *args,
    PyObject         *kwds)
{
    static char *kwlist[] = {"transport", "host", "port", NULL};
    char *transport;
    char *host;
    char *port;
    fbConnSpec_t conn;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "sss", kwlist,
                                     &transport, &host, &port))
    {
        return NULL;
    }

    if (strcmp(transport, "tcp") == 0) {
        conn.transport = FB_TCP;
    } else if (strcmp(transport, "udp") == 0) {
        conn.transport = FB_UDP;
    } else {
        PyErr_Format(PyExc_ValueError,
                     "%s is not a supported transport", transport);
        return NULL;
    }

    conn.svc = port;
    conn.host = host;
    conn.ssl_ca_file = NULL;
    conn.ssl_cert_file = NULL;
    conn.ssl_key_file = NULL;
    conn.ssl_key_pass = NULL;
    conn.vai = NULL;
    conn.vssl_ctx = NULL;

    self->exporter = fbExporterAllocNet(&conn);
    if (!self->exporter) {
        PyErr_Format(PyExc_RuntimeError,
                     "Problem setting up the Exporter on host [%s]:%s",
                     host, port);
        return NULL;
    }

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPyExporter_methods[] = {
    {"allocFile", (PyCFunction)fixbufPyExporter_allocFile, METH_VARARGS,
     ("Allocate an Exporter for an IPFIX file given input filename")},
    {"allocNet", (PyCFunction)fixbufPyExporter_allocNet,
     METH_KEYWORDS | METH_VARARGS,
     ("Allocate an Exporter for a given hostname and port")},
    {NULL, NULL, 0, NULL}
};

static PyTypeObject fixbufPyExporterType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbExporterBase",  /*tp_name*/
    sizeof(fixbufPyExporter),   /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    0,                          /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbExporter",               /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyExporter_methods,   /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbInfoElementSpec
 *
 */

/* implements InfoElementSpec.__new__ (tp_new) */
static PyObject *fixbufPyInfoElementSpec_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    fixbufPyInfoElementSpec *self;

    FBPY_UNUSED_PARAM(args);
    FBPY_UNUSED_PARAM(kwds);

    self = (fixbufPyInfoElementSpec *)type->tp_alloc(type, 0);
    if (self != NULL) {
        self->spec = PyMem_Malloc(sizeof(*self->spec));
        if (!self->spec) {
            Py_XDECREF((PyObject *)self);
            return PyErr_NoMemory();
        }
        memset(self->spec, 0, sizeof(*self->spec));
    }

    return (PyObject *)self;
}

/* implements InfoElementSpec destructor (tp_dealloc) */
static void fixbufPyInfoElementSpec_dealloc(
    fixbufPyInfoElementSpec *obj)
{
    if (obj->spec) {
        PyMem_Free(obj->spec);
    }
    Py_TYPE(obj)->tp_free((PyObject *)obj);
}


/* implements InfoElementSpec.__init__ (tp_init) */
static int fixbufPyInfoElementSpec_init(
    fixbufPyInfoElementSpec *self,
    PyObject            *args,
    PyObject            *kwds)
{
    static char *kwlist[] = {"name", "length",  NULL};
    char *name;
    int len = 0;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "s|i", kwlist, &name, &len)) {
        return -1;
    }
    if (len < 0 || len > FB_IE_VARLEN) {
        PyErr_Format(PyExc_ValueError, "Invalid element length %d", len);
        return -1;
    }

    if (self->spec) {
        strncpy(self->infoElementName, name, sizeof(self->infoElementName));
        self->infoElementName[sizeof(self->infoElementName)-1] = '\0';
        self->spec->name = self->infoElementName;
        self->spec->len_override = len;
    } else {
        return -1;
    }

    return 0;
}

/* implements comparison for InfoElementSpec (tp_richcompare) */
static PyObject *fixbufPyInfoElementSpec_richcompare(
    PyObject *self,
    PyObject *obj,
    int       cmp)
{
    long rv;

    if (!fixbufPyInfoElementSpec_Check(obj)) {
        /* set ourself to be larger than any other type other than
         * InfoElement */
        if (fixbufPyInfoElement_Check(obj)) {
            return PyBool_FromLong(cmp == Py_LT || cmp == Py_LE ||
                                   cmp == Py_NE);
        }
        return PyBool_FromLong(cmp == Py_GT || cmp == Py_GE || cmp == Py_NE);
    }

    rv = strcmp(((fixbufPyInfoElementSpec *)self)->infoElementName,
                ((fixbufPyInfoElementSpec *)obj)->infoElementName);
    if (rv == 0) {
        rv = (long)((fixbufPyInfoElementSpec *)self)->spec->len_override
            - (long)((fixbufPyInfoElementSpec *)obj)->spec->len_override;
        if (rv == 0) {
            return PyBool_FromLong(cmp == Py_LE || cmp == Py_GE
                                   || cmp == Py_EQ);
        }
    }
    if (rv < 0) {
        return PyBool_FromLong(cmp == Py_LT || cmp == Py_LE || cmp == Py_NE);
    }
    /* rv > 0 */
    return PyBool_FromLong(cmp == Py_GT || cmp == Py_GE || cmp == Py_NE);
}

/* implements InfoElementSpec.__repr__ (tp_repr) */
static PyObject *fixbufPyInfoElementSpec_repr(
    fixbufPyInfoElementSpec *obj)
{
    PyObject *args = Py_BuildValue("ssH",
                                   fixbufPyInfoElementSpecType.tp_name,
                                   obj->spec->name,
                                   obj->spec->len_override);
    PyObject *fmt = fbpyUnicode_FromString("%s(%r, %r)");
    PyObject *retval = fbpyUnicode_Format(fmt, args);
    Py_DECREF(fmt);
    Py_DECREF(args);
    return retval;
}

/* implements InfoElementSpec.__str__ (tp_str) */
static PyObject *fixbufPyInfoElementSpec_str(
    fixbufPyInfoElementSpec *obj)
{
    if (obj->spec->len_override == 0) {
        return fbpyUnicode_FromString(obj->spec->name);
    } else if (obj->spec->len_override == FB_IE_VARLEN) {
        return fbpyUnicode_FromFormat("%s[v]", obj->spec->name);
    } else {
        return fbpyUnicode_FromFormat(
            "%s[%u]", obj->spec->name, (unsigned)obj->spec->len_override);
    }
}

/* implements InfoElementSpec.length getter */
static PyObject *fixbufPyInfoElementSpec_getlength(
    fixbufPyInfoElementSpec *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return PyInt_FromLong(obj->spec->len_override);
}

/* implements InfoElementSpec.name getter */
static PyObject *fixbufPyInfoElementSpec_getname(
    fixbufPyInfoElementSpec *obj,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    return fbpyUnicode_FromString(obj->spec->name);
}

static PyGetSetDef fixbufPyInfoElementSpec_getsetters[] = {
    {"length", (getter)fixbufPyInfoElementSpec_getlength, NULL,
     "Info Element Spec Length", NULL},
    {"name", (getter)fixbufPyInfoElementSpec_getname, NULL,
     "Info Element Spec Name", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyTypeObject fixbufPyInfoElementSpecType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.InfoElementSpec", /*tp_name*/
    sizeof(fixbufPyInfoElementSpec), /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyInfoElementSpec_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)fixbufPyInfoElementSpec_repr, /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    (reprfunc)fixbufPyInfoElementSpec_str, /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,         /*tp_flags*/
    ("Creates a new Information Element Specification using the given *name*,\n"
     "and optional override *length*.  An IPFIX Template is made up of one or\n"
     "more :class:`InfoElementSpec`.\n\n"
     "The given *name* must be a defined Information Element in the\n"
     "Information Model before adding the :class:`InfoElementSpec` to a\n"
     ":class:`Template`.\n\n"
     "If *length* is nonzero, it will replace the default length of\n"
     "this Information Element (often used for reduced-length encoding).\n"
     ),                         /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    fixbufPyInfoElementSpec_richcompare, /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    fixbufPyInfoElementSpec_getsetters, /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyInfoElementSpec_init, /* tp_init */
    0,                          /* tp_alloc */
    fixbufPyInfoElementSpec_new, /* tp_new */
    0                           /* tp_free */
};


/**
 * fbSubTemplateMultiList
 *
 */

/* implements fbSTMLBase.semantic getter */
static PyObject *fixbufPySTML_getsemantic(
    fixbufPySTML *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->stml) {
        return PyInt_FromLong(fbSubTemplateMultiListGetSemantic(self->stml));
    } else {
        return PyInt_FromLong(0);
    }
}

/* implements fbSTMLBase.count getter */
static PyObject *fixbufPySTML_getcount(
    fixbufPySTML *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->stml) {
        return PyInt_FromLong(fbSubTemplateMultiListCountElements(self->stml));
    } else {
        return PyInt_FromLong(0);
    }
}

/* implements fbSTMLBase.semantic setter  */
static int fixbufPySTML_setsemantic(
    fixbufPySTML *self,
    PyObject *value,
    void *cbdata)
{
    uint8_t semantic;

    FBPY_UNUSED_PARAM(cbdata);

    if (!IS_INT(value)) {
        PyErr_SetString(PyExc_AttributeError, "Expected An Integer");
        return -1;
    }

    semantic = PyLong_AsLong(value);
    if (PyErr_Occurred()) {
        PyErr_SetString(PyExc_OverflowError,
                        "Semantic value must be an 8-bit integer");
        return -1;
    }

    if (self->stml) {
        fbSubTemplateMultiListSetSemantic(self->stml, semantic);
    }

    return 0;
}


/* implements fbSTMLBase.clear method */
static PyObject *fixbufPySTML_clear(
    fixbufPySTML      *self,
    PyObject          *args)
{
    int offset = 0;
    fixbufPyRecord *rec = NULL;

    if (!PyArg_ParseTuple(args, "|Oi", &rec, &offset)) {
        return NULL;
    }

    if (offset) {
        if (!fixbufPyRecord_Check(rec)) {
            PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.Record");
            return NULL;
        }
        self->stml = (fbSubTemplateMultiList_t *)(rec->rec + offset);
    }

    fbSubTemplateMultiListClear(self->stml);
    self->entry = NULL;
    self->stml = NULL;

    Py_RETURN_NONE;
}


/*
static PyObject *fixbufPySTML_getRecord(
    fixbufPySTML *self,
    PyObject *args)
{
    fixbufPyRecord *rec = NULL;

    if (!PyArg_ParseTuple(args, "O", &rec)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected pyfixbuf.rec");
        return NULL;
    }

    if (rec == NULL) {
        PyErr_SetString(PyExc_AttributeError, "Invalid Record - Null record");
        return NULL;
    }

    if (!self) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateMultiList - NULL STML");
        return NULL;
    }

    if (!self->entry) {
        Py_RETURN_FALSE;
    }

    rec->rec = fbSubTemplateMultiListEntryNextDataPtr(self->entry, rec->rec);

    if (rec->rec == NULL) {
          Py_RETURN_FALSE;
    }

    Py_RETURN_TRUE;
}

*/

/* implements fbSTMLBase.getNextEntry method */
static PyObject *fixbufPySTML_getNextEntry(
    fixbufPySTML        *self,
    PyObject            *args)
{
    fixbufPyRecord *rec = NULL;
    int offset = 0;

    if (!PyArg_ParseTuple(args, "Oi", &rec, &offset)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    self->stml = (fbSubTemplateMultiList_t *)(rec->rec + offset);

    self->entry =
        fbSubTemplateMultiListGetNextEntry(self->stml, self->entry);

    if (!self->entry) {
        /* We are done processing this list */
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }

    Py_RETURN_NONE;
}

/* implements fbSTMLBase.getFirstEntry method */
static PyObject *fixbufPySTML_getFirstEntry(
    fixbufPySTML        *self,
    PyObject            *args)
{
    fixbufPyRecord *rec = NULL;
    int offset = 0;

    if (!PyArg_ParseTuple(args, "Oi", &rec, &offset)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    self->stml = (fbSubTemplateMultiList_t *)(rec->rec + offset);

    self->entry =
        fbSubTemplateMultiListGetFirstEntry(self->stml);

    Py_RETURN_NONE;
}


/* implements fbSTMLBase.getIndex method */
static PyObject *fixbufPySTML_getIndex(
    fixbufPySTML        *self,
    PyObject            *args)
{
    int index = 0;
    fbSubTemplateMultiListEntry_t *entry = NULL;

    if (!PyArg_ParseTuple(args, "i", &index)) {
        return NULL;
    }

    if (!self->stml) {
        PyErr_SetString(PyExc_ValueError, "STML was not initialized.");
        return NULL;
    }

    entry = fbSubTemplateMultiListGetIndexedEntry(self->stml, index);

    if (entry == NULL) {
        PyErr_SetString(PyExc_IndexError, "Index Out of Bounds");
        return NULL;
    }

    self->entry = entry;

    Py_RETURN_NONE;
}

/* implements fbSTMLBase destructor (tp_dealloc) */
static void fixbufPySTML_dealloc(
    fixbufPySTML *obj)
{
    if (obj->stml_alloc) {
        PyMem_Free(obj->stml);
    }
    Py_TYPE(obj)->tp_free((PyObject *)obj);
}

/* implements fbSTMLBase.__init__ (tp_init) */
static int fixbufPySTML_init(
    fixbufPySTML *self,
    PyObject     *args,
    PyObject     *kwds)
{
    int tmplcount = -1;
    int offset = 0;
    fixbufPyRecord *rec = NULL;

    FBPY_UNUSED_PARAM(kwds);

    if (!PyArg_ParseTuple(args, "|Oii", &rec, &offset, &tmplcount)) {
        return -1;
    }

    /* get stml */
    if (fixbufPyRecord_Check(rec)) {
        self->stml = (fbSubTemplateMultiList_t *)(rec->rec + offset);
        self->entry = NULL;
        return 0;
    }

    if (tmplcount > -1) {
        /*if (rec->rec == NULL) {
          rec->rec = PyMem_Malloc(rec->reclen);
          if (!rec->rec) {
          Py_DECREF((PyObject *)rec);
          PyErr_NoMemory();
          return -1;
          }
          rec->memalloc = TRUE;
                }*/
        self->stml = PyMem_Malloc(sizeof(fbSubTemplateMultiList_t));
        if (!self->stml) {
            PyErr_NoMemory();
            return -1;
        }
        memset(self->stml, 0, sizeof(fbSubTemplateMultiList_t));
        self->stml_alloc = TRUE;
        fbSubTemplateMultiListInit(self->stml, 0, tmplcount);
    } else {
        self->stml = NULL;
    }

    self->entry = NULL;

    return 0;
}


/* implements fbSTMLBase.rewind method */
static PyObject *fixbufPySTML_rewind(
    fixbufPySTML *self)
{
    self->entry = NULL;

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPySTML_methods[] = {
    {"getNextEntry", (PyCFunction)fixbufPySTML_getNextEntry, METH_VARARGS,
     ("Get Next SubTemplateMultiList in the list - or NULL if done")},
    {"clear", (PyCFunction)fixbufPySTML_clear, METH_VARARGS,
     ("Clear the STML")},
    {"getFirstEntry", (PyCFunction)fixbufPySTML_getFirstEntry, METH_VARARGS,
     ("Set the entry pointer to the first item in the list.")},
    {"rewind", (PyCFunction)fixbufPySTML_rewind, METH_NOARGS,
     ("Rewind entry pointer to first item in the list.")},
    {"getIndex", (PyCFunction)fixbufPySTML_getIndex, METH_VARARGS,
     ("Get the Indexed Entry in the list.")},
    {NULL, NULL, 0, NULL}
};

static PyGetSetDef fixbufPySTML_getsetters[] = {
    {"semantic", (getter)fixbufPySTML_getsemantic,
     (setter)fixbufPySTML_setsemantic, "Get/Set STML Semantic", NULL},
    {"count", (getter)fixbufPySTML_getcount,
     NULL, "Get the number of entries in the list.", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyTypeObject fixbufPySTMLType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbSTMLBase",      /*tp_name*/
    sizeof(fixbufPySTML),       /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPySTML_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbSTML",                   /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPySTML_methods,       /* tp_methods */
    0,                          /* tp_members */
    fixbufPySTML_getsetters,    /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPySTML_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbSubTemplateMultiListEntry
 *
 */

/* implements fbSTMLEntryBase.entryInit method */
static PyObject *fixbufPySTMLEntry_entryInit(
    fixbufPySTMLEntry *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"record", "template", "template_id", "count",
                             NULL};
    fixbufPyTemplate *tmpl = NULL;
    fixbufPyRecord *rec = NULL;
    int tid;
    int count;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "OOii", kwlist,
                                     &rec, &tmpl, &tid, &count))
    {
        return NULL;
    }

    if (!fixbufPyTemplate_Check(tmpl)) {
        PyErr_SetString(PyExc_TypeError, "Expected Template");
        return NULL;
    }

    fbSubTemplateMultiListEntryInit(self->entry, tid, tmpl->template,
                                    count);
    Py_RETURN_NONE;
}


/* implements fbSTMLEntryBase.getNextRecord method */
static PyObject *fixbufPySTMLEntry_getNextRecord(
    fixbufPySTMLEntry *self,
    PyObject *args)
{
    fixbufPyRecord *rec = NULL;

    if (!PyArg_ParseTuple(args, "O", &rec)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    if (rec == NULL) {
        PyErr_SetString(PyExc_AttributeError, "Invalid Record - Null record");
        return NULL;
    }

    if (!self) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateMultiList "
                        "- NULL STML");
        return NULL;
    }

    if (!self->entry) {
        /* getNextEntry - should have returned False */
        Py_RETURN_FALSE;
    }

    rec->rec = fbSubTemplateMultiListEntryNextDataPtr(self->entry, rec->rec);

    if (rec->rec == NULL) {
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }

    Py_RETURN_NONE;
}


/* implements fbSTMLEntryBase.setIndexedEntry method */
static PyObject *fixbufPySTMLEntry_setIndexedEntry(
    fixbufPySTMLEntry *self,
    PyObject *args)
{
    fixbufPyRecord *rec = NULL;
    uint8_t *dataptr;
    int index;

    if (!PyArg_ParseTuple(args, "iO", &index, &rec)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    if (!self) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateMultiList "
                        "- NULL STML");
        return NULL;
    }

    if (!self->entry) {
        Py_RETURN_FALSE;
    }

    dataptr = fbSubTemplateMultiListEntryGetIndexedPtr(self->entry, index);

    if (dataptr == NULL) {
        PyErr_SetString(PyExc_IndexError, "Out of Bounds");
        return NULL;
    }

    memcpy(dataptr, rec->rec, rec->reclen);

    Py_RETURN_NONE;
}

/* implements fbSTMLEntryBase.getIndexedEntry method */
static PyObject *fixbufPySTMLEntry_getIndexedEntry(
    fixbufPySTMLEntry *self,
    PyObject *args)
{
    fixbufPyRecord *rec = NULL;
    int index;

    if (!PyArg_ParseTuple(args, "Oi", &rec, &index)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    if (!self) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateMultiList "
                        "- NULL STML");
        return NULL;
    }

    if (!self->entry) {
        Py_RETURN_FALSE;
    }

    rec->rec = fbSubTemplateMultiListEntryGetIndexedPtr(self->entry, index);

    if (rec->rec == NULL) {
        PyErr_SetString(PyExc_IndexError, "Index Out of Bounds");
        return NULL;
    }

    Py_RETURN_NONE;
}


/* implements fbSTMLEntryBase.containsElement method */
static PyObject *fixbufPySTMLEntry_containsElement(
    fixbufPySTMLEntry *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"model", "name", NULL};
    char *name;
    fixbufPyInfoModel *model;
    PyObject *result;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "Os", kwlist, &model, &name)) {
        return NULL;
    }

    if (!fixbufPyInfoModel_Check(model)) {
        PyErr_SetString(PyExc_TypeError, "Expected InfoModel");
        return NULL;
    }

    if (!self->entry) {
        result = Py_False;
        goto done;
    }

    if (!model->infoModel) {
        PyErr_SetString(PyExc_AttributeError, "Invalid InfoModel - NULL");
        return NULL;
    }

    if (fbSubTemplateMultiListEntryGetTemplate(self->entry)) {
        if (fbTemplateContainsElement(
                (fbTemplate_t *)fbSubTemplateMultiListEntryGetTemplate(self->entry),
                fbInfoModelGetElementByName(model->infoModel, (const char *)name)))
        {
            result = Py_True;
        } else {
            result = Py_False;
        }
    } else {
        result = Py_False;
    }

done:
    Py_INCREF(result);

    return result;
}

/* implements fbSTMLEntryBase.__init__ (tp_init) */
static int fixbufPySTMLEntry_init(
    fixbufPySTMLEntry *self,
    PyObject     *args,
    PyObject     *kwds)
{
    fixbufPySTML *stml = NULL;
    static char *kwlist[] = {"stml", NULL};

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &stml)) {
        return -1;
    }

    if (!fixbufPySTML_Check(stml)) {
        PyErr_SetString(PyExc_TypeError, "Expected STML");
        return -1;
    }

    self->entry = stml->entry;

    return 0;
}


/* implements fbSTMLEntryBase.template_id getter */
static PyObject *fixbufPySTMLEntry_gettid(
    fixbufPySTMLEntry *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self) {
        return PyInt_FromLong(
            fbSubTemplateMultiListEntryGetTemplateID(self->entry));
    } else {
        return PyInt_FromLong(0);
    }
}

/* implements fbSTMLEntryBase.count getter */
static PyObject *fixbufPySTMLEntry_getcount(
    fixbufPySTMLEntry *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self) {
        return PyInt_FromLong(self->entry->numElements);
    } else {
        return PyInt_FromLong(0);
    }
}

static PyGetSetDef fixbufPySTMLEntry_getsetters[] = {
    {"template_id", (getter)fixbufPySTMLEntry_gettid, NULL,
     "SubTemplateMultiList Entry Template ID", NULL},
    {"count", (getter)fixbufPySTMLEntry_getcount, NULL,
     "Get the number of entries in the list", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyMethodDef fixbufPySTMLEntry_methods[] = {
    {"containsElement", (PyCFunction)fixbufPySTMLEntry_containsElement,
     METH_KEYWORDS | METH_VARARGS,
     ("Find out if the entry contains the element, returns True or False")},
    {"getNextRecord", (PyCFunction)fixbufPySTMLEntry_getNextRecord,
     METH_VARARGS,
     ("Get the element we want out of the STML, returns 0 if Does not exist")},
    {"setIndexedEntry", (PyCFunction)fixbufPySTMLEntry_setIndexedEntry,
     METH_VARARGS,
     ("Set the given Record at the appropriate entry")},
    {"getIndexedEntry", (PyCFunction)fixbufPySTMLEntry_getIndexedEntry,
     METH_VARARGS,
     ("Get the Record at the appropriate entry")},
/*    {"clear", (PyCFunction)fixbufPySTML_clear, METH_VARARGS,
      ("Clear the STML")},*/
    {"entryInit", (PyCFunction)fixbufPySTMLEntry_entryInit,
     METH_KEYWORDS | METH_VARARGS,
     ("Initialize Entry in STML for Export")},
    {NULL, NULL, 0, NULL}
};

static PyTypeObject fixbufPySTMLEntryType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbSTMLEntryBase", /*tp_name*/
    sizeof(fixbufPySTMLEntry),  /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    obj_dealloc,                /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbSTMLEntry",              /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPySTMLEntry_methods,  /* tp_methods */
    0,                          /* tp_members */
    fixbufPySTMLEntry_getsetters, /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPySTMLEntry_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbSubTemplateList
 *
 */

/* implements fbSTLBase.__init__ (tp_init) */
static int fixbufPySTL_init(
    fixbufPySTL *self,
    PyObject     *args,
    PyObject     *kwds)
{
    static char *kwlist[] = {"rec", "offset", NULL};
    fixbufPyRecord *rec = NULL;
    int offset = 0;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|Oi", kwlist, &rec, &offset))
    {
        return -1;
    }

    if (rec) {
        if (fixbufPyRecord_Check(rec)) {
            /* set the subtemplate list to the first one */
            if (rec->rec == NULL) {
                rec->rec = PyMem_Malloc(rec->reclen);
                if (!rec->rec) {
                    Py_XDECREF((PyObject *)rec);
                    PyErr_NoMemory();
                    return -1;
                }
                memset(rec->rec, 0, rec->reclen);
                rec->memalloc = TRUE;
            }
            self->stl = (fbSubTemplateList_t *)(rec->rec + offset);
        } else {
            PyErr_SetString(PyExc_TypeError, "Expected Valid Record");
            return -1;
        }
    } else {
        self->stl = PyMem_Malloc(sizeof(fbSubTemplateList_t));
        if (!self->stl) {
            PyErr_NoMemory();
            return -1;
        }
        self->stl_alloc = TRUE;
        memset(self->stl, 0, sizeof(fbSubTemplateList_t));
    }

    return 0;
}


/* implements fbSTLBase destructor (tp_dealloc) */
static void fixbufPySTL_dealloc(
    fixbufPySTL *obj)
{
    if (obj->stl_alloc) {
        PyMem_Free(obj->stl);
    }
    Py_TYPE(obj)->tp_free((PyObject *)obj);
}


/* implements fbSTLBase.containsElement method */
static PyObject *fixbufPySTL_containsElement(
    fixbufPySTL *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"model", "name", NULL};
    char *name;
    fixbufPyInfoModel *model;
    PyObject *result;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "Os", kwlist, &model, &name)) {
        return NULL;
    }

    if (!fixbufPyInfoModel_Check(model)) {
        PyErr_SetString(PyExc_TypeError, "Expected InfoModel");
        return NULL;
    }

    if (!self->stl) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateList = NULL");
        return NULL;
    }

    if (!model->infoModel) {
        PyErr_SetString(PyExc_AttributeError, "Invalid InfoModel = NULL");
        return NULL;
    }

    if (fbTemplateContainsElement(
            (fbTemplate_t *)fbSubTemplateListGetTemplate(self->stl),
            fbInfoModelGetElementByName(model->infoModel, (const char *)name)))
    {
        result = Py_True;
    } else {
        result = Py_False;
    }

    Py_INCREF(result);

    return result;
}


/* implements fbSTLBase.getNext method */
static PyObject *fixbufPySTL_getNextEntry(
    fixbufPySTL        *self,
    PyObject            *args,
    PyObject            *kwds)
{
    static char *kwlist[] = {"rec", NULL};
    fixbufPyRecord *rec;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &rec)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    if (!self->stl) {
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }

    rec->rec = fbSubTemplateListGetNextPtr((fbSubTemplateList_t *)self->stl,
                                           rec->rec);
    if (!rec->rec) {
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }

    Py_RETURN_NONE;
}


/* implements fbSTLBase.clear method */
static PyObject *fixbufPySTL_clear(
    fixbufPySTL      *self,
    PyObject         *args)
{
    int offset = -1;
    fixbufPyRecord *rec = NULL;

    if (!PyArg_ParseTuple(args, "|Oi", &rec, &offset)) {
        return NULL;
    }
    if (offset != -1) {
        self->stl = (fbSubTemplateList_t *)(rec->rec + offset);
    }

    fbSubTemplateListClear(self->stl);

    Py_RETURN_NONE;
}


/* implements fbSTLBase.template_id getter */
static PyObject *fixbufPySTL_gettid(
    fixbufPySTL *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->stl) {
        return PyInt_FromLong(fbSubTemplateListGetTemplateID(self->stl));
    } else {
        return PyInt_FromLong(0);
    }
}

/* implements fbSTLBase.entryInit method */
static PyObject *fixbufPySTL_entryInit(
    fixbufPySTL *self,
    PyObject *args,
    PyObject *kwds)
{
    static char *kwlist[] = {"template", "template_id", "count", NULL};
    fixbufPyTemplate *tmpl = NULL;
    int tid;
    int count;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "Oii", kwlist,
                                     &tmpl, &tid, &count))
    {
        return NULL;
    }

    if (!fixbufPyTemplate_Check(tmpl)) {
        PyErr_SetString(PyExc_TypeError, "Expected Template");
        return NULL;
    }

    fbSubTemplateListInit(self->stl, 0, tid, tmpl->template, count);

    Py_RETURN_NONE;
}

/* implements fbSTLBase.count getter */
static PyObject *fixbufPySTL_getcount(
    fixbufPySTL *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->stl) {
        return PyInt_FromLong(fbSubTemplateListCountElements(self->stl));
    } else {
        return PyInt_FromLong(0);
    }
}

/* implements fbSTLBase.semantic getter */
static PyObject *fixbufPySTL_getsemantic(
    fixbufPySTL *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->stl) {
        return PyInt_FromLong(fbSubTemplateListGetSemantic(self->stl));
    } else {
        Py_RETURN_NONE;
    }
}

/* implements fbSTLBase.semantic setter */
static int fixbufPySTL_setsemantic(
    fixbufPySTL *self,
    PyObject *value,
    void *cbdata)
{
    uint8_t semantic;

    FBPY_UNUSED_PARAM(cbdata);

    if (!IS_INT(value)) {
        PyErr_SetString(PyExc_AttributeError, "Expected An Integer");
        return -1;
    }

    semantic = PyLong_AsLong(value);
    if (PyErr_Occurred()) {
        PyErr_SetString(PyExc_OverflowError,
                        "Semantic value must be an 8-bit integer");
        return -1;
    }

    if (self->stl) {
        fbSubTemplateListSetSemantic(self->stl, semantic);
    } else {
        PyErr_SetString(PyExc_RuntimeError, "entry_init() must be called "
                        "on STL before semantic can be set.");
        return -1;
    }

    return 0;
}

/* implements fbSTLBase.getIndexedEntry method */
static PyObject *fixbufPySTL_getIndexedEntry(
    fixbufPySTL *self,
    PyObject *args)
{
    fixbufPyRecord *rec = NULL;
    int index;

    if (!PyArg_ParseTuple(args, "Oi", &rec, &index)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    if (!self->stl) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateList "
                        "- NULL STL");
        return NULL;
    }

    rec->rec = fbSubTemplateListGetIndexedDataPtr(self->stl, index);

    if (rec->rec == NULL) {
        PyErr_SetString(PyExc_IndexError, "Index Out of Bounds");
        return NULL;
    }

    Py_RETURN_NONE;
}

/* implements fbSTLBase.setIndexedEntry method */
static PyObject *fixbufPySTL_setIndexedEntry(
    fixbufPySTL *self,
    PyObject *args)
{
    fixbufPyRecord *rec = NULL;
    uint8_t *dataptr;
    int index;

    if (!PyArg_ParseTuple(args, "iO", &index, &rec)) {
        return NULL;
    }

    if (!fixbufPyRecord_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Expected Record");
        return NULL;
    }

    if (!self->stl) {
        PyErr_SetString(PyExc_AttributeError, "Invalid SubTemplateList "
                        "- NULL STL");
        return NULL;
    }

    dataptr = fbSubTemplateListGetIndexedDataPtr(self->stl, index);

    if (dataptr == NULL) {
        PyErr_SetString(PyExc_IndexError, "Out of Bounds");
        return NULL;
    }

    memcpy(dataptr, rec->rec, rec->reclen);

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPySTL_methods[] = {
    {"getNext", (PyCFunction)fixbufPySTL_getNextEntry,
     METH_KEYWORDS | METH_VARARGS,
     ("Get Next SubTemplateList Entry in the list - or NULL if done")},
    {"clear", (PyCFunction)fixbufPySTL_clear, METH_VARARGS,
     ("Clear the STL")},
    {"getIndexedEntry", (PyCFunction)fixbufPySTL_getIndexedEntry, METH_VARARGS,
     ("get the data ptr for the given index.")},
    {"setIndexedEntry", (PyCFunction)fixbufPySTL_setIndexedEntry, METH_VARARGS,
     ("set the data ptr at the index to the given record.")},
    {"containsElement", (PyCFunction)fixbufPySTL_containsElement,
     METH_KEYWORDS | METH_VARARGS,
     ("Find out if the entry contains the element, returns True or False")},
    {"entryInit", (PyCFunction)fixbufPySTL_entryInit,
     METH_KEYWORDS | METH_VARARGS,
     ("Initialize the STL for Export")},
    {NULL, NULL, 0, NULL}
};

static PyGetSetDef fixbufPySTL_getsetters[] = {
    {"template_id", (getter)fixbufPySTL_gettid, NULL,
     "SubTemplateList Template ID", NULL},
    {"count", (getter)fixbufPySTL_getcount, NULL,
     "Get number of entries in list", NULL},
    {"semantic", (getter)fixbufPySTL_getsemantic,
     (setter)fixbufPySTL_setsemantic,
     "Get/Set the Semantic Value on the SubTemplateList", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyTypeObject fixbufPySTLType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbSTLBase",       /*tp_name*/
    sizeof(fixbufPySTL),        /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPySTL_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbSTL",                    /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPySTL_methods,        /* tp_methods */
    0,                          /* tp_members */
    fixbufPySTL_getsetters,     /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPySTL_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0                           /* tp_free */
};


/**
 * fbBasicList
 *
 *
 */

/* implements fbBLBase.__new__ (tp_new) */
static PyObject *fixbufPyBL_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    fixbufPyBL *self;

    FBPY_UNUSED_PARAM(args);
    FBPY_UNUSED_PARAM(kwds);

    self = (fixbufPyBL *)type->tp_alloc(type, 0);
    if (self != NULL) {
        self->bl = PyMem_Malloc(sizeof(fbBasicList_t));
        if (!self->bl) {
            Py_XDECREF((PyObject *)self);
            return PyErr_NoMemory();
        }
        memset(self->bl, 0, sizeof(fbBasicList_t));
    }
    self->bl_alloc = TRUE;
    return (PyObject *)self;
}

/* implements fbBLBase destructor (tp_dealloc) */
static void fixbufPyBL_dealloc(
    fixbufPyBL *obj)
{
    if (obj->bl_alloc) {
        PyMem_Free(obj->bl);
    }
    Py_TYPE(obj)->tp_free((PyObject *)obj);
}

/* implements fbBLBase.__init__ (tp_init) */
static int fixbufPyBL_init(
    fixbufPyBL   *self,
    PyObject     *args,
    PyObject     *kwds)
{
    static char *kwlist[] = {"element", "count", "semantic", NULL};
    fixbufPyInfoElement *ie = NULL;
    int count = 0;
    int semantic = 0;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|Oii", kwlist,
                                     &ie, &count, &semantic))
    {
        return -1;
    }

    if (count < 0) {
        PyErr_SetString(PyExc_ValueError,
                        "Count must be greater or equal to 0.");
        return -1;
    }

    if (ie == NULL) {
        return 0;
    }

    if (!(fixbufPyInfoElement_Check(ie))) {
        return 0;
    } else {
        if (self->bl_alloc) {
            fbBasicListInit(self->bl, semantic, ie->ptr, count);
            self->init = TRUE;
        }
    }

    return 0;
}

/* implements fbBLBase.getitems method */
static PyObject *fixbufPyBL_getitems(
    fixbufPyBL *self,
    PyObject *args)
{
    PyObject *list;
    fbBasicList_t *bl = self->bl;
    const fbInfoElement_t *ie = NULL;
    int w;
    void *v;
    int mem_len;
    int is_varlen;

    FBPY_UNUSED_PARAM(args);

    if (!self->bl) {
        PyErr_SetString(PyExc_ValueError, "Null BasicList");
        return NULL;
    }

    mem_len = fbBasicListGetElementLength(bl);

    ie = fbBasicListGetInfoElement(bl);
    if (ie == NULL) {
        PyErr_SetString(PyExc_ValueError,
                        "Null InfoElement associated with BL");
        return NULL;
    }

    list = PyList_New(fbBasicListCountElements(bl));
    if (list == NULL) {
        PyErr_SetString(PyExc_ValueError, "Could not create List Object");
        return NULL;
    }

    if (ie->len == FB_IE_VARLEN) {
        mem_len = sizeof(fbVarfield_t);
        is_varlen = 1;
    } else {
        mem_len = ie->len;
        is_varlen = 0;
    }
    for (w = 0; (v = fbBasicListGetIndexedDataPtr(bl, w)); ++w) {
        PyList_SetItem(list, w,
                       pyfixbuf_get_value(ie->type, v, mem_len, is_varlen));
    }

    return list;
}

/* implements fbBLBase.setitems method */
static PyObject *fixbufPyBL_setitems(
    fixbufPyBL *self,
    PyObject *args)
{
    PyObject *value;
    const fbInfoElement_t * ie;
    int index = 0;
    fbBasicList_t *bl = self->bl;
    void *v;

    if (!PyArg_ParseTuple(args, "iO", &index, &value)) {
        return NULL;
    }

    if (!self->bl) {
        PyErr_SetString(PyExc_ValueError, "Null BasicList: BL must be "
                        "initialized before setting.");
        return NULL;
    }

    if (self->init == FALSE) {
        PyErr_SetString(PyExc_AttributeError, "Basic List must be initialized "
                        "with InfoElement before setting.");
        return NULL;
    }

    if ( index >= fbBasicListCountElements(self->bl) ) {
        PyErr_Format(PyExc_ValueError,
                     "Invalid: Trying to add item %d to BasicList of "
                     "size %d", index + 1,
                     fbBasicListCountElements(self->bl));
        return NULL;
    }

    ie = fbBasicListGetInfoElement(self->bl);
    if (ie == NULL) {
        PyErr_SetString(PyExc_ValueError, "No InfoElement associated with "
                        "basicList.");
        return NULL;
    }

    v = fbBasicListGetIndexedDataPtr(bl, index);
    if (NULL == v) {
        PyErr_Format(PyExc_ValueError,
                     "Invalid Set for Item %d in BasicList", index);
        return NULL;
    }

    if (FB_IE_VARLEN == ie->len) {
        return pyfixbuf_set_value(ie->type, v, sizeof(fbVarfield_t), value, 1);
    }
    return pyfixbuf_set_value(ie->type, v, ie->len, value, 0);
}


/* implements fbBLBase.count getter */
static PyObject *fixbufPyBL_getcount(
    fixbufPyBL *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->bl) {
        return PyInt_FromLong(fbBasicListCountElements(self->bl));
    } else {
        return PyInt_FromLong(0);
    }
}

/* implements fbBLBase.semantic getter */
static PyObject *fixbufPyBL_getsemantic(
    fixbufPyBL *self,
    void *cbdata)
{
    FBPY_UNUSED_PARAM(cbdata);
    if (self->bl) {
        return PyInt_FromLong(fbBasicListGetSemantic(self->bl));
    } else {
        Py_RETURN_NONE;
    }
}

/* implements fbBLBase.semantic setter */
static int fixbufPyBL_setsemantic(
    fixbufPyBL *self,
    PyObject *value,
    void *cbdata)
{
    uint8_t semantic;

    FBPY_UNUSED_PARAM(cbdata);

    if (!IS_INT(value)) {
        PyErr_SetString(PyExc_AttributeError, "Expected An Integer");
        return -1;
    }

    semantic = PyLong_AsLong(value);
    if (PyErr_Occurred()) {
        PyErr_SetString(PyExc_OverflowError,
                        "Semantic value must be an 8-bit integer");
        return -1;
    }

    if (self->bl) {
        fbBasicListSetSemantic(self->bl, semantic);
    } else {
        PyErr_SetString(PyExc_RuntimeError, "BL must be initialized "
                        "before semantic can be set.");
        return -1;
    }

    return 0;
}

/* implements fbBLBase.element getter */
static PyObject *fixbufPyBL_getelement(
    fixbufPyBL *self,
    void *cbdata)
{
    fixbufPyInfoElement *element = NULL;
    const fbInfoElement_t *ie = NULL;

    FBPY_UNUSED_PARAM(cbdata);

    if (self->bl) {
        ie = fbBasicListGetInfoElement(self->bl);
    }

    if (ie) {
        element = ((fixbufPyInfoElement *)fixbufPyInfoElementType.tp_new(
                       &fixbufPyInfoElementType, NULL, NULL));
        if (element == NULL) {
            return PyErr_NoMemory();
        }
        memcpy(element->infoElement, ie, sizeof(fbInfoElement_t));

        /*Py_INCREF(element);*/

        return (PyObject *)element;
    } else {

        Py_RETURN_NONE;
    }
}

/* implements fbBLBase.clear method */
static PyObject *fixbufPyBL_clear(
    fixbufPyBL *self,
    PyObject *args)
{
    FBPY_UNUSED_PARAM(args);
    if (!self->bl) {
        PyErr_SetString(PyExc_AttributeError, "No basicList to Clear");
        return NULL;
    }

    fbBasicListClear(self->bl);

    Py_RETURN_NONE;
}

static PyMethodDef fixbufPyBL_methods[] = {
    {"clear", (PyCFunction)fixbufPyBL_clear, METH_VARARGS,
     ("Clear the BasicList")},
    {"getitems", (PyCFunction)fixbufPyBL_getitems, METH_VARARGS,
     ("returns the Python list for the basicList")},
    {"setitems", (PyCFunction)fixbufPyBL_setitems, METH_VARARGS,
     ("set the basicList")},
    {NULL, NULL, 0, NULL}
};

static PyGetSetDef fixbufPyBL_getsetters[] = {
    {"count", (getter)fixbufPyBL_getcount, NULL,
     "Get number of entries in list", NULL},
    {"semantic", (getter)fixbufPyBL_getsemantic,(setter)fixbufPyBL_setsemantic,
     "Get/Set the Semantic Value on the basicList", NULL},
    {"element", (getter)fixbufPyBL_getelement, NULL,
     "Get the InfoElement set on the basicList", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyTypeObject fixbufPyBLType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "pyfixbuf.fbBLBase",        /*tp_name*/
    sizeof(fixbufPyBL),         /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    (destructor)fixbufPyBL_dealloc, /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    0,                          /*tp_hash */
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    0,                          /*tp_getattro*/
    0,                          /*tp_setattro*/
    0,                          /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
    "fbBL",                     /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    fixbufPyBL_methods,         /* tp_methods */
    0,                          /* tp_members */
    fixbufPyBL_getsetters,      /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)fixbufPyBL_init,  /* tp_init */
    0,                          /* tp_alloc */
    fixbufPyBL_new,             /* tp_new */
    0                           /* tp_free */
};


static gboolean pyfixbuf_initializer(
    PyObject *fixbuf)
{
    int rv;

    if (fixbuf == NULL) {
        fprintf(stderr, "Could not create module pyfixbuf\n");
        return FALSE;
    }

    /* adding info model type */

    fixbufPyInfoModelType.tp_new = PyType_GenericNew;

    if (PyType_Ready(&fixbufPyInfoModelType) < 0) {
        return FALSE;
    }

    Py_INCREF(&fixbufPyInfoModelType);
    rv = PyModule_AddObject(fixbuf, "fbInfoModelBase",
                            (PyObject *)&fixbufPyInfoModelType);
    assert(rv == 0);

    /* adding info element type */

    if (PyType_Ready(&fixbufPyInfoElementType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPyInfoElementType);
    rv = PyModule_AddObject(fixbuf, "InfoElement",
                            (PyObject *)&fixbufPyInfoElementType);
    assert(rv == 0);

    /* adding session type */

    fixbufPySessionType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPySessionType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPySessionType);
    rv = PyModule_AddObject(fixbuf, "fbSessionBase",
                            (PyObject *)&fixbufPySessionType);
    assert(rv == 0);

    /* adding collector type */

    fixbufPyCollectorType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPyCollectorType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPyCollectorType);
    rv = PyModule_AddObject(fixbuf, "fbCollectorBase",
                            (PyObject *)&fixbufPyCollectorType);
    assert(rv == 0);

    /* adding listener type */

    fixbufPyListenerType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPyListenerType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPyListenerType);
    rv = PyModule_AddObject(fixbuf, "fbListenerBase",
                            (PyObject *)&fixbufPyListenerType);
    assert(rv == 0);

    /* adding exporter type */

    fixbufPyExporterType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPyExporterType) < 0) {
        return FALSE;
    }

    Py_INCREF(&fixbufPyExporterType);
    rv = PyModule_AddObject(fixbuf, "fbExporterBase",
                            (PyObject *)&fixbufPyExporterType);
    assert(rv == 0);

    /* adding Template Type */

    fixbufPyTemplateType.tp_new = PyType_GenericNew;

    if (PyType_Ready(&fixbufPyTemplateType) < 0) {
        return FALSE;
    }

    Py_INCREF(&fixbufPyTemplateType);
    rv = PyModule_AddObject(fixbuf, "fbTemplateBase",
                            (PyObject *)&fixbufPyTemplateType);
    assert(rv == 0);

    /* adding Buffer type */
    fixbufPyfBufType.tp_new = PyType_GenericNew;

    if (PyType_Ready(&fixbufPyfBufType) < 0) {
        return FALSE;
    }

    Py_INCREF(&fixbufPyfBufType);
    rv = PyModule_AddObject(fixbuf, "fBufBase",
                            (PyObject *)&fixbufPyfBufType);
    assert(rv == 0);

    /*adding info element spec type */

    if (PyType_Ready(&fixbufPyInfoElementSpecType) < 0) {
        return FALSE;
    }

    Py_INCREF(&fixbufPyInfoElementSpecType);
    rv = PyModule_AddObject(fixbuf, "InfoElementSpec",
                            (PyObject *)&fixbufPyInfoElementSpecType);
    assert(rv == 0);

    fixbufPyRecordType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPyRecordType) < 0) {
        return FALSE;
    }

    Py_INCREF(&fixbufPyRecordType);
    rv = PyModule_AddObject(fixbuf, "fbRecordBase",
                            (PyObject *)&fixbufPyRecordType);
    assert(rv == 0);

    fixbufPySTMLType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPySTMLType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPySTMLType);
    rv = PyModule_AddObject(fixbuf, "fbSTMLBase",
                            (PyObject *)&fixbufPySTMLType);
    assert(rv == 0);

    fixbufPySTMLEntryType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPySTMLEntryType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPySTMLEntryType);
    rv = PyModule_AddObject(fixbuf, "fbSTMLEntryBase",
                            (PyObject *)&fixbufPySTMLEntryType);
    assert(rv == 0);

    fixbufPySTLType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&fixbufPySTLType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPySTLType);
    rv = PyModule_AddObject(fixbuf, "fbSTLBase", (PyObject *)&fixbufPySTLType);
    assert(rv == 0);

    /* adding basicList type */
    if (PyType_Ready(&fixbufPyBLType) < 0) {
        return FALSE;
    }
    Py_INCREF(&fixbufPyBLType);
    rv = PyModule_AddObject(fixbuf, "fbBLBase",
                            (PyObject *)&fixbufPyBLType);
    assert(rv == 0);

    return TRUE;
}


/*
 *    declarations for DLL input/export changed between Python 2 and
 *    Python 3.  The entry point in Python 2 is called "initFOO" and
 *    it returns void; the entry point in Python 3 is called
 *    PyInit_FOO and it returns PyObject* (the module).
 */
#if PY_MAJOR_VERSION >= 3
PyObject *PyInit__pyfixbuf(
    void)
{
    static struct PyModuleDef pyfixbuf_module = {
        PyModuleDef_HEAD_INIT,
        "pyfixbuf._pyfixbuf",       /* m_name */
        "Fixbuf Extension Module",  /* m_doc */
        0,                          /* m_size */
        NULL,                       /* m_methods */
        NULL,                       /* m_reaload (unused) */
        NULL,                       /* m_traverse */
        NULL,                       /* m_clear */
        NULL                        /* m_free */
    };
    PyObject *fixbuf;

    fixbuf = PyModule_Create(&pyfixbuf_module);

    if (pyfixbuf_initializer(fixbuf) == FALSE) {
        Py_XDECREF(fixbuf);
        return NULL;
    }

    return fixbuf;
}

#else  /* PY_MAJOR_VERSION < 3 */

void init_pyfixbuf(
    void)
{
    PyObject *fixbuf;

    fixbuf = Py_InitModule3("_pyfixbuf", NULL, "Fixbuf Extension Module");

    if (pyfixbuf_initializer(fixbuf) == FALSE) {
        Py_XDECREF(fixbuf);
        if (PyErr_Occurred()) {
            PyErr_Print();
        }
    }
}

#endif  /* #else of #if PY_MAJOR_VERSION >= 3 */
