#!/usr/bin/env python
#######################################################################
# Copyright (C) 2013-2022 by Carnegie Mellon University.
# See license information in LICENSE-OPENSOURCE.txt
#######################################################################

from __future__ import print_function
import unittest
import warnings
import sys
import os
import tempfile
import itertools
import ipaddress
import subprocess

from pyfixbuf import *
import pyfixbuf
import pyfixbuf.cert
import pyfixbuf.yaflists

if sys.hexversion >= 0x03000000:
    long = int
else:
    # Allow Python3 syntax to invoke Python2 methods
    range = xrange
    zip = itertools.izip


SAMPLE_FILE = os.path.join(os.path.dirname(__file__), "sampleipfix.ipfix")

class TestInfoElement(unittest.TestCase):

    def testInfoElementBasicConstruction(self):
        a = InfoElement("elementOne", CERT_PEN, 722)
        self.assertRaises(TypeError, InfoElement, "elementTwo")
        self.assertRaises(TypeError, InfoElement, "elementThree", CERT_PEN)
        self.assertRaises(TypeError, InfoElement, name="elementFour", id=899)
        self.assertRaises(TypeError, InfoElement, "elementFive", enterprise_number=999)
        self.assertRaises(TypeError, InfoElement, "elementSix", reversible=False, endian=True)
        b = InfoElement("elementSeven", 722, 11, 6, True, True)
        self.assertEqual(b.length, 6)
        self.assertEqual(b.endian, True)
        c = InfoElement("elementEight", 0, 11, True, reversible=False)
        ans = c.reversible
        self.assertEqual(ans, False)

    def testInfoElementAdvancedConstruction(self):
        a = InfoElement("elementFull", CERT_PEN, 14, 1, True, True, 0, 0, 0, 0, 0, "my full element")
        self.assertEqual(a.length, 1)
        d = InfoElement("elementEight", CERT_PEN, 14, True, True, 0, 0, 0, 0, 0, 0, "my eighth element")
        self.assertEqual(d.reversible, True)
        self.assertEqual(d.endian, False)
        e = InfoElement("elementNine", 1, 25, min=5, max=15, description="niner")
        self.assertEqual(e.min, 5)
        self.assertEqual(e.max, 15)
        self.assertEqual(e.id, 25)
        self.assertEqual(e.ent, 1)
        self.assertEqual(e.length, VARLEN)
        self.assertEqual(e.description, "niner")
        self.assertEqual(e.name, "elementNine")

        f = InfoElement("element10", 34, 99, reversible=True, endian=False, type=DataType.IP4ADDR, units=Units.PACKETS, semantic=2)
        self.assertEqual(f.units, Units.PACKETS)
        self.assertEqual(f.type, DataType.IP4ADDR)
        self.assertEqual(f.length, 4)
        self.assertEqual(f.semantic, 2)
        self.assertEqual(f.description, None)
        self.assertEqual(f.min, 0)
        self.assertEqual(f.max, 0)
        self.assertEqual(f.reversible, True)
        self.assertEqual(f.endian, False)


class TestInfoModel(unittest.TestCase):

    def setUp(self):
        self.ie1 = InfoElement("elementOne", CERT_PEN, 14, 2, True, True, DataType.UINT16, 0, 65535, 0, 0, "my number one")
        self.ie2 = InfoElement("elementTwo", CERT_PEN, 16, 4)
        self.ie3 = InfoElement("elementThree", 0, 999, 2)
        self.ie4 = InfoElement("elementReplace", 0, 999, 4)
        self.ie5 = InfoElement("elementReplacePEN", 32, 999, 6)
        self.ielist = [self.ie1, self.ie2, self.ie3]

    def testInfoModelConstruction(self):
        a = InfoModel()

    def testInfoModelAddElement(self):
        a = InfoModel()
        a.add_element(self.ie1)
        self.assertEqual(a.get_element_length(self.ie1.name), 2)
        self.assertEqual(a.get_element_type(self.ie1.name), DataType.UINT16)
        ie = a.get_element(self.ie1.name)
        self.assertEqual(ie.name, self.ie1.name)
        self.assertEqual(ie.units, self.ie1.units)
        self.assertEqual(ie.type, self.ie1.type)
        self.assertEqual(ie.length, self.ie1.length)
        self.assertEqual(ie.min, self.ie1.min)
        self.assertEqual(ie.max, self.ie1.max)
        self.assertEqual(ie.semantic, self.ie1.semantic)
        self.assertEqual(ie.id, self.ie1.id)
        self.assertEqual(ie.ent, self.ie1.ent)
        self.assertEqual(ie.description, self.ie1.description)

    def testInfoModelAddElementList(self):
        a = InfoModel()
        a.add_element_list(self.ielist)
        self.assertEqual(a.get_element_length(self.ie1.name), 2)
        self.assertEqual(a.get_element_type(self.ie1.name), DataType.UINT16)
        ie = a.get_element(self.ie1.name)
        self.assertEqual(ie.name, self.ie1.name)
        self.assertEqual(ie.units, self.ie1.units)
        self.assertEqual(ie.type, self.ie1.type)
        self.assertEqual(ie.length, self.ie1.length)
        self.assertEqual(ie.min, self.ie1.min)
        self.assertEqual(ie.max, self.ie1.max)
        self.assertEqual(ie.semantic, self.ie1.semantic)
        self.assertEqual(ie.id, self.ie1.id)
        self.assertEqual(ie.ent, self.ie1.ent)
        self.assertEqual(ie.description, self.ie1.description)
        self.assertEqual(a.get_element_length(self.ie2.name), 4)
        self.assertEqual(a.get_element_type(self.ie2.name), DataType.OCTET_ARRAY)
        ie = a.get_element(self.ie2.name)
        self.assertEqual(ie.name, self.ie2.name)
        self.assertEqual(ie.units, self.ie2.units)
        self.assertEqual(ie.type, self.ie2.type)
        self.assertEqual(ie.length, self.ie2.length)
        self.assertEqual(ie.min, self.ie2.min)
        self.assertEqual(ie.max, self.ie2.max)
        self.assertEqual(ie.semantic, self.ie2.semantic)
        self.assertEqual(ie.id, self.ie2.id)
        self.assertEqual(ie.ent, self.ie2.ent)
        self.assertEqual(ie.description, self.ie2.description)
        self.assertEqual(a.get_element_type(self.ie3.name), DataType.OCTET_ARRAY)
        ie = a.get_element(self.ie3.name)
        self.assertEqual(ie.name, self.ie3.name)
        self.assertEqual(ie.units, self.ie3.units)
        self.assertEqual(ie.type, self.ie3.type)
        self.assertEqual(ie.length, self.ie3.length)
        self.assertEqual(ie.min, self.ie3.min)
        self.assertEqual(ie.max, self.ie3.max)
        self.assertEqual(ie.semantic, self.ie3.semantic)
        self.assertEqual(ie.id, self.ie3.id)
        self.assertEqual(ie.ent, self.ie3.ent)
        self.assertEqual(ie.description, self.ie3.description)

    def testInfoModelAddElementReplace(self):
        a = InfoModel()
        a.add_element(self.ie3)
        a.add_element(self.ie4)
        self.assertEqual(a.get_element_length("elementReplace"), 4)
        # This now raises an exception in libfixbuf-2.2.0:
        self.assertRaises(Exception, a.get_element_length, "elementThree")
        ie = a.get_element(id=999)
        self.assertEqual(ie.name, "elementReplace")
        self.assertEqual(ie.units, self.ie4.units)
        self.assertEqual(ie.type, self.ie4.type)
        self.assertEqual(ie.length, self.ie4.length)
        self.assertEqual(ie.min, self.ie4.min)
        self.assertEqual(ie.max, self.ie4.max)
        self.assertEqual(ie.semantic, self.ie4.semantic)
        self.assertEqual(ie.id, self.ie4.id)
        self.assertEqual(ie.ent, self.ie4.ent)
        self.assertEqual(ie.description, self.ie4.description)

        a.add_element(self.ie5)
        ie2 = a.get_element(id=999, ent=32)
        self.assertEqual(ie2.name, "elementReplacePEN")
        self.assertEqual(ie2.length, self.ie5.length)

        self.assertEqual(a.get_element_length("elementReplace"), 4)


    def testInfoModelInfoElementDNE(self):
        a = InfoModel()
        self.assertRaises(KeyError, a.get_element, id=999)
        self.assertRaises(KeyError, a.get_element, "mycoolElement")

    def testInfoModelYafElements(self):
        m = InfoModel()
        m.add_element_list(pyfixbuf.yaflists.YAF_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_STATS_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_FLOW_STATS_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_HTTP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_SLP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_FTP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_IMAP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_RTSP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_SIP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_SMTP_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_DNS_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_SSL_LIST)
        m.add_element_list(pyfixbuf.yaflists.YAF_DPI_LIST)
        big_list = []
        big_list.extend(pyfixbuf.yaflists.YAF_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_STATS_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_FLOW_STATS_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_HTTP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_SLP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_FTP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_IMAP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_RTSP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_SIP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_SMTP_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_DNS_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_SSL_LIST)
        big_list.extend(pyfixbuf.yaflists.YAF_DPI_LIST)
        for ie in big_list:
            self.assertEqual(ie, m.get_element(ie.name))
            self.assertEqual(ie, m.get_element(id = ie.id,
                                               ent = ie.enterprise_number))

#    def testInfoModelOptionsElement(self):
#        a = InfoModel()
#        r = Record(a)
#        r.add_element("elementOne")
#        r.add_element("elementTwo")


class TestInfoElementSpec(unittest.TestCase):

    def testInfoElementSpecConstructor(self):

        i = InfoElementSpec("sourceTransportPort")
        self.assertEqual(i.name, "sourceTransportPort")
        self.assertEqual(i.length, 0)

        e = InfoElementSpec("sourceTransportPort", 4)
        self.assertEqual(e.name, "sourceTransportPort")
        self.assertEqual(e.length, 4)

        self.assertRaises(TypeError, InfoElementSpec, length=2)


class TestTemplate(unittest.TestCase):

    def setUp(self):
        self.model = InfoModel()
        self.spec = InfoElementSpec("sourceTransportPort")
        self.spec2 = InfoElementSpec("destinationTransportPort")
        self.nospec = InfoElementSpec("notanElement")
        self.speclist = [self.spec, self.spec2]

        self.speclist2=[InfoElementSpec("sourceIPv4Address"),
                        InfoElementSpec("destinationIPv4Address"),
                        InfoElementSpec("octetTotalCount", 4)]

        self.invalidspec = InfoElementSpec("protocolIdentifier", 4)
        self.invalidspec2 = InfoElementSpec("sourceIPv4Address", 2)

        self.invalidspeclist = [self.invalidspec, self.invalidspec2]

    def testTemplateConstructor(self):
        tmpl = Template(self.model)

        self.assertRaises(TypeError, Template, self.spec)

        oTmpl = Template(self.model, 1)

        self.assertEqual(oTmpl.type, 1)
        self.assertEqual(tmpl.type, 0)

    def testTemplateCopy(self):
        tmpl = Template(self.model)
        tmpl.add_spec_list(self.speclist)
        tmpl.scope = 1
        tmpl2 = tmpl.copy()
        self.assertEqual(tmpl._elements, tmpl2._elements)
        self.assertEqual(tmpl.scope, tmpl2.scope)
        tmpl2.add_spec_list(self.speclist2)
        self.assertNotEqual(tmpl._elements, tmpl2._elements)
        tmpl.scope = 2
        self.assertNotEqual(tmpl.scope, tmpl2.scope)

    def testTemplateIter(self):
        tmpl = Template(self.model)
        tmpl.add_spec_list(self.speclist)
        for iter_spec,list_spec in zip(tmpl, self.speclist):
            self.assertEqual(iter_spec.name, list_spec.name)
            self.assertEqual(iter_spec.length, list_spec.length)

    def testTemplateIterIE(self):
        tmpl = Template(self.model)
        tmpl.add_spec_list(self.speclist)
        for iter_ie,list_spec in zip(tmpl.ie_iter(), self.speclist):
            list_ie = self.model.get_element(list_spec.name)
            self.assertEqual(iter_ie, list_ie)

    def testTemplateScope(self):
        tmpl = Template(self.model)
        tmpl.add_spec_list(self.speclist)
        self.assertEqual(tmpl.scope, None)
        tmpl.scope = 1
        self.assertEqual(tmpl.scope, 1)
        tmpl.scope = 0
        self.assertEqual(tmpl.scope, 2)
        tmpl.scope = None
        self.assertEqual(tmpl.scope, None)
        if sys.hexversion >= 0x020700f0:
            with self.assertRaises(AttributeError):
                tmpl.scope = 3
            with self.assertRaises(AttributeError):
                tmpl.scope = -1
        else:
            # python 2.6.x
            self.assertRaises(AttributeError, setattr, tmpl, "scope", 3)
            self.assertRaises(AttributeError, setattr, tmpl, "scope", -1)
        oTmpl = Template(self.model, True)
        self.assertEqual(oTmpl.scope, 2)

    def testAddSpecs(self):
        tmpl = Template(self.model)

        tmpl.add_spec(self.spec)

        tmpl.add_spec_list(self.speclist)
        tmpl.add_spec_list(self.speclist2)

        self.assertEqual(len(tmpl), 6)
        self.assertTrue("sourceTransportPort" in tmpl)
        self.assertTrue(self.spec in tmpl)
        self.assertFalse("fooElement" in tmpl)
        ie = InfoElement("myWeirdElement", CERT_PEN, 987)
        self.model.add_element(ie)
        tmpl.add_spec(InfoElementSpec("myWeirdElement"))
        self.assertTrue(ie in tmpl)
        self.assertRaises(Exception, tmpl.add_spec, self.nospec)
        self.assertRaises(Exception, tmpl.add_spec, self.model)

        self.assertEqual(tmpl.template_id, 0)

        self.assertRaises(Exception, tmpl.add_spec, self.invalidspec)
        self.assertRaises(Exception, tmpl.add_spec_list, self.invalidspeclist)

    def testTemplateGetItem(self):
        tmpl = Template(self.model)
        tmpl.add_spec(self.spec)
        tmpl.add_spec_list(self.speclist2)

        self.assertEqual(tmpl["octetTotalCount"].length, 4)
        self.assertEqual(tmpl["sourceTransportPort"].length, 2)

        self.assertEqual(tmpl[0].name, "sourceTransportPort")
        self.assertRaises(IndexError, tmpl.__getitem__, 10)
        self.assertRaises(KeyError, tmpl.__getitem__, "yayayaya")
        self.assertRaises(TypeError, tmpl.__getitem__, self.spec)


    def testTemplateGetIndexedIE(self):
        tmpl = Template(self.model)

        tmpl.add_spec(self.spec)
        tmpl.add_spec(self.spec2)
        tmpl.add_spec_list(self.speclist2)

        ie = tmpl.get_indexed_ie(0)
        self.assertEqual(self.spec.name, ie.name)
        self.assertRaises(IndexError, tmpl.get_indexed_ie, 9)
        self.assertRaises(TypeError, tmpl.get_indexed_ie, "keke")
        ie2 = tmpl.get_indexed_ie(4)
        self.assertEqual(ie2.name, "octetTotalCount")

class TestSession(unittest.TestCase):

    def setUp(self):
        self.model = InfoModel()
        self.spec = InfoElementSpec("sourceTransportPort")
        self.spec2 = InfoElementSpec("destinationTransportPort")
        self.nospec = InfoElementSpec("notanElement")
        self.speclist = [self.spec, self.spec2]

        self.speclist2=[InfoElementSpec("sourceIPv4Address"),
                        InfoElementSpec("destinationIPv4Address"),
                        InfoElementSpec("octetTotalCount", 4)]
        self.tmpl = Template(self.model)
        self.tmpl.add_spec_list(self.speclist)
        self.tmpl.add_spec_list(self.speclist2)
        self.oTmpl = Template(self.model, 1)

    def testSessionConstructor(self):
        sess = Session(self.model)
        self.assertRaises(TypeError, Session, self.spec)
        sess2 = Session(InfoModel())

    def testSessionAddOptionsTemplate(self):
        sess = Session(self.model)

        sess.add_template(self.oTmpl, 259)
        self.assertEqual(len(self.oTmpl), 10)

    def testSessionAddTemplate(self):
        sess = Session(self.model)
        self.assertRaises(Exception, sess.add_template, self.tmpl, 20000000)

        sess.add_template(self.tmpl, 654)
        self.assertEqual(self.tmpl.template_id, 654)

        rv = sess.add_template(self.tmpl, 999)
        self.assertEqual(self.tmpl.template_id, rv)
        self.assertEqual(rv, 999)

    def testSessionAddInternal(self):
        sess = Session(self.model)

        self.assertRaises(Exception, sess.add_internal_template, self.tmpl, 20000000)

        sess.add_internal_template(self.tmpl, 654)
        self.assertEqual(self.tmpl.template_id, 654)

        rv = sess.add_internal_template(self.tmpl, 999)
        self.assertEqual(self.tmpl.template_id, rv)
        self.assertEqual(rv, 999)

    def testSessionAddExternal(self):
        sess = Session(self.model)

        self.assertRaises(Exception, sess.add_external_template, self.tmpl, 20000000)

        sess.add_external_template(self.tmpl, 654)
        self.assertEqual(self.tmpl.template_id, 654)

        rv = sess.add_external_template(self.tmpl, 999)
        self.assertEqual(self.tmpl.template_id, rv)
        self.assertEqual(rv, 999)

    def testSessionDecodeOnly(self):
        sess = Session(self.model)

        self.assertRaises(TypeError, sess.decode_only, 999)
        self.assertRaises(Exception, sess.decode_only, ["foo"])
        self.assertRaises(Exception, sess.decode_only, [999, "foo"])

        sess.decode_only([999, 1000])
        sess.decode_only([111])

    def testSessionIgnoreTemplates(self):
        sess = Session(self.model)

        self.assertRaises(TypeError, sess.ignore_templates, 999)
        self.assertRaises(Exception, sess.ignore_templates, ["foo"])
        self.assertRaises(Exception, sess.ignore_templates, [999, "foo"])

        sess.ignore_templates([999, 1000])
        sess.ignore_templates([111])

    def testSessionAddTemplatePair(self):
        sess = Session(self.model)

        self.assertRaises(Exception, sess.add_template_pair, 999, 785)

        sess.add_internal_template(self.tmpl, 999)

        sess.add_template_pair(876, 999)


class TestCollector(unittest.TestCase):

    def testCollectorConstructor(self):
        coll = Collector()
        self.assertRaises(TypeError, Collector, "myname")

    def testCollectorInitFile(self):
        coll = Collector()

        self.assertRaises(Exception, coll.init_file, "89734kigl.txt")
        coll.init_file(SAMPLE_FILE)


class TestExporter(unittest.TestCase):

    def testExporterConstructor(self):
        exporter = Exporter()
        self.assertRaises(TypeError, Exporter, "foo")

    def testExporterInitFile(self):
        exporter = Exporter()
        collector = Collector()
        self.assertRaises(TypeError, Exporter, collector)

        exporter.init_file("testout.ipfix")

    def testExporterInitNet(self):
        exporter = Exporter()

        self.assertRaises(Exception, exporter.init_net, "localhost", "tcp", 80)
        self.assertRaises(Exception, exporter.init_net, "localhost", "sctp")
        self.assertRaises(Exception, exporter.init_net, "localhost", "udp", 0)
        exporter.init_net("localhost")
        exporter.init_net("localhost", "udp")
        exporter.init_net("localhost", "tcp", "18975")
        exporter.init_net("127.0.0.1", "udp")

class TestListener(unittest.TestCase):

    def setUp(self):
        self.model = InfoModel()
        self.session = Session(self.model)

    def testListenerConstructor(self):
        list = Listener(self.session, "localhost")

        self.assertRaises(Exception, Listener, self.session, "localhost", "udp", 90)
        self.assertRaises(Exception, Listener, self.model, "localhost")
        self.assertRaises(Exception, Listener, self.session, "localhost", "http")

        list = Listener(self.session, hostname="localhost", transport="tcp", port=18975)
        list = Listener(self.session, "localhost", transport="udp")


class TestRecord(unittest.TestCase):

    def setUp(self):
        self.model = InfoModel()
        self.template = Template(self.model)
        self.spec = InfoElementSpec("sourceTransportPort")
        self.spec2 = InfoElementSpec("destinationTransportPort")
        self.nospec = InfoElementSpec("notanElement")
        self.speclist = [self.spec, self.spec2]

        self.speclist2=[InfoElementSpec("sourceIPv4Address"),
                        InfoElementSpec("destinationIPv4Address"),
                        InfoElementSpec("octetTotalCount", 4)]
        self.template.add_spec_list(self.speclist)
        self.template.add_spec_list(self.speclist2)

    def testRecordConstructor(self):

        rec = Record(self.model)

        self.assertRaises(Exception, Record, self.template)
        self.assertRaises(Exception, Record, self.model, self.model)
        rec = Record(self.model, self.template)
        recdict = rec.as_dict()
        self.assertTrue(isinstance(recdict, dict))
        self.assertEqual(len(recdict), 5)
        self.assertTrue("sourceTransportPort" in recdict)
        self.assertTrue("destinationTransportPort" in recdict)
        self.assertTrue("sourceIPv4Address" in recdict)
        self.assertTrue("destinationIPv4Address" in recdict)
        self.assertTrue("octetTotalCount" in recdict)

        self.assertEqual(rec.length, 16)

        rec2 = Record(self.model)
        rec2.add_element("sourceTransportPort")
        rec2.add_element("destinationTransportPort")
        rec2.add_element("octetTotalCount")
        rec2.add_element("packetTotalCount", length=4)
        self.assertEqual(rec2.length, 16)
        rec2dict = rec2.as_dict()
        self.assertTrue(isinstance(rec2dict, dict))
        self.assertEqual(len(rec2dict), 4)

    def testRecordAddElementReduced(self):

        rec = Record(self.model)
        self.assertRaises(Exception, rec.add_element, "sourceTransportPort", 0, None, 6)
        self.assertRaises(Exception, rec.add_element, "sourceIPv4Address", 0, None, 2)

    def testRecordAddElement(self):
        rec = Record(self.model)

        rec.add_element("sourceTransportPort")
        rec.add_element("octetTotalCount", length=4)
        rec.add_element("weirdList", BASICLIST, "octetTotalCount")
        rec.add_element("weirdList2", BASICLIST)
        rec.add_element("subTemplateList")
        rec.add_element("subTemplateMultiList")
        rec.add_element("sourceTransportPort")
        rec.add_element("sourceTransportPort2", 0, "sourceTransportPort")

        self.assertRaises(Exception, rec.add_element, "destinationTransportPort", 99)
        rec.add_element("basicList", BASICLIST, "noelem")
#        self.assertRaises(Exception, rec.add_element, "basicList", BASICLIST, "noelem")
        rec.add_element_list(["sourceIPv4Address", "destinationIPv4Address"])
        self.assertEqual(len(rec), 11)
        self.assertTrue("sourceIPv4Address" in rec)
        self.assertFalse("destinationTransportPort" in rec)
        self.assertTrue("basicList" in rec)
        recdict = rec.as_dict()
        self.assertEqual(len(recdict), 11)
        self.assertTrue(("sourceTransportPort", 1) in rec)

    def testRecordCopy(self):
        rec = Record(self.model)

        rec.add_element("sourceTransportPort")
        rec.add_element("octetTotalCount", length=4)
        rec.add_element("weirdList", BASICLIST, "octetTotalCount")
        rec.add_element("weirdList2", BASICLIST)
        rec.add_element("subTemplateList")
        rec.add_element("subTemplateMultiList")
        rec.add_element("sourceTransportPort")
        rec.add_element("sourceTransportPort2", 0, "sourceTransportPort")

        rec['sourceTransportPort'] = 23456
        rec['octetTotalCount'] = 0xffffffff
        rec[('sourceTransportPort', 1)] = 65432
        rec['sourceTransportPort2'] = 13579

        self.assertRaises(Exception, rec.copy, self.model)
        rec2 = Record(self.model)
        rec2.add_element_list(["sourceIPv4Address", "sourceTransportPort"])
        rec2['sourceIPv4Address'] = '10.11.12.13'
        rec2['sourceTransportPort'] = 80
        rec2.copy(rec)
        rec.clear()
        self.assertEqual(len(rec2), 2)
        self.assertTrue("sourceIPv4Address" in rec2)
        self.assertTrue("sourceTransportPort" in rec2)
        self.assertTrue(rec2['sourceTransportPort'], 23456)
        self.assertTrue(str(rec2['sourceIPv4Address']), '10.11.12.13')
        rec2.clear()

    def testRecordIpaddress(self):
        v4data = [u"10.11.12.13", 0x0a0b0c0d, 168496141,
                  ipaddress.IPv4Address(0x0a0b0c0d),
                  bytearray(b'\x0a\x0b\x0c\x0d')]
        v6data = [u"2011:db8:a:b::c:d", 0x20110db8000a000b00000000000c000d,
                  long("42623843161031239560070803995474264077"),
                  ipaddress.IPv6Address(0x20110db8000a000b00000000000c000d),
                  bytearray(b'\x20\x11\x0d\xb8\x00\x0a\x00\x0b' +
                            b'\x00\x00\x00\x00\x00\x0c\x00\x0d')]

        t = Template(self.model)
        t.add_spec_list((InfoElementSpec('sourceIPv6Address'),
                         InfoElementSpec('destinationIPv4Address')))
        for s,d in zip(v6data, v4data):
            rec = Record(self.model, t)
            rec["sourceIPv6Address"] = s
            rec["destinationIPv4Address"] = d
            self.assertEqual(str(rec['sourceIPv6Address']),'2011:db8:a:b::c:d')
            self.assertEqual(str(rec['destinationIPv4Address']), '10.11.12.13')

        rec = Record(self.model)
        rec.add_element('siplist', BASICLIST, "sourceIPv6Address")
        rec.add_element('diplist', BASICLIST, "destinationIPv4Address")
        rec.init_basic_list('siplist', len(v6data))
        rec.init_basic_list('diplist', len(v4data))
        rec["siplist"] = v6data
        rec["diplist"] = v4data
        for s in rec["siplist"]:
            self.assertEqual(str(s), '2011:db8:a:b::c:d')
        for d in rec["diplist"]:
            self.assertEqual(str(d), '10.11.12.13')

    def testRecordIter(self):
        rec = Record(self.model)
        rec.add_element("sourceTransportPort")
        rec.add_element("octetTotalCount", length=4)
        rec.add_element("destinationTransportPort")
        rec.add_element("proto", 0, "protocolIdentifier")
        rec["sourceTransportPort"] = 15
        rec["octetTotalCount"] = 15
        rec["destinationTransportPort"] = 15
        rec["proto"] = 15
        for item in rec:
            self.assertEqual(item, 15)

    def testRecordIterItems(self):
        rec = Record(self.model)
        rec.add_element("sourceTransportPort")
        rec.add_element("octetTotalCount", length=4)
        rec.add_element("destinationTransportPort")
        rec.add_element("proto", 0, "protocolIdentifier")

        spec_list = (InfoElementSpec("sourceTransportPort"),
                     InfoElementSpec("octetTotalCount", 4),
                     InfoElementSpec("destinationTransportPort"),
                     InfoElementSpec("protocolIdentifier"))
        tmpl = Template(self.model)
        tmpl.add_spec_list(spec_list)
        rec.set_template(tmpl)

        rec["sourceTransportPort"] = 15
        rec["octetTotalCount"] = 15
        rec["destinationTransportPort"] = 15
        rec["proto"] = 15

        z = 0
        for field in rec.iterfields():
            self.assertEqual(field.value, 15)
            self.assertTrue(isinstance(field.ie, InfoElement),
                            "type of ie is %s" % str(type(field.ie)))
            self.assertEqual(field.ie.name, spec_list[z].name)
            z = z + 1
        self.assertEqual(z, len(spec_list))

    def testRecordInitBasicList(self):
        # Get the size of an fbBasicList_t which varies depending on
        # fixbuf version
        rec = Record(self.model)
        rec.add_element("basicList")
        basicListLen = rec.length

        rec = Record(self.model)

        rec.add_element_list(["sourceIPv4Address", "destinationIPv4Address",
                              "basicList"])
        rec.add_element("BL", BASICLIST, "octetTotalCount")
        rec.add_element("basicList2", BASICLIST)
        rec.add_element("sipbasic", BASICLIST)
        #no info element
        self.assertRaises(Exception, rec.init_basic_list, "basicList", 3)
        rec.init_basic_list("basicList", 2, "sourceTransportPort")
        rec["basicList"]=[23, 36]
        rec.init_basic_list("BL", 4)
        rec["BL"]=[long(99),long(456),long(345)]
        self.assertRaises(Exception, rec.init_basic_list, "basicList2")
        self.assertEqual(rec.length, 8 + 4 * basicListLen)
        rec.add_element("destinationTransportPort")
        rec["destinationTransportPort"]=37
        self.assertEqual(rec.length, 10 + 4 * basicListLen)

        self.assertEqual(rec["destinationTransportPort"], 37)
        self.assertEqual(len(rec["BL"]), 4)
        self.assertEqual(rec["basicList"],[23,36])
        self.assertEqual(rec[2], [23,36])
        self.assertEqual(str(rec["sourceIPv4Address"]), '0.0.0.0')
        self.assertEqual(str(rec[0]), '0.0.0.0')

        rec.init_basic_list("sipbasic", 1, "sourceIPv4Address")
        rec["sipbasic"] = "1.2.3.4"
        self.assertEqual([str(x) for x in rec["sipbasic"]], ["1.2.3.4"])

        rec.clear_basic_list("basicList")
        self.assertFalse(rec["basicList"])
        self.assertEqual(rec.length, 10 + 4 * basicListLen)

        self.assertRaises(IndexError, rec.__getitem__, 10)

    def testRecordInitBL(self):
        rec = Record(self.model)

        rec.add_element_list(["sourceIPv4Address", "destinationIPv4Address", "basicList"])
        rec.add_element("BL", BASICLIST, "octetTotalCount")
        rec.add_element("basicList2", BASICLIST)
        self.assertRaises(TypeError, BL.__init__, "sourceTransportPort", 2)
        bl = BL(self.model, "sourceTransportPort", 2)
        bl[0] = 13
        bl[1] = 15
        self.assertRaises(TypeError, BL.__setitem__, 2, 14)
        self.assertEqual(len(bl), 2)
        self.assertEqual(bl.semantic, 0)
        rec["basicList"] = bl
        bl.semantic = 12
        self.assertEqual(bl.semantic, 12)
        bl = rec["basicList"]
        ie = bl.element
        self.assertEqual(ie.type, DataType.UINT16)
        self.assertEqual(ie.name, "sourceTransportPort")

        bl2 = BL(self.model, "sourceIPv4Address", 3, 6)
        bl2[0] = "1.2.3.4"
        bl2[1] = "2.3.4.5"
        bl2[2] = "3.4.5.6"

        rec["basicList2"] = bl2

        self.assertEqual(bl2.semantic, 6)
        blg = rec["basicList2"]

        self.assertEqual(len(blg), 3)
        self.assertEqual([str(x) for x in rec["basicList2"]],
                         ["1.2.3.4", "2.3.4.5", "3.4.5.6"])

        blg.clear()
        self.assertEqual(len(blg), 0)
        bl3 = rec["BL"]
        self.assertEqual(len(bl3), 0)

        self.model.add_element_list(pyfixbuf.yaflists.YAF_HTTP_LIST)
        bl3 = pyfixbuf.BL(self.model, "httpUserAgent", 2)

        bl3[0] = "Mozilla/Firefox"
        bl3[1] = "Safari5.0"
        self.assertEqual(bl3[0], "Mozilla/Firefox")
        self.assertEqual(bl3[1], "Safari5.0")

        rec["BL"] = bl3

        self.assertEqual(rec["BL"], ["Mozilla/Firefox", "Safari5.0"])


    def testRecordInitSTML(self):
        sess = Session(self.model)
        rec = Record(self.model)
        subRec = Record(self.model, self.template)
        tmpl = Template(self.model)
        self.assertRaises(Exception, STML, rec, "subTemplateMultiList", 1)
        rec.add_element("subTemplateMultiList")

        stml0 = STML(type_count=2)
        # using a template with no template_id in a list raises an exception
        self.assertRaises(Exception, stml0[0].entry_init,
                          subRec, self.template, 2)
        tid = sess.add_template(self.template)
        stml0[0].entry_init(subRec, self.template, 2)
        subRec["sourceTransportPort"] = 3
        subRec["destinationTransportPort"] = 5
        stml0[0][0] = subRec
        subRec["sourceTransportPort"] = 6
        subRec["destinationTransportPort"] = 7
        stml0[0][1] = subRec
        self.assertEqual(stml0[0][1]["sourceTransportPort"], 6)
        rec["subTemplateMultiList"] = stml0
        # can't add elements after lists have been init'd
        self.assertRaises(Exception, rec.add_element, "protocolIdentifier")
        entry1 = rec["subTemplateMultiList"][0]
        entry1.set_record(subRec)
        newrec = entry1[0]
        self.assertEqual(newrec["sourceTransportPort"], 3)
        self.assertEqual(newrec["destinationTransportPort"], 5)
        newrec = entry1[1]
        self.assertEqual(newrec["sourceTransportPort"], 6)
        self.assertEqual(newrec["destinationTransportPort"], 7)
        self.assertRaises(IndexError, entry1.__getitem__, 2)

        # this shows how to create an entry on the fly without initializing
        # if you don't initialize, you have to use set_template and it defaults
        # to only 1 item in the list
        entry2 = stml0[1]
        self.assertRaises(IndexError, stml0.__getitem__, 2)
        recNoTmpl = Record(self.model)
        recNoTmpl.add_element("sourceTransportPort")
        recNoTmpl.add_element("protocolIdentifier")
        recNoTmpl.add_element("packetTotalCount", length=4)
        recNoTmpl["sourceTransportPort"] = 101
        recNoTmpl["protocolIdentifier"] = 20
        recNoTmpl["packetTotalCount"]=99999
        self.assertRaises(Exception, entry2.set_template, recNoTmpl)
        tmpl2 = Template(self.model)
        xspecs = [InfoElementSpec("sourceTransportPort"),
                  InfoElementSpec("protocolIdentifier"),
                  InfoElementSpec("packetTotalCount", 4)]
        tmpl2.add_spec_list(xspecs)
        sess.add_template(tmpl2)
        entry2.set_template(tmpl2)
        #entry2.entry_init(recNoTmpl, tmpl2, 1)
        self.assertRaises(Exception, entry2.__setitem__, 0, subRec)
        entry2[0] = recNoTmpl
        #rec["subTemplateMultiList"] = stml0
        self.assertRaises(IndexError, entry2.__setitem__, 1, recNoTmpl)
        newrec = entry2[0]
        self.assertEqual(newrec["sourceTransportPort"], 101)
        self.assertEqual(newrec["protocolIdentifier"], 20)
        self.assertEqual(newrec["packetTotalCount"], 99999)

        # set items
        stml = STML(rec, "subTemplateMultiList", 2)
        stml.semantic = 5
        for entry in stml:
            self.assertRaises(Exception, entry.entry_init, rec, self.model)
            self.assertRaises(Exception, entry.entry_init, rec, tmpl, 3)
            self.assertRaises(Exception, entry.entry_init, rec, self.template, 2)
            entry.entry_init(subRec, self.template, 2)
            for item in entry:
                item["sourceTransportPort"] = 3
                item["destinationTransportPort"]=5
                item["sourceIPv4Address"]="127.0.0.1"
                item["destinationIPv4Address"]="87.2.3.4"
                item["octetTotalCount"]=89
        self.assertEqual(stml.count, 2)
        self.assertEqual(stml.semantic, 5)
        stml.clear()
        self.assertEqual(stml.count, 0)
        # get items
        stml2 = rec["subTemplateMultiList"]
        for item in stml:
            self.assertTrue("sourceTransportPort" in item)
            self.assertRaises(Exception, item.set_record, self.model)
            self.assertEqual(item.count, 2)
            self.assertEqual(item.template_id, 0)
            item.set_record(subRec)
            for entry in item:
                self.assertEqual(entry["sourceTransportPort"], 3)
                self.assertEqual(entry["destinationTransportPort"], 5)
                self.assertEqual(str(entry["sourceIPv4Address"]), "127.0.0.1")
                self.assertEqual(str(entry["destinationIPv4Address"]), "87.2.3.4")
                self.assertEqual(entry["octetTotalCount"], 89)

    def testRecordInitSTL(self):
        sess = Session(self.model)
        rec = Record(self.model)
        subRec = Record(self.model, self.template)
        tmpl = Template(self.model)
        self.assertRaises(Exception, STL, rec, "subTemplateList", 1)
        rec.add_element("subTemplateList")
        rec.add_element("sourceTransportPort")
        rec.add_element("otherList", SUBTEMPLATELIST)
        rec.add_element("newList", 0, "subTemplateList")
        self.assertRaises(Exception, STL, rec, "sourceTransportPort")
        stl = STL(rec, "subTemplateList")
        # can't add elements after lists have been init'd
        self.assertRaises(Exception, rec.add_element, "protocolIdentifier")
        self.assertRaises(Exception, STL, subRec, tmpl)
        # using a template with no template_id in a list raises an exception
        self.assertRaises(Exception, stl.entry_init, subRec, self.template, 2)
        tid = sess.add_template(self.template)
        stl.entry_init(subRec, self.template, 2)
        stl.semantic=9
        for strec in stl:
            strec["octetTotalCount"] = 8888888
            strec["sourceTransportPort"] = 80
            strec["destinationTransportPort"] = 1137

        self.assertEqual(stl.template_id, tid)
        self.assertEqual(stl.count, 2)
        self.assertEqual(stl.semantic, 9)
        stl2 = STL(rec, "otherList")
        stl2.entry_init(subRec, self.template, 0)

        get_stl = rec["subTemplateList"]
        get_stl.set_record(subRec)
        self.assertEqual(get_stl.semantic, 9)
        self.assertEqual(get_stl.count, 2)
        for thing in get_stl:
            self.assertEqual(thing["octetTotalCount"], 8888888)
            self.assertEqual(thing["sourceTransportPort"], 80)
            self.assertEqual(thing["destinationTransportPort"], 1137)
            self.assertEqual(str(thing["sourceIPv4Address"]), "0.0.0.0")

        get_stl.clear()
        self.assertEqual(get_stl.count, 0)

        #test other method
        newstl = STL()
        newstl.entry_init(subRec, self.template, 1)
        subRec["octetTotalCount"] = 50
        subRec["sourceTransportPort"] = 8080
        subRec["destinationTransportPort"] = 1024
        newstl[0] = subRec

        newRec = newstl[0]

        self.assertEqual(newRec["octetTotalCount"], 50)
        self.assertEqual(newRec["sourceTransportPort"], 8080)
        self.assertEqual(newRec["destinationTransportPort"], 1024)

        rec["newList"] = newstl

        rstl = rec["newList"]
        rstl.set_record(subRec)
        self.assertEqual(rstl["octetTotalCount"], 50)
        orec = rstl[0]
        self.assertEqual(orec["sourceTransportPort"], 8080)

    def testRecordSetSTLtoList(self):
        sess = Session(self.model)
        rec = Record(self.model)
        subRec = Record(self.model, self.template)
        tmpl = Template(self.model)
        rec.add_element("subTemplateList")
        rec.add_element("otherList", SUBTEMPLATELIST)
        rec.add_element("octetTotalCount")
        rec.add_element("packetTotalCount")
        tmpl.add_spec_list(self.speclist)
        subRecAlt = Record(self.model, tmpl)
        subRecNoTmpl = Record(self.model)
        subRecNoTmpl.add_element("sourceTransportPort")
        subRecNoTmpl.add_element("destinationTransportPort")

        stl = STL()
        subRec2 = Record(self.model, record=subRec)
        subRec3 = Record(self.model, record=subRec)

        subRec["sourceTransportPort"] = 1
        subRec["destinationTransportPort"] = 2
        subRec["sourceIPv4Address"] = "1.2.3.4"
        subRec["destinationIPv4Address"] = "2.3.4.5"
        subRec["octetTotalCount"]= 100

        subRec2["sourceTransportPort"] = 3
        subRec2["destinationTransportPort"] = 4
        subRec2["sourceIPv4Address"] = "3.4.5.6"
        subRec2["destinationIPv4Address"] = "4.5.6.7"
        subRec2["octetTotalCount"]= 200

        subRec3["sourceTransportPort"] = 5
        subRec3["destinationTransportPort"] = 6
        subRec3["sourceIPv4Address"] = "5.6.7.8"
        subRec3["destinationIPv4Address"] = "6.7.8.9"
        subRec3["octetTotalCount"]= 300

        subRecAlt["sourceTransportPort"]=8
        subRecAlt["destinationTransportPort"] = 9

        subRecNoTmpl["sourceTransportPort"]=11
        subRecNoTmpl["destinationTransportPort"] = 12

        # different records will raise exception
        self.assertRaises(Exception, rec["subTemplateList"],
                          [subRec, subRec2, subRec3, subRecAlt])

        stl = [subRec, subRec2, subRec3]

        # using a template with no template_id in a list raises an exception
        self.assertRaises(Exception, rec["subTemplateList"].__setitem__, stl)

        sess.add_template(subRec.template)
        rec["subTemplateList"] = stl
        sess.add_template(subRecAlt.template)
        rec["otherList"] = [subRecNoTmpl, subRecAlt]

        newstl = rec["subTemplateList"]
        self.assertEqual(len(newstl), 3)
        newstl.set_record(subRec)
        self.assertEqual(newstl[1]["sourceTransportPort"], 3)
        self.assertEqual(newstl[1]["destinationTransportPort"], 4)
        self.assertEqual(newstl[1]["octetTotalCount"], 200)
        self.assertEqual(str(newstl[1]["sourceIPv4Address"]), "3.4.5.6")

        self.assertEqual(newstl[2]["octetTotalCount"], 300)

    def testRecordSetSTMLEntrytoList(self):
        sess = Session(self.model)
        rec = Record(self.model)
        subRec = Record(self.model, self.template)

        tmpl = Template(self.model)
        rec.add_element("subTemplateMultiList")
        rec.add_element("octetTotalCount")
        rec.add_element("packetTotalCount")
        tmpl.add_spec_list(self.speclist)
        subRecAlt = Record(self.model, tmpl)

        stml0 = STML(type_count=1)
        subRec2 = Record(self.model, record=subRec)
        subRec3 = Record(self.model, record=subRec)

        subRec["sourceTransportPort"] = 1
        subRec["destinationTransportPort"] = 2
        subRec["sourceIPv4Address"] = "1.2.3.4"
        subRec["destinationIPv4Address"] = "2.3.4.5"
        subRec["octetTotalCount"]= 100

        subRec2["sourceTransportPort"] = 3
        subRec2["destinationTransportPort"] = 4
        subRec2["sourceIPv4Address"] = "3.4.5.6"
        subRec2["destinationIPv4Address"] = "4.5.6.7"
        subRec2["octetTotalCount"]= 200

        subRec3["sourceTransportPort"] = 5
        subRec3["destinationTransportPort"] = 6
        subRec3["sourceIPv4Address"] = "5.6.7.8"
        subRec3["destinationIPv4Address"] = "6.7.8.9"
        subRec3["octetTotalCount"]= 300

        subRecAlt["sourceTransportPort"]=8
        subRecAlt["destinationTransportPort"] = 9
        # may not use a list of records with different templates
        self.assertRaises(Exception, stml0[0].__setitem__,
                          [subRec, subRec2, subRec3, subRecAlt])

        # using a template with no template_id in a list raises an exception
        self.assertRaises(Exception, stml0[0].__setitem__,
                          [subRec, subRec2, subRec3])
        sess.add_template(subRec.template)
        stml0[0] = [subRec, subRec2, subRec3]

        rec["subTemplateMultiList"] = stml0

        newstml = rec["subTemplateMultiList"]
        self.assertEqual(len(newstml), 1)
        newstml[0].set_record(subRec)
        self.assertEqual(newstml[0][1]["sourceTransportPort"], 3)
        self.assertEqual(newstml[0][1]["destinationTransportPort"], 4)
        self.assertEqual(newstml[0][1]["octetTotalCount"], 200)
        self.assertEqual(str(newstml[0][1]["sourceIPv4Address"]), "3.4.5.6")

        self.assertEqual(newstml[0][2]["octetTotalCount"], 300)

    def testRecordSetSTMLtoList(self):
        sess = Session(self.model)
        rec = Record(self.model)
        subRec = Record(self.model, self.template)
        tmpl = Template(self.model)
        rec.add_element("subTemplateMultiList")
        rec.add_element("octetTotalCount")
        rec.add_element("packetTotalCount")
        tmpl.add_spec_list(self.speclist)
        subRecAlt = Record(self.model, tmpl)
        subRecNoTmpl = Record(self.model)
        subRecNoTmpl.add_element("sourceTransportPort")
        subRecNoTmpl.add_element("destinationTransportPort")

        subRec2 = Record(self.model, record=subRec)
        subRec3 = Record(self.model, record=subRec)

        subRec["sourceTransportPort"] = 1
        subRec["destinationTransportPort"] = 2
        subRec["sourceIPv4Address"] = "1.2.3.4"
        subRec["destinationIPv4Address"] = "2.3.4.5"
        subRec["octetTotalCount"]= 100

        subRec2["sourceTransportPort"] = 3
        subRec2["destinationTransportPort"] = 4
        subRec2["sourceIPv4Address"] = "3.4.5.6"
        subRec2["destinationIPv4Address"] = "4.5.6.7"
        subRec2["octetTotalCount"]= 200

        subRec3["sourceTransportPort"] = 5
        subRec3["destinationTransportPort"] = 6
        subRec3["sourceIPv4Address"] = "5.6.7.8"
        subRec3["destinationIPv4Address"] = "6.7.8.9"
        subRec3["octetTotalCount"]= 300

        subRecAlt["sourceTransportPort"]=8
        subRecAlt["destinationTransportPort"] = 9

        subRecNoTmpl["sourceTransportPort"]=15
        subRecNoTmpl["destinationTransportPort"]=19
        self.assertRaises(Exception, subRecNoTmpl.set_template, self.template)
        subRecNoTmpl.set_template(tmpl)

        stml0 = [subRec, subRec2, subRec3, subRecAlt, subRecNoTmpl]

        # using a template with no template_id in a list raises an exception
        self.assertRaises(Exception, rec["subTemplateMultiList"].__setitem__,
                          stml0)
        sess.add_template(subRec.template)
        sess.add_template(subRecAlt.template)
        sess.add_template(subRecNoTmpl.template)
        rec["subTemplateMultiList"] = stml0

        newstml = rec["subTemplateMultiList"]
        self.assertEqual(len(newstml), 2)
        newstml[0].set_record(subRec)
        self.assertEqual(len(newstml[0]), 3)
        self.assertEqual(len(newstml[1]), 2)
        self.assertEqual(newstml[0][1]["sourceTransportPort"], 3)
        self.assertEqual(newstml[0][1]["destinationTransportPort"], 4)
        self.assertEqual(newstml[0][1]["octetTotalCount"], 200)
        self.assertEqual(str(newstml[0][1]["sourceIPv4Address"]), "3.4.5.6")

        self.assertEqual(newstml[0][2]["octetTotalCount"], 300)

        self.assertRaises(Exception, newstml[1].__getitem__, "sourceTransportPort")
        newstml[1].set_record(subRecAlt)
        self.assertEqual(newstml[1][0]["sourceTransportPort"], 8)
        self.assertRaises(Exception, newstml[1][1].__getitem__, "octetTotalCount")
        self.assertEqual(newstml[1][1]["destinationTransportPort"], 19)


class CallbackTester(object):
    def __init__(self):
        self.templates = dict()

    def __call__(self, session, tmpl, ctx):
        self.templates[tmpl.template_id] = tmpl
        return tmpl.template_id

class TestBuffer(unittest.TestCase):

    def setUp(self):
        self.model = InfoModel()
        pyfixbuf.cert.add_elements_to_model(self.model)
        self.template = Template(self.model)
        self.spec = InfoElementSpec("sourceTransportPort")
        self.spec2 = InfoElementSpec("destinationTransportPort")
        self.nospec = InfoElementSpec("notanElement")
        self.speclist = [self.spec, self.spec2]

        self.speclist2=[InfoElementSpec("sourceIPv4Address"),
                        InfoElementSpec("destinationIPv4Address"),
                        InfoElementSpec("octetTotalCount", 4)]
        self.template.add_spec_list(self.speclist)
        self.template.add_spec_list(self.speclist2)

        self.session = Session(self.model)

    def testBufferConstructor(self):
        self.assertRaises(Exception, Buffer, self.model)
        rec = Record(self.model, self.template)
        buf = Buffer(rec)

    def testBufferInitExporter(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)

        rec = Record(self.model, self.template)
        exp = Exporter()
        buf = Buffer(rec)

        self.assertRaises(Exception, buf.init_export, self.model, exp)
        self.assertRaises(Exception, buf.init_export, self.session, rec)
        self.assertRaises(Exception, buf.init_export, self.session, exp)
        exp.init_file(filename)

        buf.init_export(self.session, exp)

    def testBufferInitCollection(self):
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)

        self.assertRaises(Exception, buf.init_collection, self.model, coll)
        self.assertRaises(Exception, buf.init_collection, self.session, rec)
        self.assertRaises(Exception, buf.init_collection, self.session, coll)

        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)


    def testBufferSetTemplates(self):
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)

        self.assertRaises(Exception, buf.set_internal_template, 999)
        self.session.add_template(self.template, 999)

        self.assertRaises(Exception, buf.set_internal_template, 999)
        self.assertRaises(Exception, buf.set_export_template, 999)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        buf.set_internal_template(999)

        buf.set_export_template(999)

    def testBufferWriteOptions(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)

        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        exp = Exporter()
        coll.init_file(SAMPLE_FILE)
        exp.init_file(filename)
        ie = InfoElement("myReallyCoolElement", CERT_PEN, 6589, 2, True, True, DataType.IP4ADDR)
        self.model.add_element(ie)
        self.assertRaises(Exception, buf.write_ie_options_record, "notanElement", self.template)
        buf.init_collection(self.session, coll)
        self.assertRaises(Exception, buf.write_ie_options_record, "notanElement", self.template)

        buf.init_export(self.session, exp)
        self.assertRaises(Exception, buf.write_ie_options_record, "notanElement", self.template)

        # not an Options Template
        self.assertRaises(Exception, buf.write_ie_options_record, "myReallyCoolElement", self.template)

        oTmpl = Template(self.model, 1)
        #buf.write_ie_options_record("myReallyCoolElement", oTmpl)


    def testBufferNext(self):
        rec = Record(self.model)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)

        rec.add_element("sourceTransportPort")
        rec.add_element("sourceIPv4Address")
        rec.add_element("packetTotalCount")
        rec.add_element("subTemplateList")
        buf.init_collection(self.session, coll)
        self.assertRaises(Exception, buf.set_internal_template, 999)
        self.session.add_template(self.template, 1001)
        rec = Record(self.model, self.template)
        buf.set_record(rec)
        buf.set_internal_template(1001)

        buf.auto_insert()

        tmpl_next = buf.next_template()
        self.assertEqual(tmpl_next.template_id, 999)
        self.assertEqual(len(tmpl_next), 8)

        count = 0
        for data in buf:
            data = data.as_dict()
            count += 1
            self.assertEqual(data["sourceTransportPort"], count)
            self.assertEqual(str(data["sourceIPv4Address"]), '192.168.1.3')
            self.assertEqual(str(data["destinationIPv4Address"]), '10.5.2.3')
            self.assertEqual(data["octetTotalCount"], 0)
            self.assertEqual(data["destinationTransportPort"], 16-count)

    def testBufferSetRecord(self):
        coll = Collector()
        buf = Buffer()
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 1001)
        rec = Record(self.model, self.template)
        buf.set_record(rec)
        buf.auto_insert()
        count = 0
        for data in buf:
            data = data.as_dict()
            count += 1
            self.assertEqual(data["sourceTransportPort"], count)
            self.assertEqual(str(data["sourceIPv4Address"]), '192.168.1.3')
            self.assertEqual(str(data["destinationIPv4Address"]), '10.5.2.3')
            self.assertEqual(data["octetTotalCount"], 0)
            self.assertEqual(data["destinationTransportPort"], 16-count)

    def testBufferRecordIterItems(self):
        coll = Collector()
        buf = Buffer()
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        specs = []
        specs.extend(self.template)
        self.session.add_template(self.template, 1001)
        rec = Record(self.model, self.template)
        buf.set_record(rec)
        buf.auto_insert()
        count = 0
        for data in buf:
            count += 1
            pos = 0
            for field in data.iterfields():
                self.assertEqual(field.ie.name, specs[pos].name)
                pos = pos + 1
                if field.ie.name == "sourceTransportPort":
                    self.assertEqual(field.value, count)
                elif field.ie.name == "destinationTransportPort":
                    self.assertEqual(field.value, 16-count)
                elif field.ie.name == "sourceIPv4Address":
                    self.assertEqual(str(field.value), '192.168.1.3')
                elif field.ie.name == "destinationIPv4Address":
                    self.assertEqual(str(field.value), '10.5.2.3')
                elif field.ie.name == "octetTotalCount":
                    self.assertEqual(field.value, 0)
                else:
                    self.fail("Unexpected name " + field.ie.name)


    def testBufferBlindNext(self):
        coll = Collector()
        buf = Buffer()
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)

        buf.auto_insert()
        tmpl_next = buf.next_template()
        # shouldn't have an internal template with this ID
        self.assertRaises(Exception, self.session.get_template, tmpl_next.template_id, True)
        # this returns the external template
        tmpl = self.session.get_template(tmpl_next.template_id)
        # now add it to the session
        self.session.add_template(tmpl, tmpl_next.template_id)
        # set buffer to the new template
        buf.set_internal_template(tmpl_next.template_id)
        # no record set on buffer
        count = 0
        for data in buf:
            count += 1
            self.assertEqual(data["sourceTransportPort"], count)
            self.assertEqual(str(data["sourceIPv4Address"]), '192.168.1.3')
            self.assertEqual(str(data["destinationIPv4Address"]), '10.5.2.3')
            self.assertEqual(data["packetTotalCount"], 98)
            self.assertEqual(data["destinationTransportPort"], 16-count)
            self.assertTrue("subTemplateMultiList" in data)
            stml = data["subTemplateMultiList"]
            self.assertEqual(len(stml), 1)
            entry = stml[0]
            self.assertEqual(len(entry), 2)
            self.assertTrue("subTemplateList" in entry)
            stl = entry["subTemplateList"]
            self.assertEqual(len(stl), 1)
            self.assertTrue("basicList" in stl)
            stlrec = stl[0]
            self.assertEqual(len(stlrec), 3)
            # must call this basicList
            self.assertRaises(Exception, entry.__getitem__, "httpUserAgent")
            self.assertEqual(len(entry["basicList"]), 2)
            self.assertEqual(len(entry[("basicList", 1)]), 1)
            bl2 = entry[0][2]
            self.assertEqual(len(bl2), 1)
            self.assertEqual(bl2.element.name, "httpServerString")
            self.assertEqual(bl2, ["wikipedia.com"])
            self.assertEqual(entry[0].count("basicList"), 2)

        self.assertEqual(count, 10)

    def testBufferAutoMode(self):
        coll = Collector()
        buf = Buffer(auto=True)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)

        buf.auto_insert()

        count = 0
        for data in buf:
            count += 1
            self.assertEqual(data["sourceTransportPort"], count)
            self.assertEqual(str(data["sourceIPv4Address"]), '192.168.1.3')
            self.assertEqual(str(data["destinationIPv4Address"]), '10.5.2.3')
            self.assertEqual(data["packetTotalCount"], 98)
            self.assertEqual(data["destinationTransportPort"], 16-count)
            self.assertTrue("subTemplateMultiList" in data)
            stml = data["subTemplateMultiList"]
            self.assertEqual(len(stml), 1)
            entry = stml[0]
            self.assertEqual(len(entry), 2)
            self.assertTrue("subTemplateList" in entry)
            stl = entry["subTemplateList"]
            self.assertEqual(len(stl), 1)
            self.assertTrue("basicList" in stl)
            stlrec = stl[0]
            self.assertEqual(len(stlrec), 3)
            # must call this basicList
            self.assertRaises(Exception, entry.__getitem__, "httpUserAgent")
            self.assertEqual(len(entry["basicList"]), 2)
            self.assertEqual(len(entry[("basicList", 1)]), 1)
            bl2 = entry[0][2]
            self.assertEqual(len(bl2), 1)
            self.assertEqual(bl2.element.name, "httpServerString")
            self.assertEqual(bl2, ["wikipedia.com"])
            self.assertEqual(entry[0].count("basicList"), 2)

        self.assertEqual(count, 10)

    def testBufferNextSkipOptions(self):
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 1001)
        buf.set_internal_template(1001)

        tmpl_next = buf.next_template()
        self.assertEqual(tmpl_next.template_id, 888)
        self.assertEqual(tmpl_next.scope, 1)
        count = 0
        #skip options
        buf.next_record(rec)
        #get next
        for data in buf:
            self.assertEqual(buf.get_template().template_id, 999)
            data = data.as_dict()
            count += 1
            self.assertEqual(data["sourceTransportPort"], count)
            self.assertEqual(str(data["sourceIPv4Address"]), '192.168.1.3')
            self.assertEqual(str(data["destinationIPv4Address"]), '10.5.2.3')
            self.assertEqual(data["octetTotalCount"], 0)
            self.assertEqual(data["destinationTransportPort"], 16-count)

    def testBufferRetrieveSTML(self):
        self.template.add_element("subTemplateMultiList")
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 1001)
        buf.set_internal_template(1001)
        inrec = Record(self.model)
        inrec.add_element("subTemplateList")
        counter = 0
        buf.auto_insert()
        for data in buf:
            if "subTemplateMultiList" in data:
                stml = data["subTemplateMultiList"]
                counter += 1
                if "subTemplateList" in stml:
                    for entry in stml:
                        entry.set_record(inrec)
                        for item in entry:
                            stl = item["subTemplateList"]
                            self.assertEqual(len(stl), 1)

                        stl.clear()
                stml.clear()
        # there are 11 data records, 1 for info element info record
        self.assertEqual(counter, 10)

    def testBufferRetrieveSTL(self):
        self.template.add_element("subTemplateMultiList")
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 444)
        buf.set_internal_template(444)
        inrec = Record(self.model)
        inrec.add_element("subTemplateList")
        inrec.add_element("httpUserAgent", BASICLIST)
        inrec.add_element("httpServerString", BASICLIST)
        counter = 0
        buf.auto_insert()
        for data in buf:
            if "subTemplateMultiList" in data:
                stml = data["subTemplateMultiList"]
                if "subTemplateList" in stml:
                    counter += 1
                    for entry in stml:
                        entry.set_record(inrec)
                        for item in entry:
                            self.assertTrue("Mozilla" in item["httpUserAgent"])
                            stl = item["subTemplateList"]
                            self.assertEqual(stl.template_id, 1001)
                            template = self.session.get_template(stl.template_id)
                            self.assertTrue("octetTotalCount" in template)
                            self.assertEqual(len(stl), 1)

                        stl.clear()
                stml.clear()
        # there are 11 data records, 1 for info element info record
        self.assertEqual(counter, 10)

    def testBufferIgnoreTemplates(self):
        self.template.add_element("subTemplateMultiList")
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 444)
        self.session.ignore_templates([1000])
        buf.set_internal_template(444)
        inrec = Record(self.model)
        inrec.add_element("subTemplateList")
        inrec.add_element("httpUserAgent", BASICLIST)
        inrec.add_element("httpServerString", BASICLIST)

        buf.auto_insert()
        for data in buf:
            template = buf.get_template()
            self.assertEqual(template.template_id, 999)
            stml = data["subTemplateMultiList"]
            for entry in stml:
                entry.set_record(inrec)
                for item in entry:
                    self.assertTrue("Safari" in item["httpUserAgent"])
                    stl = item["subTemplateList"]
                    self.assertEqual(len(stl), 0)

    def testBufferDecodeOnly(self):
        self.template.add_element("subTemplateMultiList")
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 444)
        self.session.decode_only([1000])
        buf.set_internal_template(444)
        inrec = Record(self.model)
        inrec.add_element("subTemplateList")
        inrec.add_element("httpUserAgent", BASICLIST)
        inrec.add_element("httpServerString", BASICLIST)

        buf.auto_insert()
        for data in buf:
            template = buf.get_template()
            self.assertEqual(template.template_id, 999)
            stml = data["subTemplateMultiList"]
            for entry in stml:
                entry.set_record(inrec)
                for item in entry:
                    self.assertTrue("wikipedia.com" in item["httpServerString"])
                    stl = item["subTemplateList"]
                    self.assertEqual(len(stl), 0)

    def testIgnoreOptions(self):
        self.template.add_element("subTemplateMultiList")
        rec = Record(self.model, self.template)
        coll = Collector()
        buf = Buffer(rec)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(self.session, coll)
        self.session.add_template(self.template, 444)
        buf.set_internal_template(444)
        self.assertRaises(Exception, buf.ignore_options, -1)
        buf.ignore_options(True)
        inrec = Record(self.model)
        inrec.add_element("subTemplateList")
        inrec.add_element("httpUserAgent", BASICLIST)
        inrec.add_element("httpServerString", BASICLIST)

        for data in buf:
            template = buf.get_template()
            self.assertEqual(template.template_id, 999)
            stml = data["subTemplateMultiList"]
            for entry in stml:
                entry.set_record(inrec)
                for item in entry:
                    self.assertTrue("wikipedia.com" in item["httpServerString"])
                    stl = item["subTemplateList"]
                    self.assertEqual(len(stl), 1)

    XMLTest = """
    <registry>
        <record>
           <name>testRecordOne</name>
           <dataType>string</dataType>
           <dataTypeSemantics>identifier</dataTypeSemantics>
           <elementId>32767</elementId>
           <units>inferred</units>
        </record>
        <record>
           <name>testRecordTwo</name>
           <dataType>unsigned16</dataType>
           <enterpriseId>32473</enterpriseId>
           <elementId>1</elementId>
           <units>entries</units>
        </record>
    </registry>
    """

    def checkXMLBeforeRead(self, model):
        self.assertRaises(KeyError, model.get_element, id=32767)
        self.assertRaises(KeyError, model.get_element, "testRecordOne")
        self.assertRaises(KeyError, model.get_element, ent=32473, id=1)
        self.assertRaises(KeyError, model.get_element, "testRecordTwo")

    def checkXMLAfterRead(self, model):
        ie = model.get_element(id=32767)
        self.assertEqual(ie.name, "testRecordOne")
        self.assertEqual(ie.type, DataType.STRING)
        self.assertEqual(ie.id, 32767)
        self.assertEqual(ie.ent, 0)
        self.assertEqual(ie.semantic, Semantic.IDENTIFIER)
        self.assertEqual(ie.units, Units.INFERRED)
        self.assertEqual(ie.reversible, True)
        self.assertEqual(ie.length, VARLEN)
        ie = model.get_element("testRecordOne")
        self.assertEqual(ie.name, "testRecordOne")
        self.assertEqual(ie.type, DataType.STRING)
        self.assertEqual(ie.id, 32767)
        self.assertEqual(ie.ent, 0)
        self.assertEqual(ie.semantic, Semantic.IDENTIFIER)
        self.assertEqual(ie.units, Units.INFERRED)
        self.assertEqual(ie.reversible, True)
        self.assertEqual(ie.length, VARLEN)
        ie = model.get_element(ent=32473, id=1)
        self.assertEqual(ie.name, "testRecordTwo")
        self.assertEqual(ie.type, DataType.UINT16)
        self.assertEqual(ie.id, 1)
        self.assertEqual(ie.ent, 32473)
        self.assertEqual(ie.semantic, Semantic.QUANTITY)
        self.assertEqual(ie.units, Units.ENTRIES)
        self.assertEqual(ie.reversible, False)
        self.assertEqual(ie.length, 2)
        ie = model.get_element("testRecordTwo")
        self.assertEqual(ie.name, "testRecordTwo")
        self.assertEqual(ie.type, DataType.UINT16)
        self.assertEqual(ie.id, 1)
        self.assertEqual(ie.ent, 32473)
        self.assertEqual(ie.semantic, Semantic.QUANTITY)
        self.assertEqual(ie.units, Units.ENTRIES)
        self.assertEqual(ie.reversible, False)
        self.assertEqual(ie.length, 2)

    def testXMLRead(self):
        model = InfoModel()
        self.checkXMLBeforeRead(model)
        model.read_from_xml_data(self.XMLTest)
        self.checkXMLAfterRead(model)

    def testXMLReadFile(self):
        model = InfoModel()
        self.checkXMLBeforeRead(model)
        tmp = tempfile.NamedTemporaryFile('w', delete=False)
        name = tmp.name
        try:
            with open(name, 'w') as f:
                f.write(self.XMLTest)
                f.close()
            model.read_from_xml_file(name)
            self.checkXMLAfterRead(model)
        finally:
            os.remove(name)

    def testBufferTemplateCallback(self):
        coll = Collector()
        buf = Buffer()
        session = Session(self.model)
        coll.init_file(SAMPLE_FILE)
        buf.init_collection(session, coll)
        cb = CallbackTester()
        session.add_template_callback(cb)
        tmpl = buf.next_template()
        self.assertEqual(tmpl.template_id, tmpl.get_context())
        self.assertTrue(tmpl.template_id in cb.templates)

    def testBufferRecordStrings(self):
        sport = 'sourceTransportPort'
        string_values = [
            "The quick brown fox jumps over the lazy dog",
            "How much wood would a woodchuck chuck?"
        ]
        rec_keys = (sport, "destinationTransportPort",
                    "wlanSSID", "natPoolName")
        start_val = 33
        rec_vals = {
            sport: start_val,
            'destinationTransportPort': 54321,
            'wlanSSID': string_values[0],
            'natPoolName': string_values[1]
        }

        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)

        # set up the templates
        tmpl_0 = Template(self.model)
        for k in rec_keys:
            tmpl_0.add_spec(InfoElementSpec(k))

        tmpl_1 = Template(self.model)
        tmpl_1.add_spec_list((InfoElementSpec(sport),
                              InfoElementSpec('subTemplateList')))

        tmpl_2 = Template(self.model)
        tmpl_2.add_spec_list((InfoElementSpec(sport),
                              InfoElementSpec('subTemplateMultiList')))

        tmpl_3 = Template(self.model)
        tmpl_3.add_spec_list((InfoElementSpec(sport),
                              InfoElementSpec('basicList')))

        # write the records
        export = Exporter()
        export.init_file(filename)
        session = Session(self.model)
        writer = Buffer()
        writer.init_export(session, export)

        tid_0 = session.add_template(tmpl_0)
        tid_1 = session.add_template(tmpl_1)
        tid_2 = session.add_template(tmpl_2)
        tid_3 = session.add_template(tmpl_3)
        session.export_templates()

        rec_0 = Record(self.model, tmpl_0)
        for k in rec_keys:
            rec_0[k] = rec_vals[k]
        rec_vals[sport] = rec_vals[sport] + 1

        writer.set_internal_template(tid_0)
        writer.set_export_template(tid_0)
        writer.append(rec_0)

        rec_1 = Record(self.model, tmpl_1)
        rec_1[sport] = rec_vals[sport]
        rec_vals[sport] = rec_vals[sport] + 1

        rec = Record(self.model, tmpl_0)
        stl = STL(rec_1, 'subTemplateList')
        stl.entry_init(rec, tmpl_0, 2)

        for i in range(2):
            for k in rec_keys:
                rec[k] = rec_vals[k]
            rec_vals[sport] = rec_vals[sport] + 1
            stl[i] = rec
            rec.clear()

        rec = None
        writer.set_internal_template(tid_1)
        writer.set_export_template(tid_1)
        writer.append(rec_1)

        rec_2 = Record(self.model, tmpl_2)
        rec_2[sport] = rec_vals[sport]
        rec_vals[sport] = rec_vals[sport] + 1

        stml = STML(type_count = 2)

        rec = Record(self.model, tmpl_0)
        stml[0].entry_init(rec, tmpl_0, 2)

        for i in range(2):
            for k in rec_keys:
                rec[k] = rec_vals[k]
            rec_vals[sport] = rec_vals[sport] + 1
            stml[0][i] = rec
            rec.clear()

        rec = Record(self.model, tmpl_3)
        stml[1].entry_init(rec, tmpl_3, 2)

        rec[sport] = rec_vals[sport]
        rec_vals[sport] = rec_vals[sport] + 1

        bl = BL(self.model, 'wlanSSID', len(string_values))
        for k in range(len(string_values)):
            bl[k] = string_values[k]
        rec['basicList'] = bl

        stml[1][0] = rec
        rec.clear()

        rec[sport] = rec_vals[sport]
        rec_vals[sport] = rec_vals[sport] + 1
        bl = BL(self.model, 'natPoolName', len(string_values))
        for k in range(len(string_values)):
            bl[k] = string_values[len(string_values) - 1 - k]
        rec['basicList'] = bl

        stml[1][1] = rec
        rec.clear()

        rec_2['subTemplateMultiList'] = stml

        writer.set_internal_template(tid_2)
        writer.set_export_template(tid_2)
        writer.append(rec_2)

        writer.emit()
        writer.free()
        writer = None
        session = None
        export = None
        rec = None
        rec_1 = None
        rec_2 = None

        last_val = rec_vals[sport]
        rec_vals[sport] = start_val

        # read the file
        collect = Collector()
        collect.init_file(filename)
        session = Session(InfoModel())
        reader = Buffer(auto=True)
        reader.init_collection(session, collect)
        reader.auto_insert()

        rec = next(reader)
        for k in rec_keys:
            self.assertEqual(rec_vals[k], rec[k])
        rec_vals[sport] = rec_vals[sport] + 1

        rec_1 = next(reader)
        self.assertEqual(rec_vals[sport], rec_1[sport])
        rec_vals[sport] = rec_vals[sport] + 1

        stl = rec_1['subTemplateList']
        self.assertTrue(isinstance(stl, STL))

        for rec in stl:
            for k in rec_keys:
                self.assertEqual(rec_vals[k], rec[k])
            rec_vals[sport] = rec_vals[sport] + 1

        rec_2 = next(reader)
        self.assertEqual(rec_vals[sport], rec_2[sport])
        rec_vals[sport] = rec_vals[sport] + 1

        stml = rec_2['subTemplateMultiList']
        self.assertTrue(isinstance(stml, STML))
        it = iter(stml)

        entry = next(it)
        self.assertTrue(isinstance(entry, pyfixbuf.STMLEntry))
        for rec in entry:
            for k in rec_keys:
                self.assertEqual(rec_vals[k], rec[k])
            rec_vals[sport] = rec_vals[sport] + 1

        entry = next(it)

        it2 = iter(entry)
        rec = next(it2)

        self.assertEqual(rec_vals[sport], rec[sport])
        rec_vals[sport] = rec_vals[sport] + 1

        self.assertTrue(isinstance(rec['basicList'], BL))
        for x,y in zip(rec['basicList'], string_values):
            self.assertEqual(x, y)
        rec = next(it2)

        self.assertEqual(rec_vals[sport], rec[sport])
        rec_vals[sport] = rec_vals[sport] + 1

        for k in range(len(string_values)):
            self.assertEqual(rec['basicList'][k],
                             string_values[len(string_values) - 1 - k])

        self.assertRaises(StopIteration, it2.next)
        self.assertRaises(StopIteration, it.next)
        self.assertRaises(StopIteration, reader.next)
        self.assertEqual(rec_vals[sport], last_val)

        # done
        os.remove(filename)

    def testBufferMacAddr(self):
        values = (
            (u"01:02:03:04:05:06", u"ff:fe:fd:fc:fb:fa"),
            (1108152157446, 281470647991290),
            (str(u"01:02:03:04:05:06"), str(u"ff:fe:fd:fc:fb:fa")),
            (bytearray(b"\x01\x02\x03\x04\x05\x06"),
             bytearray(b"\xff\xfe\xfd\xfc\xfb\xfa"))
        )
        rec_keys = ("sourceMacAddress", "destinationMacAddress")

        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)

        # set up the templates
        tmpl_0 = Template(self.model)
        for key in rec_keys:
            tmpl_0.add_spec(InfoElementSpec(key))

        tmpl_1 = Template(self.model)
        tmpl_1.add_spec(InfoElementSpec('basicList'))
        tmpl_1.add_spec(InfoElementSpec('basicList'))

        # write the records
        export = Exporter()
        export.init_file(filename)
        session = Session(self.model)
        writer = Buffer()
        writer.init_export(session, export)

        tid_0 = session.add_template(tmpl_0)
        tid_1 = session.add_template(tmpl_1)

        writer.set_internal_template(tid_0)
        writer.set_export_template(tid_0)

        rec_0 = Record(self.model, tmpl_0)
        for pair in values:
            for k,key in enumerate(rec_keys):
                rec_0[key] = pair[k]
            writer.append(rec_0)

        rec_1 = Record(self.model, tmpl_1)
        for k,key in enumerate(rec_keys):
            bl = BL(self.model, key, len(values))
            for j,pairs in enumerate(values):
                bl[j] = pairs[k]
            rec_1[k] = bl

        writer.set_internal_template(tid_1)
        writer.set_export_template(tid_1)
        writer.append(rec_1)

        writer.emit()
        writer.free()
        writer = None
        session = None
        export = None
        rec_0 = None
        rec_1 = None

        # read the file
        collect = Collector()
        collect.init_file(filename)
        session = Session(InfoModel())
        reader = Buffer(auto=True)
        reader.init_collection(session, collect)
        reader.auto_insert()

        for pairs in values:
            rec = next(reader)
            for k,key in enumerate(rec_keys):
                self.assertEqual(values[0][k], rec[key])

        rec = next(reader)
        for k,key in enumerate(rec_keys):
            bl = rec[k]
            self.assertTrue(isinstance(bl, BL))
            for j,pairs in enumerate(values):
                self.assertEqual(values[0][k], bl[j])

        self.assertRaises(StopIteration, reader.next)

        # done
        os.remove(filename)


class TestDataTypes(unittest.TestCase):

    def setUp(self):
        self.model = InfoModel()
        self.ie_list = [
            InfoElement("pyfixbufFLOAT32", CERT_PEN, 0x7ff0, 4,
                        type=DataType.FLOAT32, endian=True),
            InfoElement("pyfixbufINT8",    CERT_PEN, 0x7ff1, 1,
                        type=DataType.INT8),
            InfoElement("pyfixbufINT16",   CERT_PEN, 0x7ff2, 2,
                        type=DataType.INT16, endian=True),
            InfoElement("pyfixbufINT32",   CERT_PEN, 0x7ff3, 4,
                        type=DataType.INT32, endian=True),
            InfoElement("pyfixbufINT64",   CERT_PEN, 0x7ff4, 8,
                        type=DataType.INT64, endian=True)
        ]
        self.model.add_element_list(self.ie_list)

    def doOptionsTempate(self, session, buffer):
        """Writes to `buffer` the options templates that describe the info
elements defined by this test.
        """
        elem_tmpl = Template(self.model, 1)
        elem_tid = session.add_template(elem_tmpl)
        for ie in self.ie_list:
            buffer.write_ie_options_record(ie.name, elem_tmpl)

    def testFloat32(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        name = "pyfixbufFLOAT32"
        vals = [
            12345.67,
            -3456.789,
            2.345e-22,
            -3.456e17,
            -8.765e-8,
            2.71828182845905
        ]
        exponents = [
            1e4,
            1e3,
            1e-22,
            1e17,
            1e-8,
            1
        ]

        # write the file
        spec = InfoElementSpec(name)
        sess = Session(self.model)
        tmpl = Template(self.model)
        tmpl.add_spec(spec)
        tmpl.add_spec(InfoElementSpec("basicList"))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for v,e in zip(vals, exponents):
            #sys.stderr.write("setting %s to %s\n" % (name, str(v)))
            bl = BL(self.model, name, 1)
            bl[0] = v
            rec[name] = v
            rec["basicList"] = bl
            self.assertAlmostEqual(rec[name]/e, v/e, 6)
            self.assertAlmostEqual(rec["basicList"][0]/e, v/e, 6)
            buf.append(rec)
            rec.clear()
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            v = vals[count]
            e = exponents[count]
            count = count + 1
            self.assertAlmostEqual(rec[name]/e, v/e, 6)
            if 0 == count:
                for i in range(0, len(vals)):
                    self.assertAlmostEqual(rec["basicList"][i]/exponents[i],
                                           vals[i]/exponents[i], 6)
            rec.clear()
        self.assertEqual(count, len(vals))

        # done
        os.remove(filename)

    def testFloat64(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        name = "absoluteError"
        vals = [
            34560000000000000000000000000000000000000000.789,
            3.1415927,
            6.02e23,
            67e-89,
            -2.0,
            -9876543.21
        ]
        exponents = [
            1e43,
            1,
            1e23,
            1e-90,
            1,
            1e6
        ]

        # write the file
        spec = InfoElementSpec(name)
        sess = Session(self.model)
        tmpl = Template(self.model)
        tmpl.add_spec(spec)
        tmpl.add_spec(InfoElementSpec("basicList"))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for v,e in zip(vals, exponents):
            #sys.stderr.write("setting %s to %s\n" % (name, str(v)))
            rec[name] = v
            self.assertAlmostEqual(rec[name]/e, v/e, 6)
            bl = BL(self.model, spec, len(vals))
            for i in range(0, len(vals)):
                bl[i] = vals[i]
            rec["basicList"] = bl
            for i in range(0, len(vals)):
                self.assertAlmostEqual(rec["basicList"][i]/exponents[i],
                                       vals[i]/exponents[i], 6)
            buf.append(rec)
            rec.clear()
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            v = vals[count]
            e = exponents[count]
            count = count + 1
            self.assertAlmostEqual(rec[name]/e, v/e, 6)
            if 0 == count:
                for i in range(0, len(vals)):
                    self.assertAlmostEqual(rec["basicList"][i]/exponents[i],
                                           vals[i]/exponents[i], 6)
            rec.clear()
        self.assertEqual(count, len(vals))

        # done
        os.remove(filename)

    def testMixed(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "pyfixbufINT64",
            "absoluteError",
            "pyfixbufFLOAT32",
            "pyfixbufINT32",
            "pyfixbufINT16",
            "pyfixbufINT8",
        ]
        data = [
            123456789012345,
            15e-16,
            -6.02e23,
            87654321,
            31975,
            -2
        ]
        exponents = [
            None,
            1e-15,
            1e23,
            None,
            None,
            None
        ]
        data = [data, [-v for v in data]]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            if n == "absoluteError":
                tmpl.add_spec(InfoElementSpec(n, 4))
            else:
                tmpl.add_spec(InfoElementSpec(n))
        list_tmpl = Template(self.model)
        list_tmpl.add_spec(InfoElementSpec("basicList"))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        list_tid = sess.add_template(list_tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for vals in data:
            for n,v,e in zip(names, vals, exponents):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                if e is None:
                    self.assertEqual(rec[n], v)
                else:
                    self.assertAlmostEqual(rec[n]/e, v/e, 6)
            buf.append(rec)
        buf.set_internal_template(list_tid)
        buf.set_export_template(list_tid)
        rec = Record(self.model, list_tmpl)
        for i in range(0, len(names)):
            e = exponents[i]
            bl = BL(self.model, InfoElementSpec(names[i]), len(data))
            for j in range(0, len(data)):
                bl[j] = data[j][i]
            rec["basicList"] = bl
            if e is None:
                for j in range(0, len(data)):
                    self.assertEqual(rec["basicList"][j], data[j][i])
            else:
                for j in range(0, len(data)):
                    self.assertAlmostEqual(rec["basicList"][j]/exponents[i],
                                           data[j][i]/exponents[i], 6)
            buf.append(rec)
            rec.clear()
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            if rec.__contains__("basicList"):
                self.assertTrue(count >= len(data))
                i = count - len(data)
                e = exponents[i]
                if e is None:
                    for j in range(0, len(data)):
                        self.assertEqual(rec["basicList"][j], data[j][i])
                else:
                    for j in range(0, len(data)):
                        self.assertAlmostEqual(rec["basicList"][j]/exponents[i],
                                               data[j][i]/exponents[i], 6)
            else:
                vals = data[count]
                for n,v,e in zip(names, vals, exponents):
                    if e is None:
                        self.assertEqual(rec[n], v)
                    else:
                        self.assertAlmostEqual(rec[n]/e, v/e, 6)
            count = count + 1
            rec.clear()
        self.assertEqual(count, len(data) + len(names))

        # done
        os.remove(filename)

    def testReducedLength1(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "protocolIdentifier",
            "sourceTransportPort",
            "ingressInterface",
            "octetTotalCount",
            "pyfixbufINT8",
            "pyfixbufINT16",
            "pyfixbufINT32",
            "pyfixbufINT64"]
        data = [
            0xff,
            0x7f,
            0x3f,
            0x1f,
            0x0f,
            0x07,
            0x03,
            0x01]
        data = [data, [v - 0x10 for v in data]]
        bad = [1 << 9, 1 << 12, 1 << 25, 1 << 62,
               -(1 << 9), -(1 << 12), -(1 << 25), -(1 << 62)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 1))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                #sys.stderr.write("# %d, read %s as %s\n" % (count, n, str(v)))
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength2(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "sourceTransportPort",
            "ingressInterface",
            "octetTotalCount",
            "pyfixbufINT16",
            "pyfixbufINT32",
            "pyfixbufINT64"]
        data = [
            0xffff,
            0x7fff,
            0x3fff,
            0x07ff,
            0x03ff,
            0x01ff]
        data = [data, [v - 0x1000 for v in data]]
        bad = [1 << 18, 1 << 25, 1 << 62,
               -(1 << 18), -(1 << 25), -(1 << 62)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 2))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength3(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "ingressInterface",
            "octetTotalCount",
            "pyfixbufINT32",
            "pyfixbufINT64"]
        data = [
            0xffffff,
            0x3fffff,
            0x03ffff,
            0x01ffff]
        data = [data, [v - 0x100000 for v in data]]
        bad = [1 << 25, 1 << 27, 1 << 62,
               -(1 << 25), -(1 << 27), -(1 << 62)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 3))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength4(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "ingressInterface",
            "octetTotalCount",
            "pyfixbufINT32",
            "pyfixbufINT64"]
        data = [
            0xffffffff,
            0x3fffffff,
            0x03ffffff,
            0x01ffffff]
        data = [data, [v - 0x10000000 for v in data]]
        bad = [1 << 34, 1 << 62, -(1 << 34), -(1 << 62)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 4))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength5(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "octetTotalCount",
            "pyfixbufINT64"]
        data = [
            0xffffffffff,
            0x01ffffffff]
        data = [data, [v - 0x1000000000 for v in data]]
        bad = [1 << 41, -(1 << 41)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 5))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength6(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "octetTotalCount",
            "pyfixbufINT64"]
        data = [
            0xffffffffffff,
            0x01ffffffffff]
        data = [data, [v - 0x100000000000 for v in data]]
        bad = [1 << 50, -(1 << 50)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 6))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength7(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "octetTotalCount",
            "pyfixbufINT64"]
        data = [
            0xffffffffffffff,
            0x01ffffffffffff]
        data = [data, [v - 0x10000000000000 for v in data]]
        bad = [1 << 58, -(1 << 58)]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 7))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for n,v in zip(names, bad):
            #sys.stderr.write("Setting rec[%s] to %d\n" % (n, v))
            self.assertRaises(ValueError, rec.__setitem__, n, v)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)

    def testReducedLength8(self):
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        names = [
            "octetTotalCount",
            "pyfixbufINT64"]
        data = [
            0xffffffffffffffff,
            0x01ffffffffffffff]
        data = [data, [v - 0x1000000000000000 for v in data]]

        # write the file
        sess = Session(self.model)
        tmpl = Template(self.model)
        for n in names:
            tmpl.add_spec(InfoElementSpec(n, 8))
        exp = Exporter()
        exp.init_file(filename)
        buf = Buffer()
        buf.init_export(sess, exp)
        self.doOptionsTempate(sess, buf)
        tid = sess.add_template(tmpl)
        buf.set_internal_template(tid)
        buf.set_export_template(tid)
        rec = Record(self.model, tmpl)
        for vals in data:
            for n,v in zip(names, vals):
                #sys.stderr.write("setting %s to %s\n" % (n, str(v)))
                rec[n] = v
                self.assertEqual(rec[n], v)
            buf.append(rec)
        buf.emit()
        buf.free()
        buf = None
        exp = None
        sess = None
        tmpl = None

        # read the file
        coll = Collector()
        coll.init_file(filename)
        sess = Session(InfoModel())
        buf = Buffer(auto=True)
        buf.init_collection(sess, coll)
        buf.auto_insert()
        count = 0
        for rec in buf:
            vals = data[count]
            count = count + 1
            for n,v in zip(names, vals):
                self.assertEqual(rec[n], v)
        self.assertEqual(count, len(data))

        # done
        os.remove(filename)


class TestSamplePrograms(unittest.TestCase):

    def setUp(self):
        if os.path.isdir("samples"):
            self.samples_dir = "samples"
        elif sys.hexversion >= 0x020700f0:
            self.skipTest("cannot find samples directory")
        else:
            self.samples_dir = None
        expect_dir = os.path.join("samples", "expected")
        if os.path.isdir(expect_dir):
            self.expect_dir = expect_dir
        elif sys.hexversion >= 0x020700f0:
            self.skipTest("cannot find samples/expected directory")
        else:
            self.expect_dir = None
        self.PYTHON = sys.executable
        self.dev_null = open("/dev/null", "w")

    def tearDown(self):
        if self.dev_null:
            self.dev_null.close()

    def getAppAndExpectedPaths(self, name):
        if self.samples_dir is None:
            return (None, None)
        app = os.path.join(self.samples_dir, name)
        exp = os.path.join(self.expect_dir, name + ".out")
        if not os.path.isfile(app) or not os.path.isfile(exp):
            return (None, None)
        return (app, exp)

    def compareFiles(self, a, b):
        rv = subprocess.call(["cmp", a, b], stdout=self.dev_null,
                             stderr=subprocess.STDOUT)
        self.assertEqual(0, rv, "files differ: '%s' '%s'" % (a, b))

    def test_IPFinder(self):
        (app, expected) = self.getAppAndExpectedPaths("IPFinder.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, "--in", SAMPLE_FILE,
               "--ip", "192.168.1.3", "--sip",
               "--outfields", "destinationIPv4Address",
               "destinationTransportPort"];
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_dnsQNameFinder(self):
        # sample data does not have any dns data, so the test is not
        # fully exercised
        (app, expected) = self.getAppAndExpectedPaths("dnsQNameFinder.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, "--in", SAMPLE_FILE, "--count",
               "--dnsq", "www.example.com"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    # COMMENTED OUT -- ITEMS APPEAR IN UNDEFINED ORDER
    #def test_dump_ipfix_metadata(self):
    #    (app, expected) = self.getAppAndExpectedPaths("dump_ipfix_metadata.py")
    #    if app is None:
    #        if sys.hexversion >= 0x020700f0:
    #            self.skipTest("Skipping test")
    #        return
    #    prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
    #    (fp, filename) = tempfile.mkstemp(prefix=prefix)
    #    cmd = [self.PYTHON, app, "--in", SAMPLE_FILE, "--screen"]
    #    rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
    #    os.close(fp)
    #    self.assertEqual(rv, 0, "Failure while running " + str(cmd))
    #    self.compareFiles(expected, filename)

    def test_list_iterator(self):
        (app, expected) = self.getAppAndExpectedPaths("list_iterator.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, "--in", SAMPLE_FILE]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_nmsg_to_pipeline(self):
        (app, expected) = self.getAppAndExpectedPaths("nmsg_to_pipeline.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        # write textual input to a file
        (fp, textfile) = tempfile.mkstemp(prefix=prefix, suffix=".txt")
        os.write(fp, b"express.cites.example.com.,2,29280,192.168.0.62\n")
        os.write(fp, b"dns1.example.com.,1,35038,192.168.0.63,192.168.1.63\n")
        os.write(fp, b"dns1.cso.example.com.,1,11208,192.168.0.64\n")
        os.write(fp, b"dns2.cso.example.com.,28,126,2011:db8::41\n")
        os.close(fp)
        # read text and produce ipfix
        (fp, ipfixfile) = tempfile.mkstemp(prefix=prefix, suffix=".ipfix")
        cmd = [self.PYTHON, app, textfile, "-"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        # read ipfix and produce text
        (app, ignored) = self.getAppAndExpectedPaths("sample_exporter_collector_blind.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, ipfixfile]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_portMismatch(self):
        (app, expected) = self.getAppAndExpectedPaths("portMismatch.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, "--in", SAMPLE_FILE, "--count"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    # COMMENTED OUT -- USES NETWORK
    #def test_sample_blind_listener(self):
    #    (app, expected) = self.getAppAndExpectedPaths("sample_blind_listener.py")
    #    if app is None:
    #        if sys.hexversion >= 0x020700f0:
    #            self.skipTest("Skipping test")
    #        return
    #    prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
    #    (fp, filename) = tempfile.mkstemp(prefix=prefix)
    #    cmd = [self.PYTHON, app, SAMPLE_FILE]
    #    rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
    #    os.close(fp)
    #    self.assertEqual(rv, 0, "Failure while running " + str(cmd))
    #    self.compareFiles(expected, filename)

    def test_sample_collector(self):
        (app, expected) = self.getAppAndExpectedPaths("sample_collector.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, SAMPLE_FILE, "-"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_sample_exporter(self):
        (app, expected) = self.getAppAndExpectedPaths("sample_exporter.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        # produce ipfix
        (fp, ipfixfile) = tempfile.mkstemp(prefix=prefix, suffix=".ipfix")
        cmd = [self.PYTHON, app, "-"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        # read ipfix and produce text
        (app, ignored) = self.getAppAndExpectedPaths("sample_exporter_collector_blind.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, ipfixfile]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_sample_exporter_collector(self):
        (app, expected) = self.getAppAndExpectedPaths("sample_exporter_collector.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, SAMPLE_FILE, "-"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_sample_exporter_collector_blind(self):
        (app, expected) = self.getAppAndExpectedPaths("sample_exporter_collector_blind.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, SAMPLE_FILE]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    # COMMENTED OUT -- USES NETWORK
    #def test_sample_listener(self):
    #    (app, expected) = self.getAppAndExpectedPaths("sample_listener.py")
    #    if app is None:
    #        if sys.hexversion >= 0x020700f0:
    #            self.skipTest("Skipping test")
    #        return
    #    prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
    #    (fp, filename) = tempfile.mkstemp(prefix=prefix)
    #    cmd = [self.PYTHON, app, SAMPLE_FILE, filename]
    #    rv = subprocess.call(cmd, stdout = self.dev_null,
    #                         stderr = subprocess.STDOUT)
    #    self.assertEqual(rv, 0, "Failure while running " + str(cmd))
    #    self.compareFiles(expected, filename)

    # COMMENTED OUT -- bytearray vs str for Python 3 vs Python 2
    #def test_sample_mediator(self):
    #    (app, expected) = self.getAppAndExpectedPaths("sample_mediator.py")
    #    if app is None:
    #        if sys.hexversion >= 0x020700f0:
    #            self.skipTest("Skipping test")
    #        return
    #    # read ipfix and produce ipfix
    #    prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
    #    (fp, ipfixfile) = tempfile.mkstemp(prefix=prefix, suffix=".ipfix")
    #    cmd = [self.PYTHON, app, SAMPLE_FILE, "-"]
    #    rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
    #    os.close(fp)
    #    self.assertEqual(rv, 0, "Failure while running " + str(cmd))
    #    # read ipfix and produce text
    #    (app, ignored) = self.getAppAndExpectedPaths("sample_exporter_collector_blind.py")
    #    if app is None:
    #        if sys.hexversion >= 0x020700f0:
    #            self.skipTest("Skipping test")
    #        return
    #    (fp, filename) = tempfile.mkstemp(prefix=prefix)
    #    cmd = [self.PYTHON, app, ipfixfile]
    #    rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
    #    os.close(fp)
    #    self.assertEqual(rv, 0, "Failure while running " + str(cmd))
    #    self.compareFiles(expected, filename)

    def test_sample_mpls_collector(self):
        (app, expected) = self.getAppAndExpectedPaths("sample_mpls_collector.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, SAMPLE_FILE, "-"]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    # COMMENTED OUT -- USES NETWORK
    #def test_sample_net_exporter(self):
    #    (app, expected) = self.getAppAndExpectedPaths("sample_net_exporter.py")
    #    if app is None:
    #        if sys.hexversion >= 0x020700f0:
    #            self.skipTest("Skipping test")
    #        return
    #    prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
    #    (fp, filename) = tempfile.mkstemp(prefix=prefix)
    #    cmd = [self.PYTHON, app, SAMPLE_FILE, filename]
    #    rv = subprocess.call(cmd, stdout = self.dev_null,
    #                         stderr = subprocess.STDOUT)
    #    self.assertEqual(rv, 0, "Failure while running " + str(cmd))
    #    self.compareFiles(expected, filename)

    def test_yaf_mediator(self):
        (app, expected) = self.getAppAndExpectedPaths("yaf_mediator.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, SAMPLE_FILE]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)

    def test_yaf_mediator_new(self):
        (app, expected) = self.getAppAndExpectedPaths("yaf_mediator_new.py")
        if app is None:
            if sys.hexversion >= 0x020700f0:
                self.skipTest("Skipping test")
            return
        prefix = "pyfixtest-%s." % sys._getframe().f_code.co_name
        (fp, filename) = tempfile.mkstemp(prefix=prefix)
        cmd = [self.PYTHON, app, SAMPLE_FILE]
        rv = subprocess.call(cmd, stdout = fp, stderr = self.dev_null)
        os.close(fp)
        self.assertEqual(rv, 0, "Failure while running " + str(cmd))
        self.compareFiles(expected, filename)




"""
def suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestInfoModel))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestInfoElement))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestInfoElementSpec))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestTemplate))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestSession))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestCollector))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestExporter))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestListener))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestRecord))
    suite.addTest(unittest.TestLoader().loadTestsFromTestCase(TestBuffer))

    return suite
"""

if __name__ == '__main__':
    unittest.main()
#    result = unittest.TextTestRunner(verbosity=3).run(suite())
#    if result.errors or result.failures:
#        sys.exit(1)
#    sys.exit(0)
