#! /usr/bin/env python
## ------------------------------------------------------------------------
## Copyright (C) 2018-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt
## ------------------------------------------------------------------------

from __future__ import print_function
import argparse
import os
import sys
import shared_utils
import portMismatch_utils


class Args(object):
    pass


if __name__ == '__main__':
    shared_utils.print_warning_message()

    in_files = []
    args = Args()

    # Create command-line argument parser and add arguments
    parser = argparse.ArgumentParser(prog='portMismatch.py',
                                     description="""
Find IPFIX records where the silkAppLabel is different than either
the source or destination port.  The output may be a count of the
mis-matched records, the mis-matched records as IPFIX, or the values
of the specified IPFIX information elements.  """)

    parser.add_argument('--input', '--in', nargs=1, required=True,
                        help='input file or directory (no *)')

    output_group = parser.add_mutually_exclusive_group(required=True)
    output_group.add_argument('--outfields', nargs='+',
                              help="Fields to print from filtered records")
    output_group.add_argument('--outfile', nargs=1,
                              help="IPFIX file to output filtered records to")
    output_group.add_argument('--count', action='store_true',
                              help="Print total count of found records")

    parser.add_argument('--delim', nargs=1, default='|',
                        help="field delimiter for printed fields")

    # Add all command-line arguments to instance of class Args
    parser.parse_args(namespace=args)

    # handle input directory vs single input file
    if os.path.isdir(args.input[0]):
        for filename in os.listdir(args.input[0]):
            in_files.append(args.input[0] + '/' + filename)
    else:
        in_files.append(args.input[0])

    # handle outputting field values to console or records to ipfix file
    if args.outfields:
        # output selected fields to console
        portMismatch_utils.output_to_screen(in_files, args.outfields, args.delim[0])
    elif args.count:
        # output total count of filtered records
        print('Counting filtered records...')
        count = portMismatch_utils.count_records(in_files)
    else:
        # output filtered records to IPFIX files
        print('Filtering to %s...' % args.outfile[0], file=sys.stderr)
        portMismatch_utils.output_to_ipfix(in_files, args.outfile[0])
