#! /usr/bin/env python
## ------------------------------------------------------------------------
## sample_listener.py
##
## sample IPFIX listener using pyfixbuf.
## This script listens for connections from an exporter and writes
## to a text file.
## Usage: hostname port {tcp,udp} file
## ------------------------------------------------------------------------
## Copyright (C) 2013-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## Authors: Emily Sarneso
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt

from __future__ import print_function
import os
import sys
import pyfixbuf
import pyfixbuf.cert

# Test that the argument number is correct

if (len(sys.argv) < 4 or sys.argv[1] in ['-h', '--help']):
    print("Usage: sample_listener.py hostname port transport [out_file].")
    print("This script runs forever")
    sys.exit()

# Create an InfoModel

infomodel = pyfixbuf.InfoModel()

# Add YAF elements to the infomodel
pyfixbuf.cert.add_elements_to_model(infomodel)

# Create a Template

tmpl = pyfixbuf.Template(infomodel)

# Create a Stats Template

stats_tmpl = pyfixbuf.Template(infomodel)

# Add some elements to the data template

data_list = [pyfixbuf.InfoElementSpec("flowStartMilliseconds"),
             pyfixbuf.InfoElementSpec("flowEndMilliseconds"),
             pyfixbuf.InfoElementSpec("octetTotalCount"),
             pyfixbuf.InfoElementSpec("reverseOctetTotalCount"),
             pyfixbuf.InfoElementSpec("packetTotalCount"),
             pyfixbuf.InfoElementSpec("reversePacketTotalCount"),
             pyfixbuf.InfoElementSpec("sourceIPv4Address"),
             pyfixbuf.InfoElementSpec("destinationIPv4Address"),
             pyfixbuf.InfoElementSpec("sourceTransportPort"),
             pyfixbuf.InfoElementSpec("destinationTransportPort"),
             pyfixbuf.InfoElementSpec("flowAttributes"),
             pyfixbuf.InfoElementSpec("reverseFlowAttributes"),
             pyfixbuf.InfoElementSpec("protocolIdentifier"),
             pyfixbuf.InfoElementSpec("flowEndReason"),
             pyfixbuf.InfoElementSpec("silkAppLabel"),
             pyfixbuf.InfoElementSpec("subTemplateMultiList")]

# Add elements to stats template (this is only a subset of stats that YAF exports)
stats_list = [pyfixbuf.InfoElementSpec("exportedFlowRecordTotalCount"),
              pyfixbuf.InfoElementSpec("packetTotalCount"),
              pyfixbuf.InfoElementSpec("droppedPacketTotalCount"),
              pyfixbuf.InfoElementSpec("ignoredPacketTotalCount")]

# Add the lists to their respective template

tmpl.add_spec_list(data_list)
stats_tmpl.add_spec_list(stats_list)

# Create a session

session = pyfixbuf.Session(infomodel)

# Add the data template to the session

tid = session.add_internal_template(tmpl, 999)

# Add the Stats Template

stats_tid = session.add_internal_template(stats_tmpl, 911)

# create a Rec for each template or subtemplate to give the buffer

rec = pyfixbuf.Record(infomodel, tmpl)

statsrec = pyfixbuf.Record(infomodel, stats_tmpl)

transport = sys.argv[3]
# Create a listener

listener = pyfixbuf.Listener(session, hostname=sys.argv[1], transport=transport.lower(), port=sys.argv[2])

# open our output file
if len(sys.argv) > 4:
    outFile = open(sys.argv[4], "w")
else:
    outFile = sys.stdout

flowcount = 0

# Start listening
while True:
    if (flowcount):
        print("Received %d flows" % flowcount)
    buf = listener.wait()
    buf.set_record(rec)
    # Set your internal template to the data template
    buf.set_internal_template(tid)

    for data in buf:
        flowcount += 1
        for field in data.iterfields():
            if field.ie.type == pyfixbuf.DataType.IP4ADDR:
                outFile.write(field.name + ": " + str(field.value) + "\n")
            elif field.ie.type != pyfixbuf.DataType.SUB_TMPL_MULTI_LIST:
                outFile.write(field.name + ": " + str(field.value) + "\n")
        stml = data["subTemplateMultiList"]
        for entry in stml:
            if "tcpSequenceNumber" in entry:
                for record in entry:
                    for field in record.iterfields():
                        outFile.write(field.name + ": " + str(field.value) + "\n")
            elif entry.template_id == 0xCE00:
                for record in entry:
                    stl = record["subTemplateList"]
                    for dnsrec in stl:
                        for field in dnsrec.iterfields():
                            if (field.ie.type != pyfixbuf.SUBTEMPLATELIST):
                                outFile.write(field.name + ": "
                                              + str(field.value) + '\n')

                stl.clear()

        stml.clear()
        outFile.write("----------------------------\n")

        # check to see if the next template is a stats template
        try:
            tmpl_next = buf.next_template()
        except StopIteration:
            if not listener:
                break

        if tmpl_next and tmpl_next.scope and stats_list[0] in tmpl_next:
            sys.stdout.write("STATS Record:\n")
            # set the interal template to the stats template
            buf.set_internal_template(stats_tid)
            # get the record
            stats = buf.next_record(statsrec)
            if (stats != None):
                # print all the items in stats
                for field in stats.iterfields():
                    sys.stdout.write(field.name + ": " + str(field.value) + "\n")
            sys.stdout.write("-------------------------\n")
            # set the template back to the data template
            buf.set_internal_template(tid)


