## ------------------------------------------------------------------------
## Copyright (C) 2018-2022 Carnegie Mellon University. All Rights Reserved.
## ------------------------------------------------------------------------
## See license information in LICENSE-OPENSOURCE.txt
## ------------------------------------------------------------------------

from __future__ import print_function
import pyfixbuf
from time_utils import make_datetime
from string import punctuation
import sys

SEEN = [0 for i in range(65535)]
EXP_SESSION = None


def print_warning_message():
    '''
    Print warning message explaning why this script is not made to be efficient
    '''
    warning_message = "*** WARNING:This script is very slow ***"
    print(warning_message, file=sys.stderr)


def add_export_templates(session, tmpl, ctx):
    global EXP_SESSION
    if EXP_SESSION is not None:
        if SEEN[tmpl.template_id] == 0:
            EXP_SESSION.add_template(tmpl, template_id=tmpl.template_id)
            SEEN[tmpl.template_id] = 1


def process_files(in_files, infomodel, exp_session=None):
    # Set up collection capabilities
    collector = pyfixbuf.Collector()
    # Set up global export session
    global EXP_SESSION
    EXP_SESSION = exp_session
    # Iterate through all passed in_files
    for in_file in in_files:
        # Initialize collector, buffer, and session
        collector.init_file(in_file)
        session = pyfixbuf.Session(infomodel)
        session.add_template_callback(add_export_templates)
        buf = pyfixbuf.Buffer(auto=True)
        buf.init_collection(session, collector)
        # Read the data
        for data in buf:
            yield data


def print_fields(record, out_fields, delim, infomodel):
    string_to_print = ""
    for field in out_fields:
        if field in record:
            elem_type = infomodel.get_element_type(field)
            if elem_type == pyfixbuf.DataType.MILLISECONDS:
                string_to_print += delim + str(make_datetime(record[field] / 1000))
            elif elem_type != pyfixbuf.DataType.SUB_TMPL_MULTI_LIST:
                string_to_print += delim + str(record[field])
        else:
            string_to_print += delim + str(None)
    print(string_to_print.strip(punctuation))


def print_to_ipfix(data, exp_buf):
    if EXP_SESSION is not None:
        tmpl_id = data.template.template_id
        exp_buf.set_internal_template(tmpl_id)
        exp_buf.set_export_template(tmpl_id)
        exp_buf.append(data)
        exp_buf.emit()
