#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-discard-when-ipset-v6.pl 04bb49edf544 2018-05-31 18:25:33Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsetbuild = check_silk_app('rwsetbuild');
my $rwsort = check_silk_app('rwsort');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');
$file{v6data} = get_data_or_exit77('v6data');

# verify that required features are available
check_features(qw(ipv6));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the IPset files
my %ipset = (
    internal_v4 => "$tmpdir/internal-v4.set",
    external_v4 => "$tmpdir/external-v4.set",
    null_v4     => "$tmpdir/null-v4.set",
    internal_v6 => "$tmpdir/internal-v6.set",
    external_v6 => "$tmpdir/external-v6.set",
    null_v6     => "$tmpdir/null-v6.set",
    );
my %blocks = (
    internal_v4 => '192.168.x.x',
    external_v4 => '10.0.0.0/8',
    null_v4     => '172.16.0.0/13',
    internal_v6 => '2001:db8:c0:a8::x:x',
    external_v6 => '2001:db8:a:x::x:x',
    null_v6     => '2001:db8:ac:10-17::x:x',
    );

for my $k (keys %ipset) {
    my $cmd = "echo '$blocks{$k}' | $rwsetbuild - '$ipset{$k}'";
    check_md5_output('d41d8cd98f00b204e9800998ecf8427e', $cmd);
}

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
{
    my $regex = join "|", keys %ipset;
    my $sensor_conf_text;
    get_config_body(\$sensor_conf_text);
    $sensor_conf_text =~ s,\$\{($regex)\},$ipset{$1},g;
    make_config_file($sensor_conf, \$sensor_conf_text);
}

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--copy $file{data}:incoming",
                     "--copy $file{v6data}:incoming",
                     "--limit=972788",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--flat-archive",
    );

# run it and check the MD5 hash of its output
check_md5_output('69f68cb84613ff60f57979dc63605dfd', $cmd);

# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental sender));

# input files should now be in the archive directory
verify_directory_files("$tmpdir/archive", $file{data}, $file{v6data});

# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  Use rwsort since input has two files,
    # and we don't know which was processed first.  IPv6 files do not
    # use the bytes/packet ratio, so no rounding issue to work-around
    check_md5_output($md5_map{$_},
                     ("$rwsort --fields=stime,sip $path "
                      ." | $rwcat --compression=none --byte-order=little"));
}

sub get_config_body
{
    my ($string_ref) = @_;

    $$string_ref = <<'EOF';
# the sensor.conf file for this test
probe P0 silk
    poll-directory ${incoming}
    quirks zero-packets
end probe

group internal
    ipsets ${internal_v4}
    ipsets "${internal_v6}"
end group
group external
    ipsets "${external_v6}" ${external_v4}
end group
group null
    ipsets "${null_v4}"
    ipsets ${null_v6}
end group

sensor S0
    silk-probes P0

    discard-when source-ipsets @null
    discard-when destination-ipsets ${null_v6} ${null_v4}

    internal-ipsets  @internal
    external-ipsets  @external
    null-ipsets      ${null_v4} "${null_v6}"
end sensor
EOF
}

__END__
5c1202799db1611bc2208ba0fc08a3c2 in-S0_20090212.00
e49d9864f8a310b18c992e096e94c90f in-S0_20090212.01
72c0b5d8dbb3d3bfd944217cd49d8a65 in-S0_20090212.02
eb35e9be06a6112db7705ef6e43e01a2 in-S0_20090212.03
c19bff77a31283af3145b4baf51a3576 in-S0_20090212.04
2daebbfa0bc5677497898802edd221d3 in-S0_20090212.05
734a02a9529967f2e6ab30815a4323fa in-S0_20090212.06
e2ec0965c46690ffb1fd347aa9d0d04c in-S0_20090212.07
295a03629ccc512028b1b1175b237bbe in-S0_20090212.08
8278f3317b43d2691fbc215360d3a223 in-S0_20090212.09
2bcd088ec78c121600dadcd507ca91c1 in-S0_20090212.10
094cf8df69d990d8d1b2f09005952e33 in-S0_20090212.11
1d6a1a4ab9961cb3904ccc2fa007a60a in-S0_20090212.12
78b8f4f54cb49e9911b6c2c57fd37f88 in-S0_20090212.13
e86ccc07f2e9313a957e155620398c17 in-S0_20090212.14
bfdb8addaf026be01165796d36fd5fc0 in-S0_20090212.15
726088454164437f1e34ca362881e718 in-S0_20090212.16
d3aa3f68724a63645af25397e4070467 in-S0_20090212.17
8e48bc8ce02f7a63249c83c161b6aefd in-S0_20090212.18
8da19d4d1ef286d3693ee03894f85fd7 in-S0_20090212.19
78b327d3d03d8d0e6e185f4fe1c8edba in-S0_20090212.20
e28697ed0a2312ee1330e318f2ee8445 in-S0_20090212.21
9180df63d28faabebe88ec3a4b1308c6 in-S0_20090212.22
758dff197b5f6fcbbce5a063a4a02b62 in-S0_20090212.23
b7936efb7d3305ae7eb761ec01ecabff in-S0_20090213.00
d769b77a4be7713e29a04c3b977dc556 in-S0_20090213.01
1ec16a4594480d6c62dd27f9f8cd9a0a in-S0_20090213.02
93262ed17a33164b84e5f23460458659 in-S0_20090213.03
b2e0a1c2108acd8d502193370f92236d in-S0_20090213.04
3896d5b39a42cdf58f13ce3549951de0 in-S0_20090213.05
3833039b8b172dcb62d57f9976a72449 in-S0_20090213.06
36e7f21b9f612122012499c6fbf90b52 in-S0_20090213.07
400a5b81fb12d93d79fea4493bd26438 in-S0_20090213.08
dc2bef11e7c9552b80d74f4e77edcc8b in-S0_20090213.09
01eb357868b5e89861a06c340a922239 in-S0_20090213.10
e9daaf3f6c5819800c34777a04a92973 in-S0_20090213.11
2a46c1a9306702569201848ef8c42329 in-S0_20090213.12
5fb0c310a2e21eed5953f57587b8f176 in-S0_20090213.13
bd25c31b2f9d038a1e4b87a9b18ff4fb in-S0_20090213.14
5ebfdbc8959d30ae78663df72854c355 in-S0_20090213.15
81858554995fabb0007747c7a29c65e2 in-S0_20090213.16
1788d4e43cf6b70be1a57f8fdfece65b in-S0_20090213.17
e3489e8488eca5759e25d8fbd0e228f1 in-S0_20090213.18
72c21f34c4aa8f187bfb8173e19c6629 in-S0_20090213.19
18a12790a7b73edc12a0fc596695820f in-S0_20090213.20
b186a316a6fafa3d42dfc9b0191a340c in-S0_20090213.21
bcccb7bd83ebc72b6eb62592dfc34c5b in-S0_20090213.22
cb7b045dfa3c1e4046847978afa00859 in-S0_20090213.23
973015ba7d6cc5d6965db87a8eef0a0a in-S0_20090214.00
38a210b59856718723b9706872f088e7 in-S0_20090214.01
a05278f9e4cd40d5fc48a4baa52fc8f1 in-S0_20090214.02
29537ab708c05502ebfa7792dabd69e2 in-S0_20090214.03
01ad3854af7a5d71b208ea89f8c8a056 in-S0_20090214.04
9eb0a4f7b659f952037e8fff795e2e95 in-S0_20090214.05
f9d2b84106b6cb9660906dd6046e5841 in-S0_20090214.06
749412c7760852c2e8e208cfd917779f in-S0_20090214.07
0224f37e6b416cac5d567c6e9771655f in-S0_20090214.08
95e02bcdef0fca2837441c10c5b92235 in-S0_20090214.09
9bf2de351d40823b89acc9b689d1ecc2 in-S0_20090214.10
c8f48091e7668ff6f83c979dfdd063c9 in-S0_20090214.11
9413ff606305aa922f4bdf966598b1fa in-S0_20090214.12
c93bf08c03398aebadd904fed86413da in-S0_20090214.13
4818ce06ba752c3837a449b731aaa1b7 in-S0_20090214.14
8b79c4a15f14fb1734d59c9a6810abac in-S0_20090214.15
b4e96ea406e2d1355be3cedeecaa2d0d in-S0_20090214.16
1ca7d4e644e55609dc0b13e832b7d98a in-S0_20090214.17
764244a864dd008f999b7d0bb3f7f2c5 in-S0_20090214.18
2642c421166d58c177072a2ef1c27e2b in-S0_20090214.19
d7f95135f006725149494b13154a0290 in-S0_20090214.20
1239c1468163fe8753c218dca04364a1 in-S0_20090214.21
357cb41c32ae395e25d722dab67b4859 in-S0_20090214.22
7079fb8af89912e74f6f85d1350cfddf in-S0_20090214.23
1bb69f6b9eedc88f8bb70a7e79c713da iw-S0_20090212.00
58f84ee3545461519dc010a8d756033f iw-S0_20090212.01
81cb1bd50277331930e2492ceafd41b2 iw-S0_20090212.02
73ad0f3e86ffaa2fd2fea8e213512777 iw-S0_20090212.03
87d563a9dae0f3c1923b2b33daf7bd5c iw-S0_20090212.04
8b98a3e80cb53a1f3497ef332acdd6b1 iw-S0_20090212.05
3f55df39562fcfb37abf6e017a6724eb iw-S0_20090212.06
cc404842551db2295b1212fe014651c9 iw-S0_20090212.07
d44e26d95c7d0528b6ec0f4922d83882 iw-S0_20090212.08
2e8a9b7ec66bb539cb0f37bd7ae7c471 iw-S0_20090212.09
f0d05a7a997f8ea2983f4bf339f7b75c iw-S0_20090212.10
d6a2a46d53d14bc1f159bf43c78069b7 iw-S0_20090212.11
d3c44bfae726f09c919ae51fc1fbf79c iw-S0_20090212.12
4164de27ac2b62df9318b7efbe71f68e iw-S0_20090212.13
839a3281c6dc0a9b0bc83f3e7d916760 iw-S0_20090212.14
20b27e9f285574bff5cc590f60edda07 iw-S0_20090212.15
0e2fe90d905fac8f264f77636a5fb89e iw-S0_20090212.16
cc19586841365d7fe7f8630463f63e5b iw-S0_20090212.17
4eed64cecf1dce34c132855d507109c9 iw-S0_20090212.18
7d40fc41529da3da7551d0d85398e26e iw-S0_20090212.19
cd9eac258b0b56608ed4c5a440598c0e iw-S0_20090212.20
caf803409b982c3b356c0e7613b3ec0d iw-S0_20090212.21
355225c535d7d55d3d67b18c40023839 iw-S0_20090212.22
dabb88498abfeb538d2551312139fb7f iw-S0_20090212.23
8f412cbe1f3952e7df209c028834e4bc iw-S0_20090213.00
c98742c8bb641a3a54abaa7df541c51d iw-S0_20090213.01
5016dac31dbb9fe98159e2e31eb3d57a iw-S0_20090213.02
9a26a9dc7c99d7b26c20c3158bdd5e13 iw-S0_20090213.03
4df30d18b7ec254682c39d07b874f999 iw-S0_20090213.04
c2e145a63e4706866cd959dcf953362a iw-S0_20090213.05
0d873bd800ea7be2ac63c01a0706a917 iw-S0_20090213.06
06f22118768b7ec53c7a9dad51b7a68c iw-S0_20090213.07
15aa878b61df648b12fccb43c3e61bdc iw-S0_20090213.08
8a8110a050555b6625d833f64aee6257 iw-S0_20090213.09
0f15bfc18e16176b407e63a32dbb71d9 iw-S0_20090213.10
8f087951e474dc2af116c7f5febf0d23 iw-S0_20090213.11
e3bb8bac0a671b22930525c3bde176e0 iw-S0_20090213.12
05f5d9aeb47348629fdee0f814f90f69 iw-S0_20090213.13
69e5a57defe28ee957041c077a7e15be iw-S0_20090213.14
4bf8ed575a440c22beb97cdf0553a7f6 iw-S0_20090213.15
b8cb55933a712e2fd3af7c5df7263c6d iw-S0_20090213.16
9401a15cedd78fac15a25de8ead6a12e iw-S0_20090213.17
0227f81ce595d34dd578940f839c85c8 iw-S0_20090213.18
fa63bd1548ae7d5997393db64b63d2ed iw-S0_20090213.19
c4e06248454124e77db33adfc37ce9de iw-S0_20090213.20
cbfa51d8f20e8da4fb9ffa3af7874c28 iw-S0_20090213.21
b7da1fd9b22382856b2dff5502d282cf iw-S0_20090213.22
17125c721d80f4c7ab7d1500c3eda2e5 iw-S0_20090213.23
977daceaefa1056d66048fdcbfb5a9b1 iw-S0_20090214.00
5c710668dad923f4cede80540ddb1734 iw-S0_20090214.01
b50ec55b0358c4b3ba3075a24a7fb5b8 iw-S0_20090214.02
12a80b0cc881e6528b4db23155f267f1 iw-S0_20090214.03
49f42b9e57dd14b859164fe308105319 iw-S0_20090214.04
13d003fabf485bca827ec151a3243c67 iw-S0_20090214.05
3ca9848a51df351eb71b1ecff735ea43 iw-S0_20090214.06
3c76e37e571ac3b534492a70a0542a27 iw-S0_20090214.07
76602d2ab7c7cc754f13a97f8ee6ac27 iw-S0_20090214.08
edecb3665d0d716d6b20cdaabdff446b iw-S0_20090214.09
3f24e1d03e3d2152edf8f313a56831de iw-S0_20090214.10
48ad0d33d8199a14bca9876a9228349d iw-S0_20090214.11
96abcfe83430ba8960a6c9deabffb133 iw-S0_20090214.12
9c98849d728a491e228ec7b051065266 iw-S0_20090214.13
8636587e1211c772a048c7535d4d0fbf iw-S0_20090214.14
3bd08dfe8bae1764192f6bfcd6cb7ae4 iw-S0_20090214.15
d5239b9e9e340b741086a5bafc96595d iw-S0_20090214.16
01b221fe6684f02bc5d20da2ec970484 iw-S0_20090214.17
6707e1fbd0e13f4a4e758ffb2a444b99 iw-S0_20090214.18
b0fe3b773f8a1f809fd4b935b5089e6f iw-S0_20090214.19
df74fdc099d6e33698ceb4356865bda7 iw-S0_20090214.20
7c2a9a40074753dbb53ea579e8986c07 iw-S0_20090214.21
0ff872ea7deff6e21fb0419831050ba8 iw-S0_20090214.22
7b06da464ccebda33ddaf216e8c5d67d iw-S0_20090214.23
a4214b50846b4c425b30f80e54374e18 other-S0_20090212.00
139329e7dbbcdf722158f57c3e04bde1 other-S0_20090212.01
f7c2884bdb0d3eb23e85f27f207d204f other-S0_20090212.02
c3a6ea43ea76e1c6f1e83aaaae7e397b other-S0_20090212.03
42eb021f5a9375aafaf92502969184a9 other-S0_20090212.04
9dede8549832997ace686dff66e766e8 other-S0_20090212.05
f3a8dc8449b031515884015168e3dff6 other-S0_20090212.06
a0ca8ee456e38488225c9fd2a01c309f other-S0_20090212.07
ca4e60af1030ba27c3fa50499aa4dc80 other-S0_20090212.08
f0976dc98214e703519f98ef04c0897a other-S0_20090212.09
e17d293dc59983a6085f47fdbb993bb1 other-S0_20090212.10
00966ae3a06247a669265b8ae5c485ab other-S0_20090212.11
2125615b6c8d0de5bd6f0663ace81072 other-S0_20090212.12
954fa0fa76631701988ab36e4100d3b6 other-S0_20090212.13
378413272ddaf794d23e5a0e69a1cf9d other-S0_20090212.14
c68c7ab01a843e3473eab17661e409cb other-S0_20090212.15
d65c4841ca9fe0c1fcc8f18794afb9a6 other-S0_20090212.16
c98cfc89aff2445ff2c9207d7736dcba other-S0_20090212.17
1c60d48ec4dd48222a60a973a782a6e6 other-S0_20090212.18
301f225133ca71aca56c85c1b18ef505 other-S0_20090212.19
ceb04950a2cb0e2e3db0549e14ef3153 other-S0_20090212.20
444f4a9d8f7768dc2595990f2ef2c8c8 other-S0_20090212.21
222a0e798bf0d2ca7e68df73fac39bcc other-S0_20090212.22
24b07783a3c84e4ac3c6eed9a198ee0c other-S0_20090212.23
f46807f6f82c93a2ddc37ad131510ad1 other-S0_20090213.00
a884aae0ab1f1e6dd31015a6538a827f other-S0_20090213.01
30dd520b08d2bad79bac0c4fabe3cf3e other-S0_20090213.02
0d81fee99323640ee88ff240102d179a other-S0_20090213.03
d13db5b2f5a068eb852a652707f6d005 other-S0_20090213.04
5b252f10fc53b7178d5796714bc84994 other-S0_20090213.05
37b773bdce2ccc141dc38f4fc46f35f5 other-S0_20090213.06
015090252cd14ac6d95bc3183000749e other-S0_20090213.07
80972bf3cfc3bb907285a578a3ae9428 other-S0_20090213.08
b2950bae4c7922a4ca9b1a0f18a01ef7 other-S0_20090213.09
d2f3cdd685dc110e983b32a1d30caac9 other-S0_20090213.10
5fddd47afb2c88e0e64d13036002d74d other-S0_20090213.11
6982781bcd5f3124099fb3ec3933038f other-S0_20090213.12
fb2c3839c8f1caa8bfe6a914254deeb7 other-S0_20090213.13
ec93b024ba16d6ecbc882b97da713807 other-S0_20090213.14
9d22174ccef39c4f6a90e2abc22f74d5 other-S0_20090213.15
a068ba1fd4e4ddc0370132614df87101 other-S0_20090213.16
93e0fd5f9b6455836d42054bc9bbf056 other-S0_20090213.17
47412eb45c538a01794c6ff1d8ec4b81 other-S0_20090213.18
58904923a0cbd2da0e2fbc9ffcbd3005 other-S0_20090213.19
f71f7658298901f5605781c04da9b731 other-S0_20090213.20
8076276db188853867f788599b9732fb other-S0_20090213.21
d2872542aca30fd4d2692f1526818304 other-S0_20090213.22
895e14ee7b6ede4a3c41a26fffd7b918 other-S0_20090213.23
ff6c6524c03b7a193d242ac0317477a6 other-S0_20090214.00
4baa96ef6cc13c6d6ea1b6cade3b6b1c other-S0_20090214.01
53325d2e220626179b8f5d9c93756e03 other-S0_20090214.02
89122e9b3a37330ffcfe66f88784d059 other-S0_20090214.03
27ab625c30f11bf838aca5a497b7980b other-S0_20090214.04
eb3e54947a1af4a703b4781b3bfddfe1 other-S0_20090214.05
ab8825a2575d38a8148339d00f549593 other-S0_20090214.06
aa2da99b65a76155820776b2927a0e8f other-S0_20090214.07
1f4d4f7b8c016cbe0697043e656ab3d6 other-S0_20090214.08
eb1b848230537e1965b16bc0a10f8543 other-S0_20090214.09
51d01a6194ed755656e9c67d7db85ae8 other-S0_20090214.10
4531ef50802f90037aaf7b4be957ff56 other-S0_20090214.11
a22472b11ba613f25e14c045cb4d0fc1 other-S0_20090214.12
5ac508ce6d7c71745dc59cb1b62f52cd other-S0_20090214.13
39b4d41490ed3f0a89992dd7cbbba2b8 other-S0_20090214.14
5f8c09d1acdbf38f674eb15a11cadb8f other-S0_20090214.15
f2895ee3ad8836d15a66dc55d05633e7 other-S0_20090214.16
4c861564bc8c57d11205ac0bed144837 other-S0_20090214.17
b9c25ff8b8cc048dec2e331f68476587 other-S0_20090214.18
3ed4b57b2db45ad7014ff99ba67ac191 other-S0_20090214.19
fa43c168bfe9805946c57cd46268acb9 other-S0_20090214.20
9af3e8556139049b08874c863cf4ccb7 other-S0_20090214.21
dd824d08b6c28d8084e5b47ebe43669f other-S0_20090214.22
8a2bd6133722e9ac01a924f39740e642 other-S0_20090214.23
38b4b361a41b36c099942c2890aece30 out-S0_20090212.00
5531264e6cd7080c2fa931ad14e1a18c out-S0_20090212.01
53a21a3c3cf86d051b647b478bc8a292 out-S0_20090212.02
b4d85d2a7a778d20be863f02b5f53f23 out-S0_20090212.03
13e67a308b520b2a91f54743caa0ff4f out-S0_20090212.04
551dfa47a45413a37c71d7b14f20ace1 out-S0_20090212.05
7d133f7fc9a9ef9992cf4d72c1dc524e out-S0_20090212.06
f76ce676a952de16aae31c0dbddfe581 out-S0_20090212.07
ebbf71f3745e08a75648c65e28d715ab out-S0_20090212.08
cf8768c9c2ea9a2bbf74bdec781d8e2f out-S0_20090212.09
283d1c90e40de6d9fa03096c976ba877 out-S0_20090212.10
a346caa1feb7e482d59f915af5c2eed6 out-S0_20090212.11
720f7e482448cb1e80a4f24e184215df out-S0_20090212.12
0786f5fe84b87eb3fc21e41aa303b33d out-S0_20090212.13
22a814de4263a7afbbde3cb215a0e73b out-S0_20090212.14
472f99ba3bda878aff31bfd79cbe38e1 out-S0_20090212.15
aa75eb5877efddeffd41b6fd1ebe6abd out-S0_20090212.16
1782fa8ac547c16313aa2a7eec9d6fc9 out-S0_20090212.17
6ecb29fde2a636b5c7f9980bfea26055 out-S0_20090212.18
94897e233d82f6d1e568d78353bff810 out-S0_20090212.19
8952ef5d9bdb00eba53986f6f833a6e6 out-S0_20090212.20
e1a6633c3289c41c7f4037e339fb934e out-S0_20090212.21
1d5610eb4a04678fc6bd349c01d2b83c out-S0_20090212.22
7893a7231dcb606de4fceedda2a787f8 out-S0_20090212.23
126bccc96550ec87b103e3ded8af1541 out-S0_20090213.00
33f2d92e13bcb388d91a9a983ddee1c7 out-S0_20090213.01
e7aa165d6c9f1b644fe4063217609d1a out-S0_20090213.02
958d6cf8ddef48f3bb3b7cd3eae4b133 out-S0_20090213.03
abcaa997f77db5a81320d9ebd0bd855b out-S0_20090213.04
3d5a9a3e9f32f14e0961497bc4e3f713 out-S0_20090213.05
b25ff66e1238c25c352d3ce87e4aea98 out-S0_20090213.06
95491042bd13654f06e956475a0c0f47 out-S0_20090213.07
1e5dc52e5a48b4e81973d9164a61d663 out-S0_20090213.08
bfad275cb3ae485a0125f745f45adf0c out-S0_20090213.09
eac13563aaaf0d4eb133cd019f498805 out-S0_20090213.10
2bc39b05e8ea89950dcc808b498657fa out-S0_20090213.11
1f692166a4f569c15628102fe348879a out-S0_20090213.12
a4fb1ef604a0014f004f934f0f1d0ec7 out-S0_20090213.13
5e33b3ce73af126ef09ac974a31e8631 out-S0_20090213.14
49effeb60ba1389c4e26bddd772e9b91 out-S0_20090213.15
ce571cdc971351ba68feae1aa2d214d4 out-S0_20090213.16
82add68ae7f95596bfc2df6a3d63e9c7 out-S0_20090213.17
d587367d38cee70fe639ab5ee3592642 out-S0_20090213.18
e9403ce7a6f8deca1439ff11309280a9 out-S0_20090213.19
e6a0f9fab68785ba34e4cbf7db490732 out-S0_20090213.20
22cd35aa337901813f3593f0e6980cc0 out-S0_20090213.21
a74fe1d6a606b18115ffd5ad89e4e337 out-S0_20090213.22
02d18caf7b4eb7533d83239c5386c763 out-S0_20090213.23
84efc560e368391b41b9e346d2ebb236 out-S0_20090214.00
fc664beb2da2657d2b646b8f0a048871 out-S0_20090214.01
6ba57d57970fd4dc4cff5d8569319307 out-S0_20090214.02
8fe3edce8ca285445cdbff1ead1ec286 out-S0_20090214.03
4f6db403de16cc1927c871d1557f43b4 out-S0_20090214.04
f264d0e9d008207f7d84f43c669c9679 out-S0_20090214.05
bb0abdf6c4a7fa0a333879cfeba70e40 out-S0_20090214.06
18a924410cd8c7e1a3e5f68af0cd517e out-S0_20090214.07
01fbc9c0d4649d78056389d4ff461744 out-S0_20090214.08
4c1511286097c0afbe6ca33360dcd3c4 out-S0_20090214.09
f29f75ff1c271d969bc17e6e332a615c out-S0_20090214.10
580e4d0a34586779a8250f8b7bf09f38 out-S0_20090214.11
8206e0b50e23f46f7052676d60a523cc out-S0_20090214.12
3ac08955af12311b54c831b54e2e41fa out-S0_20090214.13
390873aaa3dc44fe3d29c1164ee51e78 out-S0_20090214.14
e8a42ae77a9bdacfabdfd793a25098cf out-S0_20090214.15
2e553dd3d7701e7bd879052c338ebe55 out-S0_20090214.16
d372eb046b5ccf9c4da2d690dde026c8 out-S0_20090214.17
5af69df9297fb4e97725150c5517f890 out-S0_20090214.18
85445cda501281ba42b461f59b60bfba out-S0_20090214.19
ab83a3468bc4bbf3c7a3ad71ecafafb5 out-S0_20090214.20
69286b615c3cacff375a2f72a1240367 out-S0_20090214.21
cec5a4372aae2da78c9135d548506819 out-S0_20090214.22
91dca756aa95cd43c913d898bf0a394e out-S0_20090214.23
9b3e883113ed1a2ea5ab5273bff42430 ow-S0_20090212.00
6ce42eefe725e0af1b73e86fe9578882 ow-S0_20090212.01
ab29836b33f9e9b1f865f72ce4d35de8 ow-S0_20090212.02
daee83e9ef13578f620000e3049f1d7b ow-S0_20090212.03
a9467801402c0002719c9635a6627a07 ow-S0_20090212.04
c7686ca5aa846073a1bf410176e38158 ow-S0_20090212.05
015c71459a07e9d2bf6494d0ebd71ec0 ow-S0_20090212.06
1c94fedf52fe4e31ccbce3e9645cf309 ow-S0_20090212.07
19029a7194ecad3abe8dd430ce90d92e ow-S0_20090212.08
3546c35a096fdd73e9c25e7d79299cfc ow-S0_20090212.09
f03c2b7a183a1ee78666741d5164337c ow-S0_20090212.10
10e512a3c9b8acc48fa6a3617fd34612 ow-S0_20090212.11
313eb47ba620c174ea8ac640495514a5 ow-S0_20090212.12
ae9a37a35d934d6e76fceb42d09f7f7c ow-S0_20090212.13
32c8d949316d99d557ea37dc2d1ab23b ow-S0_20090212.14
51b038f9c82a05f82084d8c27502ad5b ow-S0_20090212.15
3ada1e67c9a88a30ac22219f798ab48b ow-S0_20090212.16
e273efca152e0b19b37bde55cb096c67 ow-S0_20090212.17
07b261bb9bf31524f0db6e2cfddead9c ow-S0_20090212.18
29f12e162f3cab1507513f1c9c730c55 ow-S0_20090212.19
11c88b25703d68e5673aa37a07a3088e ow-S0_20090212.20
abbacfd05310a1c04a98ded064d09e04 ow-S0_20090212.21
36d2c2d327f8ae6098b3fd96130250ef ow-S0_20090212.22
407e34f251b073336876163029577b62 ow-S0_20090212.23
7e1218a91a4b5f20a4fc5bbe5dd80709 ow-S0_20090213.00
0afed1e0eb42e5fc58141c92aa54b1be ow-S0_20090213.01
8ba82162eb6319e42c628f2f8d958ca3 ow-S0_20090213.02
fc9492172703707ff8b5ce35144d478d ow-S0_20090213.03
ea767d26877143303d81f1a6021d131b ow-S0_20090213.04
c34e3d6a8ced1a1ce5ad17bdd314da4c ow-S0_20090213.05
924de890c216739865099c7e00f5c3a5 ow-S0_20090213.06
e20cfbcf0fe46cc464ada47250d3f29f ow-S0_20090213.07
0b975dab106a370e78307f8a52806e9e ow-S0_20090213.08
4584634c04584ddd9ec422355dfb994f ow-S0_20090213.09
3bf45871730614f0cdf2903ef2d929a7 ow-S0_20090213.10
2fd151450a8a0fadd9bde935f4b487d5 ow-S0_20090213.11
44d857209e3b4d024703b8884f419e37 ow-S0_20090213.12
00acd9aa9f8a9defc158b8be3f4808e0 ow-S0_20090213.13
b91e39944738a1b03732c0daedb527fd ow-S0_20090213.14
83707c2ce99098335a7beb6d73b6a5e8 ow-S0_20090213.15
4184ec4239ea6dc79cd670c5cd300601 ow-S0_20090213.16
5647033b8efd520056652f1bc0169d63 ow-S0_20090213.17
f96b2e2e8495d0765ade1b577ebe260b ow-S0_20090213.18
e5541799034c3b978d33eca8ccb3db93 ow-S0_20090213.19
4cad2f85da583ba8ed82e0f8695721a5 ow-S0_20090213.20
f1d963adef76564ff3e081a9a26eb765 ow-S0_20090213.21
596c5df7b66f883d7ea0767eb82c5398 ow-S0_20090213.22
456ca5f836b9947db64ae53f2ebbf64a ow-S0_20090213.23
63ad3957f8ce86a2124eea9653b345b1 ow-S0_20090214.00
4a9f5ac39314b8036067f49817b31a14 ow-S0_20090214.01
fba2ee06614f81e193520cf6b3e40a8c ow-S0_20090214.02
db5f780f30d8d2fc9634975da2579815 ow-S0_20090214.03
4d141e3e829f3ed06374b7e957f6c351 ow-S0_20090214.04
07df95645e5ebd598fb3b7c98b97b2a9 ow-S0_20090214.05
db3ce312e0e323dbca6df16bdcb548f9 ow-S0_20090214.06
513c62f7cab183920c0422c57a3e8a33 ow-S0_20090214.07
8cd43629e03d62fd8d022aa5b75d0a01 ow-S0_20090214.08
ccca01d212bfb0d0801ff75762cbf306 ow-S0_20090214.09
7b45f3b6ac9f5d4ad1deb5a1a0128683 ow-S0_20090214.10
2488ab5e4ccd16d6871e2181ba11e8e6 ow-S0_20090214.11
5b2fa827382a6079d8a251015c604a23 ow-S0_20090214.12
9914aadf9ab6b8c3e7ead160f7b414b7 ow-S0_20090214.13
54d3a65adaac6329588480b12853b3b6 ow-S0_20090214.14
20122a4077256fecbafdc79e28c31cbe ow-S0_20090214.15
313ab1d90af606db6dcdf6d750116baf ow-S0_20090214.16
6dff3b4cd3968d021b0c261be4cc1ac4 ow-S0_20090214.17
04e18084e7511fcab5d2a3eadcd71770 ow-S0_20090214.18
220627a806de1e4a693d44bc2a89b939 ow-S0_20090214.19
f54e228708f9941b9b386936e1388471 ow-S0_20090214.20
12b869c761d508fb011d13fe1141d186 ow-S0_20090214.21
d6bbd779810ec6132987c7ade68027bc ow-S0_20090214.22
21ce14fa98960505779e1c3d76184f69 ow-S0_20090214.23
