#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-send.pl 40a363507ed0 2014-04-01 14:09:52Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwuniq = check_silk_app('rwuniq');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'silk', 0, 'polldir');

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--output-mode=sending",
                     "--copy $file{data}:incoming",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--flush-timeout=5",
                     "--",
                     "--pack-interfaces",
                     "--polling-interval=5",
                     "--file-cache-size=8",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);


# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental root));

# In sending mode, files are named with a mkstemp()-type extension.
# We must find all files having the same name minus this extension,
# and combine them before checking the output.

# path to the data directory
my $sender_dir = "$tmpdir/sender";
die "ERROR: Missing data directory '$sender_dir'\n"
    unless -d $sender_dir;

# number of files representing a unique {flowtype,sensor,hour} tuple
# to find; i.e., where we have mapped out-S0_20090212.00.LELcDc and
# out-S0_20090212.00.PzEA5g into one entry
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every {flowtype,sensor,hour} we expect to find.
# Read these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

my %file_list = ();

# find the files in the data directory
File::Find::find({wanted => \&find_files, no_chdir => 1}, $sender_dir);

# combine the files and check the output
while (my ($basename, $files) = each %file_list) {
    $cmd = ("$rwuniq --fields=1-5,type,in,out"
            ." --values=records,packets,sTime-Earliest,eTime-Latest"
            ." --ipv6-policy=ignore --sort-output"
            .$files);
    check_md5_output($md5_map{$basename}, $cmd);
    ++$file_count;
}

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files types; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub find_files
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename, stripping the mkstemp
    # extension
    s,^.*/,,;
    s,([^.]+\.\d\d)\..+$,$1,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};

    $file_list{$_} .= " $path";
}


__DATA__
77f29ff118ec8f8d9854720aad1b5daf  in-S0_20090212.00
6d3cb955aec81e58a517f55b18e39b39  in-S0_20090212.01
30fe140b75b349dc60108a15b0f09089  in-S0_20090212.02
e28a14cb50bc5105fef43d72db564748  in-S0_20090212.03
758e6eba47178e0d663d87666d55ed7c  in-S0_20090212.04
e471c4b70280b88898ae222e2776073d  in-S0_20090212.05
ad0e090687a78c0e1639da01bee6559f  in-S0_20090212.06
f6595994e94b9ba804b0052fcb071d11  in-S0_20090212.07
e94c6132cdfd8f8e4ccb9afe938a34f9  in-S0_20090212.08
3a82ad1281818a43d5bbfdbc8bca27f8  in-S0_20090212.09
9651d0c506bc6e80aca888695d890d2b  in-S0_20090212.10
4105ee1cf6b3d27899507c42639d9110  in-S0_20090212.11
3e3d9d547d3e9423991db159f6c737e2  in-S0_20090212.12
b083113d6492ea0b2b161a4a77ab7128  in-S0_20090212.13
6f9a9a7d6f5901856bc479b6885695cf  in-S0_20090212.14
fa31379238c26ef9e1c68e4c13e05bf5  in-S0_20090212.15
dcd810086cbd7f91cb910f2dca64ad54  in-S0_20090212.16
2de393639f0aa77fefcaa6ffdc36a111  in-S0_20090212.17
00f87d5bad430a19e71ff1d166b64ad8  in-S0_20090212.18
e029a30cac3f61836caea418835b4683  in-S0_20090212.19
50cdebfffb888c3d41e0d3237ee97b9e  in-S0_20090212.20
a8637137925ab10def285aa23f4d55ff  in-S0_20090212.21
54327825923dc3b4f6d4167aa8c14625  in-S0_20090212.22
c5f4f2137a9e78c1451e11ad3a9e8f94  in-S0_20090212.23
e0a1613e402585ecd1b631b95ff647bb  in-S0_20090213.00
0450bd7fc62a0eda7aef08187a54f3ce  in-S0_20090213.01
236f8eb9d2347790d594e113de25a43e  in-S0_20090213.02
8c824fbf3ad12a7cfdbec037bdb96e7c  in-S0_20090213.03
e5287bd2c7198903a3806a0e7413cf5f  in-S0_20090213.04
0f4a16d745e5e44e03a5afcc401a1ad8  in-S0_20090213.05
6d557cef4f14df43518300674ef817f0  in-S0_20090213.06
57e7a502bd456324d79962edde8bec34  in-S0_20090213.07
8eb585550fd3abb6364515072ee96830  in-S0_20090213.08
b853cdcea1ee93a86b9ebb4b2f2356bf  in-S0_20090213.09
474a10aebe3ce4866cc38b5de6383189  in-S0_20090213.10
60d7969cadb20b871ef44878fa4053ca  in-S0_20090213.11
763782f13e2c3129878278091979d225  in-S0_20090213.12
89d4d9668d183669b46040f046ab5f25  in-S0_20090213.13
7c34ecdafbfff97b039674e770c50229  in-S0_20090213.14
f2158ed69384057a4895f74c8ebfae3b  in-S0_20090213.15
1800716a23083210d34e16c94162739f  in-S0_20090213.16
da0bdf85b18f6ce1ab82a0e96861280b  in-S0_20090213.17
321f44dc69fd6991408e35fc7731fe29  in-S0_20090213.18
fbc45dd1516208ea0a03846efc66f280  in-S0_20090213.19
a288a8e4438fb4ca1f45b7f24721c4c9  in-S0_20090213.20
35283133191f723784d272b91afb9db0  in-S0_20090213.21
9a138efa5a012330934453093221f8a2  in-S0_20090213.22
8d464cb3dc12857c5ed3d14a20a8481e  in-S0_20090213.23
a9d9bc9af15c699176dffb00575f2170  in-S0_20090214.00
f3ad3b47a2b31406781a55f566af9af1  in-S0_20090214.01
c1be6f756657ac104c408e3303eb3e11  in-S0_20090214.02
f941437ced4d3e9b12f392b2981cce8b  in-S0_20090214.03
3890f79f06e9a0e33226e4906e568798  in-S0_20090214.04
ba335d09f4c2b521f2260dff998a359a  in-S0_20090214.05
22749dd93ee0eb34603df79334016623  in-S0_20090214.06
a17a501750f5fe12e5808789d36302f8  in-S0_20090214.07
6d98fa4a19e823ab0d58c10cbc11a376  in-S0_20090214.08
c96ac8de3410215118c6a253deff4745  in-S0_20090214.09
3a49909d13f6fa9a1de767145dbfd993  in-S0_20090214.10
28ac0b60a602aef649881fa763df0699  in-S0_20090214.11
7e7d462719c2f66632bac257dac8596b  in-S0_20090214.12
78a9f1548e3e68144ce80bea7e3ec2b3  in-S0_20090214.13
23a97fd944f3a76419f106097d8ec181  in-S0_20090214.14
90f6b518dd33cbeb91fae80dd10922b7  in-S0_20090214.15
1e349cec6a65fa6fe18d5be5dfa0d034  in-S0_20090214.16
79f94e633af33d40619cb40270b25fc2  in-S0_20090214.17
99b2301062258820b06207158323458b  in-S0_20090214.18
f91d9637705b4eb12197f6a09dc6450b  in-S0_20090214.19
6871518a672bf8e3b71851907b35dab7  in-S0_20090214.20
ea5484d7851443cc913d523d5d2c99bc  in-S0_20090214.21
11961c234485ea15269e39c742033ea5  in-S0_20090214.22
00f7f8d46735d834ba615c928f5d31d5  in-S0_20090214.23
30c37e221bc059b08f381d430e620ad1  iw-S0_20090212.00
b7626bd8ec89159ba880457f0cdca93c  iw-S0_20090212.01
47d96d9b641c71c1c101ac3e33cf11ca  iw-S0_20090212.02
c8cee2d278ce67547439be7889a7d0d9  iw-S0_20090212.03
7c9cddad1d9aada59f8f309fadef37a2  iw-S0_20090212.04
8d176f3de55364d3fd46c38f90179449  iw-S0_20090212.05
ff20b6cedd0a88976d8e1a3f2de79baf  iw-S0_20090212.06
207959c0e4f73308b2a1057bc77d6e31  iw-S0_20090212.07
c5a8050892b07c3d64a009afde67f0ea  iw-S0_20090212.08
a7b98a95a5f123e7ea1bfe0d4beb8bbf  iw-S0_20090212.09
bd0707b207f52bed95eb0f191c838424  iw-S0_20090212.10
8971bc8cdd6c51bddfb5b5999a261dcc  iw-S0_20090212.11
d1dc5ce3e363c81a093a26b0401081ef  iw-S0_20090212.12
f81e4ee54e84aaf2082d8d6255900b4d  iw-S0_20090212.13
3fa3470afd70ca96b0aa6d41a22adaa5  iw-S0_20090212.14
755e446c31b606f3f52c5026890cdba4  iw-S0_20090212.15
5fe95145dc560e91f9b6c6dc414ff9ad  iw-S0_20090212.16
bf9225a6f83914c41453d9613cc28b43  iw-S0_20090212.17
0ea0cf9b0002dd7a73f3df1890870398  iw-S0_20090212.18
e70c8260321ba89fc3e028afc14598c9  iw-S0_20090212.19
0d174aed45051a942bf0fc0086504631  iw-S0_20090212.20
c2b9001361f79564e4f336b8a7d2aefd  iw-S0_20090212.21
3e028b1c170fbcd8768d1883a363d95f  iw-S0_20090212.22
fc6fb30f684c818df029b1147af2e8a5  iw-S0_20090212.23
c2d637097f0fb09b967b925271492d6f  iw-S0_20090213.00
d4fcb30041436c3420ca550e4f5c89df  iw-S0_20090213.01
8422393ce60a42362598223af77755d9  iw-S0_20090213.02
339652d5523d4411e2ffc2e7f7d2d65f  iw-S0_20090213.03
54810293bb4478c24199073edf6560a8  iw-S0_20090213.04
98f95ef0f0879c97abfde6099741876d  iw-S0_20090213.05
30491ef7a14451d6a41b5339452806be  iw-S0_20090213.06
9e8647b8ea7297215130d703b8d4eee9  iw-S0_20090213.07
12de42cab40083eda97247683047a71a  iw-S0_20090213.08
86e8514dbfe139cb835374edc80350db  iw-S0_20090213.09
2f917cc5cfc7053a71e55a2b4fd635fd  iw-S0_20090213.10
5642b831cff267617e8a670ed312c02b  iw-S0_20090213.11
4fd1717cab32737ee6da220b46db82d3  iw-S0_20090213.12
aaa04a84dc86511cc489b14c92bc4301  iw-S0_20090213.13
ebaa1efbbc5e981cfe9e57580a67ab86  iw-S0_20090213.14
5703b92975d021f8fc218531847ecad4  iw-S0_20090213.15
d246f1791f01743992740eae2f44703c  iw-S0_20090213.16
047de40843d6b296952dbb327c6f6b05  iw-S0_20090213.17
aa81a049c1a639cafc080de9ab7383c9  iw-S0_20090213.18
2b8ed8f39d7b05934481b9b97dc7a949  iw-S0_20090213.19
a9798777e2273965b6572652616429e7  iw-S0_20090213.20
3b8f2e45bac1525e0e785b8d1261e862  iw-S0_20090213.21
b672f1f6f431b2718e3d2247477ffcb8  iw-S0_20090213.22
d61c0e5dd38e9eaec88841f2ad87d205  iw-S0_20090213.23
2da813ecfee7a2eae4a356d14df712bc  iw-S0_20090214.00
f9c916b912399435a97a46c6fa0b048a  iw-S0_20090214.01
1568e533147da68e84b3ea64f23be4a8  iw-S0_20090214.02
fe85bb6fbc1e9a9a831d59bd904dbc69  iw-S0_20090214.03
f640090a61eb33393d7e2ea2202aaf48  iw-S0_20090214.04
3463f34cb03369db10467d2170fe75d6  iw-S0_20090214.05
ee26ca573c50d171e09906a1bba50047  iw-S0_20090214.06
a550b21d335ad382361616746b409aae  iw-S0_20090214.07
ccc1694c9ddf5f9faeeeaf8a6a994cb9  iw-S0_20090214.08
d7e76c712e1091a3769b7545768d7667  iw-S0_20090214.09
7bdb6861b3a983deab8542cb99febbf8  iw-S0_20090214.10
3724778dc228a3a1f9833e5cbdc762c1  iw-S0_20090214.11
ee10f74d0f2afa25e2b51ceab9e2b38c  iw-S0_20090214.12
bd131fc0226e694d23040780c71f08b0  iw-S0_20090214.13
e319743b5835f200b31192e3863896b4  iw-S0_20090214.14
bda48057483ab821033dcb8d15592f7a  iw-S0_20090214.15
d94c8a5fe25ac87f6c218d9ac58c5eab  iw-S0_20090214.16
897394f9b7079d674c181356c6985988  iw-S0_20090214.17
de08d51a1c82d29dc17a9b71e2979466  iw-S0_20090214.18
e6b5fcfff87533842a3f4ef0f9a43802  iw-S0_20090214.19
1d45fd27964b6f076eb8fccaf9044aca  iw-S0_20090214.20
c67b62de9e11467b5b11998655557a41  iw-S0_20090214.21
a8c40fd68529ed5bca5f6cfd6aabfc55  iw-S0_20090214.22
48417f7c827da89f6aa6548f12594466  iw-S0_20090214.23
f56fe3e2e6b199d5da39adcbb9017532  other-S0_20090212.00
f968f8389f1a1156d69e2709e771ecaf  other-S0_20090212.01
f38befa04e57d7c24f8833da80b0d680  other-S0_20090212.02
667fb61ce623a3b33c73e9bfcc517919  other-S0_20090212.03
47974de3b2a494598b0f87608d65856a  other-S0_20090212.04
12dcfc9b4641c32482916703db862899  other-S0_20090212.05
2c9b38cdeab5b5d078fff888b00a8d54  other-S0_20090212.06
d6f1a6eeafcca90e061c0ad266228524  other-S0_20090212.07
1e3c5edc5b19e4a98e298fd4afdd787d  other-S0_20090212.08
87c67f76d41b4f97529b8f08a12769c4  other-S0_20090212.09
6467f66489c116d85b63a96b2d395fd7  other-S0_20090212.10
a5a9e42bdee8f6e463e4e0f43d8e4657  other-S0_20090212.11
a1d9a6c64bfbc9a1eb0c933d609e1ef4  other-S0_20090212.12
dcbf6677f9c3c4e44722c2c21f7fd535  other-S0_20090212.13
72fe4173f9d7443edbdab85803bb4627  other-S0_20090212.14
ef6d54ae6a3c2763d6d3e1ae7cb73053  other-S0_20090212.15
675b7590d59ed03065a1697de6b4437c  other-S0_20090212.16
861d6831999c1f4f51758281895003bf  other-S0_20090212.17
0cb6b54714fc2e0b6f44b229a3e28283  other-S0_20090212.18
4e0b9d53b2e85bcfc4bfcc10ce33cff2  other-S0_20090212.19
a78c2b9886fff12fa43af34b8b424f77  other-S0_20090212.20
812904d99090b6c145c904b82453b98e  other-S0_20090212.21
a1aa90c87896adb00ec7b7b75bf33dc9  other-S0_20090212.22
925087f241c9e6aec5e5b5307d57d52a  other-S0_20090212.23
238cf0be53ac1b15c6e79fd2c9af9e7a  other-S0_20090213.00
a8e31a07430485e9ab56991ac6eb754d  other-S0_20090213.01
76ec54b662b4007b1b0995043e43c082  other-S0_20090213.02
b75bc9cbe60968464eeda56bb0e8030a  other-S0_20090213.03
27961637a4742a0b65fbbfb6753b269e  other-S0_20090213.04
0f01c8891ee2f26a5a74b9060bcf3526  other-S0_20090213.05
df4369cdd51ff81d964a5274893b45b8  other-S0_20090213.06
9dbb3c0491ac452f0fcec61d60641337  other-S0_20090213.07
e93a565b6dff02e847decd4eb63c883f  other-S0_20090213.08
33b4f1f8ce175959f1ea9c6b52e02b7b  other-S0_20090213.09
f57ff177cac97e35430f48bfb340b9a0  other-S0_20090213.10
fe94ec57a41a838e74a27af5fec1a55c  other-S0_20090213.11
535f9e099e0337bbcbf79cee3ba0cb34  other-S0_20090213.12
eca5d954cfa56f5931d59f78bc884f5b  other-S0_20090213.13
1db20b8a010b418ad5c825697812b4df  other-S0_20090213.14
17a48bf2e56d648747e7973d584109e8  other-S0_20090213.15
277ba6109da427987836d640d6375e4e  other-S0_20090213.16
30dbfd71986159bbf1a758a2ef0584cc  other-S0_20090213.17
7c9dc506b6ca15ffafeae9daf82535a5  other-S0_20090213.18
2191f24be75533ea61ad3e4fe716b3f8  other-S0_20090213.19
6de50bedf3779f83a1231d698e65b025  other-S0_20090213.20
eb492830d114b6a704dd90c4aa5fb618  other-S0_20090213.21
0187b323781282419c3c8306e5049f8a  other-S0_20090213.22
527c0480f40e184b9a073df00e03418a  other-S0_20090213.23
4728a656a3a9b76635e53c50d61c82bf  other-S0_20090214.00
f252db400be3f9bd02a009f02013bb23  other-S0_20090214.01
846da2632cec636170795f2d41553af1  other-S0_20090214.02
5b4d45d3e56d729aed17cfeab5decb2c  other-S0_20090214.03
5c201d47dedb46216604ea84966dd487  other-S0_20090214.04
d948c6d9210d039cf38d3a2bbce7274d  other-S0_20090214.05
006c23e1a53fe1e245dd3a6b3627f82f  other-S0_20090214.06
93d33fd6142892877eda09ff1ac8b9eb  other-S0_20090214.07
ade18c7a1b3a54091b9bea0b791a03da  other-S0_20090214.08
b02a617e16b3ca34cf4ab87b9c4294d7  other-S0_20090214.09
0e5478558046a3eca22f5668f253fd39  other-S0_20090214.10
299eea86a06e89bf99f63395c6dbc747  other-S0_20090214.11
d5aa3a5d9de20bd4b5221aae3a5da5f6  other-S0_20090214.12
87a487b613d8a3828037a6dcd84901fd  other-S0_20090214.13
768d1d7f88d8cf647585fbaa4d8296bb  other-S0_20090214.14
50616b16969740ab6b20e18afa89549a  other-S0_20090214.15
52f8831ef43fd3f7ed33c26e6796d0cb  other-S0_20090214.16
071989c0d99c48e3fd038e68e69f6f9e  other-S0_20090214.17
97b7a7388e6a2229ece42b5104654780  other-S0_20090214.18
d180e64415fc5fd1d4cfe067dfac8cb8  other-S0_20090214.19
0d15419b7a996b49705fbad7d7b67fe6  other-S0_20090214.20
2f9c10796f87cfb5f852429740ae1da3  other-S0_20090214.21
39383ed3fb3d303d0765f43061984b6c  other-S0_20090214.22
1e0b90ed2d612bfe0148acc52f14bb91  other-S0_20090214.23
c3149c11692cc939ff212a6708ce2c55  out-S0_20090212.00
e22bb11570e2264a38a55845a9c0de93  out-S0_20090212.01
b87a23a8e2c3f1e3a5bff37cf0a8bdf2  out-S0_20090212.02
34cea4decdf32b90460d47b5651b40bb  out-S0_20090212.03
7be2458209fe3c6ab3db80c4a33e044e  out-S0_20090212.04
8eb31497340a13469c751280afc966c4  out-S0_20090212.05
6227f71fdb39c0bd164b7ab6240754b9  out-S0_20090212.06
b057c5b2d2f5e0fd55b2da965e915406  out-S0_20090212.07
86d6d84b26440ab3783b9896039a9d49  out-S0_20090212.08
784f41a1b802e9fd8597c2c9883ba25e  out-S0_20090212.09
0cd43f9ed0d41edf09b2f4459bbe5263  out-S0_20090212.10
8cab0e8e1fefaedb2b9b98c1f79c4c3c  out-S0_20090212.11
485ea52c6f785aa8a4309880204e4c81  out-S0_20090212.12
08fbc05675f9dae71abd0578ce984cbb  out-S0_20090212.13
7085389ef23c336f817653c2090a3c4c  out-S0_20090212.14
c8656d1db1ccacaf322a37a4c30a8300  out-S0_20090212.15
55b64cf7f9ce00113cd1edfb47f39925  out-S0_20090212.16
624a5b8a60b3971bf05411a5f03c8a9d  out-S0_20090212.17
2fa53a56a08a85931a86c94570788b0f  out-S0_20090212.18
f67537eac5bdd4c9d329d482953eb23a  out-S0_20090212.19
03d89a918f6121a33cc7a7ff73ead242  out-S0_20090212.20
ee33bd6aadecd5f17590265b5eeca93e  out-S0_20090212.21
f0880deaddbfdf3b21d58eb16800d3ab  out-S0_20090212.22
838c34af99fc83c4a755dfae53e3294e  out-S0_20090212.23
da6aa31da4b5f524740da5792082f5e9  out-S0_20090213.00
4bcd970ca7033f02746c23a5fbf893a3  out-S0_20090213.01
6522331a57485d6a71c988f7e73832ef  out-S0_20090213.02
4e354a6f661f7e0de11ab4ccdd5dfab2  out-S0_20090213.03
e3682ef6911e876a0482ebb31136ed54  out-S0_20090213.04
0815e8a1f4b2a293804c4cbe6b944e97  out-S0_20090213.05
694cfb6f06a432a2db075a6f053bd4e1  out-S0_20090213.06
b2449321dfb99fc2e3c768852ad23b53  out-S0_20090213.07
e700bbf1e385c2247aba9289e932d8c9  out-S0_20090213.08
403baa5e6988bfd078109cec8e836357  out-S0_20090213.09
4dac2003d70f7d549305b6d2707c45b8  out-S0_20090213.10
fae8b034d87a5e5c911f373aa7da7a58  out-S0_20090213.11
fb2566846f6c2feb8d5ad9f448a87497  out-S0_20090213.12
9a5c4e89f7b095da4bc4610556a27a89  out-S0_20090213.13
46dd9b8fa3a7949f1cf32b67c8cc5071  out-S0_20090213.14
a3d3a8317378221e43f11d45d42c4ad1  out-S0_20090213.15
7ded609e9b6fdb6a211bcd4c2fabd829  out-S0_20090213.16
5eb44f5f5a347e0b38d77ff349c806fa  out-S0_20090213.17
82c33b0b96d561795ace3fe34f19d527  out-S0_20090213.18
7c45e83825af42ab50d67d5a3953d51f  out-S0_20090213.19
523d9cb5ff627f6bed9c347aebc0e8bd  out-S0_20090213.20
37b817e1915b7c47a3daa2e8e8eaff0d  out-S0_20090213.21
9b95f30b15ecddf54f8df75f8925e56f  out-S0_20090213.22
9c0122abcce77a465a19c1da642de696  out-S0_20090213.23
f79a81203b697a41343b55254824e93a  out-S0_20090214.00
0ede1ffce53ee32eb915d190b8b6903d  out-S0_20090214.01
1c8b3bd5f818685d48e4ec2772d4fbf5  out-S0_20090214.02
770f96f35c1149926cfa84deb52fff3a  out-S0_20090214.03
3fd64ba5f7f5f005531310886ba3ab1c  out-S0_20090214.04
c4b68a796eec471ed1ba584a9c9413b2  out-S0_20090214.05
4ab857a41a94234e9f1d847c7fcfde4e  out-S0_20090214.06
cb5dde5c9a55fafc0575de4b6d0a3b91  out-S0_20090214.07
0c91d83321835b81d56abd32311c1839  out-S0_20090214.08
dd2849e6f1d97672b08342448b2a9d43  out-S0_20090214.09
b4d4760c77626d7aa2f26a1821cc9409  out-S0_20090214.10
465ea62b3a8b7e3165e1c59628d92483  out-S0_20090214.11
f377be8ea32c8edc7ec611bf2088f1b4  out-S0_20090214.12
04eac0f411954861e697edf01fdeaef1  out-S0_20090214.13
a5bb8f1dac1daad03b3607df7c775789  out-S0_20090214.14
c2c2dec76fcd4f3134b7437b6fecbda6  out-S0_20090214.15
baa746a4ccfda7ac13065b30f873a1c3  out-S0_20090214.16
25c7f1bd2f43a465a7cce4d43460d847  out-S0_20090214.17
93193931a67f6c271bd6f6b5bad28cfa  out-S0_20090214.18
a7f42c92b070a30e3218c407942d9fb7  out-S0_20090214.19
fbf38e1cc2735ce95e4e2c198c12982a  out-S0_20090214.20
68b2701b266badefe6a4ab270075c7bb  out-S0_20090214.21
2540e454cdf4a7661a771754daa8cff7  out-S0_20090214.22
3140b93073575130bb33e5c4e2c96353  out-S0_20090214.23
fcc0358fedf6b76536b3e1fbf1c9659c  outnull-S0_20090212.00
b2aa94264201fdaca5928e464e10abfc  outnull-S0_20090212.01
6863cb5a3b5c68803471c34a714f1bf3  outnull-S0_20090212.02
83ca2b2a57831fdd8449024c195fe78b  outnull-S0_20090212.03
2cbc3f96ff39cc3d735b15f5a311ca7b  outnull-S0_20090212.04
9a6567a7b76596766c36e9edf7425fe3  outnull-S0_20090212.05
de3213456ad266a37cd9ca8d219b9f73  outnull-S0_20090212.06
fcc28a84ed3d4176080778f96815a625  outnull-S0_20090212.07
f294668970c57f2a1831f3f99180976e  outnull-S0_20090212.08
c91745ecc0f13396b62b4acc8845420e  outnull-S0_20090212.09
e862839ecc2b9749f6ba4fefd45df080  outnull-S0_20090212.10
7880b10278b01fa300fa9a63d2ca30cb  outnull-S0_20090212.11
d53e07baab53273879672e91a1f22aa6  outnull-S0_20090212.12
7fdcf68a07ea1918b213fc5c2857506c  outnull-S0_20090212.13
2e6bfe430bb4e5195918180a87fa6521  outnull-S0_20090212.14
031e1e3c409eab411a78af9cecce57ba  outnull-S0_20090212.15
80d120f43fa5d2951429f2fa27535c95  outnull-S0_20090212.16
de4f2ce7aa5ba0cbdc9126cf75f9d351  outnull-S0_20090212.17
30d069d5068fb3e32a14b3e0cbb34e06  outnull-S0_20090212.18
dccd410f347f952debdab1b8bc131068  outnull-S0_20090212.19
b3bc1bd4366e0af34770e1b3f22e4a9d  outnull-S0_20090212.20
646cdedfd0474dcd4c1d7663271fdc11  outnull-S0_20090212.21
c45a0d140fa3346145ce5e1fab5edc27  outnull-S0_20090212.22
761541aa20f0d9e09f5c75afc4127f0e  outnull-S0_20090212.23
366f038f9ac645e5fac204b047e71240  outnull-S0_20090213.00
cb400eb1822f42397465ad39e2ffb98b  outnull-S0_20090213.01
b556231bddd62a10d59b1afa4b80588f  outnull-S0_20090213.02
3035f7c1b209a3f481857aead6a5eeb3  outnull-S0_20090213.03
d12c8ae68b66a66f1e6a7428afd5add8  outnull-S0_20090213.04
c735a4fde406db32a5d4753f018282f5  outnull-S0_20090213.05
0328a61ce99215de47136030dbb9b02a  outnull-S0_20090213.06
edd4b06ecb241f691a0405c6fac3c562  outnull-S0_20090213.07
80160b05611682fefbc140dc9a2a3a8d  outnull-S0_20090213.08
3efa0c7b8ee78a7e3882036d9e4d398e  outnull-S0_20090213.09
6d31bdec7117cbac334b19f3dc5d270d  outnull-S0_20090213.10
25a5bcdc9bc6a083ca73d6ce13cee042  outnull-S0_20090213.11
f3376725035a5102e3d119f4ab61785e  outnull-S0_20090213.12
938eb99b8c18c40c01d2a6e263326226  outnull-S0_20090213.13
93eabd8c29299fd54472658a416edefc  outnull-S0_20090213.14
5d044726176fdd80a4d7ac1198e08a15  outnull-S0_20090213.15
919c310feb6cd836e81fe4d15177b9c4  outnull-S0_20090213.16
3a8431cbf53011f27e88e06dff138445  outnull-S0_20090213.17
483f56e7009a5f1840f42ca20ca78cc4  outnull-S0_20090213.18
beba7b17dddd958c456569d016124ff8  outnull-S0_20090213.19
424501347bd65861991ed48d41308e7a  outnull-S0_20090213.20
0e0cc98dc9644c7ba70085d0b4a9ae05  outnull-S0_20090213.21
0a89bb3a4894b8ca6104fb1b53cc9e07  outnull-S0_20090213.22
58be4ca4f4b671fdf1bbd600e4c2bcf6  outnull-S0_20090213.23
8389ef2b046c30effd4060c297bab678  outnull-S0_20090214.00
3909698c48635085bf43bc229b022fbd  outnull-S0_20090214.01
1f370f19e5d8912155807e574a1374d4  outnull-S0_20090214.02
5ff43e98cd6ecc7a5ce041130ae04ecf  outnull-S0_20090214.03
11d909a38d27ceec290ef0895fe9c0b8  outnull-S0_20090214.04
5e063f85782f5aafae2470b1bcdec8ee  outnull-S0_20090214.05
e4c047ef227fbb5443ae1952cd3fddd2  outnull-S0_20090214.06
2cf5ea6ad1ee422b201d180d8160e341  outnull-S0_20090214.07
0ed0f03310113bcf3a52d1de57217633  outnull-S0_20090214.08
c2faf7ad1587acd7f20b415ee4bb8d0f  outnull-S0_20090214.09
b1eaccdc2237ef8bc7836419b90d85ed  outnull-S0_20090214.10
e1180b23aac831a6c5eeb9928052d42c  outnull-S0_20090214.11
5e30a3129152d6392e9a022339388e5c  outnull-S0_20090214.12
623e3938a9c24f30bb8e5b782bea6e79  outnull-S0_20090214.13
2493fe3049e63aa7114c541b1a7033de  outnull-S0_20090214.14
f532a108abcc5efa11b5b1677819ff7f  outnull-S0_20090214.15
8ff9a41510e5adb6d3b72b5456db6495  outnull-S0_20090214.16
a4b62debb48775a66a10af584907754f  outnull-S0_20090214.17
b7cfa02ed2bd2bc2803191a2847e8cf0  outnull-S0_20090214.18
9cc2594fb361592f921e41b19c8442ca  outnull-S0_20090214.19
21606dba70b3126253c96e5ffcbaba32  outnull-S0_20090214.20
a30efb49c148778bf0f5dd513185d104  outnull-S0_20090214.21
9ea97f7e0927eda9cb46031387c8bddb  outnull-S0_20090214.22
ad99c8cad59b4cd7c7498f5176069ac4  outnull-S0_20090214.23
2a298daf79f12fbc398d697d4d7b17e3  ow-S0_20090212.00
8de1851e9345fbb77bb8d157d3716547  ow-S0_20090212.01
60b07338d2bce80c60ba20dbddf79157  ow-S0_20090212.02
b26a211fd0fa33dc6d3c6b27dcba051d  ow-S0_20090212.03
ca3fa7f20cd9c7f746e06acfd8c552cf  ow-S0_20090212.04
7b30238b5b3adeb9cfeab25d863f23df  ow-S0_20090212.05
2d7e68ce3d252065fe8e8ddd4702028b  ow-S0_20090212.06
c68e8179018e263984dc52c0304e69f6  ow-S0_20090212.07
f3602b85b4da206ebcd8dc0eeadb1b1a  ow-S0_20090212.08
e8f2009ce4a6d8a2d28006dd99eb400f  ow-S0_20090212.09
76175ff0a16b11e58cca09e776735fcc  ow-S0_20090212.10
4be82aaba3f38e91dc3d8ba7f691a6c0  ow-S0_20090212.11
9632689e4d9dd6747cd7b3cfacfbbcbb  ow-S0_20090212.12
f36655abbcb520d7e781b5a29f0c6778  ow-S0_20090212.13
ffed198a1c6dc337d39b992e43da3383  ow-S0_20090212.14
54b2fd72f1ac6a4df3baeb0e2b85a047  ow-S0_20090212.15
d4411fc476cc4c7fe804abb973dff91d  ow-S0_20090212.16
8f5f4e44824a483f400f8af23f1df18d  ow-S0_20090212.17
0a6f64b3080f45ab077d7edd551bf175  ow-S0_20090212.18
18a196e6cd8cd5b2711d025e0030c622  ow-S0_20090212.19
2c18983636f1921ab5acad9a6f560716  ow-S0_20090212.20
77937df6d25a28d2937107e5a3993f4e  ow-S0_20090212.21
de20912d50be7d9ebffde6e75374a39e  ow-S0_20090212.22
3b62659b655758af21b0d4f10726fbbf  ow-S0_20090212.23
7c2c2f17d7f040b4f11806399d10c3fa  ow-S0_20090213.00
db67cee28506ca84b4331dff53613bcf  ow-S0_20090213.01
4a2180f2e5261207837cf4b0740dde66  ow-S0_20090213.02
2457c85ba5468033a5c5bbc3db401394  ow-S0_20090213.03
8f8ddc73dc72e5182957caed6c03244e  ow-S0_20090213.04
9b7efef6f579c57e69eccce4c73dbbc3  ow-S0_20090213.05
75735450ea551e79b8967344ee9f9c47  ow-S0_20090213.06
3b518a490e6f8ec184d6c00a60ff02e9  ow-S0_20090213.07
4b8023b2372e6ea33bf7c02e75b112a5  ow-S0_20090213.08
bc8f88caa2ab8e86e41abf90b6850a02  ow-S0_20090213.09
2e58cec546c3820656cbba027c798e86  ow-S0_20090213.10
ee316626beda42b76516e43c1a6d4278  ow-S0_20090213.11
33777314081952815ceab7fd44f3dbb5  ow-S0_20090213.12
a6fb2d5efd8caa7ef02eba1392ee94cc  ow-S0_20090213.13
15feeb1e22f7d253702c8c922e8ba2aa  ow-S0_20090213.14
ef335f553782aed8387354d470e8b388  ow-S0_20090213.15
990c87ac0333b59a7a9cb286e7135396  ow-S0_20090213.16
72db300c6986fc6ac078168ab491a653  ow-S0_20090213.17
d4625142374250e8db905b55a4771782  ow-S0_20090213.18
ef7190c2e94de8bced1c59b96e51649d  ow-S0_20090213.19
85d2803fd1ae362ba06d88f7466da76c  ow-S0_20090213.20
ea8bcb8ec108ddf31bfff78cc125591c  ow-S0_20090213.21
d1caa84ede160177be59afdfc72e2612  ow-S0_20090213.22
fabaca1408aa6a4204294e17ae07e730  ow-S0_20090213.23
4c66b348f796e91081b47c99229aa7f1  ow-S0_20090214.00
a05c8a0eb17fbedbbe469792a63add76  ow-S0_20090214.01
2169e4edc3cb9abbe74ceeb41fd55fb0  ow-S0_20090214.02
4a14a1f0cfcfc60436f6b04d827efd7f  ow-S0_20090214.03
53754a2ad433f2b76f08181083c49d37  ow-S0_20090214.04
15a8b8acc50b5b4b8260f9da7b4b0478  ow-S0_20090214.05
d830a5548f8cf1065119fbb9e8e459ce  ow-S0_20090214.06
4f0c0803079a5d935b9dc5258f826080  ow-S0_20090214.07
065a3df392e62741a38781180dac0756  ow-S0_20090214.08
732b72e04015745e8281c7f5a39b8712  ow-S0_20090214.09
4ec2c209d98e6f2463a001a7aacea9f0  ow-S0_20090214.10
499de0ddd6be8310ea352f0a30999c1f  ow-S0_20090214.11
1ae6a78a1c479a5f1b48c15e459d8554  ow-S0_20090214.12
16f798118335e39d3b4083f638dab2dd  ow-S0_20090214.13
360550a8fbbb8e7de3344a1a71e624d4  ow-S0_20090214.14
257e5f9d6de28b343dd93bbd19bda76e  ow-S0_20090214.15
19b5cff38856af2f357fb038c9b4a94f  ow-S0_20090214.16
cf3f35af3df1a55236511fcd1497d950  ow-S0_20090214.17
e250f725aba42ed6d8bddf11b9313f38  ow-S0_20090214.18
f2edfe3299936eb5ddd0b877c398e66f  ow-S0_20090214.19
85efa6e66b830e60121ea9b5f46f681e  ow-S0_20090214.20
e18f67e265bd949d25883f1cab56ea40  ow-S0_20090214.21
3b0c78beb091caec48009b83096401ba  ow-S0_20090214.22
762d1f9abb58dcd87707cec16761439b  ow-S0_20090214.23
