/*
** Copyright (C) 2001-2025 Carnegie Mellon University
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.24
**
** Copyright 2025 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM25-0915
**
** @OPENSOURCE_LICENSE_END@
*/

/*
**  sku-times.c
**
**  various utility functions for dealing with time
**
**  Suresh L Konda
**  1/24/2002
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: sku-times.c c7d6bb438741 2025-01-17 20:52:04Z mthomas $");

#include <silk/utils.h>


/* declared in sktime.h; used to mark SKTIMESTAMP_NOMSEC as deprecated */
const int SKTIMESTAMP_NOMSEC = SKTIMESTAMP_NOFRAC;

/* Maximum sktime_t value */
const sktime_t SKTIME_MAX = { INT64_MAX };


/* Time to ASCII */
char *
sktimestamp_r(
    char               *outbuf,
    sktime_t            t,
    unsigned int        timestamp_flags)
{
    struct tm ts;
    struct tm *rv;
    int64_t t_frac;
    time_t t_sec;
    const int form_mask = (SKTIMESTAMP_EPOCH | SKTIMESTAMP_MMDDYYYY
                           | SKTIMESTAMP_ISO);

    switch (timestamp_flags & sktimestamp_flags_precision) {
      case SKTIMESTAMP_MILLI:
        sktimeGetSecondsMilli(t, &t_sec, &t_frac);
        break;
      case 0:
      case SKTIMESTAMP_MICRO:
        sktimeGetSecondsMicro(t, &t_sec, &t_frac);
        break;
      case SKTIMESTAMP_NANO:
        sktimeGetSecondsNano(t, &t_sec, &t_frac);
        break;
      case SKTIMESTAMP_NOFRAC:
      default:
        t_sec = sktimeGetSeconds(t);
        t_frac = 0;
        break;
    }

    if ((timestamp_flags & form_mask) == SKTIMESTAMP_EPOCH) {
        switch (timestamp_flags & sktimestamp_flags_precision) {
          case SKTIMESTAMP_MILLI:
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1, ("%" PRId64 ".%03" PRId64),
                     (int64_t)t_sec, t_frac);
            break;
          case 0:
          case SKTIMESTAMP_MICRO:
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1, ("%" PRId64 ".%06" PRId64),
                     (int64_t)t_sec, t_frac);
            break;
          case SKTIMESTAMP_NANO:
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1, ("%" PRId64 ".%09" PRId64),
                     (int64_t)t_sec, t_frac);
            break;
          case SKTIMESTAMP_NOFRAC:
          default:
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1, ("%" PRId64),
                     (int64_t)t_sec);
            break;
        }
        return outbuf;
    }

    switch (timestamp_flags & (SKTIMESTAMP_UTC | SKTIMESTAMP_LOCAL)) {
      case SKTIMESTAMP_UTC:
        /* force UTC */
        rv = gmtime_r(&t_sec, &ts);
        break;
      case SKTIMESTAMP_LOCAL:
        /* force localtime */
        rv = localtime_r(&t_sec, &ts);
        break;
      default:
        /* use default timezone */
#if  SK_ENABLE_LOCALTIME
        rv = localtime_r(&t_sec, &ts);
#else
        rv = gmtime_r(&t_sec, &ts);
#endif
        break;
    }
    if (NULL == rv) {
        memset(&ts, 0, sizeof(struct tm));
    }

    switch (timestamp_flags & (form_mask | sktimestamp_flags_precision)) {
      case (SKTIMESTAMP_MMDDYYYY | SKTIMESTAMP_MILLI):
        /* "MM/DD/YYYY HH:MM:SS.sss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%02d/%02d/%04d %02d:%02d:%02d.%03" PRId64),
                 ts.tm_mon + 1, ts.tm_mday, ts.tm_year + 1900,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case SKTIMESTAMP_MMDDYYYY:
      case (SKTIMESTAMP_MMDDYYYY | SKTIMESTAMP_MICRO):
        /* "MM/DD/YYYY HH:MM:SS.ssssss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%02d/%02d/%04d %02d:%02d:%02d.%06" PRId64),
                 ts.tm_mon + 1, ts.tm_mday, ts.tm_year + 1900,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case (SKTIMESTAMP_MMDDYYYY | SKTIMESTAMP_NANO):
        /* "MM/DD/YYYY HH:MM:SS.sssssssss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%02d/%02d/%04d %02d:%02d:%02d.%09" PRId64),
                 ts.tm_mon + 1, ts.tm_mday, ts.tm_year + 1900,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case (SKTIMESTAMP_MMDDYYYY | SKTIMESTAMP_NOFRAC):
        /* "MM/DD/YYYY HH:MM:SS" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1, "%02d/%02d/%04d %02d:%02d:%02d",
                 ts.tm_mon + 1, ts.tm_mday, ts.tm_year + 1900,
                 ts.tm_hour, ts.tm_min, ts.tm_sec);
        break;

      case (SKTIMESTAMP_ISO | SKTIMESTAMP_MILLI):
        /* "YYYY-MM-DD HH:MM:SS.sss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%04d-%02d-%02d %02d:%02d:%02d.%03" PRId64),
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case SKTIMESTAMP_ISO:
      case (SKTIMESTAMP_ISO | SKTIMESTAMP_MICRO):
        /* "YYYY-MM-DD HH:MM:SS.ssssss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%04d-%02d-%02d %02d:%02d:%02d.%06" PRId64),
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case (SKTIMESTAMP_ISO | SKTIMESTAMP_NANO):
        /* "YYYY-MM-DD HH:MM:SS.sssssssss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%04d-%02d-%02d %02d:%02d:%02d.%09" PRId64),
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case (SKTIMESTAMP_ISO | SKTIMESTAMP_NOFRAC):
        /* "YYYY-MM-DD HH:MM:SS" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1, "%04d-%02d-%02d %02d:%02d:%02d",
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec);
        break;

      case SKTIMESTAMP_MILLI:
        /* "YYYY/MM/DDTHH:MM:SS.sss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%04d/%02d/%02dT%02d:%02d:%02d.%03" PRId64),
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case 0:
      case SKTIMESTAMP_MICRO:
        /* "YYYY/MM/DDTHH:MM:SS.ssssss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%04d/%02d/%02dT%02d:%02d:%02d.%06" PRId64),
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case SKTIMESTAMP_NANO:
        /* "YYYY/MM/DDTHH:MM:SS.sssssssss" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                 ("%04d/%02d/%02dT%02d:%02d:%02d.%09" PRId64),
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec, t_frac);
        break;

      case SKTIMESTAMP_NOFRAC:
        /* "YYYY/MM/DDTHH:MM:SS" */
        snprintf(outbuf, SKTIMESTAMP_STRLEN-1, "%04d/%02d/%02dT%02d:%02d:%02d",
                 ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                 ts.tm_hour, ts.tm_min, ts.tm_sec);
        break;

      default:
        /* multiple formats and/or multiple fractional seconds flags were
         * given; suppress fractional seconds */
        switch (timestamp_flags & form_mask) {
          case SKTIMESTAMP_MMDDYYYY:
            /* "MM/DD/YYYY HH:MM:SS" */
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                     "%02d/%02d/%04d %02d:%02d:%02d",
                     ts.tm_mon + 1, ts.tm_mday, ts.tm_year + 1900,
                     ts.tm_hour, ts.tm_min, ts.tm_sec);
            break;
          case SKTIMESTAMP_ISO:
            /* "YYYY-MM-DD HH:MM:SS" */
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                     "%04d-%02d-%02d %02d:%02d:%02d",
                     ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                     ts.tm_hour, ts.tm_min, ts.tm_sec);
            break;
          default:
            /* "YYYY/MM/DDTHH:MM:SS" */
            snprintf(outbuf, SKTIMESTAMP_STRLEN-1,
                     "%04d/%02d/%02dT%02d:%02d:%02d",
                     ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday,
                     ts.tm_hour, ts.tm_min, ts.tm_sec);
            break;
        }
        break;
    }

    return outbuf;
}


char *
sktimestamp(
    sktime_t            t,
    unsigned int        timestamp_flags)
{
    static char t_buf[SKTIMESTAMP_STRLEN];
    return sktimestamp_r(t_buf, t, timestamp_flags);
}


int
sktimestamp_width(
    unsigned int        timestamp_flags)
{
    const int form_mask = (SKTIMESTAMP_EPOCH | SKTIMESTAMP_MMDDYYYY
                           | SKTIMESTAMP_ISO);

    switch (timestamp_flags & sktimestamp_flags_precision) {
      case SKTIMESTAMP_MILLI:
        return 4 + (((timestamp_flags & form_mask) == SKTIMESTAMP_EPOCH)
                    ? 10 : 19);
      case 0:
      case SKTIMESTAMP_MICRO:
        return 7 + (((timestamp_flags & form_mask) == SKTIMESTAMP_EPOCH)
                    ? 10 : 19);
      case SKTIMESTAMP_NANO:
        return 10 + (((timestamp_flags & form_mask) == SKTIMESTAMP_EPOCH)
                     ? 10 : 19);
      case SKTIMESTAMP_NOFRAC:
      default:
        return (((timestamp_flags & form_mask) == SKTIMESTAMP_EPOCH)
                ? 10 : 19);
    }
}


/*
 *  max_day = skGetMaxDayInMonth(year, month);
 *
 *    Return the maximum number of days in 'month' in the specified
 *    'year'.
 *
 *    NOTE:  Months are in the 1..12 range and NOT 0..11
 *
 */
int
skGetMaxDayInMonth(
    int                 yr,
    int                 mo)
{
    static int month_days[12] = {31,28,31,30,31,30,31,31,30,31,30,31};

    /* use a 0-based array */
    assert(1 <= mo && mo <= 12);

    /* if not February or not a potential leap year, return fixed
     * value from array */
    if ((mo != 2) || ((yr % 4) != 0)) {
        return month_days[mo-1];
    }
    /* else year is divisible by 4, need more tests */

    /* year not divisible by 100 is a leap year, and year divisible by
     * 400 is a leap year */
    if (((yr % 100) != 0) || ((yr % 400) == 0)) {
        return 1 + month_days[mo-1];
    }
    /* else year is divisible by 100 but not by 400; not a leap year */

    return month_days[mo-1];
}


/* like gettimeofday returning an sktime_t */
sktime_t
sktimeNow(
    void)
{
    struct timeval tv;

    (void)gettimeofday(&tv, NULL);
    return sktimeCreateFromTimeval(&tv);
}


/*
 *    Convert the NTP timestamp (RFC1305) contained in 'ntp' to epoch
 *    milliseconds.  The 'is_micro' field should be 0 if the function
 *    is decoding dateTimeNanoseconds and non-zero when decoding
 *    dateTimeMicroseconds.
 *
 *    An NTP timestamp is a 64 bit value that has whole seconds in the
 *    upper 32 bits and fractional seconds in the lower 32 bits.  Each
 *    fractional second represents 1/(2^32)th of a second.
 *
 *    In addition, NTP uses an epoch time of Jan 1, 1900.
 *
 *    When the 'is_micro' flag is set, decoding must ignore the 11
 *    lowest bits of the fractional part of the timestamp.
 *
 *    If 'ntp' is 0, return 2036-02-07 06:28:16Z for the start of NTP Era 1;
 *    that value in decimal epoch milliseconds is 2085978496000.
 */
sktime_t
sktimeCreateFromNTP(
    const uint64_t *ntp,
    int             is_micro)
{
    /* The number of seconds between the NTP epoch (Jan 1, 1900) and the UNIX
     * epoch (Jan 1, 1970).  Seventy 365-day years plus 17 leap days, at 86400
     * sec/day: ((70 * 365 + 17) * 86400) */
    const uint64_t NTP_EPOCH_TO_UNIX_EPOCH = UINT64_C(0x83AA7E80);

    /* use all lower 32 bits for the Nanosecond mask */
    const uint64_t NANO_MASK      = UINT64_C(0xffffffff);
    /* nanoseconds per whole second, 1e9 */
    const uint64_t NANO_PER_SEC   = UINT64_C(1000000000);

    /* IETF says the Microsecond mask must ignore the lowest 11 bits */
    const uint64_t MICRO_MASK     = UINT64_C(0xfffff800);
    /* microseconds per whole second, 1e6 */
    const uint64_t MICRO_PER_SEC  = UINT64_C(1000000);

    /* When the NTP value rolls over in 2036, must add 2^32 seconds to the
     * UNIX time */
    const uint64_t NTP_ROLLOVER = UINT64_C(0x100000000);

    /* To get a proper value when converting to fractional seconds, add 1<<31
     * before the >>32 to round up values.  Not doing so introduces errors
     * that can accumulate with repeated conversions. */
    const uint64_t ROUNDING_DIFF = UINT64_C(0x80000000);

    /* sktime_t value to return */
    sktime_t t;

    /* Seconds: Right shift `ntp` by 32 to get the whole seconds since 1900.
     * Subtract the difference between the epochs to get a UNIX time, then
     * multiply by NANO_PER_SEC to get nanoseconds.
     *
     * Use the highest bit of ntp to determine (assume) the NTP Era and add
     * NTP_ROLLOVER if Era 1; this is valid from 1968 to 2104. */
    if (*ntp >> 63) {
        /* Assume NTP Era 0 */
        /* valid for 1968-01-20 03:14:08Z to 2036-02-07 06:28:15Z */
        t.t = ((*ntp >> 32) - NTP_EPOCH_TO_UNIX_EPOCH) * NANO_PER_SEC;
    } else {
        /* Assume NTP Era 1 */
        /* valid for 2036-02-07 06:28:16Z to 2104-02-26 09:42:23Z */
        t.t = (((*ntp >> 32) + NTP_ROLLOVER - NTP_EPOCH_TO_UNIX_EPOCH)
               * NANO_PER_SEC);
    }

    /* Handle fractional seconds */
    if (!is_micro) {
        /* Mask the lower 32 bits of `ntp` to get the fractional second part.
         * Divide by 2^32 to get a floating point number that is a fraction of
         * a second, and multiply by NANO_PER_SEC to get nanoeconds, but do
         * those in reverse order, use shift for the division, and handle
         * rounding before the division. */
        t.t += (((*ntp & NANO_MASK) * NANO_PER_SEC + ROUNDING_DIFF) >> 32);
    } else {
        /* Do something similar as for nanoseconds but using microseconds,
         * then multiply by 1000 at the end to get nanoseconds as a whole
         * number of microseconds */
        t.t += ((((*ntp & MICRO_MASK) * MICRO_PER_SEC + ROUNDING_DIFF) >> 32)
                * 1000);
    }

    return t;
}


uint64_t
sktimeGetNTP(
    const sktime_t  t)
{
    /* The number of seconds between the NTP epoch (Jan 1, 1900) and the UNIX
     * epoch (Jan 1, 1970).  Seventy 365-day years plus 17 leap days, at 86400
     * sec/day: ((70 * 365 + 17) * 86400) */
    const uint64_t NTP_EPOCH_TO_UNIX_EPOCH = UINT64_C(0x83AA7E80);

    /* Seconds and fractional-seconds divisor */
    const intmax_t divisor = INTMAX_C(1000);

    /* seconds */
    lldiv_t split = lldiv(sktimeGetEpochMilli(t), divisor);
    uint64_t sec;
    uint64_t frac;

    /* Adjust seconds for the difference in epochs.  When NTP rolls over in
     * 2036, sec will be > UINT32_MAX, but those are chopped off when the <<32
     * is applied. */
    sec = (uint64_t)split.quot + NTP_EPOCH_TO_UNIX_EPOCH;

    /* Divide number of milliseconds by 1e3 to get a fractional second, then
     * multiply by 2^32.  Do those in the reverse order and use shift for the
     * multiplication.  */
    frac = (((uint64_t)split.rem) << UINT64_C(32)) / divisor;

    return ((sec << UINT64_C(32)) | frac);
}


/* Deprecated */
sktime_t
sktimeCreate(
    int64_t   seconds,
    int64_t   millis)
{
    return sktimeCreateFromMilli(seconds, millis);
}

/* DEPRECATED */
uint32_t
sktimeGetMilliseconds(
    sktime_t    t)
{
    return sktimeGetFractionalMilli(t);
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
