#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-ipfix-ipv6.pl 323d2cb74c99 2024-05-30 18:17:09Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsilk2ipfix = check_silk_app('rwsilk2ipfix');

# find the data files we use as sources, or exit 77
my %file;
$file{v6data} = get_data_or_exit77('v6data');

# verify that required features are available
check_features(qw(ipv6 ipfix));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'ipfix', 0, 'polldir', 'ipv6');

# Generate the test data
my $ipfixdata = "$tmpdir/v6data.ipfix";
unlink $ipfixdata;
system "$rwsilk2ipfix --ipfix-output=$ipfixdata $file{v6data}"
    and die "ERROR: Failed running rwsilk2ipfix\n";

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--move $ipfixdata:incoming",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--byte-order=big",
                     "--compression-method=none",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);

# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental sender));

# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  IPv6 files do not use the bytes/packet
    # ratio, so no rounding issue to work-around
    check_md5_file($md5_map{$_}, $path);
}


__DATA__
e674a60e0426d3a865c2416fe74371ae in-S0_20090212.00
58cf874fda7708b56a08cf938c6e4cb2 in-S0_20090212.01
2450043792f414f940412b00ab2c18dc in-S0_20090212.02
900a0f8697faa381e7781c9d460c95b9 in-S0_20090212.03
7cf79b9bd3fd51e7f990504b48b940e8 in-S0_20090212.04
2bb4c770590b9c0f86c3b34fa3d6e941 in-S0_20090212.05
6099a437f1bc3273e8f79d28f206a522 in-S0_20090212.06
81ecc130dd02994d88d05610b4acb56c in-S0_20090212.07
3380b086d036fdbe8ead174b74417a12 in-S0_20090212.08
b0e7c5d16085a02713a596671f2715df in-S0_20090212.09
04ff650949cf6ddc28c60e3ec6918a18 in-S0_20090212.10
ae9546d43cb9ac56eec5d28b6d54c41b in-S0_20090212.11
3db8edc84b5af84be7857aa15a49b93c in-S0_20090212.12
d9f0489f851050b393e9ef353a9d7142 in-S0_20090212.13
8616bc8554f2d28aedbeb7c62a5c1b44 in-S0_20090212.14
47066270942641f4b753a2f9f44a1432 in-S0_20090212.15
8f68ff67f8818d2ac3393df0c4246bf1 in-S0_20090212.16
d3594760ba8d46a647971f59bc9873ab in-S0_20090212.17
7b46ba60c6a40ed5fb08837ac6b3df20 in-S0_20090212.18
39204d84dd08b3b3ed9941b85e36b03e in-S0_20090212.19
200da1ba19c4d9f9ee4577e705c0a4ec in-S0_20090212.20
7374b764e5dea30f7568bcbfac4144db in-S0_20090212.21
d1e722b2f80b880d07006ca9e0337ece in-S0_20090212.22
2a5e1acea4ca5e3bd9134309570a419c in-S0_20090212.23
060e2cbc391c53f0eceb5dafdd3c789d in-S0_20090213.00
e36ad54e9c7d9ec5f0073d5e10c66a7a in-S0_20090213.01
132cdcb3db947f5917b04696a2b953d0 in-S0_20090213.02
4236a3bc9d8b12d0b24ec44d262c815f in-S0_20090213.03
493033e18b1fd423e284cddf663b237b in-S0_20090213.04
126dfee137df9dcf9f4981dd96d76e6c in-S0_20090213.05
2043901e3a02ac89d0cf208122e08570 in-S0_20090213.06
979f4c5336aac06e10c1e956c283bcb2 in-S0_20090213.07
25c182ce1d1375547998a77f909ffbc4 in-S0_20090213.08
d0471ba63d9a8259816584f405e03bb8 in-S0_20090213.09
a1e15b56813d749ede9458302d70e09a in-S0_20090213.10
8cf68d95ca64038049e95a42c18642a7 in-S0_20090213.11
771d432e12a93269f6c7760444ca72c7 in-S0_20090213.12
64312b223e673464a581bd67273a4c6a in-S0_20090213.13
a5c6c616c8032bbcf85d7cadf4cb3443 in-S0_20090213.14
edc92aa3af018463202d9cb5eaf7bcb4 in-S0_20090213.15
7f014ac55b52d69a26260ff09795807c in-S0_20090213.16
4c4c49ad751993c1bea88d6830793f15 in-S0_20090213.17
0a39990efd384c850606684d2b1612e0 in-S0_20090213.18
a2f9f13abf91cb373eddd403d17de1be in-S0_20090213.19
c342132a773046a19673de026408fa82 in-S0_20090213.20
8ebef2b59d70201019b4ec3eed145b4f in-S0_20090213.21
0ec8685c64191efe9b224ef0677c9008 in-S0_20090213.22
69460ba17b24b32c6e2d1a0143684071 in-S0_20090213.23
8350b75179be8adc13c0c355a4c12b80 in-S0_20090214.00
f731c02f3e36986641932889c64bcb6f in-S0_20090214.01
b74c128c975c4ed6a8dd553b393619d7 in-S0_20090214.02
dac1c226fa5031598818b6faffe95974 in-S0_20090214.03
4b256ab94c68f1f3d47df17a5a5b21d7 in-S0_20090214.04
edc2d95c7fb347522fb285a993254021 in-S0_20090214.05
f07bcb568e92c2d68e51b2ab55e3fc16 in-S0_20090214.06
e474ec12c21bd13a4a17018a367f5fe6 in-S0_20090214.07
f09e6da0c829b95749366df69d154a1f in-S0_20090214.08
1f0c303ca75751a534d26a4cde9e5dfd in-S0_20090214.09
4a5a50a96f403ef840a6b063100b4cf6 in-S0_20090214.10
524ba64eb6021923dad68a23555a1193 in-S0_20090214.11
140a9fece1a9097d9e5d609f75a0926e in-S0_20090214.12
d5efc9040012d36b4f844d6566355361 in-S0_20090214.13
ce3fa3338b4e87166941f70c5d93064b in-S0_20090214.14
f88406d9bf9a2d422c96f02635f664e0 in-S0_20090214.15
311d5de1023d18eda6289576136e87a1 in-S0_20090214.16
11bcd38f417fd68e92a834641072d2b9 in-S0_20090214.17
31922d8e1f0cab2c2052388394fa1685 in-S0_20090214.18
055ecabb8e99ffdc51096c5e9880dcab in-S0_20090214.19
38fe3f41e40ad264546438147b9f4d61 in-S0_20090214.20
6b381bd7ee209bed42a15250856b642c in-S0_20090214.21
46f04fc037c669492bec531cd3c7e2d2 in-S0_20090214.22
c3123cf47e3b8f578cf2b3b515a55d02 in-S0_20090214.23
acf098a846ae8a1de13c822e90aa7fc3 iw-S0_20090212.00
78aaad594ff55e4e9f44360ed00d7adf iw-S0_20090212.01
0725f09b049adc1a39569d1d4cdd536d iw-S0_20090212.02
7e50964f58c1eac5e7342a6b56cfa6cd iw-S0_20090212.03
c1500b39fa6ece4fd7aa6ef7a9dcd2d2 iw-S0_20090212.04
e1b4ec566786292c894c011633e190fd iw-S0_20090212.05
5bd2a157ceaa38ae179bfb26814e770d iw-S0_20090212.06
072d7cd6492b966a73cb5f46e5d2c546 iw-S0_20090212.07
92944c3774ad5060bbc1919776241c37 iw-S0_20090212.08
9ec0d3859e1abc311a05f313e12dde89 iw-S0_20090212.09
d62f6dee2e2d6267b8b1f99ff2ad58ee iw-S0_20090212.10
403191d6c44c8f1b1a946f2fedc2e179 iw-S0_20090212.11
09e63a63dd9dbba6f2a57059ab30efc1 iw-S0_20090212.12
75a10ca4e2be59515c93cd6a37a0e4c6 iw-S0_20090212.13
be8c779f118a79d34332773496b0e368 iw-S0_20090212.14
a15ca4b9df60749902ba944206dfd316 iw-S0_20090212.15
b04d82acb92d4de6650d8fee28206890 iw-S0_20090212.16
e9973826f41899c7f0968bdcbfb68333 iw-S0_20090212.17
0a0e002dd33b6ad5491f368e366ddc79 iw-S0_20090212.18
bcae625271faaa43b02d124fee3e9792 iw-S0_20090212.19
52732c238b782bd3f0c91d0263b24c48 iw-S0_20090212.20
de7cc5417f656a3b5959bae75db70faf iw-S0_20090212.21
f56c6d624b603aac1bc651709a0bb51a iw-S0_20090212.22
a70bd5f7eaae356b1af121835d6707d2 iw-S0_20090212.23
c5f1c62c34ac9d7cf0b0fc08363d6420 iw-S0_20090213.00
a5a7f1184a845c12640cd9dfe43c6031 iw-S0_20090213.01
bdb537d0c73b08721bda1bd982a380e3 iw-S0_20090213.02
914453fb7777faeb49373f988a05eff3 iw-S0_20090213.03
c23883f6e51e9054282fb2754505a467 iw-S0_20090213.04
8ec6b60f6c434ff965e7ffdc39c8e879 iw-S0_20090213.05
4449f9333c24a23ba94673413be122a3 iw-S0_20090213.06
6faad94b188fd7ef558cd2dfad2fdecf iw-S0_20090213.07
af96a140c2fbc0568f6d57636b4efd19 iw-S0_20090213.08
780b9c68c88a5297d33a82f49e8db5ff iw-S0_20090213.09
db1c3d97d3a62f2cd9bd8f5e877ce7b4 iw-S0_20090213.10
cd9d6cfca9bf78c2b0847befea301a71 iw-S0_20090213.11
280909e86dc1d29c94616ddd0a3ad622 iw-S0_20090213.12
e8ceeb5e43decc53dbd8a262ea0c014e iw-S0_20090213.13
5bdc83a6fe7d948d64cfd034cca9f8e5 iw-S0_20090213.14
b9c54e53c3ca1a14dc2c3eae7a73f5c7 iw-S0_20090213.15
790d88e8fb0ba3bcb1160edef97bc20f iw-S0_20090213.16
e6caab6c96e9b7989719baf7736b540e iw-S0_20090213.17
2f0eb95cdeeb1ea50abec777518a4af5 iw-S0_20090213.18
056b4c5f1c9133320a7a03e052d0863d iw-S0_20090213.19
98d8fc72d6923c524db6ca73461442ed iw-S0_20090213.20
d79b571947467655c7152793e3157b05 iw-S0_20090213.21
95b711c0ca8a0740d4d174dfbff56be7 iw-S0_20090213.22
814d25e102cabe404af5282575f4179c iw-S0_20090213.23
cae3b82e612755f82f517ae6a9462d11 iw-S0_20090214.00
504edfb1c68c9fc3cc19a5b233d25492 iw-S0_20090214.01
211c36ee8481b32ce3396fcf1f713268 iw-S0_20090214.02
05fa898ca0a77caa7d0b23aed8987069 iw-S0_20090214.03
03cdfb4ed54664bf531c1d018e6a14f3 iw-S0_20090214.04
17c4e79d46a0712951652458f0cceabe iw-S0_20090214.05
706a4e7145d6e25332359bcefa1cd341 iw-S0_20090214.06
82c3598dcf1a8848327e6ff6486cb11b iw-S0_20090214.07
99d7025beffadfe0fc1ec2169210a048 iw-S0_20090214.08
8a7dbc0249a1e5ebc3001e1d80a4f418 iw-S0_20090214.09
f472636c17af5c4325d7decfbcd78e23 iw-S0_20090214.10
550ad8156f72237e50fb0f7dc1bbd051 iw-S0_20090214.11
d355b99866c420e3e2b5bc439d466ed2 iw-S0_20090214.12
adb3a8f68dc16623457ac7b91d287dc8 iw-S0_20090214.13
824becae71de53da58e80c4dd36b461d iw-S0_20090214.14
5566c78c0bb40ee15ce7081b136c505e iw-S0_20090214.15
6ce8e7ccfaa4cadad3b1ee46f715502f iw-S0_20090214.16
b44442e8e571de9585d1a5bc2246082b iw-S0_20090214.17
32e5fd849a8f7f7dd0bcf17969836891 iw-S0_20090214.18
9f829edfd67a3fa98f15b0fa11b083fe iw-S0_20090214.19
a5b560867c0e2fe21af387cb2056a16e iw-S0_20090214.20
9ae26d88a691a99615398ff0ff3d62cf iw-S0_20090214.21
e7b0a1f294f3d532f515e0b85d00ab89 iw-S0_20090214.22
49fbb81aef2bae48b7ad74a313e46c98 iw-S0_20090214.23
40bb8e0cdeeccddb32a03d756bd04c6a other-S0_20090212.00
f8acccf40a75713df7c447287ab21588 other-S0_20090212.01
a1a8319815040967560758dd58e16888 other-S0_20090212.02
4a426f219ec266a7d0c3eabb9a650514 other-S0_20090212.03
86ed9975ad81618b9b7a18452a51a81b other-S0_20090212.04
047b7c598668ca59a11d23e11703ec19 other-S0_20090212.05
783f9e59af25429d959e768e26696dae other-S0_20090212.06
a1071f3c21a6ef3f611b6a158cd5c8ab other-S0_20090212.07
0eb2f6051919c173078cd422a1794954 other-S0_20090212.08
5dd50c8cf5d7159f35f6bdf1144bc275 other-S0_20090212.09
524ccd574243f28fa56a60a4d6db552f other-S0_20090212.10
e67d390dbeb6099089cc842941f549dd other-S0_20090212.11
15827e9e18d736e71d2d9de65fca970d other-S0_20090212.12
6ccc1631bf72c42774d9987042262bcd other-S0_20090212.13
e8ddb3fd48582a997f77cea3d6e13f97 other-S0_20090212.14
5efd0a6c136de4cde5461463678bde39 other-S0_20090212.15
77e242455f74743730eb18e14a666ef6 other-S0_20090212.16
9b71a2c4cfc066ac76de12e14c3b10c2 other-S0_20090212.17
2db8acf7c30a7312f8962c44b8f6812f other-S0_20090212.18
38ccaef892d6b8f0b4b49b2db18f49ff other-S0_20090212.19
34e4bb940178dbf3d87cc35123b50f4c other-S0_20090212.20
d09bfac2d8734acf6cc1435be33705b3 other-S0_20090212.21
d9e645d06300780cd32a280d3d33377d other-S0_20090212.22
f4ba3c4541d10b0e719dcd8880bb4d2f other-S0_20090212.23
dfd8f8969e376d12f42b7f6e93c467bf other-S0_20090213.00
d0daa9953d7f2e41c533e70cd0bab948 other-S0_20090213.01
2e87327c84de975c6269310903cb86ab other-S0_20090213.02
d6d9a16ece5ec9e80b1cbe71a4389f30 other-S0_20090213.03
9f8652b6c1cb031ca7712b57098bd5ca other-S0_20090213.04
f7b85649135d35db302913ff1127fee1 other-S0_20090213.05
d74fd4d2188cc85ede54f7d8723cfce7 other-S0_20090213.06
d4617210b2ff40ac7130a2ffd26286b8 other-S0_20090213.07
ae4df98c2d8673e9cbf28701b7994b78 other-S0_20090213.08
46222a004ecf4d77f3772db09b635a55 other-S0_20090213.09
dbcbf4be225bdbf39e90180de8c924fb other-S0_20090213.10
68633fb2e0434a20cc5f369b9231424c other-S0_20090213.11
020442aa54849b1bdc4597aaaed293d4 other-S0_20090213.12
6ea82d1769e900b632eeafea5255e835 other-S0_20090213.13
d58c6bbd18412c5eb1d900636d34050f other-S0_20090213.14
8674c9bfffaa5337f7b76c931e179e27 other-S0_20090213.15
4d9c1defdae8192e3328c0697e7cce01 other-S0_20090213.16
99f0c2e9142ac490e9a4bdbf47f3307d other-S0_20090213.17
02fe333e537bb02d74f1379aad3431b1 other-S0_20090213.18
a9b5c13ee2db500599556bea3b6d55e8 other-S0_20090213.19
8569c5b544189f77acd116f0040e3095 other-S0_20090213.20
dbd1ee451df2df2bd2a912a6eead7792 other-S0_20090213.21
a04eb726a9330e35858d335002837dfd other-S0_20090213.22
fadeaa57d9a74359646a081a824c148b other-S0_20090213.23
a6ce10bed56198a6a3c1817812774a93 other-S0_20090214.00
31f3cbdc0441f12ce17d11ec12d37b30 other-S0_20090214.01
050b3f68195df5dec2b890356afc1533 other-S0_20090214.02
90b245f2ca360693a9432e899df7779f other-S0_20090214.03
0121ae389cde845d5b692ace80d5c2da other-S0_20090214.04
f616c2b3578803e444ed30089f3888c7 other-S0_20090214.05
5afe2a562471c8b0a5fd10618be33feb other-S0_20090214.06
6fffcfb3954184a25206e64a866ebf27 other-S0_20090214.07
d2f62507ee1feaa909f32762b2e189c8 other-S0_20090214.08
22ba813ea34a3bfc3c0844b479302a8f other-S0_20090214.09
b480727dd9b6f0fafb3a0d9b371e3954 other-S0_20090214.10
b6bcc6c305c0b0fc3f472d0c19e26196 other-S0_20090214.11
4c4e67b9178befe3dd434ff236fcbbe7 other-S0_20090214.12
89c36a9b48b74d60f42f9920e009a002 other-S0_20090214.13
01f3d3947ae75fc216a07f934bf702f8 other-S0_20090214.14
65edc89bdba679dec340a75acb4c93ad other-S0_20090214.15
fbae48be15ecd7d0b5edbd20c016c314 other-S0_20090214.16
f11c634aebd66b69182db99840ec29da other-S0_20090214.17
edf04bc5280b16947cd75e6e573dff3d other-S0_20090214.18
8e284adc557b9aeece4b86badf037ad1 other-S0_20090214.19
193efb768e8ffe7ffdf57faaedb67647 other-S0_20090214.20
3c25ed23e865178677833095bd007a89 other-S0_20090214.21
541bcb87f31b56ca7548afeaea315315 other-S0_20090214.22
ce6622fe6574d3f18474eafda24bc812 other-S0_20090214.23
cb497468d4c05ce6afdaa97407a414cd out-S0_20090212.00
b20980b0b4b59335e2b76ae3a9895175 out-S0_20090212.01
3316d89ce9aa903e6fab2524b03e13a7 out-S0_20090212.02
fd8c4d80d029e017496366402a980e8d out-S0_20090212.03
6231c5b4c0a760d040465fd9418fea5a out-S0_20090212.04
f7d75212c451b526da6cd0c836e7df0d out-S0_20090212.05
46cd58f04c20d4cbe3a18c487bd19cdf out-S0_20090212.06
5e3b34d6017db2c4aec16fcb753c9bf3 out-S0_20090212.07
e8d9e9fea2665a748ffe7abc1cc0da19 out-S0_20090212.08
0945e0b96e54e7a05fc4fb46ab5221c8 out-S0_20090212.09
5a118dbf83afaf94d0e2502e2d5a04ef out-S0_20090212.10
7891d749848bba79b172017374ec5f03 out-S0_20090212.11
bbae4fe7e2ced4f882ae5fc1ef5755fb out-S0_20090212.12
fabb249bd3ea0e7d5e017eb6dc138c49 out-S0_20090212.13
42c2f020d649fdf9abfcc20b6ed8b3f0 out-S0_20090212.14
b0841050c831e8aa4ee288cb94bb790d out-S0_20090212.15
f40615e56a81c297191dc05b2d29bf9c out-S0_20090212.16
0353e168e6241d771a49aeb7ce3f452d out-S0_20090212.17
0f580412b1fc4d80f6f153fbb48a906a out-S0_20090212.18
31697558ecee2977db75f1c77ff5b428 out-S0_20090212.19
a939a7b73ac155464a29718220a742e9 out-S0_20090212.20
3baee8068e037d75f3ccc81e5ab7fd86 out-S0_20090212.21
082f0718d1ecf8a052185d90efff0385 out-S0_20090212.22
9750578e504abd6badcc9ebffb3f3bba out-S0_20090212.23
41f1c3eac481bb3df6e36887363ac3f6 out-S0_20090213.00
75453bdd44b892be6a70fc9b41a975f4 out-S0_20090213.01
34fdb03038ad9ed7950c37388a5ddd43 out-S0_20090213.02
b3712d9d2b80d7e51ae3c5d28df9c75f out-S0_20090213.03
a82d18673bae83c1cd45fc7305b59b63 out-S0_20090213.04
01b5f48706cbc18f21cea039d396f0e5 out-S0_20090213.05
dd73c3d7a665182db53978c0d316d99d out-S0_20090213.06
f85f703a905cda5dcba005e6aaf1cdf5 out-S0_20090213.07
33b1d2e7fb48c571affead4e5be7c91c out-S0_20090213.08
a99e0deae5a1b5bcf3c040621253a448 out-S0_20090213.09
19021459112fb6f35a3d8964f1c48248 out-S0_20090213.10
2a8219e361ec0041d5e4f44ede3653be out-S0_20090213.11
68882e3037b41e7716b7e676741b7367 out-S0_20090213.12
3d10d7d8f6b95504f96d1684c5236a5f out-S0_20090213.13
52182853913744b156b3756f03e0528a out-S0_20090213.14
895bf96f6c1ca91d481d80abbb2c1124 out-S0_20090213.15
b1d868bd63ae51c975f9c5a9f1e2a62c out-S0_20090213.16
6b30b1330f31cd6b5f4c22d53d91d5e7 out-S0_20090213.17
e5646608b3d16ed152d70b56d553ae72 out-S0_20090213.18
811bf5b7fe5d19e2f29d21d395325bb9 out-S0_20090213.19
3d401ca4439b56105a870c318dd68016 out-S0_20090213.20
9a1a07fe2bd513230fd74e55f67143f8 out-S0_20090213.21
64f1fb675a979e0a407d3fe3327e4810 out-S0_20090213.22
9cff4dc321726d67311e042e764ec6ef out-S0_20090213.23
087d14fac1ac751650281a25828f264f out-S0_20090214.00
2d4c62ea447df813df9be53852cc5cc1 out-S0_20090214.01
eef060779801663f134e659bcac4ba5a out-S0_20090214.02
a40f86598944c7ba47800c4c027fc567 out-S0_20090214.03
2056abf4ba1d445e238db22588854733 out-S0_20090214.04
4b5af79638ec981c8f1acf14480981ab out-S0_20090214.05
3bd317d0a78a8d4953e99657b51564b2 out-S0_20090214.06
2c425da8cfb906726717a30eb80720f9 out-S0_20090214.07
99a696c0eda2b6ce1aa80d95e00da022 out-S0_20090214.08
237bde6619d6c0f84be048e7bccfc49e out-S0_20090214.09
1a080a4845e0bcb5459903bf20193597 out-S0_20090214.10
eebb2e02e1890c11df51a44d3963970b out-S0_20090214.11
15792aeae58f7bc0d7b702b8d598bffc out-S0_20090214.12
dd249aad9352208457f463595e9cd3e9 out-S0_20090214.13
975eb0902b14053e7d06a55c1651aa71 out-S0_20090214.14
2ab0cd2d87c99652856e826fb736e389 out-S0_20090214.15
eb4a54fcdfdfa107dd75043576dc1a43 out-S0_20090214.16
55be19815156626a65ad78291e758e5e out-S0_20090214.17
09150bd19d85f65d488555f195689078 out-S0_20090214.18
4f68db3272d120fd537d71d7361193aa out-S0_20090214.19
ff686585644c61363396005e3474d432 out-S0_20090214.20
725fcd55ae2c350c0dce93229547f7ae out-S0_20090214.21
6c8ab9cb214ba4e2bc5144fd1412350c out-S0_20090214.22
0f47938df19ea6d3c86e0490fc68b758 out-S0_20090214.23
45e4540e7a9d5d310cc7f1747ca4ecfc outnull-S0_20090212.00
c73d9f3e3b470c432520e24d8488bf2f outnull-S0_20090212.01
33c676b449c469bc07e0eb1622f4c396 outnull-S0_20090212.02
3447b2749cc1be91971d8086db3074db outnull-S0_20090212.03
ed4c5c2b442f550035d2791fa711ec9b outnull-S0_20090212.04
45f2acb7ac2344dbe586af4bcbfe153d outnull-S0_20090212.05
41c980dcca71bcca1809abc9ff6024f5 outnull-S0_20090212.06
9e3c0b681e3b99209917e09fece27785 outnull-S0_20090212.07
c31dddb41faa5e03977cd22df6f0684e outnull-S0_20090212.08
2326ec2435acad6b488bb12b867eca1a outnull-S0_20090212.09
92c80545abab8f051c1d413bd1c07c60 outnull-S0_20090212.10
045f5c9f36e1c4844c63256336f2fef6 outnull-S0_20090212.11
faf5d6c54fbc2416f285ae13c622f645 outnull-S0_20090212.12
544c5e9e66fee5c1b4d2ce9356960399 outnull-S0_20090212.13
af5220a340e6d3e046a1d2a050da5cac outnull-S0_20090212.14
380f712be92861256882c0baff8a10c7 outnull-S0_20090212.15
29e9b7801e8213c4f719b17d3fa8aab3 outnull-S0_20090212.16
0fe4c00cfc9d83886b079cff1f5f1e31 outnull-S0_20090212.17
84f6ef5b5a2d87e34b7198ac06c260c2 outnull-S0_20090212.18
c1c9d8178e1ac27be02bbb3408884d8d outnull-S0_20090212.19
4443ff484600dafc601f3eea55636765 outnull-S0_20090212.20
6a8ff27710893e8289c777035a33cd54 outnull-S0_20090212.21
019c2f5aa1c1fa3aa2b37228795e9a27 outnull-S0_20090212.22
0d727c1648077dff4509fb86619249bc outnull-S0_20090212.23
e42980986f087eea035c3a868e3e9673 outnull-S0_20090213.00
3bf717f2dd607e3402ca8a5fde25c20d outnull-S0_20090213.01
db85cc7208851a080fcb899b63ba8e08 outnull-S0_20090213.02
cabb415677cdc09c21bd5511374d0cea outnull-S0_20090213.03
c934eb562be9a4805db38be5857d13bf outnull-S0_20090213.04
9d2a99825b33ff9dc6c9f56b924393a4 outnull-S0_20090213.05
89571eb9139baee83ac10b17d509d387 outnull-S0_20090213.06
f05fd2fe4397bc22b85ac9aff1409fc9 outnull-S0_20090213.07
816ec9bb716e82ea2541c1116af12323 outnull-S0_20090213.08
9bfab76c535bd4f39913517044319e7d outnull-S0_20090213.09
448806680e2f5cdcf5abca5ce9161369 outnull-S0_20090213.10
6deb75387622cab60747c8b54dfbe44c outnull-S0_20090213.11
d4b496732e2f7ad4942bc0aedf390155 outnull-S0_20090213.12
015f07347ea9d58151460817207264c3 outnull-S0_20090213.13
cac8f35b961126e18861715372db82a5 outnull-S0_20090213.14
5301e97a278d2180b196748f6b9a1dd3 outnull-S0_20090213.15
553b458421b4a668c9919d307be87838 outnull-S0_20090213.16
48dce62c4472a856d3a3654b2f2a1694 outnull-S0_20090213.17
e111d153d88a73e5f88de5606329665f outnull-S0_20090213.18
668555bb679203513aed653c5d207ba5 outnull-S0_20090213.19
0dddbf041592c96731e6de8c436fc679 outnull-S0_20090213.20
81c1e520d20067cc810e7af47e495f7d outnull-S0_20090213.21
0dab80e307c79a84a8980d481df69fd5 outnull-S0_20090213.22
5ac8d4789f6827a35852161d0d9d450f outnull-S0_20090213.23
5d29891eed3e3547c4bfc76fa8bc4f85 outnull-S0_20090214.00
ba62075782ec318df93b1516f352b58e outnull-S0_20090214.01
a2e9d34799eed16301dfb6180b23d235 outnull-S0_20090214.02
afc4e5b993033acd53ea88625cb307f5 outnull-S0_20090214.03
21085050b0797ee3af4627e341bd59df outnull-S0_20090214.04
8a446287e85c8953d969e8f33bf86823 outnull-S0_20090214.05
b5a781ded0c145954674d48f5812a57e outnull-S0_20090214.06
418c7eec224cf55e720b979929d38640 outnull-S0_20090214.07
80d9f713103ad6c4e9af007759e8da0c outnull-S0_20090214.08
a92e06ee8d29eac18eda14d0b1c3029d outnull-S0_20090214.09
2c4b5368e58e4bfefa3a65f35a6075d4 outnull-S0_20090214.10
14bddc19e5656e850209fcb7cc9dd957 outnull-S0_20090214.11
c7a0e191a07b0a2595c44e60ddab4c05 outnull-S0_20090214.12
1dcab9758a5a552acbb590c8febf7e5a outnull-S0_20090214.13
2b2f1f90d11949d804525fdcbf4b43b6 outnull-S0_20090214.14
87819c798b3a7c4b81b73a4547874b5e outnull-S0_20090214.15
4f1b8fc73ddf8a4ef99f5b854e917d92 outnull-S0_20090214.16
ea99e9ddc532a8c519b867b677aa53cf outnull-S0_20090214.17
782867f24ac6c713e717a9412d36be4e outnull-S0_20090214.18
fbdf0911eb783922c4336a1bd1909a0a outnull-S0_20090214.19
a7e9d2a2d6992f18a8d4947287d094e8 outnull-S0_20090214.20
75200fab4a0dd115452254bdf8958fd5 outnull-S0_20090214.21
aeed9569912f9a5ac42a7275cf0fe7ad outnull-S0_20090214.22
76e9343ac6c1148f5391fcd608f09e96 outnull-S0_20090214.23
03b949ae21b28df3bb9e9232223f6f47 ow-S0_20090212.00
1fa664522e9b01b32e16fa13ca64991f ow-S0_20090212.01
3798c25b3a84871b09b36d03532668de ow-S0_20090212.02
f8535b5f22c66c5a3d4253e17b2b2a52 ow-S0_20090212.03
59bad14fc5bb149b7e0ca8609667c867 ow-S0_20090212.04
554f7ff6c9c6c5c9bb12469df9347c4c ow-S0_20090212.05
0cde1a87b83f03e9d82c18ecf7f37fad ow-S0_20090212.06
7fc15cb3d8956c76a29de895dee0d4e8 ow-S0_20090212.07
e970f64909beb8111e0b1e083e53c9b3 ow-S0_20090212.08
5e09b3abe8a82455c366cc46259f3c4f ow-S0_20090212.09
34029a022871bd7f27257a69dc0373c2 ow-S0_20090212.10
b019347951f890fa651286a08949ecc8 ow-S0_20090212.11
149bff38c79b515ea73bf1dee85f754a ow-S0_20090212.12
bc66b626bcb3e33d4e1afbdfaad774b5 ow-S0_20090212.13
810289b45dd0903d011c730ac1f16791 ow-S0_20090212.14
0ecf7a10a225a48e87e7e03d53688c54 ow-S0_20090212.15
f0f618fc16b2be1eb436b3c0a319d1e3 ow-S0_20090212.16
8281d1041ea723e3ebe07a8a20d1dc2d ow-S0_20090212.17
5da91ccdf82462ec6b2a97e2ee1444a9 ow-S0_20090212.18
3be26b04f24a52aa34b1b4e464d10db1 ow-S0_20090212.19
053a54aae0203cdc5953a271ed585c4f ow-S0_20090212.20
0a828ed9564ebeb9b926003ba2a148db ow-S0_20090212.21
e830d105b74dcb2405b049b618f84d11 ow-S0_20090212.22
f1c8b521f672269c5ab061a30d6421c0 ow-S0_20090212.23
f6edc7a7060d7ef52ac8a8e247da88fb ow-S0_20090213.00
1885675e51671cd72bd7bc2aa0c49b59 ow-S0_20090213.01
d663043893dfcc91d01a9c50e5a621bd ow-S0_20090213.02
1d58e92116c861cc332d30e07f052347 ow-S0_20090213.03
4dc04254df6a83cefce3bf623f9b6ed8 ow-S0_20090213.04
8cbd43f567ed884f5f89418cba2d7362 ow-S0_20090213.05
19c7c1c2572760be5f32adf4fedb182e ow-S0_20090213.06
b60c05e8d033384818d6d2adb9c3f97b ow-S0_20090213.07
9fc03eddc5a248380e69cd6209268d04 ow-S0_20090213.08
a5708c93a944fe527a77cebd072efbe8 ow-S0_20090213.09
4bebef4128c2b4105f2bc07c4bae4ec9 ow-S0_20090213.10
e88858e3e1e47ab93e3ea982e296f7e8 ow-S0_20090213.11
3b0a3a3c919476441c4dd3ab2d41e79f ow-S0_20090213.12
1f966985bae7ee2d26a49e2d398e59e9 ow-S0_20090213.13
0a4cb9164bc48fd8693865acfed2e45a ow-S0_20090213.14
9162a2e90700d0352a54b642de3b53ba ow-S0_20090213.15
fa51c1a29288584f949f2c341c36bffb ow-S0_20090213.16
1ec1ee54723a10034eae0e1c7efef31b ow-S0_20090213.17
c0cf24beb026c60baefb218c79309af0 ow-S0_20090213.18
c015600464f313343ec9f51f18ce269a ow-S0_20090213.19
1e834b03cc13a3f73d0f908fe3d3fc1d ow-S0_20090213.20
d9a4dff0a8a5f57a05a7ec0194e02677 ow-S0_20090213.21
f289472925f9fbfeb5864e1b534f70b3 ow-S0_20090213.22
2dff182e7a29ae53bc8359ddedfbda55 ow-S0_20090213.23
1012d589d9665675bf7fff1d7125db44 ow-S0_20090214.00
59c0adb563deeb61adcce26135c27622 ow-S0_20090214.01
2eb79cdaefdbf43f8a1865b9d54d711b ow-S0_20090214.02
3319034d3ee2d9084b986c0f4ee02b79 ow-S0_20090214.03
d3b7f68ac833f70f8a547da8cf39f6fc ow-S0_20090214.04
5b981d6810572ec5159b3af7ed8fda7e ow-S0_20090214.05
df1d4d54580b0a2072409d40328a28a9 ow-S0_20090214.06
9c878b70e5d25495f93871287a525ae5 ow-S0_20090214.07
b9ad84567aad9a4033a6f9e8db072a5e ow-S0_20090214.08
8b1b0ce4ed57547166ed7a6e5828bb73 ow-S0_20090214.09
97146ec036fcf7afe8ab81a96348461e ow-S0_20090214.10
6401e166fb419eb4e7d0921ead36c56a ow-S0_20090214.11
f1874450ac68f095157026f72772854d ow-S0_20090214.12
1c1a62fec0e5b0f46bb58e9722670f3b ow-S0_20090214.13
677e9c05f9605ab44e37d9219249a33e ow-S0_20090214.14
30a21234a2defe14468b1194504e6390 ow-S0_20090214.15
52b2a7189b56213080c1aa963e522af7 ow-S0_20090214.16
76b557d885d0062b2402cb9a10e1ba25 ow-S0_20090214.17
1bd95ab6d512c1422728b2b297e76bb4 ow-S0_20090214.18
ada16bd1cc4b8e865c7b5f65d7962774 ow-S0_20090214.19
22b61076284d0d69f3b943bfb62cb2a3 ow-S0_20090214.20
e8add1185829565889ea135a0faf7ae3 ow-S0_20090214.21
7fb2069c07a846fbe3598cd553df8ce6 ow-S0_20090214.22
52dc5d6612aa4567b4bb4ba8bb726489 ow-S0_20090214.23
