#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-ipv6.pl 323d2cb74c99 2024-05-30 18:17:09Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsort = check_silk_app('rwsort');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');
$file{v6data} = get_data_or_exit77('v6data');

# verify that required features are available
check_features(qw(ipv6));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'silk', 0, 'polldir', 'ipv6');

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--copy $file{data}:incoming",
                     "--copy $file{v6data}:incoming",
                     "--limit=1003752",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--flat-archive",
                     "--file-cache-size=8",
    );

# run it and check the MD5 hash of its output
check_md5_output('9b8040c3af8489905de6018edd08cd1b', $cmd);

# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental sender));

# input files should now be in the archive directory
verify_directory_files("$tmpdir/archive", $file{data}, $file{v6data});


# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  Use rwsort since input has two files,
    # and we don't know which was processed first.  IPv6 files do not
    # use the bytes/packet ratio, so no rounding issue to work-around
    check_md5_output($md5_map{$_},
                     ("$rwsort --fields=stime,sip $path "
                      ." | $rwcat --compression=none --byte-order=little"));
}


__DATA__
5183a9d82976c6e01ebb86fc6e485095 in-S0_20090212.00
b207b958735c508876d11076a583d24c in-S0_20090212.01
d6e4d27e2b6825fcce08bd873627f0fe in-S0_20090212.02
0daaeea2e8941236bcf49611daf9cb7c in-S0_20090212.03
20689138ee0f2c9139f6cfeda94510ed in-S0_20090212.04
2386a66b36e7d8e5e6317e17f109e53a in-S0_20090212.05
96165bdae630b41585e912c5570db131 in-S0_20090212.06
e7db3ba10780d317249e585548cfcbef in-S0_20090212.07
4555982bb21317c6822209075ea1447e in-S0_20090212.08
abdb8ee9df3cd81f92c192fb99a5a5c7 in-S0_20090212.09
bae6fae358e65ec16cf36eb72be844fe in-S0_20090212.10
bd5cdfe3bfe2f46f90ef5592291f6098 in-S0_20090212.11
d35886691e6f4fc1d637f50ddd1eff32 in-S0_20090212.12
6eec3db5b97bfb1ee6b5f6e830aaf19b in-S0_20090212.13
26fbe66574938670341b541a21486de9 in-S0_20090212.14
5346c77f13e5e5e35bf307bef2e1cab5 in-S0_20090212.15
f5338cfd38fc270becb5cb513fc27401 in-S0_20090212.16
f93b24130ed2218379c473a3e14b9334 in-S0_20090212.17
84c21b69ad4deee4868f1fe16ac59af3 in-S0_20090212.18
973c891399c169aab1a0ec15e3d93ad4 in-S0_20090212.19
bd7bdc0dfdcfb851bd4f2b5fa8f1e32f in-S0_20090212.20
6f73c6e3e185af0d3f1374139198bf93 in-S0_20090212.21
97eadd9c181f1a90184272a2ef7980bf in-S0_20090212.22
d94895d7979e592e10ef3b72d23389bc in-S0_20090212.23
01e75e94d3b067095f3958211f0c1a8f in-S0_20090213.00
a4e4dff88719c5dc8aeaf4020705e74e in-S0_20090213.01
485a6d96e5c1a124466282d57cd57447 in-S0_20090213.02
e68687af075577ac1ccd00ba550fca83 in-S0_20090213.03
80b8ead86a1c3da8b5d83452b32d33fe in-S0_20090213.04
66523d4f93c057d578ca7ac564ef0f0c in-S0_20090213.05
cd27a84d09fdcb22d5a80544dda36b48 in-S0_20090213.06
87ba5662c239ee7ebf03c12a20d0164d in-S0_20090213.07
9f699c10398654f4c4f7409ac228d52e in-S0_20090213.08
da68201697a0589b945cbd37eb8072da in-S0_20090213.09
283cd7c506ae973cc30d326ea2e3db02 in-S0_20090213.10
bad1aac973e60fb89597a49ae443ba2c in-S0_20090213.11
6f9686146d84727fbb596b94b064148a in-S0_20090213.12
01637cd55dd95040dcf187fc70aa3d2a in-S0_20090213.13
5c5bd8f4a07501c753da1cd18d1e0902 in-S0_20090213.14
f1bfb4d58a73dd57a49a270121ff3e2f in-S0_20090213.15
5c0bc0ddb319f24616e57b7a18cc64d6 in-S0_20090213.16
3a20ddaee1d69dcfd36f6bc2a2dd49f2 in-S0_20090213.17
815728ed36b9197cf1b635bc9cf2485c in-S0_20090213.18
2f9d6ae762e35c0a35de90f039c0ddfc in-S0_20090213.19
f452378ebc0b6457121c8dce4e634d0e in-S0_20090213.20
d3eae49109559a621ebd3ad164e17a40 in-S0_20090213.21
ac5ca2e0cf1bcca8978afb26c49842f2 in-S0_20090213.22
153379021b0a647c491a31fba67a2aa5 in-S0_20090213.23
d81feb1e2ae7b012215fb45e7453f591 in-S0_20090214.00
00d87b210892d3ef12471b04e6b15842 in-S0_20090214.01
711d72979f2b9d3c59be7aa5cec3db21 in-S0_20090214.02
6fafd111e1f791299f86577e4e37c832 in-S0_20090214.03
0d148a6c9ae270cdc18b1d80b1e8fec9 in-S0_20090214.04
96978a43e035dee2630ed92212a02828 in-S0_20090214.05
2e4e0793dad5b01ee7aa55e79395bf8e in-S0_20090214.06
ea4dced0c4cdae34e14fae1c2221c53a in-S0_20090214.07
53cb873286342b9a134e23a8dd9f9690 in-S0_20090214.08
020be76b39a229a15d931f370ecd3937 in-S0_20090214.09
1969dba0adda199f9f699a4f098fc39e in-S0_20090214.10
bbc4da5a09cc8a243d9255e6dcfa76c5 in-S0_20090214.11
efbfaaca5bd4fb7dbd20944d34dae59b in-S0_20090214.12
b108116a9ab86d69e6c69450cadf687f in-S0_20090214.13
381dea87ed0f508208e363e2856ee98b in-S0_20090214.14
24d261c5ea663eaf39e34c0da2e2767e in-S0_20090214.15
1b930c6d6a0cab7fe8779e5a772e5269 in-S0_20090214.16
acdbd1a798306a8019a47439c5ddfa99 in-S0_20090214.17
5b26e006aef13917a97fa7319a76ec6e in-S0_20090214.18
34c30ffabd644ead3ca71e76e7b1e3a5 in-S0_20090214.19
c6ed0b30da0b481ddfd0606e698e7101 in-S0_20090214.20
269ae41558a5c7d86c592ba90134a51f in-S0_20090214.21
9382c66c52c2bd13edf8ec089f8575ba in-S0_20090214.22
1378bfb5055d9ca7b8e5ddd5f9b2d9ef in-S0_20090214.23
29523d925540ae1d9c5b2468eff64c40 iw-S0_20090212.00
c21c90ea1622bafb3e295bac58c48716 iw-S0_20090212.01
0cca1eb272f2bcdca465618f423989fc iw-S0_20090212.02
2d26e78087a7e471b783a18f5d7733a9 iw-S0_20090212.03
bab1662b50238b29f4be11e454ad088e iw-S0_20090212.04
82ea4cd19d26a5bda14015be7a62423b iw-S0_20090212.05
6197a58d1163261ad82a5a74ccd21e7f iw-S0_20090212.06
a311715a8edd00132071a5a8f33cc5da iw-S0_20090212.07
2ccfd5324a8f922d5f7f9e22b5f425b7 iw-S0_20090212.08
e1428fc6e8acdcb33bf29c78d84563fb iw-S0_20090212.09
6b7efa0659ba410d39bb363342f196f6 iw-S0_20090212.10
dca8e01e1c3022e6b2c0a569733c70db iw-S0_20090212.11
2b2b496443a6a624cf460cfc1a2d4d78 iw-S0_20090212.12
ea7fdb01df80b04375b98aceeed619e5 iw-S0_20090212.13
82f3a64a7e55814237f0fa2aa05f90b8 iw-S0_20090212.14
cf64df13abde70b396239d0caafbea1c iw-S0_20090212.15
0fcbaeaa2d797401a52fbed3870b9de1 iw-S0_20090212.16
20a84a3a68ec251b7eea75c0abdbf2c5 iw-S0_20090212.17
16d62a48917ad4809158ade8a0598634 iw-S0_20090212.18
cd47022ad4368123fef23f23e3814433 iw-S0_20090212.19
b919dfa40e6346beb41e6ae9ecb44cd3 iw-S0_20090212.20
2fcdb32874d230450b5f505418543df6 iw-S0_20090212.21
1b44b76e5f54925facd3f8041ce42a62 iw-S0_20090212.22
a7d55b94b2d603a3e462b920f46b9347 iw-S0_20090212.23
fe3e4e2ed14fc62dfad7e3dddcd1d631 iw-S0_20090213.00
4460dddd6bb0fe70c7c772c0ecc2fc8e iw-S0_20090213.01
ac37f5e9fa314c5e5af4ef065f24128a iw-S0_20090213.02
7e7daaae01aa1564c631e069d2500e0d iw-S0_20090213.03
32dbc89c099c692489149fdcaf014e31 iw-S0_20090213.04
758bc43723462bdb84dfd8edcd815e32 iw-S0_20090213.05
99d452ea63f85762ef8e0489d1688561 iw-S0_20090213.06
768339c0db0853c96f082bdee13f219d iw-S0_20090213.07
14b5f49eec30e7f5226b1588fac3b3da iw-S0_20090213.08
a90e22f7eb9c40fd0f1102e88dcfd8c3 iw-S0_20090213.09
83daa0821597b49be22a4fc849d011ee iw-S0_20090213.10
de869c001adbfc6a2fd0097f358357a2 iw-S0_20090213.11
1ba5dce53638ee01d30c261ad786ad5e iw-S0_20090213.12
dc9b7ccfedf251029ee1053288fe4db2 iw-S0_20090213.13
2ae89d1e91f4ef28345e7f164e813dac iw-S0_20090213.14
4b0e611add386d1132675239bbbd7bce iw-S0_20090213.15
a274b162cc4d39389a279033925c8a5f iw-S0_20090213.16
5b107b812d3b650cdf9dd90ac5e39c6f iw-S0_20090213.17
31952931a179cd759427f9f59637f6a8 iw-S0_20090213.18
1ad0249e58eff981765714f4ba15746f iw-S0_20090213.19
007dea425cd85071919a69a6fd01c690 iw-S0_20090213.20
9389eee9f703cb258c120b4d84f78ab9 iw-S0_20090213.21
667505a4fb87b2aa04def139c328908d iw-S0_20090213.22
936eb95c3c16f34c68dae4cb950da5e7 iw-S0_20090213.23
0a511cebd4f50fc8b70f2acc792ff165 iw-S0_20090214.00
93bd95fab9bf77b0bd507578713d8f5e iw-S0_20090214.01
4a2f873370272d6d2a70de4bbc6ce751 iw-S0_20090214.02
1fbc37220f9c0d6bbe8e6e45c9004d41 iw-S0_20090214.03
af30b3b2ceba34b6c9d8423d4315c560 iw-S0_20090214.04
aeb6bb6163d8feb452b86d5b35373195 iw-S0_20090214.05
45718d5286609a7c650ac11a9810f98b iw-S0_20090214.06
5c0c1ad221a33cf48325a46e640b03cc iw-S0_20090214.07
507d35731208931184be30824a4132ca iw-S0_20090214.08
603e42b1033de3265856959518edefe9 iw-S0_20090214.09
9c072011eba65af7dfb391d5e6604347 iw-S0_20090214.10
eb1a5b5596b5f93ef2b9a25a1ffa75f3 iw-S0_20090214.11
d7aa980b48141bee21119a9cb263df6e iw-S0_20090214.12
74e72b79853d4b616dfcaf0ec6071dba iw-S0_20090214.13
7226eb58fb678eb6f1243d601860807a iw-S0_20090214.14
5dc14d4860e922effc504398c407eb62 iw-S0_20090214.15
8a36df5ecc0ec98a0c596b515180b3f3 iw-S0_20090214.16
1a1304a31d707a7b705a5896af051eaa iw-S0_20090214.17
2400dbefce9e8213b5ce6c2ede8ad05a iw-S0_20090214.18
19bd9670f509b3b2e67374c3dc816620 iw-S0_20090214.19
49a794b7776a6ec0bfbcace258eeb229 iw-S0_20090214.20
482758eea3bf19f0e291f866b2ef7a38 iw-S0_20090214.21
a03e859b12f835e95ffc1f2c89eade05 iw-S0_20090214.22
59d4a22ec958901077b59035aa883cfa iw-S0_20090214.23
9107871d14f5209547926f63665be456 other-S0_20090212.00
46a7ffdd77d2ebfc6a7c7a34b4279c6e other-S0_20090212.01
a20f2f23503699233261051337a785d2 other-S0_20090212.02
83e3de17a458f721e21aa9dd00c168a9 other-S0_20090212.03
6afa22f21bf48ff87cfbaed16c52af6f other-S0_20090212.04
4429a5d111cb00dd73a826211129773b other-S0_20090212.05
3ae4c70be9ade619a24e47b3876913aa other-S0_20090212.06
c812739b6200902de665a0301a4dbc4e other-S0_20090212.07
d674a9eae92910e42ec33f690aafb490 other-S0_20090212.08
37240c3ebb635bd2ae61d253617b8928 other-S0_20090212.09
bd62ae4a57f6320ed5bee56300b04d9b other-S0_20090212.10
61929cf8cebf24e528861729fe707ea2 other-S0_20090212.11
110cbd47ff625fd4b985ee4b16fbbb94 other-S0_20090212.12
08c3c63703fc187f4bdb0ebce1a38681 other-S0_20090212.13
9b13e380bf1532d15b4ce9622a4123f8 other-S0_20090212.14
f30eaad1b35f8422f44b5dbc2e9a9349 other-S0_20090212.15
463d5f1435d9eefb443acf00caf3f90f other-S0_20090212.16
55a767d2fb2f93ef98aedb1d41053b7c other-S0_20090212.17
8cf3203f6d49cd5dd8b61f6fff74a7bb other-S0_20090212.18
241419b8cdee045e093d5e4181e61a7b other-S0_20090212.19
0b02f7689dd19157d49fd44824a0c0a5 other-S0_20090212.20
a937f7e1e1f35d96e4d1fb263266b04d other-S0_20090212.21
fdf882256653d6dc623d43b87da00599 other-S0_20090212.22
447ab101ebe0d4c78db0053cdca00802 other-S0_20090212.23
70f839be687ec834fb5832c72df8e978 other-S0_20090213.00
8af5521d2c46d35050bb20b780e4fe95 other-S0_20090213.01
cc040a5a61c74e48fd93d1471d121b00 other-S0_20090213.02
27e1bc0d67ec22c8fe35a3059ff93bd5 other-S0_20090213.03
d2052bf1a5c760a78781c561021dfe59 other-S0_20090213.04
f48a11728ec3f7a1b6e3b45efa2419a6 other-S0_20090213.05
d392912f0f506b9e024762df57fe09a1 other-S0_20090213.06
f37f2f8e988d824911e6fb9589a61715 other-S0_20090213.07
cb076f5f40774c1fbccb5c233432fccd other-S0_20090213.08
d06dee275dc93c2d7f7fa503cfcf29cd other-S0_20090213.09
8528e1621c7fde9aec2f9f5f61c00358 other-S0_20090213.10
10b12b9a71264006fedd72c445df529a other-S0_20090213.11
5a395566bfd68986c394d569327a2edd other-S0_20090213.12
bdb8589b4bd3bbc044b7ec814387daeb other-S0_20090213.13
a70a9463e577a5cd99e5cd88d0170d9d other-S0_20090213.14
70f2337722856b0d1cb6accd9466ea9c other-S0_20090213.15
038f1ada5bfd96dac47fa00624697b77 other-S0_20090213.16
acf1d6de68e4c7c1e86a50fecfe63fd6 other-S0_20090213.17
0ceed3bf6a91d14a1f87414fc65ad690 other-S0_20090213.18
8eb152fd41ece1315142dc6385976e4a other-S0_20090213.19
247788879c3deb510ebaa547227d50bf other-S0_20090213.20
61874bfb2bb176f28a6618fb9bc8d6e1 other-S0_20090213.21
68cb69109895017012a44c146a132ecd other-S0_20090213.22
89dabb51349cccf54780bbd6142e4cb4 other-S0_20090213.23
a5387b55a4da0de9f3009757a5d54cbf other-S0_20090214.00
016b05f562f967fb36a6cfb3be22f7ef other-S0_20090214.01
3da1bf5191187cdfe2578e86e78ea82d other-S0_20090214.02
3b2ad5ede8b36000ee7aad15526c34c2 other-S0_20090214.03
8da55f6c1564e208322c6c3f390ea9ee other-S0_20090214.04
543f3de44ecc62b31e202ab6e73f6e94 other-S0_20090214.05
c899d3997d95b77a15d02dd65e63c8db other-S0_20090214.06
9d19b866d2e2f37b9c0bf98aa0350769 other-S0_20090214.07
a14610cd2c356cf162f009257f3a1eb7 other-S0_20090214.08
6d2be11604b4a64901476b9c4b9718e1 other-S0_20090214.09
b9c1231c9b0d4ae5e0aa33d753de0fda other-S0_20090214.10
43ee43efe30e6afc3041ce583f7a9b21 other-S0_20090214.11
3b01e37590efe0fa829c95b015a9125c other-S0_20090214.12
02bb8fece9363ecaa4ada32911726f6b other-S0_20090214.13
ed91f0e2000e505e0ecf563e5b9343f0 other-S0_20090214.14
d03b0fc73c0d8ce492d7b4233e051bcd other-S0_20090214.15
a79e5f4d55440b86daea6a99701ad8ef other-S0_20090214.16
a156d268215bd4cd0bf6208a13449656 other-S0_20090214.17
13277ea1f971731b94879b16e8d963a3 other-S0_20090214.18
5446849abbe5d91f075edc5a212c9da4 other-S0_20090214.19
7c8c87c3c5db6479d829557994f0b195 other-S0_20090214.20
9be5d854309f828ed0ff20f2b1db2beb other-S0_20090214.21
ac92f5824e7c784b15420397af3fb62d other-S0_20090214.22
32f1d39fd0e8ef2968348441de0ee5d4 other-S0_20090214.23
60f7597a3e55b9f745197b0060e478f2 out-S0_20090212.00
5692d0ee737e058df3bb7cb5f6ff73a6 out-S0_20090212.01
d8d62287262b84dc0b076aedc6e9f203 out-S0_20090212.02
ddb33e0418eec4f7d2f9fd657cb28a83 out-S0_20090212.03
fa910f28bb8d338ffaac655b3629022b out-S0_20090212.04
aa6b09bfb00f1ae48f32561d6c9ae092 out-S0_20090212.05
4e7778df1f79ce1161ef3b466dbe6262 out-S0_20090212.06
8994884e1d654537c0bc5a0976584f77 out-S0_20090212.07
fab8ec5cc4954c322503c90e83768944 out-S0_20090212.08
049cc66dcaeac78bb856dcc4ba515e0b out-S0_20090212.09
ca3a8651a59e0e006113acdec9e76555 out-S0_20090212.10
d42130431ab2457b01dd23189b8c8b76 out-S0_20090212.11
8593dcb41972ed714c475ae2655c0e0f out-S0_20090212.12
ee3d92a3b19d17e679541712246eb5c7 out-S0_20090212.13
48dac75b887298f729a48b5629b9e34a out-S0_20090212.14
a016df031baa41bfdfd784ed7faffe31 out-S0_20090212.15
4fc4717fd7d02f8d73930be2c715f9c7 out-S0_20090212.16
a8a9fb1c48f10b4e3bd8c809e5992600 out-S0_20090212.17
d4f46c3f3494b221fcdebf41807d00c5 out-S0_20090212.18
fb37e66f556ecaec30f25a5ae14c29ae out-S0_20090212.19
ded5c1e627c34c4cf5664e058ec06fbc out-S0_20090212.20
1527b22c96ecff44ff0e1962b2e2dd88 out-S0_20090212.21
53897aa40779d516f3f601f490b259fe out-S0_20090212.22
a06763af353140766fa103ab7fbb633d out-S0_20090212.23
8d634c813cf3f4bcf925096ac99523ab out-S0_20090213.00
a31148c96d6155ae08ce6f85d65ec6c4 out-S0_20090213.01
462f8fa8fcd3af3942313994863c552e out-S0_20090213.02
87d85a3543b08f08f5a79a297839f576 out-S0_20090213.03
aa76f57f709c7978543140919958c4ed out-S0_20090213.04
c1a3ee7cc5edb70aa13d66ffe8059536 out-S0_20090213.05
4efb3e3b7729cb9cb73ee28201e59266 out-S0_20090213.06
52c356c0c23e05c89ebbe52b124440ed out-S0_20090213.07
73a03a9b8fcd8c9671fab2623e0c0a13 out-S0_20090213.08
2a8beef13989a523b828d41f8d79a370 out-S0_20090213.09
d0e0aea0fb39b13624d9c0335122e811 out-S0_20090213.10
7c03f5b95ba31a7276c4da636eb577df out-S0_20090213.11
6b0af2a0993e2f9da83f9e54c7d74f93 out-S0_20090213.12
6c91b185eed48cda75d39e1dfb8a618c out-S0_20090213.13
df050bf942d6df96166b5adc388ce617 out-S0_20090213.14
4d26af88f90a76437474675ec726b629 out-S0_20090213.15
70fc19207aa989df446e7bb17120bd01 out-S0_20090213.16
a51363a61f256a27b5235376939aed69 out-S0_20090213.17
5defceac7915d5ab52b0504b9ef4767f out-S0_20090213.18
a694d4d68388f93ae5ed955e31037129 out-S0_20090213.19
3b24cb6319abb96e53b973f262b6a0dc out-S0_20090213.20
5b7dacd4b3a1c9fe4047404f164ea789 out-S0_20090213.21
b675323fd5942058286424cbbdb03132 out-S0_20090213.22
2f476e0d851c8e362191d13feb79e2f4 out-S0_20090213.23
990808b779e502930ea85345675398d6 out-S0_20090214.00
220a0ce382de5769570c525cd3d4e041 out-S0_20090214.01
e29860550bfc83e7fb0185be97c01d61 out-S0_20090214.02
9ed33bd1a86604efad5a14feb7f5d2a7 out-S0_20090214.03
464cacbe113a9e2b48dfa6b864218614 out-S0_20090214.04
8365e8c3521f953d4ab87bcdc0eebd27 out-S0_20090214.05
1608f290a389e63b13f509fc1a70c84a out-S0_20090214.06
9ab456724fe46879623bf3546a48919e out-S0_20090214.07
0df29921c8176bb4da0f4cc49cb3de82 out-S0_20090214.08
1bb0ed1ba490ba372fe46dedeb2ce6aa out-S0_20090214.09
a28cca8f5063c45e646316d919e22930 out-S0_20090214.10
0ed973e358557a7ce68214005d027f78 out-S0_20090214.11
db70ce7a9599a5b294c9c4d3e062e921 out-S0_20090214.12
f12c375513e77a06fdf51c6efe5e508d out-S0_20090214.13
f8744f00cd6cd59cbff159496b5f45ae out-S0_20090214.14
7b3679787f9734a7aaa64139d64730ac out-S0_20090214.15
a6d80008e466800d1e51668352ce4e61 out-S0_20090214.16
0e90a0e5dbf3447eb19563dcabaf2213 out-S0_20090214.17
fab9d161fc315d2f4e08a6fc5e318432 out-S0_20090214.18
782bbc2240ec7bbeebe6db6536218f7f out-S0_20090214.19
03dd1a5f7eedebcbefb5314659d425c2 out-S0_20090214.20
66f3e3d16a8c9d3b7f6c03275d87fe31 out-S0_20090214.21
f8965cecd3b22556053a93633a6d1f19 out-S0_20090214.22
e6b88efc3252e3c4f30a90364d92d485 out-S0_20090214.23
119e56a1bca9a9d007df33a7c12f4cdc outnull-S0_20090212.00
6de662e52e3755db6b548af843544e40 outnull-S0_20090212.01
28865b0138dbdf1f30992b2558c345ca outnull-S0_20090212.02
4b03543f993f5a657284651063a7e729 outnull-S0_20090212.03
8bc198146a3003d547b2e54563633088 outnull-S0_20090212.04
1bbb3aa45ee1e948c617e2cc15a219cb outnull-S0_20090212.05
d5b1f51961d9bcd40148d955287b8062 outnull-S0_20090212.06
703f487c69f0d574224b0e904cd3b2f3 outnull-S0_20090212.07
fd4583c098c48fa843ec09a0ad58b7fd outnull-S0_20090212.08
807981c2a3cce80aace8da473c1f881c outnull-S0_20090212.09
c7a495fd4ba1a28ff4e360dc29b4246f outnull-S0_20090212.10
943a29b35da5c8b3c48b7c76be61ae69 outnull-S0_20090212.11
c41000ffae3fe9d2a1321da7e6641f8d outnull-S0_20090212.12
fd8c5b6dd6021af68c32f1b6f02e1d49 outnull-S0_20090212.13
5180940257471a1103b18466440480a2 outnull-S0_20090212.14
c6d923d7c67a207c66443485dd0b880f outnull-S0_20090212.15
c375ff5f191dbd8ef2a6e219be94586e outnull-S0_20090212.16
6cac8d61990f14020006d6a67cb7d3e6 outnull-S0_20090212.17
15c755680791c919521fa0f683b43ce8 outnull-S0_20090212.18
9e171a3ec5ee0244a89a132a573e98ee outnull-S0_20090212.19
58152c8caab4ceec788aa2eec453a151 outnull-S0_20090212.20
a90437223bcbbe8efaa8f9a2bf7eb1c4 outnull-S0_20090212.21
af19794ff87a4e2e86a47f516dae4403 outnull-S0_20090212.22
1940180828eb067b4fde088be20c4e9c outnull-S0_20090212.23
5d7a92947709ac6dfc8bd6e1895e510f outnull-S0_20090213.00
e1c2193f3accbd65388953e78c9e5c7c outnull-S0_20090213.01
7c6a1efe27e43d36a5fc66de21d5d61a outnull-S0_20090213.02
ec8e39596b55388114915e56b3168124 outnull-S0_20090213.03
bba89a055676880485f273c5ac579e59 outnull-S0_20090213.04
4a03de78ef878de719ca55303e8fb845 outnull-S0_20090213.05
43f746ef86c1be0aebaa46d7719fc28c outnull-S0_20090213.06
4fe2e0b31719222b63529e3d354d37dc outnull-S0_20090213.07
339ddec716999fd7c5c6aea6d8e2b9ac outnull-S0_20090213.08
a0d398dd8235c29e5a24ff6ece196e04 outnull-S0_20090213.09
e2323724d444235df156aba57783e48c outnull-S0_20090213.10
a7791eab546b599cc1753bd0b11e3322 outnull-S0_20090213.11
34ddecfb275f7763c8c42755570c73dc outnull-S0_20090213.12
f4723100271cad23961a1ee2894236ac outnull-S0_20090213.13
ac162f5bb530057b4d990083dcabb03f outnull-S0_20090213.14
92c336fef04b00f2f6479b0eca550716 outnull-S0_20090213.15
91e3d9bf4933ae2a0e23e6449d773f2f outnull-S0_20090213.16
ab361c13f55c3a93c5df42c539beb1d5 outnull-S0_20090213.17
0d9ce7e64a8d2ee2df940e482f90c8c0 outnull-S0_20090213.18
9b21354e75c0d2ac50c2e333ca28ac3f outnull-S0_20090213.19
37ae1e18656768561868c4e34ebc6ddf outnull-S0_20090213.20
1877475f11b3d69c0042642baee6c4ef outnull-S0_20090213.21
143ae11c25ba41370521fc37fc87cbbf outnull-S0_20090213.22
053c7cac6409286756b4415ef986958c outnull-S0_20090213.23
e92a0dad9291901bfbf4ab2ab61f8c4d outnull-S0_20090214.00
ac6298d93b27514efe4e17b8d5d238f3 outnull-S0_20090214.01
2e9e6cd6b575845f6093c335c84ebcd0 outnull-S0_20090214.02
53e217670a750370619b9944cd5feb12 outnull-S0_20090214.03
667a5fdd263e66cd653fad11e2717ad5 outnull-S0_20090214.04
f17edba03b52a56d6779402999d08536 outnull-S0_20090214.05
08dcb1566fa9f4e052a3553026c3f519 outnull-S0_20090214.06
85dac7e7dbee30ff2ed85bbebee9a858 outnull-S0_20090214.07
6224b4adca59a3eb5e89d3de8b32b737 outnull-S0_20090214.08
e645d3ca838f61239be4dd71c7a56282 outnull-S0_20090214.09
9ebfc499ff52e95ce211496ed4fc1790 outnull-S0_20090214.10
7836332f0305d7882cb82b57cc1f3713 outnull-S0_20090214.11
71f8fccb36a428308a43320a407c5682 outnull-S0_20090214.12
0747455c2207cc345d97dd8a72dd9928 outnull-S0_20090214.13
0be9b9e10e084b3fda62aaeca1c197ba outnull-S0_20090214.14
320d7b5cc0ce8338d974809a8fe9b734 outnull-S0_20090214.15
791a2f186aa7de8f382575d9121f7d99 outnull-S0_20090214.16
303aa40138836abc07ab2f3253eacb14 outnull-S0_20090214.17
4aef6f3f24724297f74ff8eed4e819ca outnull-S0_20090214.18
aed3f1e70010a356cdf1e6814dde4b25 outnull-S0_20090214.19
d1071ee3fa3fa2876ad1f398c998519f outnull-S0_20090214.20
e02971edb328db6bb30121c08b6d72fe outnull-S0_20090214.21
8784867c9583e6e5331d032b8195d57d outnull-S0_20090214.22
122291154d7cae8590bbc47c9e9d0c01 outnull-S0_20090214.23
516c52caae1571961ff74703557d42f2 ow-S0_20090212.00
060a95a5545bee6deb3bea6cfff531b7 ow-S0_20090212.01
49b0eaf820ea0e1332e65029c4a6b4da ow-S0_20090212.02
e6c8784d1e25b4492ff275fee0047e7e ow-S0_20090212.03
253dfd848776b30004789ef7705ef78f ow-S0_20090212.04
cb59b1bf8c88456116e3604c61e6db57 ow-S0_20090212.05
881d2710cb61ed6d7892106231f636d3 ow-S0_20090212.06
b2d6468548a41aac10d3e220706c8f0a ow-S0_20090212.07
d3348e199fe7940c05075999b0694229 ow-S0_20090212.08
ff853dd30fe410142f44d9ea06c160db ow-S0_20090212.09
90f795c2dbaeb6276dc253bad60dd997 ow-S0_20090212.10
939e858bd86e694ef11d6f9b3a36262b ow-S0_20090212.11
a3b4726dd623396be2550f893696bd2c ow-S0_20090212.12
e5965b183c3b478e89b4f2e7f0ce3221 ow-S0_20090212.13
82c332cd5205ffd3ab4d8238ad6b5d99 ow-S0_20090212.14
32dc81d4c80700419ccb718ac35df91c ow-S0_20090212.15
5bb36a40f177cb3768e7896bedd4d1f7 ow-S0_20090212.16
37258ffe2675fb9a0a8a892a604523d6 ow-S0_20090212.17
3f2e4707023859028c8ef1fea8e264b8 ow-S0_20090212.18
3484aa7a6fdc906bce1748523fd62ca7 ow-S0_20090212.19
dde85ffec754972709566e4086b18a16 ow-S0_20090212.20
572987c4d448617d482a444890a0c460 ow-S0_20090212.21
67b119524631773f87fb3a2fb16c43e1 ow-S0_20090212.22
27bf04040645cdd5dbd35fdc5ba158e7 ow-S0_20090212.23
cf2491e6a077550264bb260e9f72a2f0 ow-S0_20090213.00
7d3efdab0c732e29389964f34997273c ow-S0_20090213.01
7aaa089a2badba5def1ac551dd6072d3 ow-S0_20090213.02
05c2b5d4cdfc042e5f31abb5574f9eb9 ow-S0_20090213.03
0b02925ef647817278e346d0f019d4ef ow-S0_20090213.04
19faad768c844281516c5dfb0d50649a ow-S0_20090213.05
b1d0c0aedb09252b6d99e99b203f363a ow-S0_20090213.06
c31fb8e230c3ee54ceb848f491962d2a ow-S0_20090213.07
73233246967fea92e63cd8f574908c8c ow-S0_20090213.08
0512e5d804b79f9e1da81d90ef0dc9bf ow-S0_20090213.09
bd906a05688778419ee24f492858d821 ow-S0_20090213.10
aa4714f30832b75c6805624e763cb23c ow-S0_20090213.11
f16fc870825ca358c9be2b875413df98 ow-S0_20090213.12
531d99221a7857af16e8a914bd3b36d7 ow-S0_20090213.13
c1bf2c4a5831a4ac8f0e8252b22b6ddd ow-S0_20090213.14
7b0c740a74315f3c7e398342511c1868 ow-S0_20090213.15
42a2bfb1fed335624756121130fd46b4 ow-S0_20090213.16
c8d4b37620da44f6d3795dff865f064f ow-S0_20090213.17
cd4ad1fbc8bc07c76e7b7de5131005db ow-S0_20090213.18
73e0cdb3f84792da9b50ee98ecf51035 ow-S0_20090213.19
c52e8731d038abdd06384a31eb6af67b ow-S0_20090213.20
d932f97905f51cfab9826401ba421992 ow-S0_20090213.21
bb5894ad73c94fb9bdf71cfaec111933 ow-S0_20090213.22
daa4c4752b9aa03d790aace405cfb52a ow-S0_20090213.23
bc1e5771ad462e19164872fd479f5de2 ow-S0_20090214.00
fe79f48a6eca2a973c65c146732d4729 ow-S0_20090214.01
91d477a917e641b68d072afcf354399e ow-S0_20090214.02
9bf77d64b85fdc825e975613416fe024 ow-S0_20090214.03
6b5677775de84e581a5dac47ec663ced ow-S0_20090214.04
329885bed2738080379d0baa5d879c5d ow-S0_20090214.05
6080e3d847e75ca13161bf2c2dc39098 ow-S0_20090214.06
1529bdb495a731daf3f9b3b129a00086 ow-S0_20090214.07
f89462972219648bc345eab949bfdf23 ow-S0_20090214.08
a2ac3c9bf000dc66825d6645b1f12be7 ow-S0_20090214.09
5d764cd38c7b2de9383402993e26959e ow-S0_20090214.10
35315d36b5fb60e9c85b0c1fa9ea0c27 ow-S0_20090214.11
2d08fcad6455ce7529af60213b313032 ow-S0_20090214.12
2a21f16a90675914f5aced6ce9fca9ea ow-S0_20090214.13
c85624dd3de76325f63b2d2d40016958 ow-S0_20090214.14
91d4dfab3b6f1f8508aae7c6ebe7d151 ow-S0_20090214.15
f0b8ebe9ea28a6a579b2f40072e592dc ow-S0_20090214.16
c3c76225bc371115fe7ac044fdbc7b66 ow-S0_20090214.17
2634926eb65cee405d4b60e0aae4f7ff ow-S0_20090214.18
2858e136fc699fee1c2b225c3392e594 ow-S0_20090214.19
289f8b0234cbd07bd2817dd39ade9f91 ow-S0_20090214.20
9b13494e2b99624006aa1ad7b22573d4 ow-S0_20090214.21
9838950e543453bc99241bb57ab6325a ow-S0_20090214.22
c4de9cf5055789c0d40cd5e0133fb8b1 ow-S0_20090214.23
