#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: flowcap-ipfix-v6.pl 323d2cb74c99 2024-05-30 18:17:09Z mthomas $")

use strict;
use SiLKTests;

my $NAME = $0;
$NAME =~ s,.*/,,;

# set envvar to run app under valgrind when SK_TESTS_VALGRIND is set
check_silk_app('flowcap');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsort = check_silk_app('rwsort');
my $rwsilk2ipfix = check_silk_app('rwsilk2ipfix');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');

# verify that required features are available
check_features(qw(ipfix inet6));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# create our tempdir
my $tmpdir = make_tempdir();

# send data to this port and host
my $host = '::1';
my $port = get_ephemeral_port($host, 'tcp');

# verify IPv6 capability using Python
my $verify_ipv6 = "$tmpdir/verify-ipv6.py";
open VERIFY_IPV6, ">$verify_ipv6"
    or die "$NAME: ERROR: Cannot open $verify_ipv6: $!\n";
print VERIFY_IPV6 <<EOF;
import socket
socket.getaddrinfo("$host", $port, 0, 0, 0, socket.AI_ADDRCONFIG);
EOF
close VERIFY_IPV6
    or die "$NAME: ERROR: Cannot close $verify_ipv6: $!\n";
system $SiLKTests::PYTHON, $verify_ipv6
    and skip_test("IPv6 addressing not available");

# create the sensor.conf
my $sensor_conf = "$tmpdir/sensor.conf";
my $sensor_conf_text = <<EOF;
probe P0 ipfix
    protocol tcp
    listen-on-port $port
    listen-as-host $host
    accept-from-host $host
end probe
EOF
make_config_file($sensor_conf, \$sensor_conf_text);

# Generate the test data
my $ipfixdata = "$tmpdir/data.ipfix";
unlink $ipfixdata;
system "$rwsilk2ipfix --ipfix-output=$ipfixdata $file{data}"
    and die "$NAME: ERROR: Failed running rwsilk2ipfix\n";

# the command that wraps flowcap
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/flowcap-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--tcp $ipfixdata,$host,$port",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--",
                     "--sensor-conf=$sensor_conf",
                     "--max-file-size=100k",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);

# path to the directory holding the output files
my $data_dir = "$tmpdir/destination";
die "$NAME: ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# check for zero length files in the directory
opendir D, "$data_dir"
    or die "$NAME: ERROR: Unable to open directory $data_dir: $!\n";
for my $f (readdir D) {
    next if (-d "$data_dir/$f") || (0 < -s _);
    warn "$NAME: WARNING: Zero length files in $data_dir\n";
    last;
}
closedir D;

# create a command to sort all files in the directory and output them
# in a standard form.
$cmd = ("find $data_dir -type f -print "
        ." | $rwcat --xargs "
        ." | $rwsort --fields=stime,sip "
        ." | $rwcat --byte-order=little --compression-method=none"
        ." --ipv4-output");

# because the IPv4 flowcap file format encodes bytes as a
# bytes-per-packet and the IPv6 does not, the results will be slightly
# different depending on whether SiLK was compiled with IPv6 support
my $md5 = (($SiLKTests::SK_ENABLE_IPV6)
           ? 'e9560963078c92393095c50afbe12739'
           : 'efa8123f93c841bda61b4e8c8fec5740');

exit check_md5_output($md5, $cmd);
