#!/bin/sh -e

usage() {
  cat <<__HERE__
$0 OUTPUT_FILE [ --program-name NAME ] INPUT_FILE ...

This script is only expected to be used via the GEN_STARTUP_FILE
definition in ../../build.mk, which defines needed environemtn variables
for the build process, and provides the output file. Makefile rules
should look like this:

rwsender_startup: Makefile \$(RWSENDER_SYSTEMD_SNIPPETS)
	\$(GEN_STARTUP_FILE) \$(RWSENDER_SYSTEMD_SNIPPETS)

With any '--program-name' option immediately following \$(GEN_STARTUP_FILE).

Given an output filename and a set of input snippets, composes an output
file by concatenating those snippets and performing @var@ substitions on
the result.

    OUTPUT_FILE
        The path to the output file to produce. This is used to derive the
        program name if '--program-name' is not provided by removing
        everything after the *first* dot '.', underscore '_', dash '-',
        or at sign '@' in the filename.  For example, "x/y/z/test-conf.txt"
        would produce "test" as the program name by default. "foo@.service"
        would use "foo".

    --program-name PROGRAM_NAME
        Optionally use 'PROGRAM_NAME' as the program name instead of
        deriving it from the output filename. If this is included, it
        must occur as the second argument.

    INPUT_FILE ...
        The path to input files to be included in the output.

ENVIRONMENT

    This program requires the following variables to be set in the
    environment for substitution purposes. Each is substituted for an
    @variable-name@ substitution of the same name.

        SILK_DATA_ROOTDIR
        libdir
        libexecdir
        localstatedir
        prefix
        sbindir
        sysconfdir

    It will also use srcdir, if it is set, as an alternative source for
    input files.

EXAMPLES

    $0 test_startup ../snippet-part_1.sh snippet-middle_bit.sh ../snippet-part_2.sh

        Will produce a file named "test_startup" by concatenating the
        given snippets and performing variable substitution. The
        program_name variable will be substituted with "test".

    $0 config-for-thing.conf --program-name thing snippet-part_a.conf snippet-part_b.conf

        Will produce a file named "config-for-thing.conf" from the given
        snippets using "thing" as the program name.

__HERE__
}

if [ $# -lt 2 ] || [ "$2" = "--program-name" ] && [ $# -lt 4 ]; then
    usage
    exit 1
elif [ "$1" = "-h" ] || [ "$1" = "--help" ]; then
    usage
    exit 0
fi

# Required. The name of the output file.
OUTPUT_FILE="$1"
shift

# Optionally, --program-name X
if [ "$1" = "--program-name" ]; then
    program_name="$2"
    shift 2
else
    program_name="$( echo "$OUTPUT_FILE" | ${SED:-sed} -e 's|^.*/||' -e 's/[-._@].*$//' )"
fi

# Removing any existing output files
rm -f "$OUTPUT_FILE" "$OUTPUT_FILE.tmp"

# Transform the program name if needed.
if [ -n "$transform" ]; then
    program_name="$( echo "$program_name" | ${SED:-sed} -e "$transform" )"
fi

# For each input file, see if it's in the working or separate srcdir,
# then cat it. All of this output goes through sed, and if there are any
# missing substitution variables (which should be provided in the
# environment, it will error out.
for f in "$@"; do
    if [ -f "$f" ]; then
        input_file="$f"
    elif [ -n "$srcdir" ] && [ -f "$srcdir/$f" ]; then
        input_file="$srcdir/$f"
    else
        echo "Missing input snippet: '$f'" 1>&2
        exit 1
    fi
    cat "$input_file"
done \
| ${SED:-sed} \
       -e "s|@SILK_DATA_ROOTDIR[@]|${SILK_DATA_ROOTDIR:?}|g" \
       -e "s|@libdir[@]|${libdir:?}|g" \
       -e "s|@libexecdir[@]|${libexecdir:?}|g" \
       -e "s|@localstatedir[@]|${localstatedir:?}|g" \
       -e "s|@prefix[@]|${prefix:?}|g" \
       -e "s|@sbindir[@]|${sbindir:?}|g" \
       -e "s|@sysconfdir[@]|${sysconfdir:?}|g" \
       -e "s|@program_name[@]|$program_name|g" \
> "$OUTPUT_FILE"

# Make sure to document new substitution variables added to the above
# sed invocation in the usage, and to add them to ../../build.mk's
# GEN_STARTUP_FILE definition.
