
### Below this from SiLK's src/startup/snippet-common_prelude.sh

# PROGRAM_NAME and SCRIPT_CONFIG_FILENAME should be set before this point.

# PROGRAM_NAME is the name of the executable program being managed by
# this script.

# SCRIPT_CONFIG_FILENAME is the name of the configuration file from
# which to read environment variables. Typically some variation on
# "${PROGRAM_NAME}.conf".

# SCRIPT_CONFIG_LOCATION is the directory where the
# ${SCRIPT_CONFIG_FILENAME} config file is located. It can be set via an
# environment variable. If the envar is not set, then @sysconfdir@ is
# used.

# If $SCRIPT_CONFIG_LOCATION/sysconfig/$SCRIPT_CONFIG_FILENAME exists, a
# warning will be emitted (since the default location of these files has
# changed on some systems.)

SCRIPT_CONFIG="${SCRIPT_CONFIG_LOCATION:=@sysconfdir@}/${SCRIPT_CONFIG_FILENAME}"
DEPRECATED_SCRIPT_CONFIG="${SCRIPT_CONFIG_LOCATION}/sysconfig/${SCRIPT_CONFIG_FILENAME}"

if [ -f "$DEPRECATED_SCRIPT_CONFIG" ]; then
    echo "$0: *** WARNING: Config files should be placed directly in '${SCRIPT_CONFIG_LOCATION}', but '${DEPRECATED_SCRIPT_CONFIG}' exists." 1>&2
    echo "$0: *** WARNING: Move any existing configuration from '${DEPRECATED_SCRIPT_CONFIG}' to '${SCRIPT_CONFIG}'." 1>&2
fi

if [ ! -f "${SCRIPT_CONFIG}" ] ; then
    echo "$0: '${SCRIPT_CONFIG}' does not exist, exiting."
    exit 0
fi

. "${SCRIPT_CONFIG}"

if [ "x$ENABLED" = "x" ] ; then
    echo "$0: Not enabled in '${SCRIPT_CONFIG}', exiting." 1>&2
    exit 0
fi

#######################################################################
# SHELL FUNCTIONS

# check_empty VARNAME VALUE
#
#    Verifies that VALUE has a value.  If it doesn't, a message is
#    printed that the VARNAME variable is unset and script exits.
check_empty()
{
    if [ "x$2" = "x" ] ; then
        echo "$0: the \${$1} variable has not been set."
        exit 1
    fi
}

# check_dir VARNAME DIR
#
#    Verifies that VARNAME is set.  Next, verifies that the directory
#    DIR exists.  If not and if $CREATE_DIRECTORIES is set, the
#    directory is created.  Otherwise, an error is printed and the
#    script exits.
check_dir()
{
    check_empty "$1" "$2"
    if [ ! -d "$2" ] ; then
        if [ "${CREATE_DIRECTORIES}" = "yes" ] ; then
            mkdir -p "$2" || { echo "$0: Could not create $2" ; exit 1 ; }
            chown -h "${USER}" "$2" || { echo "$0: Could not chown $2 to ${USER}"; exit 1 ; }
        else
            echo "$0: the $2 directory does not exist."
            exit 1
        fi
    else
        chown -h "${USER}" "$2" || { echo "$0: Could not chown $2 to ${USER}"; exit 1 ; }
    fi
}

# Variables for building Basic Regular Expressions
TAB="	"                       # Contains a literal tab
WHITE="[ ${TAB}]"
WHITEPLUS="${WHITE}${WHITE}*"
NONWHITE="[^ ${TAB}]"
NONWHITEPLUS="${NONWHITE}${NONWHITE}*"
STRINGWHITE="\\(${NONWHITEPLUS}\\)${WHITEPLUS}"

# ws_strip STRING
#
#   Strips leading and trailing whitespace from STRING and returns it
ws_strip()
{
    echo $1 | sed -e "s/^${WHITE}*\\(${NONWHITEPLUS}\\(${WHITE}*${NONWHITEPLUS}\\)*\\)${WHITE}*\$/\\1/"
}

# cut_1 STRING
#
#    Returns first "token" from STRING or the empty string for an
#    empty STRING
cut_1()
{
    expr "x$1" : "x\\(${NONWHITE}*\\)"
}

# cut_2 STRING
#
#    Returns second "token" from STRING
cut_2()
{
    expr "x$1" : "x${NONWHITEPLUS}${WHITEPLUS}\\(${NONWHITEPLUS}\\)"
}

# cut_2rest STRING
#
#    Returns from second "token" to end of STRING
cut_2rest()
{
    expr "x$1" : "x${NONWHITEPLUS}${WHITEPLUS}\\(.*\\)"
}

# cut_3rest STRING
#
#    Returns from third "token" to end of STRING
cut_3rest()
{
    expr "x$1" : "x${NONWHITEPLUS}${WHITEPLUS}${NONWHITEPLUS}${WHITEPLUS}\\(.*\\)"
}

#######################################################################

RETVAL=0

PROGRAM_PATH="${BIN_DIR}/${PROGRAM_NAME}"

PROG_OPTIONS=""

if [ ! -x "${PROGRAM_PATH}" ] ; then
    echo "$0: could not find an executable ${PROGRAM_PATH}."
    exit 1
fi

