#! /usr/bin/perl

##  Copyright 2006-2022 Carnegie Mellon University
##  See license information in LICENSE.txt.

##  This script modifies the HTML man pages generated by pod2html.  It
##  adds a header and footer, and creates links to other man pages.
##  ------------------------------------------------------------------------
##  @DISTRIBUTION_STATEMENT_BEGIN@
##  YAF 3.0.0
##
##  Copyright 2022 Carnegie Mellon University.
##
##  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
##  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
##  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
##  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
##  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
##  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
##  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
##  INFRINGEMENT.
##
##  Released under a GNU GPL 2.0-style license, please see license.txt or
##  contact permission@sei.cmu.edu for full terms.
##
##  [DISTRIBUTION STATEMENT A] This material has been approved for public
##  release and unlimited distribution.  Please see Copyright notice for
##  non-US Government use and distribution.
##
##  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
##  Trademark Office by Carnegie Mellon University.
##
##  This Software includes and/or makes use of Third-Party Software subject
##  to its own license.
##
##  DM22-0007
##  @DISTRIBUTION_STATEMENT_END@
##  ------------------------------------------------------------------------

use warnings;
use strict;
use File::Copy  qw//;
use File::Temp  qw//;
use FindBin qw($Bin);

# name of this script
my $NAME = $0;
$NAME =~ s,.*/,,;

# name of the project
my $PROJECT = 'YAF';

# Man pages to link to: hard-coding this for now, should generate from
# build the way SiLK does
our $man_re = "(".join("|", map { "\Q$_\E" }
                       qw(
                           airdaemon
                           applabel
                           filedaemon
                           getFlowKeyHash
                           yaf
                           yaf.init
                           yafMeta2Pcap
                           yafdhcp
                           yafdpi
                           yafscii
                           yafzcbalance
                       )).")";

# Get the location of the body-open and body-close files which should
# be next to this script.
my $body_open  = "body-open.html";
my $body_close = "body-close.html";

for my $b ($body_open, $body_close) {
    unless (-f "$Bin/$b") {
        die "$NAME: Did not find file '$b' in '$Bin'\n";
    }
    $b = "$Bin/$b";
}


our $old = $ARGV[0];
unless ($old) {
    die "Usage: $NAME <filename>\n\tAdd header and footer to named file\n";
}

open(OLD, '<', $old)
    or die "$NAME: Cannot open '$old' for reading: $!\n";

our ($fh, $new) = File::Temp::tempfile(UNLINK => 1)
    or die "$NAME: Unable to create a temporary file: $!\n";
*NEW = $fh;

do_manpage();

close OLD;
close NEW
    or die "$NAME: Cannot close '$new': $!\n";

File::Copy::move $new, $old
    or die "$NAME: Cannot replace '$old' with '$new': $!\n";

exit;


sub add_header
{
    my ($title) = @_;

    if ($title)
    {
        $title = "Documentation - $title";
    }
    else
    {
        $title = "Documentation";
    }

    my $text = "";
    $text .= <<EOF;
<head>
<title>$PROJECT - $title</title>
<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1" />
<link rel="stylesheet" type="text/css" href="../site/style.css" />
</head>
<body>
EOF

    open BODY_OPEN, "<", $body_open
        or die "$NAME: Cannot open '$body_open': $!";
    while (<BODY_OPEN>) {
        $text .= $_;
    }
    close BODY_OPEN;

    return $text;
}


sub add_footer
{
    my $text = "";

    open BODY_CLOSE, "<", $body_open
        or die "$NAME: Cannot open '$body_open': $!";
    while (<BODY_CLOSE>) {
        $text .= $_;
    }
    close BODY_CLOSE;

    $text .= "</body>\n";
    return $text;
}


sub do_manpage
{
    # We'll overwrite this with name of man page
    my $title = '';

    # ignore everything until we get to the end of the index, but do
    # cash the title
    my $saw_index = 0;

    # whether we saw the <head>
    my $saw_head = 0;

    $/ = "";  # read one paragraph at a time
    while (<OLD>) {
        # # Downgrade all <Hn> tags by one
        # s{(</?h)(\d)\b([^>]*>)}{$1.($2 +1 ).$3}gieo;

        # Get rid of any mailto: links
        s{<link[^>]+mailto:[^>]+>}{}i;

        # Remove all <hr>
        s{<hr\b[^>]*>}{}iog;

        # # Change <a name> to <a id>
        # s{<a name="}{<a id="}iog;

        # Make links to other man pages
        s{(<b>$man_re\(\d\)</b>)}{<a href="$2.html">$1</a>}og;

        # Stash the title
        if (m{<title>(.+)</title>}io) {
            $title = $1;
        }

        # Don't print anything until we see the Perl-generated index
        if ( !$saw_index) {
            #old perldoc
            if (m{(<!-- INDEX END -->)}) {
                $saw_index = 1;
                print NEW add_header($title);
                next;
            }
            next unless m{(<h1 id="NAME">NAME</h1>)};

            $saw_index = 1;
            print NEW add_header($title);
            print NEW "<h1><a name=\"name\">NAME</a></h1>";
            next;
        }
        s{</body\b[^>]*>}{add_footer()}eio;

        print NEW;
    }
}
# do_manpage
